/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef GAME_H
#define GAME_H

#include <string.h>
#include <stdbool.h>
#include "assert.h"
#include "config.h"

#include "move.h"
#include "board.h"
#include "hashtable.h"
#include "pawn_table.h"
#include "piece_tables.h"
#include "evalhash.h"
#include "book.h"

typedef struct gamestate_t {
   board_t *board;      /* pointer to the current board position */

#ifdef UNMAKE_MOVE
   uint8_t *status;     /* Castle flags */
#endif

   uint8_t side_to_move;/* Who's turn it is to move */
   size_t moves_played; /* Number of moves played to current position */
   size_t last_move;    /* Number of the last move played in the game; useful when we take back a move */

   size_t max_moves;    /* Maximum number of moves that can be stored */
   board_t *board_list; /* Past board positions, so moves can be taken back */
   move_t *move_list;   /* list of moves played in the game */
   int16_t *score;          /* Evaluation score associated with each position */
   int8_t *fifty_counter;  /* Counter for the 50-moves rule */ 

   /* Opening book */
   book_t *book;
   bool out_book;

   uint8_t num_moves[MAX_TOTAL_DEPTH+1];

   /* Killer moves, storage space requirements must come from the search
    * function.
    */
#ifdef USE_THIRD_KILLER
   move_t killer[3][MAX_TOTAL_DEPTH];
#else
   move_t killer[2][MAX_TOTAL_DEPTH];
#endif
#ifdef USE_MATE_KILLER
   move_t mate_killer[MAX_TOTAL_DEPTH];
#endif

   /* Botwinnik-Markov extension.
    * This is like a killer move or a refutation move, but for NULL moves.
    * When encountering it on consecutive plies, extend the search.
    */
#ifdef USE_NULL_KILLER
    move_t null_killer[MAX_TOTAL_DEPTH];
#endif

   /* Good captures: these are captures that are nominally bad (based on SEE score)
    * but actually turn out to be ok after trying them. These don't qualify for killer moves,
    * but they should still be sorted slightly higher up in the movelist than other bad captures.
    */
#ifdef USE_GOOD_CAPTURE
   move_t good_capture[2][MAX_TOTAL_DEPTH];
#endif

   /* Data structure for retrieving the principle variation. At each depth,
    * there are two branches for the tree: the principle variation and the
    * "current" branch. If the current branch turns out to be better than
    * the PV, it becomes the new PV.
    * A binary tree would be enough to store this information (and be easy
    * to manipulate), but for now we use a plain NxN array (of which half
    * the space is wasted, obviously)
    */
   move_t principle_variation[MAX_TOTAL_DEPTH][MAX_TOTAL_DEPTH];
   int length_of_variation[MAX_TOTAL_DEPTH];
   int quiescence_depth_of_variation[MAX_TOTAL_DEPTH];
   bool hash_hit[MAX_TOTAL_DEPTH];

   move_t best_move[MAX_TOTAL_DEPTH];

#ifdef USE_COUNTER_MOVES
   /* Counter moves, for use with the counter move heuristic.
    * Indexed by from- and to square and side to move.
    */
   move_t counter[64][64][2];
#endif

#ifdef USE_COMBINATION_MOVES
   /* Combination moves, similar to counter moves except indexed by our own
    * previous moves.
    * Indexed by from- and to square and side to move.
    */
   move_t combo[64][64][2];
#endif

   /* History reductions, as discussed by Ed Schroeder.
    * The idea is to reduce the depth of moves that have failed low in the
    * past. This requires a history table.
    * This is indexed by colour, piece type and destination square.
    * NB: I tried colour/from square/destination square, and it seems to be
    * worse, at least with the parameters given by Ed Schroeder. Probably
    * because using the piece type means we can catch a bad move more
    * easily even if we moved the piece first (Rf5 exf5 is bad independent
    * of whether the rook was on f8 or f7, this situation catches both).
    */
#ifdef HISTORY_REDUCTION
   int history_reduce[2][6][64];
#endif

   /* History table: each move that generates a cut-off gets recorded in
    * this table.
    */
#ifdef USE_HISTORY_HEURISTIC
#ifdef USE_BUTTERFY_HISTORY
   int16_t history[2][64][64];
#else
   int16_t history[2][6][64];
#endif
   int16_t max_history;
#endif

   /* Transposition (hash) table, with a separate one for QS */
   hash_table_t *transposition_table;
   hash_table_t *qs_table;
   size_t default_hash_size;

   /* Evaluation hash table */
   eval_hash_table_t *eval_hash;

   /* Pawn structure hash table */
   pawn_hash_table_t *pawn_structure;

   /* hash table for dynamic queen PSQ */
   piece_square_hash_table_t *queen_psq;

   /* hash table for dynamic rook PSQ */
   piece_square_hash_table_t *rook_psq;

   /* hash table for dynamic knight PSQ */
   piece_square_hash_table_t *knight_psq;

   /* hash table for dynamic bishop PSQ */
   piece_square_hash_table_t *bishop_psq;

   /* hash table for dynamic king PSQ */
   piece_square_hash_table_t *king_psq;

   /* Hash table for repetition detection */
   int8_t repetition_hash_table[0xFFFF+1];

   combined_psq_t dynamic_psq;

   /* various things having to do with time management */
   int time_left[2];       /* Total time left on the clock, in msec */
   int time_inc[2];        /* Time increment per move, in msec */
   int time_per_move;      /* Time per move, in msec */
   uint64_t start_time;    /* Timer value when the search started */
   int movestotc;
   int movestogo;
   int extra_time;
   int running_clock;
   int root_moves_played;
   board_t *root_board;    /* pointer to board position at root */

   /* Whether the engine is currently pondering or not, if it is, disable
    * time control.
    */
   bool pondering;
   move_t ponder_move;

   bool analysing;

   bool random_ok;
   bool trace_ok;

   /* Meta-data */
   char *name_white;
   char *name_black;

   /* Various function pointers, so we can easily customise things and
    * adjust to different UIs.
    */
   void (*output_iteration)(const char *, ...) __attribute__((format(printf, 1, 2)));
   void (*uci_output)(const char *, ...) __attribute__((format(printf, 1, 2)));
   void (*xboard_output)(const char *, ...) __attribute__((format(printf, 1, 2)));

   bool (*check_clock)(const struct gamestate_t *game);
   bool (*check_keyboard)(struct gamestate_t *game);
} gamestate_t;

extern void playgame_init(void);
extern void playgame_shutdown(void);

/* Setters for virtual methods, so we can customise the engine output
 * easily
 */
extern void set_default_output_function(void (*func)(const char *, ...));
extern void set_uci_output_function(void (*func)(const char *, ...));
extern void set_xboard_output_function(void (*func)(const char *, ...));

extern void set_transposition_table_size(gamestate_t *game, uint64_t nelem);

extern gamestate_t *create_game(void);
extern void start_new_game(gamestate_t *game);
extern void end_game(gamestate_t *game);



/* Calculate the new hash key that would result if the move were played on
 * a board with hashkey "key"
 */
static inline uint64_t update_hashkey_from_move(move_t move, uint64_t key, uint8_t ep_square)
{
   int piece, colour, square;
   /* En-passant status */
   if (get_ep_square(move) != ep_square) {
      key ^= en_passant_key[get_ep_square(move)];
      key ^= en_passant_key[ep_square];
   }

   /* Side to move */
   key ^= side_to_move_key;

   /* Move piece */
   piece = get_move_piece(move); 
   colour = (piece>>7);
   key ^= piece_key[colour][piece&PIECE][get_move_origin(move)];
   key ^= piece_key[colour][piece&PIECE][get_move_destination(move)];

   /* Remove captured piece */
   if (is_capture_move(move)) {
      piece = get_captured_piece(move);
      square = get_capture_square(move);
      colour = piece_colour(piece);
      key ^= piece_key[colour][piece&PIECE][square];
   }

   /* Check special moves: pawn promotions and castling */
   if (is_promotion_move(move)) {
      key ^= piece_key[colour][get_move_piece(move) & PIECE][get_move_destination(move)];
      key ^= piece_key[colour][get_promotion_piece(move) & PIECE][get_move_destination(move)];
   }

   /* Castling; the King has already been moved, now just need to move the
    * rook.
    */
   if (is_castle_move(move)) {
      key ^= castle_rook_key[get_move_player(move)>>7][get_castle_flags(move)&1];
   }

   return key;
}



static inline void backup_principle_variation(gamestate_t *game, int depth, move_t move)
{
   int c;
   /* Copy principle variation */
   if (game->length_of_variation[depth+1] >= (depth+1)) {
      game->principle_variation[depth][depth] = move;
      for (c=depth+1; c<game->length_of_variation[depth+1]; c++) {
         game->principle_variation[c][depth] =
            game->principle_variation[c][depth+1];
      }
      game->length_of_variation[depth] = game->length_of_variation[depth+1];
      game->quiescence_depth_of_variation[depth] = game->quiescence_depth_of_variation[depth+1];
      game->hash_hit[depth] = game->hash_hit[depth+1];
   }
}

#endif
