/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef INLINE_GAME_H
#define INLINE_GAME_H

/* We don't expect to take some branches - most moves are not promotions or
 * castling moves, so we tell the compiler about this.
 */
#ifndef __GNUC__
#define __builtin_expect(x,y) (x)
#endif

/* These functions are static inline in header files so the compiler might
 * be able to inline them.
 */

/* We can check whether the move was a checking move without needing to do
 * the (slightly expensive) attack lookup: if neither the target nor the
 * destination square are in the 'superpiece' attack sphere
 */
static inline bool move_checked_player(gamestate_t *game, int side, move_t move)
{
   bitboard_t king_bb = game->board->bbc[side>>7] & game->board->bbp[KING];
   bitboard_t bb;
   uint8_t king_square = bitscan64(king_bb);

   /* If the move didn't originate on a square connecting to the king, then
    * it can't possibly have been a checking move (exceptions: castling, en-passant).
    * Break out early in this case.
    */
   bb = make_bitboard_square(get_move_origin(move)) | 
        make_bitboard_square(get_move_destination(move)) |
        make_bitboard_square(get_capture_square(move));

   if (!is_castle_move(move) && !(bb & super_attack[king_square]))
      return false;
 
   /* Knight checks: 
    * the piece on the target square is a knight, see if it attacks the
    * enemy king. By explicitly using the target square piece, we
    * automatically consider under promotion to a knight.
    * NB: this check is slower than doing a super-piece check below when
    * doing a perft-test from the starting position, but it is slightly
    * faster in more open positions where other pieces may move to the
    * knight's square.
    */
   if (__builtin_expect((get_move_piece(move)&PIECE)==KNIGHT, false) ||
       __builtin_expect((get_promotion_piece(move)&PIECE)==KNIGHT, false)) {
      if (knight_attack[get_move_destination(move)] & king_bb)
         return true;
   }

   bool on_ray = squares_on_ray(king_square, get_move_origin(move)) ||
                 squares_on_ray(king_square, get_move_destination(move)) ||
                 squares_on_ray(king_square, get_capture_square(move));

   /* Catch vertical check by the rook after castling */
   if (!on_ray && is_castle_move(move)) {
      int square = (get_move_origin(move) + get_move_destination(move))/2;
      on_ray = squares_on_ray(king_square, square);
   }
   return on_ray && square_is_attacked(game->board, super_attack[king_square], king_square, side^BLACK);
}

static inline bool player_in_check(const gamestate_t *game, int colour)
{
   /* This test is only needed if:
    *  - the player was in check (unless we only do legal move generation
    *    in that case)
    * - the move is a king move (non-castling)
    * - the piece is pinned (unless we do legal move generation for pinned
    *   pieces)
    * - the move is an en-passant capture (pins/xray attacks are
    *   complicated in this case, can omit if king is not on the same
    *   rank as the captured pawn)
    */
   uint8_t king_square = bitscan64(and_bitboards(game->board->bbc[colour>>7],
                                     game->board->bbp[KING]));
   if (square_is_attacked(game->board, super_attack[king_square], king_square, colour^BLACK))
      return true;
   return false;
}

/* Test whether a move leaves the player in check.
 * We can assume that the player was not in check prior to the move (since
 * we generate only legal evasions) so we only need to test whether the
 * piece was pinned, or is a king move into check.
 */
static inline bool move_leaves_player_in_check(const gamestate_t *game, move_t move, int side)
{
   const board_t *board = game->board;
   if ((get_move_piece(move)&PIECE) == KING)
      return player_in_check(game, side);

   bitboard_t king = board->bbp[KING] & board->bbc[side >> 7];
   bitboard_t bb;
   int square = get_move_origin(move);
   bb = orth[square];
   if (bb & king) {
      int king_square = bitscan64(king);
      bb &= orth[king_square];
      bitboard_t rooks = get_rook_movers(board) & board->bbc[1 - (side >> 7)];
      if (bb & rooks)
         return square_is_attacked(game->board, bb & rooks, king_square, side^BLACK);
   }

   bb = diag[square];
   if (bb & king) {
      int king_square = bitscan64(king);
      bb &= diag[king_square];
      bitboard_t bishops = get_bishop_movers(board) & board->bbc[1 - (side>>7)];
      if (bb & bishops)
         return square_is_attacked(game->board, bb & bishops, king_square, side^BLACK);
   }

   assert(!player_in_check(game, side));
   return false;
}


/* Play a null-move; we don't simply flip the current player because that
 * will mess up the killer tables (and possibly the transposition table,
 * since the hash keys are not properly updated).
 */
static inline void play_null_move(gamestate_t *game)
{

   game->move_list[game->moves_played] = 0;
   game->moves_played++;
   game->board++;
   memcpy(game->board, game->board-1, sizeof *game->board);

   game->side_to_move ^= BLACK;
   game->board->hash ^= side_to_move_key;
   game->board->hash ^= en_passant_key[game->board->ep_square];
   game->board->ep_square = 0;
   game->repetition_hash_table[game->board->hash&0xFFFF]++;
}

static inline void playmove(gamestate_t *game, move_t move)
{
   assert(game);

   /* The following assertion should be prevented from a higher level: we
    * don't want to check whether we have the space to store the next move
    * in a low level function like this. The search algorithm should make
    * sure that there is enough space to store to the end of the current
    * search depth before trying to play any moves. In other words,
    * max_moves-moves_played needs to be larger than the horizon depth,
    * allowing for quiescence search. We could take the horizon depth + 100
    * and always be safe.
    */
   assert(game->moves_played < game->max_moves);

#ifdef COPY_MOVE
   game->board++;
   memcpy(game->board, game->board-1, sizeof *game->board);
#else
   game->status[game->moves_played+1] = game->status[game->moves_played]&0x0f;
#endif

   /* This should not be a NULL move */
   assert(move);

   if (is_capture_move(move)) {
      uint8_t ptaken = get_captured_piece(move);

      assert((ptaken & PIECE) < KING);
      assert((ptaken & PIECE) >= PAWN);

      /* Remove piece from the capture square (normally the same as the
       * destination, except on e-p moves)
       */
      remove_known_piece(game->board, ptaken, get_capture_square(move));
   }

   game->move_list[game->moves_played] = move;
   game->moves_played++;

   /* Advance the 50-move counter if this move was reversible, reset if it
    * wasn't.
    */
   game->fifty_counter[game->moves_played] =
      is_irreversible_move(move)?0:
      (game->fifty_counter[game->moves_played-1] + 1);

   /* Move the piece to its new location */
   move_piece(game->board, get_move_piece(move), get_move_origin(move), get_move_destination(move));

   /* Store en-passant square
    * The hash key for this needs some clarification. If there is no EP
    * square, then move.ep = board->ep_square = 0 and this is a NOP. If
    * there is, then this places an extra white and black pawn on the board
    * on this square, which should be a fairly unique combination.
    * Note that the en-passant square is not set if no enemy pawn is
    * present to exact the capture, which is the common case. This
    * prevents pollution of the hash table where identical positions would
    * otherwise be stored differently.
    */
   if (__builtin_expect(game->board->ep_square != get_ep_square(move), false)) {
      game->board->hash ^= en_passant_key[get_ep_square(move)];
      game->board->hash ^= en_passant_key[game->board->ep_square];
      game->board->ep_square = get_ep_square(move);
   }

   /* Check special moves: pawn promotions and castling */
   if (__builtin_expect(is_promotion_move(move), false)) {
      int piece = get_move_piece(move);
      assert((piece & PIECE) == PAWN);
      remove_known_piece(game->board, piece, get_move_destination(move));
      piece = get_promotion_piece(move);
      assert((piece & PIECE) >= KNIGHT);
      assert((piece & PIECE) <= QUEEN);
      place_promotion_piece(game->board, get_move_destination(move), piece);
   }

   /* Castling; the King has already been moved, now just need to move the
    * rook.
    */
   if (__builtin_expect(is_castle_move(move), false)) {
      game->board->did_castle[get_move_player(move)>>7] = true;
      castle_rook(game->board, get_move_player(move)>>7, get_castle_flags(move));
   }

   game->side_to_move ^= BLACK;
   game->board->hash ^= side_to_move_key;
   game->repetition_hash_table[game->board->hash&0xFFFF]++;

   /* Update check state */
   /* FIXME: do we really need this here? It slows things down quite a
    * bit... there might be a more clever way to do this.
    * Note that the perft test is misleading here - the actual search will
    * also do the test that we do here.
    */
   game->board->in_check = move_checked_player(game, game->side_to_move, move);
#ifdef UNMAKE_MOVE
   game->status[game->moves_played] |= 0x80 * game->board->in_check;
#endif
}

/* Like the regular takeback function, but doesn't check if any moves have
 * been played. Can be used instead of takeback in the search function to
 * avoid an unnecessary branch there.
 */
static inline void takeback_no_check(gamestate_t *game)
{
#ifdef COPY_MOVE
   __builtin_prefetch(game->board-1);
   __builtin_prefetch(&(game->repetition_hash_table[game->board->hash&0xFFFF]));
   game->side_to_move ^= BLACK;
   game->moves_played--;
   game->repetition_hash_table[game->board->hash&0xFFFF]--;
   game->board--;
#else
   move_t move;

   game->repetition_hash_table[game->board->hash&0xFFFF]--;
   game->board->in_check = (game->status[game->moves_played]>>7)&0x01;
   game->moves_played--;
   move = game->move_list[game->moves_played];
   game->side_to_move ^= BLACK;
   game->board->hash ^= side_to_move_key;

   if (__builtin_expect(is_castle_move(move), false)) {
      game->board->did_castle[get_move_player(move)>>7] = false;
      castle_rook(game->board, get_move_player(move)>>7, get_castle_flags(move));
   }

   if (__builtin_expect(is_promotion_move(move), false)) {
      remove_known_piece(game->board, get_promotion_piece(move), get_move_destination(move));
      place_piece(game->board, get_move_destination(move), get_move_piece(move));
   }


   game->board->hash ^= en_passant_key[get_ep_square(move)];
   game->board->hash ^= en_passant_key[game->board->ep_square];
   game->board->ep_square = get_ep_square(move);

   move_piece(game->board, get_move_piece(move), get_move_origin(move), get_move_destination(move));

   if (is_capture_move(move))
      place_piece(game->board, get_capture_square(move), move.ptaken);

   if (game->moves_played) {
      move = game->move_list[game->moves_played-1];
      game->board->hash ^= en_passant_key[get_ep_square(move)];
      game->board->hash ^= en_passant_key[game->board->ep_square];
      game->board->ep_square = get_ep_square(move);
   }
#endif
}

/* To take back a move, simply go back one spot in the array. We could save
 * memory by actually reversing the move, but what's the point? Memory is
 * cheap these days!
 */
static inline void takeback(gamestate_t *game)
{
   assert(game);
   if (game->moves_played) {
      takeback_no_check(game);
   }
}

/* Redo the last move: if a move has been reverted, but no new move has
 * been played yet.
 * Mainly for player interaction.
 */
static inline void redo_last_move(gamestate_t *game)
{
   if (game->moves_played < game->last_move) {
      game->moves_played++;
      game->board++;
      game->side_to_move ^= BLACK;
      game->repetition_hash_table[game->board->hash&0xFFFF]++;
   }
}

/* Again mainly for the benefit of user interaction: check if the move was
 * valid or left the king in check. Don't destroy the move history of the
 * game if it does.
 */
static inline bool playmove_was_ok(gamestate_t *game, move_t move)
{
   move_t old_move = game->move_list[game->moves_played+1];
   board_t old_board = game->board_list[game->moves_played+1];

   memcpy(&old_board, game->board+1, sizeof *game->board);

   playmove(game, move);

   if (player_in_check(game, game->side_to_move^BLACK)) {
      takeback(game);
      game->move_list[game->moves_played+1] = old_move;
      memcpy(game->board+1, &old_board, sizeof *game->board);
      return false;
   }
   return true;
}

/* Again mainly for the benefit of user interaction: check if the current
 * position is a mate (check or stale). The search function will detect
 * this in a different way.
 */
static inline bool player_is_mate(gamestate_t *game)
{
   /* By making a move, we affect the *current* move and the *next* board. */
   move_t old_move = game->move_list[game->moves_played];
   board_t old_board = game->board_list[game->moves_played+1];
   movelist_t movelist;
   int n;

   memcpy(&old_board, game->board+1, sizeof *game->board);

   generate_moves(&movelist, game->board, game->side_to_move);
   for (n=0; n<movelist.num_moves; n++) {
      int check;
      playmove(game, movelist.move[n]);
      check = player_in_check(game, game->side_to_move^BLACK);
      takeback(game);

      game->move_list[game->moves_played] = old_move;
      memcpy(game->board+1, &old_board, sizeof *game->board);
      if (!check)
         return false;
   }
   return true;
}
#endif
