/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef MOVEGEN_H
#define MOVEGEN_H

#include <stdbool.h>
#include "move.h"
#include "board.h"

/* Attack and move bitboards */
/* Normal moves */
extern bitboard_t king_attack[64];
extern bitboard_t king_attack2[64];
extern bitboard_t knight_attack[64];
extern bitboard_t super_attack[64];
extern bitboard_t orth[64];
extern bitboard_t diag[64];
extern bitboard_t rays[64];
extern bitboard_t attack_row[64][64];
extern bitboard_t attack_file[64][64];
extern bitboard_t attack_diagonal_a8h1[64][64];
extern bitboard_t attack_diagonal_a1h8[64][64];
extern bitboard_t ep_neighbours[16];
extern bitboard_t connecting_ray[64][64];

/* X-ray attacks */
extern bitboard_t xattack_row[64][64];
extern bitboard_t xattack_file[64][64];
extern bitboard_t xattack_diagonal_a8h1[64][64];
extern bitboard_t xattack_diagonal_a1h8[64][64];

/* Population count/mobility */
extern int8_t mobility_row[64][64];
extern int8_t mobility_file[64][64];
extern int8_t mobility_diagonal_a8h1[64][64];
extern int8_t mobility_diagonal_a1h8[64][64];


void generate_movement_tables(void);
void generate_legal_moves(movelist_t *movelist, const board_t *board, int colour);
void generate_moves(movelist_t *movelist, const board_t *board, int colour);
void generate_tactical_moves(movelist_t *movelist, const board_t *board, int colour);
void generate_quiescence_moves(movelist_t *movelist, const board_t *board, int colour);
void generate_checking_moves(movelist_t *movelist, const board_t *board, int side_to_move, bool include_captures);

bool check_move_exists(move_t move, const board_t *board);

int validate_move(const movelist_t *movelist, int source, int dest);

bitboard_t get_attack_bitboard(const board_t *board, int side);
bitboard_t get_xrayattack_bitboard(const board_t *board, int side);
void get_xrayattack_and_block_bitboard(const board_t *board, int side, bitboard_t *attack, bitboard_t *blockers);
bitboard_t get_attackers(const board_t *board, int side, int square);
bitboard_t get_all_attackers(const board_t *board, bitboard_t mask, int square);
bitboard_t get_pinned_pieces(const board_t *board, bitboard_t mask, int side, int square);

void print_movelist(const movelist_t *movelist);

/* Simplified version of the move generator that only checks if any piece
 * belonging to the given player can reach the square indicated.
 */
static bool inline square_is_attacked(const board_t *board, bitboard_t mask, int square, int player)
{
   bitboard_t bb_where = make_bitboard_square(square);
   bitboard_t occupied;
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops_or_queens;
   bitboard_t rooks_or_queens;
   bitboard_t king;
   bitboard_t bb;
   bitboard_t moves;
   bitboard_t own_movers;
   int c_index = (player != 0);
   int row_nr, col_nr, a8h1_nr, a1h8_nr;

   /* Initilize appropriate bitboards */
   own_movers = board->bbc[c_index] & mask;
   occupied = or_bitboards(board->bbc[0], board->bbc[1]);

   /* Check rook moves and horizontal/vertical queen moves */
   rooks_or_queens = and_bitboards(own_movers, or_bitboards(board->bbp[ROOK], board->bbp[QUEEN]));
   if (rooks_or_queens) {
      row_nr = (get_row_occupancy_number(board, square) >> 1) & 63;
      col_nr = (get_file_occupancy_number(board,square) >> 1) & 63;
      moves = or_bitboards(attack_row[square][row_nr], attack_file[square][col_nr]);
      bb = and_bitboards(moves, rooks_or_queens);
      if (bb) return true;
   }

   /* Check bishop moves and diagonal queen moves */
   bishops_or_queens = and_bitboards(own_movers, or_bitboards(board->bbp[BISHOP], board->bbp[QUEEN]));
   if (bishops_or_queens) {
      a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
      a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;
      moves = or_bitboards(attack_diagonal_a8h1[square][a8h1_nr], attack_diagonal_a1h8[square][a1h8_nr]);
      bb = and_bitboards(moves, bishops_or_queens);
      if (bb) return true;
   }

   /* Check knight and king moves in one go */
   knights = and_bitboards(own_movers, board->bbp[KNIGHT]);
   king = and_bitboards(own_movers, board->bbp[KING]);
   bb = and_bitboards(knights, knight_attack[square]);
   bb = or_bitboards(bb, and_bitboards(king, king_attack[square]));
   if (bb) return true;

   /* Finally check for pawns. */
   pawns = and_bitboards(own_movers, board->bbp[PAWN]);
   if (pawns) {
      if (player==WHITE) {
         bb = and_bitboards(bb_where, shift_bitboard_right_up(pawns)|shift_bitboard_left_up(pawns));
         if (bb) return true;
      } else {
         bb = and_bitboards(bb_where, shift_bitboard_right_down(pawns)|shift_bitboard_left_down(pawns));
         if (bb) return true;
      }
   }

   return false;
}


/* Return true if two squares are on a connecting ray (ie, a super piece
 * can move from one square to the other in one move on an ampty board)
 */
static inline bool squares_on_ray(int square1, int square2)
{
   return (rays[square1] & make_bitboard_square(square2)) != 0;
}

/* Special: get attacks and xray attacks for a rook on a given square */
static inline void get_rook_attacks(const board_t *board, int square, bitboard_t *attack, bitboard_t *xray)
{
   int row_nr, col_nr;

   row_nr = (get_row_occupancy_number(board, square) >> 1) & 63;
   col_nr = (get_file_occupancy_number(board,square) >> 1) & 63;
   *attack = (attack_row[square][row_nr] | attack_file[square][col_nr])&
                           ~make_bitboard_square(square);
   *xray = (xattack_row[square][row_nr] | xattack_file[square][col_nr]);
}

/* Special: get attacks and xray attacks for a bishop on a given square */
static inline void get_bishop_attacks(const board_t *board, int square, bitboard_t *attack, bitboard_t *xray)
{
   int a8h1_nr, a1h8_nr;

   a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
   a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;
   *attack = (attack_diagonal_a8h1[square][a8h1_nr] |
               attack_diagonal_a1h8[square][a1h8_nr])& ~make_bitboard_square(square);
   *xray = xattack_diagonal_a8h1[square][a8h1_nr]|
                           xattack_diagonal_a1h8[square][a1h8_nr];
}

/* Special: get attack and xray attacks for a bishop on a given square and
 * for a given occupancy bitboard.
 */
static inline void get_bishop_attacks_for_occupancy(bitboard_t occ, int square, bitboard_t *attack, bitboard_t *xray)
{
   int a8h1_nr, a1h8_nr;

   a8h1_nr = (get_bitboard_antidiag_occupancy_number(occ, square) >> 1) & 63;
   a1h8_nr = (get_bitboard_diagonal_occupancy_number(occ, square) >> 1) & 63;
   *attack = (attack_diagonal_a8h1[square][a8h1_nr] | attack_diagonal_a1h8[square][a1h8_nr]) & ~make_bitboard_square(square);
   *xray   = xattack_diagonal_a8h1[square][a8h1_nr] | xattack_diagonal_a1h8[square][a1h8_nr];
}



/* Combination of the above two: get queen attacks */
static inline void get_queen_attacks(const board_t *board, int square, bitboard_t *attack, bitboard_t *xray) 
{
   bitboard_t a, x;
   get_rook_attacks(board, square, &a, &x);
   get_bishop_attacks(board, square, attack, xray);
   *attack |= a;
   *xray |= x;
}

/* Pawn attacks */
static inline void get_pawn_attacks(const board_t *board, bitboard_t *white, bitboard_t *black)
{
   bitboard_t pawns = board->bbp[PAWN] & board->bbc[0];

   *white = shift_bitboard_right_up(pawns) | shift_bitboard_left_up(pawns);
   pawns ^= board->bbp[PAWN];
   *black = shift_bitboard_right_down(pawns) | shift_bitboard_left_down(pawns);
}

#endif
