/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <inttypes.h>
#include <limits.h>
#include <ctype.h>
#include "config.h"
#include "computer.h"
#include "bits.h"
#include "alphabeta.h"
#include "genrand.h"
#include "board.h"
#include "movegen.h"
#include "game.h"
#include "inline/game.h"
#include "names.h"
#include "timer.h"
#include "config.h"
#include "evaluate.h"
#include "see.h"
#include "keypressed.h"
#include "history.h"

#undef DEBUG
#ifdef DEBUG
#define abtrace printf
#else
#define abtrace(s,...)
#endif

#undef PRINT_MOVE_EVALUATIONS
#undef CLEAR_KILLER_TABLE
#undef CLEAR_TRANSPOSITION_TABLE

#define print_iter if (game->output_iteration) game->output_iteration
#define uci if (game->uci_output) game->uci_output
#define xb if (game->xboard_output) game->xboard_output

#define check_keyboard(game) ((game->check_keyboard) ?  game->check_keyboard(game) : false)

static bool stop_analysing;
bool break_search_early = false;
bool root_is_mate;

static int min(int x, int y)
{
   return (x<y)?x:y;
}

static int max(int x, int y)
{
   return (x>y)?x:y;
}

#define move_string(m, null) short_move_string(m, NULL, null)
/* Retrieve principle variation from the hash table */
void retrieve_principle_variation(gamestate_t *game, move_t move)
{
   static int n = 30;
   hash_table_entry_t *hash;

   if (n == 0) return;
   n--;

   if (count_repetition(game) > 1) return;
   
   print_iter("%s ", move_string(move, NULL));
   playmove(game, move);

   /* Look up th resulting position in the hash table */
   hash = query_table_entry(game->transposition_table, game->board->hash);

   /* Recursive call to get next move from the table */
   if (hash && hash->best_move)
      retrieve_principle_variation(game, hash->best_move);

   takeback(game);
}

#undef move_string
#define move_string(m, null) short_move_string(m, &movelist, null)
#define move_stringv(m, null) short_move_string(m, NULL, null)

static void insert_pv_in_transposition_table(gamestate_t *game, int score)
{
   int n;

   if (score == 0)
      return;

   uint8_t ep_square = game->board->ep_square;
   uint64_t hash_key;
   hash_key = game->board->hash;

   for (n=0; n<game->length_of_variation[0]; n++) {
      int horizon_distance = game->length_of_variation[0] - n;
      move_t move = game->principle_variation[n][0];
      hash_table_entry_t *hash;

      assert(move);

      hash = query_table_entry(game->transposition_table, game->board->hash);

      /* Don't store capture moves from the PV into the transposition table - we don't want to insert
       * moves from the quiescence search, afterall.
       */
      if (!hash && !is_capture_move(move))
         store_table_entry(game->transposition_table, game->board->hash, 
                           horizon_distance, score_to_hashtable(score, n), HASH_TYPE_EXACT, move);

      /* Play the move, just update the hashkey. */
      hash_key = update_hashkey_from_move(move, hash_key, ep_square);
      ep_square = get_ep_square(move);
   }
}

static void print_principle_variation(const gamestate_t *game)
{
   int c;
   int root_move_number = 0;

   print_iter("   ");
   if (game->side_to_move) {
      root_move_number = 1;
      print_iter("%d. ... ", (int)(game->moves_played)/2+1);
   }
   for (c=0; c<game->length_of_variation[0]; c++) {
      if (((root_move_number+c) & 1) == 0)
         print_iter("%d. ", (int)(root_move_number+game->moves_played+c)/2+1);
      print_iter("%-5s ", move_stringv(game->principle_variation[c][0], NULL));
   }
}


static void print_principle_variation_xb(const gamestate_t *game)
{
   int c;
   int root_move_number = 0;

   if (game->side_to_move) {
      root_move_number = 1;
      xb("   %d. ... ", (int)(game->moves_played)/2+1);
   }
   for (c=0; c<game->length_of_variation[0]; c++) {
      if (((root_move_number+c) & 1) == 0)
         xb("%d. ", (int)(root_move_number+game->moves_played+c)/2+1);
      xb("%-5s ", move_stringv(game->principle_variation[c][0], NULL));
   }
}

bool computer_play(gamestate_t *game, int max_depth)
{
   static const char *output_lines[] = {
      "<%2d>   ---   % 7.2f %8d %+7.2f ->%+7.2f  %2d /%2d",
      "<%2d>         % 7.2f %8d %+7.2f ->%+7.2f  %2d /%2d",
      "<%2d>   !!!   % 7.2f %8d %+7.2f ->%+7.2f  %2d /%2d",
      "[%2d] %+7.2f % 7.2f %8d   (EBR = %5.2f)    %2d /%2d"
   };
   uint64_t start_time, time, prev_time, prev_prev_time;
   movelist_t movelist;
   int move_perm[MAX_MOVES];
   int score[MAX_MOVES];
   int eval[MAX_MOVES];
   int node_count[MAX_MOVES];
   int alpha, beta;
   int best_move;
   int legal_moves;
   int mate_depth;
   int depth;
   int sdepth, sqdepth;
   int me;
   int n;
   hash_table_entry_t *hash;
   move_t hash_move;
   bool hash_hit = false;
   int root_move_number = 0;

   assert(game->board->bbp[KING] && !onebit64(game->board->bbp[KING]));

   game->extra_time = 0;
   game->running_clock = game->side_to_move>>7;
   game->root_moves_played = game->moves_played;
   game->root_board = game->board;
   game->ponder_move = 0;
   root_is_mate = false;

   /* Check a number of obvious things: repetition draw, 50 move rule */
   if (draw_by_repetition(game)) {
      print_iter("Draw (position repeated 3 times)\n");
      return false;
   }
   if (game->fifty_counter[game->moves_played] > 101) {
      print_iter("Draw (50 moves without pawn move or capture)\n");
      return false;
   }

   /* Start the clock */
   start_time = get_timer();
   start_clock(game);

   /* Setup data structures */
#ifdef CLEAR_KILLER_TABLE
   memset(game->killer, 0, sizeof game->killer);
   //memset(game->mate_killer, 0, sizeof game->mate_killer);
#endif

#ifdef CLEAR_TRANSPOSITION_TABLE
   if (game->transposition_table) {
      memset(game->transposition_table->data, 0, game->transposition_table->number_of_elements*sizeof(hash_table_entry_t));
      memset(game->transposition_table->depth_data, 0, game->transposition_table->number_of_elements*sizeof(hash_table_entry_t));
   }
#endif

   /* Prepare the transpostion table for a new search iteration:
    * Resets the write count and increases the generation counter.
    */
   prepare_hashtable_search(game->transposition_table);

   memset(score, 0, sizeof score);

   clear_history(game);

   memset(game->principle_variation, 0, sizeof game->principle_variation);
   game->length_of_variation[0] = 0;

   generate_moves(&movelist, game->board, game->side_to_move);

   if (max_depth>MAX_SEARCH_DEPTH) max_depth = MAX_SEARCH_DEPTH;
   sdepth = sqdepth = 0;

   me = game->side_to_move;
   legal_moves = movelist.num_moves;

   abort_search = false;
   positions_evaluated = 0;
   moves_searched = 0;
   positions_in_hashtable = 0;
   branches_pruned = 0;
   branches_pruned_1st = 0;

   game->score[game->moves_played+1] = -game->score[game->moves_played];

   if ((game->moves_played & 1) && (me == WHITE)) root_move_number++;

   /* A small optimisation for the evaluation function: if a player cannot
    * castle in the root position, then we will claim that that player has
    * castled.
    * The evaluation function punishes the loss of castling rights since
    * the root of the search to encourage the engine to castle - but it
    * should do so only on the search where the castling rights are lost.
    */
   if (!may_castle(game->board, 0))
      game->board->did_castle[0] = true;
   if (!may_castle(game->board, 1))
      game->board->did_castle[1] = true;

   /* Look up the root position in the hash table, re-search the move from
    * the hash table first
    */
   hash_move = 0;
   hash = query_table_entry(game->transposition_table, game->board->hash);
   if (hash) {
      hash_move = hash->best_move;
      print_iter("Found TT entry, score %d\n", hash->score);
   }

   /* First step: initial ordering of the moves, filter out illegal moves */
   prev_time = time = get_timer();
   for (n=0; n<movelist.num_moves; n++) {
      move_t move = movelist.move[n];
      bool check;
      move_perm[n] = n;

      playmove(game, move);
      check = player_in_check(game, me);
      //eval[n] = static_evaluation(game, me, -CHECKMATE, CHECKMATE);
      //score[n] = eval[n];
      if (!check) {
         eval[n] = -alphabeta(game, 0, 0, -CHECKMATE, CHECKMATE);
         score[n] = eval[n];
      }
      //printf("% 3d % 8s %d %d\n", n, move_string(move, NULL), check, score[n]);
      takeback(game);
      if (check) {   /* Illegal move */
         legal_moves--;
         score[n] = -3*CHECKMATE;
         node_count[n] = -3*CHECKMATE;
         continue;
      }

      /* Score moves */

      /* Give the highest priority to the move from the hash table, if it
       * exists.
       */
      if (moves_are_equal(hash_move, move)) {
         score[n] = eval[n] = hash->score;
         score[n] += CHECKMATE;
         continue;
      }

      if (game->moves_played < 10 && game->random_ok)
         score[n] += genrandui() & 0x7f;
   }

   /* Hmm... no legal moves? */
   if (legal_moves == 0) {
      root_is_mate = true;
      stop_analysing = true;
      return false;
   }

   load_dynamic_psq(game);

   /* Sort the move list based on the score */
   sort_moves(move_perm, movelist.num_moves, score);
   if (moves_are_equal(hash_move, movelist.move[move_perm[0]]))
      score[move_perm[0]] -= CHECKMATE;

   /* Check whether the current position is in the opening book */
   if (!game->out_book && game->book && !game->analysing) {
      movelist_t book_moves;
      int move_score[256];
      int total_score = 0;
      book_moves.num_moves = 0;
      opening_position_t *entry = get_book_move(game->book, game->board, game->side_to_move);
      /* If the current position is not in the book and we've already
       * played a fair number of games, assume we're not going to transpose
       * back into the book.
       */
      if (!entry && game->moves_played > 10)
         game->out_book = true;
      while(entry) {
         if (entry->weight == 0) continue;
         int from, to;
         from = (entry->move >> 6) & 63;
         to = entry->move & 63;
         int n;
         for (n=0; n<movelist.num_moves; n++) {
            if (get_move_origin(movelist.move[n]) == from && get_move_destination(movelist.move[n]) == to) {
               int k = book_moves.num_moves;
               book_moves.num_moves++;
               total_score += entry->weight;
               book_moves.move[k] = movelist.move[n];
               move_score[k] = total_score;
               break;
            }
         }
         free_book_move(entry);
         entry = get_next_book_move(game->book);
      }

      if (book_moves.num_moves) {
         int s = genrandui() % total_score;
         int n = 0;
         while (move_score[n] < s) n++;
         playmove(game, book_moves.move[n]);

         //printf("Book move: %s\n", move_string(book_moves.move[n], NULL));
         print_iter(output_lines[3], 1,
               0, (get_timer() - start_time)/1000000.0,
               0, 0.0, sdepth, sqdepth);
         print_iter("   %d. ", (int)(root_move_number+game->moves_played)/2+1);
         if (me == BLACK) print_iter("... ");
         print_iter(" %-6s", move_string(book_moves.move[n], NULL));
         print_iter(" (book)\n");

         xb("% 3d % 4d %5"PRIu64" %9d (", 0, 0, (get_timer() - start_time)/10000, 0);
         for (n=0; n<book_moves.num_moves; n++) {
            move_t move = book_moves.move[n];
            double f = 100. * move_score[n] / total_score;
            if (n) f =  100. * (move_score[n]-move_score[n-1])  / total_score;
            if (n) xb(", ");
            xb("%s %.2f%%", move_string(move, NULL), f);
         }
         xb(")\n");

         return true;
      }
   }

   best_move = move_perm[0];
   print_iter(output_lines[3], 1,
         eval[best_move]/100.0, (get_timer() - start_time)/1000000.0,
         moves_searched, 0.0, sdepth, sqdepth);
   print_iter("   %d. ", (int)(root_move_number+game->moves_played)/2+1);
   if (me == BLACK) print_iter("... ");
   print_iter("%-5s", move_string(movelist.move[best_move], NULL));
   print_iter("\n");

   /* Uh oh... */
   if (legal_moves == 1 && !game->analysing) {
      playmove(game, movelist.move[best_move]);
      return true;
   }

   /* If we're trying to mate a lone king, allocate some extra time, it
    * can't take very long anyway.
    */
   if ((onebit64(game->board->bbc[0]) || onebit64(game->board->bbc[1])) && (game->fifty_counter[game->moves_played] > 40))
      game->extra_time = peek_timer(game) / 4;

   prev_time = time;
   time = get_timer();

   uci("info depth 1 score cp %d nodes %lld time %d hashfull %d\n", score[best_move], moves_searched, peek_timer(game), min(1000, game->transposition_table->write_count*1000/game->transposition_table->number_of_elements));

   int best_move_kept = 0;
   mate_depth = 0;
   for (depth=1; depth<max_depth; depth++) {
      int best_score = score[best_move];
      int new_best_move = best_move;

      scale_history(game);

      for (n=0; n<legal_moves; n++) {
         int previous_moves_searched = moves_searched;
         int open_alpha_window = 100;
         int open_beta_window = 100;
         move_t move;
         int r;
         int new_score;

         move = movelist.move[move_perm[n]];
         hash_hit = false;

         uci("info currmove %s%s\n", square_str[get_move_origin(move)], square_str[get_move_destination(move)]);

         /* Don't waste time re-searching moves for which we already know
          * they lead to a forced mate.
          */
         //if (score[move_perm[n]] < -(CHECKMATE-1000)) continue;

         /* Aspiration search */
         if (move_perm[n] == new_best_move) {
            alpha = best_score - 50;
            beta = best_score + 50;

            if (best_score > CHECKMATE - 1000) {
               beta = best_score+1;
               alpha = best_score;
            }
         } else {
            alpha = best_score;
            beta = best_score + 1;
         }

#ifdef PRINT_MOVE_EVALUATIONS
       print_iter("     %2d/%d (%+3.2f) %-5s %+3.2f",
             n+1, legal_moves, best_score/100.0, 
             move_string(move, NULL), score[move_perm[n]]/100.0);
      fflush(stdout);
#endif

#ifdef DEBUG_SEARCH_TREE
         printf("\n  [%ld %c", game->moves_played-game->root_moves_played+1, game->side_to_move?'b':'w');
         printf(" %s % d [%d %d] (Root)\n", move_string(move, NULL), score[move_perm[n]], alpha, beta);
#endif

         /* Late move reductions */
         r = (n > 2 && depth > 1)*PLY;
         r += (n > 10 && depth > 6)*PLY;
         if (is_promotion_move(move) || is_capture_move(move) || game->board->in_check)
            r = 0;

         playmove(game, move);
         while (true) {
            abtrace("Searching move %s (%d/%d) with window [%d %d] at %d\n",
                     move_stringv(move, NULL), n, legal_moves, -beta, -alpha, depth-r/PLY);
            assert(beta > alpha);
            new_score = -alphabeta(game, 0, depth*PLY-r, -beta, -alpha);
            abtrace("[%d %d %d]\n", alpha, new_score, beta);

            /* Fail high/low, adjust window.
             * NB: this assumes fail soft alpha-beta, which may return a
             * score outside of [alpha, beta]!
             */
            /* Non-PV moves are expected to fail low, so only research fail
             * low for PV-nodes.
             */
            if (new_score <= alpha && n==0) {
               abtrace("alphabeta- [%d %d] returned %d\n",alpha,beta,new_score);
               if (new_score > CHECKMATE - 1000) {
                  alpha = new_score - 2;
                  //beta = new_score + 4;
               } else {
                  alpha = max(new_score - open_alpha_window, -CHECKMATE);
                  open_alpha_window *= 100;
               }
               abtrace("\n");
               abtrace("Set alpha to %d\n", alpha);
               if (n == 0) {  /* Best move failed low, allocate extra time */
                  game->extra_time = peek_timer(game) / 2;
               }
               continue;
            }
            if (new_score >= beta && beta < CHECKMATE) {
               abtrace("alphabeta+ [%d %d] returned %d\n",alpha,beta,new_score);
               if (new_score > CHECKMATE - 1000) {
                  beta = min(new_score + depth, CHECKMATE);
                  //alpha = new_score-2;
                  //alpha = min(beta - depth - 1, new_score - depth);
               } else {
                  beta = min(new_score + open_beta_window, CHECKMATE);
                  open_beta_window *= 100;
               }
               r = 0;
               abtrace("\n");
               abtrace("Set beta to %d\n", beta);
               //abtrace("%d (%c)\n", game->length_of_variation[1], game->hash_hit[1] ? 'H' : '-');
               continue;
            }
            break;
         } 
         takeback(game);


#ifdef DEBUG_SEARCH_TREE
         printf("-->]%ld %c", game->moves_played-game->root_moves_played+1,
               game->side_to_move?'b':'w');
         printf(" %s (%d) % d -> % d [%d %d]\n",
               move_string(move, NULL), n, score[move_perm[n]], new_score, alpha, beta);
#endif

#ifdef PRINT_MOVE_EVALUATIONS
         if (abort_search) {
            print_iter("(interrupted)\n");
            break;
         }
         {
         int sqdepth = game->quiescence_depth_of_variation[0];
         int sdepth = game->length_of_variation[0] - sqdepth;

         print_iter(" -> %+3.2f %2d/%d (%d nodes)\n",
                     new_score/100.0, sdepth, sqdepth,
                     moves_searched - previous_moves_searched);
         }
#endif
         if (abort_search) break;

         /* store principle variation */
         if (n == 0) {
            backup_principle_variation(game, 0, move);
            sqdepth = game->quiescence_depth_of_variation[0];
            sdepth = game->length_of_variation[0] - sqdepth;
         }

         /* Update score for this move */
         score[move_perm[n]] = new_score;
         node_count[move_perm[n]] = moves_searched - previous_moves_searched;

         /* Best move score decreased? */
         if (new_best_move == move_perm[n] && new_score<(best_score-50)) {
            if (depth > 1) {
               int sqdepth = game->quiescence_depth_of_variation[0];
               int sdepth = game->length_of_variation[0] - sqdepth;
               print_iter(output_lines[0],
                     depth+1, (get_timer() - start_time)/1000000.0,
                     moves_searched,
                     best_score/100.0, new_score/100.0,
                     sdepth, sqdepth);
               print_principle_variation(game);
               if (game->hash_hit[1]) print_iter("<H>");
               print_iter("\n");
            }
         } else if (new_score>(best_score+50) && depth>1) {
            /* New best move, or score improved substantially */
            int sqdepth = game->quiescence_depth_of_variation[0];
            int sdepth = game->length_of_variation[0] - sqdepth;
            print_iter(output_lines[2],
               depth+1, (get_timer() - start_time)/1000000.0,
               moves_searched,
               best_score/100.0, new_score/100.0,
               sdepth, sqdepth);
            print_principle_variation(game);
            if (game->hash_hit[1]) print_iter("<H>");
            print_iter("\n");
         }

         /* New best move, or score increased substantially */
         if ( (new_score > best_score) || (new_best_move == move_perm[n])) {
            if (new_best_move != move_perm[n] && new_score<=(best_score+50) && depth>1) {
               int sqdepth = game->quiescence_depth_of_variation[0];
               int sdepth = game->length_of_variation[0] - sqdepth;
               print_iter(output_lines[1],
                     depth+1, (get_timer() - start_time)/1000000.0,
                     moves_searched,
                     best_score/100.0, new_score/100.0,
                     sdepth, sqdepth);
               print_principle_variation(game);
               if (game->hash_hit[1]) print_iter("<H>");
               print_iter("\n");
            }

            /* Copy principle variation */
            if (new_best_move != move_perm[n]) {
               backup_principle_variation(game, 0, move);
            }

            best_score = new_score;
            new_best_move = move_perm[n];
            hash_hit = game->hash_hit[0];

            sqdepth = game->quiescence_depth_of_variation[0];
            sdepth = game->length_of_variation[0] - sqdepth;
            store_table_entry(game->transposition_table, game->board->hash, 
                  depth, score_to_hashtable(best_score, 0), HASH_TYPE_EXACT,
                  movelist.move[new_best_move]);
         }
      }  /* end of loop over all moves */

      if (abort_search) break;

      /* Sort the move list based on the node count of the subtree.
       * Make sure that the "best move" is still sorted as "best move"
       * after the call to sort(); this is not guaranteed because there
       * might be other moves that have the same evaluation and that might
       * therefore "replace" this move after the sort, even if they were
       * not reported as alternative moves by the search above. This is
       * probably mainly a cosmetic issue, but it might have some
       * implications for the history of PV/non-PV nodes.
       */
      score[new_best_move]++;
      node_count[new_best_move] = moves_searched + 1;
      sort_moves(move_perm, movelist.num_moves, node_count);
      score[move_perm[0]]--;

      if (best_move == new_best_move)
         best_move_kept++;
      else
         best_move_kept = 0;
      best_move = new_best_move;

      prev_prev_time = prev_time;
      prev_time = time;
      time = get_timer();

      /* Inject principle variation into the transposition table, in case
       * it was lost.
       */
      insert_pv_in_transposition_table(game, score[best_move]);

      /* Write detailed iteration output */
      print_iter(output_lines[3], depth+1,
            score[best_move]/100.0, (get_timer() - start_time)/1000000.0,
            moves_searched,
            //positions_evaluated,
            //(float)moves_searched/positions_evaluated,
            (float)(time - prev_time)/(prev_time - prev_prev_time),
            sdepth, sqdepth);
      print_principle_variation(game);
      if (hash_hit) print_iter("<H>");
      print_iter("\n");
#ifdef DEBUG_SEARCH_TREE
      print_iter("\n");
#endif

      uci("info nps %lld\n", 1000000ll*moves_searched / (get_timer() - 1 - start_time));

      uci("info depth %d score cp %d nodes %llu time %d hashfull %d ", depth+1,
         score[best_move], moves_searched, peek_timer(game), min(1000, game->transposition_table->write_count*1000/game->transposition_table->number_of_elements));

      if (game->uci_output) {
         uci("pv ");
         for (n=0; n<game->length_of_variation[0]; n++) {
            move_t move;
            const char *p = "";
            static const char *lower_piece_str[] = { "p","n","b","r","q","k" };
            move = game->principle_variation[n][0];
            if (is_promotion_move(move))
               p = lower_piece_str[get_promotion_piece(move)&PIECE];

            uci("%s%s%s ", square_str[get_move_origin(move)], square_str[get_move_destination(move)], p);
         }
      }
      uci("\n");

      xb("% 3d % 4d %5"PRIu64" %9"PRIu64" ", depth+1, score[best_move], (get_timer() - start_time)/10000, (uint64_t)moves_searched);
      print_principle_variation_xb(game);
      xb("\n");

      /* Break out early if we've found a checkmate */
      if (abs(best_score) >= CHECKMATE-1000) {
         int ply = CHECKMATE - abs(best_score);
         if (mate_depth == 0 || ply < mate_depth)
            mate_depth = ply;
         print_iter("Mate in %d (%d ply)\n", ply/2 + 1, ply);

         stop_analysing = true;

         /* Break early if we find a forced mate, but only if it is within
          * the horizon.
          */
         if (mate_depth <= depth || game->length_of_variation[0] == mate_depth || game->analysing)
            break;
      }

      /* Break out early if we're told to do so */
      if (check_keyboard(game))
         break;

      /* Check if we have enough time for the next iteration, assuming an
       * effective branching ratio of 2.
       */
      if ( peek_timer(game) > get_game_time_for_move(game)/2 )
         break;

      /* Problem solving mode: if the best move was kept for at least 4
       * iterations in a row with a score of +500, then abort the search.
       */
      if (break_search_early && best_move_kept>=4 && score[best_move]>500)
         break;
   }

   print_iter(" --> %s %.2f\n", move_string(movelist.move[best_move], NULL), score[best_move]/100.0);
   print_iter("     ");
   retrieve_principle_variation(game, movelist.move[move_perm[0]]);
   print_iter("\n");
   print_iter("%llu nodes visited (%llu [%.2f%%] in transposition table), "
              "%g nodes/s\n"
              "%llu branches pruned (%.1f%% on first move)\n"
              "%llu positions evaluated (average branching ratio %.2f)\n",
         moves_searched,
         positions_in_hashtable,
         100.0*(float)positions_in_hashtable/moves_searched,
         1.0e6*moves_searched / (time - start_time),
         branches_pruned,
         100. * branches_pruned_1st / branches_pruned,
         positions_evaluated,
         (float)moves_searched/positions_evaluated);

   if (!game->analysing)
      playmove(game, movelist.move[best_move]);

   if (draw_by_repetition(game)) {
      print_iter("Draw (position will repeat for the third time after %s)\n", move_string(movelist.move[best_move], NULL));
      stop_analysing = true;
      return true;
   }

   /* Store the best reply as the move we would like to ponder on (the hint move).  */
   if (!game->pondering && game->length_of_variation[0] > 1)
      game->ponder_move = game->principle_variation[1][0];

   /* Delete this position form the transposition table: if the game
    * repeats here, it should b a draw anyway, and we don't want to pick up
    * this position as one that would lead to mate later on (which might
    * happen otherwise).
    */
#if 0
   hash = query_table_entry(game->transposition_table, game->board->hash);
   if (hash) {
      hash->key = 0;
   }
#endif

   //printf("Table entries unused: %d (out of %d)\n",

   //         count_unused_table_entries(game->transposition_table), 2*game->transposition_table->number_of_elements);

   game->score[game->moves_played] = score[best_move];

   return true;
}

static bool interrupt_ponder(gamestate_t *game)
{
   return keyboard_input_waiting();
}

bool computer_ponder(gamestate_t *game)
{
   move_t ponder_move = game->ponder_move;

   /* Verify that the ponder move is legal.
    * There is a corner-case where the ponder move is illegal if the game
    * was terminated by 50-move or three-fold repetition.
    * There may be a more elegant way to resolve this, but this works.
    */
   movelist_t movelist;
   generate_moves(&movelist, game->board, game->side_to_move);
   game->pondering = false;
   for (int n = 0; n<movelist.num_moves; n++) {
      if (movelist.move[n] == ponder_move) {
         game->pondering = true;
         break;
      }
   }
   if (!game->pondering)
      return false;
   playmove(game, game->ponder_move);

   /* Search the current position */
   void *old_keyboard_handler = game->check_keyboard;
   void *old_clock_handler = game->check_clock;
   game->check_keyboard = interrupt_ponder;
   set_ponder_timer(game);

   if (computer_play(game, MAX_SEARCH_DEPTH))
      takeback(game);

   game->check_keyboard = old_keyboard_handler;
   game->check_clock = old_clock_handler;

   takeback(game);
   game->pondering = false;

   game->ponder_move = ponder_move;

   /* Report whether we stopped pondering because of keyboard input, or
    * because the main thinking loop terminated (as in the case of a legal
    * draw or a forced mate).
    */
   if (!keyboard_input_waiting())
      game->ponder_move = 0;
   return true;
}

bool computer_analyse(gamestate_t *game)
{
   game->pondering = true;

   /* Search the current position */
   void *old_keyboard_handler = game->check_keyboard;
   void *old_clock_handler = game->check_clock;

   /* TODO: we need a timer that checks for keyboard input and either
    * handles it directly or tells the search to drop to the root first
    * ("abort") and handle it there.
    */
   set_ponder_timer(game);

   stop_analysing = false;
   while (game->analysing && !stop_analysing) {
      computer_play(game, MAX_SEARCH_DEPTH);
      check_keyboard(game);
   }

   game->check_keyboard = old_keyboard_handler;
   game->check_clock = old_clock_handler;

   game->pondering = false;
   game->ponder_move = 0;
   return true;
}
