/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdint.h>
#include "bitboard.h"
#include "board.h"
#include "game.h"
#include "pawns.h"
#include "config.h"
#include "evaluate.h"

#undef PRINT_PAWN_EVALUATION_BITBOARDS
#undef DEBUG_EVALUATION

#ifdef DEBUG_EVALUATION

#define WRITE_LOG(...) printf(__VA_ARGS__)
#define WRITE_SQUARES(bb) print_bitboard_squares(bb)

#else

#define WRITE_LOG(...) (void)0
#define WRITE_SQUARES(bb) (void)0

#endif


/* Penalty for isolated pawns. The base index is the number of isolated
 * pawns, but we add one extra point for pawns that are on semi-open files
 * and one extra point for pawns that can't advance (because their stop
 * square is attecked by an enemy pawn).
 * Of course, there can never be 8 isolated pawns, so we're being a bit
 * pedantic here.
 */
static const int isolated_penalty[] = {
   0, 12, 24, 36, 48, 60, 72, 84, 96, 108, 120
// 0, 12, 30, 48, 72, 72, 84, 84, 96, 108, 120
};

const int backward_penalty[] = {
   0, 7, 14, 21, 28, 35, 42, 49, 56, 63, 70
// 0, 7, 17, 28, 35, 35, 49, 49, 56, 63, 70
};

static const int candidate_bonus[] = {
   0, 5, 10, 15, 30, 50, 50, 0
};

/* Piece square table for pawns */
static const int pawn_psq[] = {
   -15,  0,  0,  0,  0,  0,  0, -15,
    -5,  5,  5,  0,  0,  5,  5,  -5,
   -15,  0,  0,  0,  0,  0,  0, -15,
   -15,  0,  0, 20, 20,  0,  0, -15,
   -15,  0,  0,  0,  0,  0,  0, -15,
   -15,  0,  0,  0,  0,  0,  0, -15,
    -5,  5,  5,  0,  0,  5,  5,  -5,
   -15,  0,  0,  0,  0,  0,  0, -15,
};

static void print_uint8_t(uint8_t b) __attribute__((unused));
static void print_uint8_t(uint8_t b)
{
   int c;
   
   for (c=0; c<8; c++) {
      printf ("%d", (b>>c)&1);
   }
}

static inline size_t map_key_to_index(uint64_t key, size_t nelem)
{
   return key & (nelem - 1);
}

pawn_hash_table_t *create_pawn_hash_table(size_t nelem)
{
   pawn_hash_table_t *table = calloc(1, sizeof *table);

   table->number_of_elements = nelem;
   table->data = calloc(nelem, sizeof *table->data);
   return table; 
}

void destroy_pawn_hash_table(pawn_hash_table_t *table)
{
   if (table) {
      free(table->data);
      free(table);
   }
}

pawn_structure_t *query_pawn_table_entry(pawn_hash_table_t *table, uint64_t key)
{
   size_t index;

   if (!table)
      return NULL;

   /* Map the key onto the array index, check if entry is there */
   index = map_key_to_index(key, table->number_of_elements);

   if (table->data[index].key == key)
      return table->data + index;
   else
      return NULL;
}

void prefetch_pawntable(pawn_hash_table_t *table, uint64_t key)
{
   size_t index;

   index = map_key_to_index(key, table->number_of_elements);
   __builtin_prefetch(table->data+index);
}

/* Find all positions that white pawns may still advance to */
static inline bitboard_t blocked_span_white(bitboard_t pawns, bitboard_t blockers)
{
   bitboard_t free, span;
   free = ~blockers;

   span = pawns;
   pawns = (pawns << 8) & free;
   span |= pawns;
   pawns = (pawns << 8) & free;
   span |= pawns;
   pawns = (pawns << 8) & free;
   span |= pawns;
   pawns = (pawns << 8) & free;
   span |= pawns;
   pawns = (pawns << 8) & free;
   span |= pawns;
   pawns = (pawns << 8) & free;
   span |= pawns;
   return span;
}

/* Find all positions that black pawns may still advance to */
static inline bitboard_t blocked_span_black(bitboard_t pawns, bitboard_t blockers)
{
   bitboard_t free, span;
   free = ~blockers;

   span = pawns;
   pawns = (pawns >> 8) & free;
   span |= pawns;
   pawns = (pawns >> 8) & free;
   span |= pawns;
   pawns = (pawns >> 8) & free;
   span |= pawns;
   pawns = (pawns >> 8) & free;
   span |= pawns;
   pawns = (pawns >> 8) & free;
   span |= pawns;
   pawns = (pawns >> 8) & free;
   span |= pawns;
   return span;
}

static inline bitboard_t fill_north(bitboard_t bb)
{
   bb |= bb<<8;
   bb |= bb<<16;
   bb |= bb<<32;
   return bb;
}

static inline bitboard_t fill_south(bitboard_t bb)
{
   bb |= bb>>8;
   bb |= bb>>16;
   bb |= bb>>32;
   return bb;
}


#define store_pawn_structure()\
{                              \
   int n;   \
   int index = map_key_to_index(board->pawn_hash,\
                        game->pawn_structure->number_of_elements);\
   pawn_structure = game->pawn_structure->data+index;\
   \
   pawn_structure->key = board->pawn_hash;\
   pawn_structure->score = ev;\
   pawn_structure->free_pawns_mg = free_pawns_mg;\
   pawn_structure->free_pawns_eg = free_pawns_eg;\
   pawn_structure->open_files = open_files;\
   pawn_structure->strong_squares[0] = ( (isolated_pawns[1] | backward_pawns[1]) >> 8 ) & board_black;\
   pawn_structure->strong_squares[1] = ( (isolated_pawns[0] | backward_pawns[0]) << 8 ) & board_white;\
   pawn_structure->doubled_pawns = board_empty;\
   pawn_structure->weak = board_empty;\
   pawn_structure->connected_free = board_empty;\
   pawn_structure->free = board_empty;\
   for (n=0; n<2; n++) {\
      pawn_structure->half_open_files[n] = half_open_files[n];\
      pawn_structure->connected_free |= free_pawn_span[n]&free_pawns[n];\
      pawn_structure->doubled_pawns |= doubled_pawns[n];\
      pawn_structure->free |= free_pawns[n];\
      pawn_structure->weak |= isolated_pawns[n]|backward_pawns[n]|advanced_pawns[n];\
      pawn_structure->outposts[n] =\
               (attack_span[n] & pawn_structure->strong_squares[n])|\
               (attack[n] & ~defend_span[1-n]);\
   }\
}

bitboard_t get_passed_pawns(const board_t *board)
{
   bitboard_t pawns[2], span[2], attack_span[2], doubled_pawns[2], open[2];
   bitboard_t left_attack_span, right_attack_span;

   /* Construct white and black pawn boards */
   pawns[0] = board->bbp[PAWN] & board->bbc[0];
   pawns[1] = board->bbp[PAWN] & board->bbc[1];

   /* Construct forward black/white spans and attack spans: these are
    * squares on a file containing a pawn, or that are adjacent to a file
    * containing a pawn (forward attack files).
    * FIXME: there are different classes of weak pawn, all of them
    * identified separately. In fact, the way we detect advanced pawns
    * also detects all of the other types, so we may be able to
    * simplify the code a bit.
    */
   span[0] = fill_north(pawns[0]);
   span[1] = fill_south(pawns[1]);

   left_attack_span     = (span[0]&board_minus_a)<<7;
   right_attack_span    = (span[0]&board_minus_h)<<9;
   attack_span[0]       = right_attack_span | left_attack_span;

   left_attack_span     = (span[1]&board_minus_h)>>7;
   right_attack_span    = (span[1]&board_minus_a)>>9;
   attack_span[1]       = right_attack_span | left_attack_span;

   /* Find doubled (or tripled) pawns. We mark the *backward* member of
    * the doubled pawn.
    */
   doubled_pawns[0] = pawns[0] & fill_south(pawns[0] >> 8);
   doubled_pawns[1] = pawns[1] & fill_north(pawns[1] << 8);

   /* Open pawns: pawns that don't have a pawn between them and the
    * promotion square (back members of double pawns are never open)
    */
   open[0] = pawns[0] & ~(span[1] | doubled_pawns[0]);
   open[1] = pawns[1] & ~(span[0] | doubled_pawns[1]);

   /* Find passed pawns: any pawns that can never be captured or
    * stopped by an enemy pawn.
    */
   return (open[0] & ~attack_span[1]) | (open[1] & ~attack_span[0]);
}

/* Evaluate pawn structure, from the point-of-view of *white*
 * Pawn structure evaluation terms:
 *  - Passed (free) pawns
 *  - Candidate passed pawns
 *  - Isolated pawns
 *  - Backward pawns
 */
pawn_structure_t *evaluate_pawn_structure(gamestate_t *game)
{
   pawn_structure_t *pawn_structure;
   bitboard_t pawns[2];
   bitboard_t duo[2];
   bitboard_t backward_pawns[2];
   bitboard_t isolated_pawns[2];
   bitboard_t advanced_pawns[2];
   bitboard_t connected_pawns[2];
   bitboard_t doubled_pawns[2];
   bitboard_t back_double_span[2];
   bitboard_t free_pawns[2];
   bitboard_t free_pawn_span[2];
   bitboard_t span[2];
   bitboard_t left_attack_span[2];
   bitboard_t right_attack_span[2];
   bitboard_t attack[2];
   bitboard_t attack_span[2];
   bitboard_t defend_span[2];
   bitboard_t back_attack_span[2];
   bitboard_t safe_span[2];
   bitboard_t stop;
   bitboard_t bb;
   bitboard_t semi_open_bb;
   bitboard_t open[2];
   bitboard_t candidates[2];
   board_t *board;
   uint8_t pawn_files[2];
   uint8_t open_files;
   uint8_t half_open_files[2];
   int n;
   int index;
   int ev = 0;
   int centre_ev, wing_ev;
   int free_pawns_mg = 0;
   int free_pawns_eg = 0;

   assert(game);
   board = game->board;

   pawn_structure =
      query_pawn_table_entry(game->pawn_structure, board->pawn_hash);

   if (!pawn_structure) {
      WRITE_LOG("Asses pawn structure:\n");

      /* Construct white and black pawn boards */
      pawns[0] = board->bbp[PAWN] & board->bbc[0];
      pawns[1] = board->bbp[PAWN] & board->bbc[1];

      /* Piece square tables; mainly to penalise pawns on the edge of the
       * board and encourage centre pawns to advance.
       */
      /* Central pawns */
      ev += popcount64(pawns[0] & board_centre) * 20;
      ev -= popcount64(pawns[1] & board_centre) * 20;
      WRITE_LOG("  Centralised pawns: ");
      WRITE_SQUARES(board->bbp[PAWN] & board_centre); WRITE_LOG("\n");

      /* Minor bonus: pawns that support the central pawns. Be careful to
       * not wreck the king's pawn shelter.
       * Don't award this bonus if the centre pawn hs not advanced,
       * otherwise we might trap the knight.
       */
      if ((pawns[0] & (D3_MASK|D4_MASK))) ev += popcount64(pawns[0] & C3_MASK) * 5;
      if ((pawns[1] & (D6_MASK|D5_MASK))) ev -= popcount64(pawns[1] & C6_MASK) * 5;

      ev += popcount64(pawns[0] & C4_MASK) * 10;
      ev -= popcount64(pawns[1] & C5_MASK) * 10;
      centre_ev = ev;

      /* Rook pawns get a penalty */
      ev -= popcount64(pawns[0] & (board_afile | board_hfile)) * 15;
      ev += popcount64(pawns[1] & (board_afile | board_hfile)) * 15;

      /* Encourage wing pawns to remain where they are, to get some more
       * protection for the king.
       * FIXME: this should not be a general evaluation term, but should be
       * handled by the king safety code.
       */
      //ev += popcount64(pawns[0] & (board_rank2 | G3_MASK) & ~(board_efile|board_dfile)) * 5;
      //ev -= popcount64(pawns[1] & (board_rank7 | G6_MASK) & ~(board_efile|board_dfile)) * 5;
      wing_ev = ev;


      /* Find open and semi-open files.
       * Collapse all rows into one row to get file masks.
       */
      pawn_files[0] = fill_south(pawns[0]);
      pawn_files[1] = fill_south(pawns[1]);
      open_files = ~(pawn_files[0]|pawn_files[1]);
      half_open_files[0] = (~pawn_files[0]) & ~open_files;
      half_open_files[1] = (~pawn_files[1]) & ~open_files;

      /* Bitboard of semi-open files, for awarding bonuses/penalties for
       * weak pawns.
       * It's ok to put these on one BB, since there can only be a pawn for
       * any one colour on a semi open file.
       */
      semi_open_bb = half_open_files[0] | half_open_files[1];
      semi_open_bb = fill_north(semi_open_bb);

      /* Construct forward black/white spans and attack spans: these are
       * squares on a file containing a pawn, or that are adjacent to a file
       * containing a pawn (forward attack files).
       * FIXME: there are different classes of weak pawn, all of them
       * identified separately. In fact, the way we detect advanced pawns
       * also detects all of the other types, so we may be able to
       * simplify the code a bit.
       */
      span[0] = fill_north(pawns[0]);
      span[1] = fill_south(pawns[1]);

      attack[0] = (pawns[0]&board_minus_a)<<7 | (pawns[0]&board_minus_h)<<9;
      attack[1] = (pawns[1]&board_minus_h)>>7 | (pawns[1]&board_minus_a)>>9;

      left_attack_span[0]  = (span[0]&board_minus_a)<<7;
      right_attack_span[0] = (span[0]&board_minus_h)<<9;
      attack_span[0]       = right_attack_span[0] | left_attack_span[0];

      left_attack_span[1]  = (span[1]&board_minus_h)>>7;
      right_attack_span[1] = (span[1]&board_minus_a)>>9;
      attack_span[1]       = right_attack_span[1] | left_attack_span[1];

      /* Calculate backward extension of attack spans (needed to distinguish
       * between isolated and backward pawns.
       */
      back_attack_span[0] = fill_south(attack_span[0]);
      back_attack_span[1] = fill_north(attack_span[1]);

      /* Safe span: the squares pawns can reach unopposed. */
      safe_span[0] = blocked_span_white(pawns[0], pawns[1] | attack[1]);
      safe_span[1] = blocked_span_white(pawns[1], pawns[0] | attack[0]);

      /* Defend spans: find all pawns that are or could be defended by a
       * friendly pawn. Pawns that don't fall in this category are
       * isolated, backward, or maybe advanced too far.
       */
      bb = blocked_span_white(pawns[0], pawns[1]);
      defend_span[0] = shift_bitboard_left_up(bb) | shift_bitboard_right_up(bb);
      bb = blocked_span_black(pawns[1], pawns[0]);
      defend_span[1] = shift_bitboard_left_down(bb) | shift_bitboard_right_down(bb);

      /* Find doubled (or tripled) pawns. We mark the *backward* member of
       * the doubled pawn.
       */
      doubled_pawns[0] = pawns[0] & fill_south(pawns[0] >> 8);
      doubled_pawns[1] = pawns[1] & fill_north(pawns[1] << 8);
      WRITE_LOG("  Doubled pawns:     ");
      WRITE_SQUARES(doubled_pawns[0]); WRITE_SQUARES(doubled_pawns[1]); WRITE_LOG("\n");

      /* Open pawns: pawns that don't have a pawn between them and the
       * promotion square (back members of double pawns are never open)
       */
      open[0] = pawns[0] & ~(span[1] | doubled_pawns[0]);
      open[1] = pawns[1] & ~(span[0] | doubled_pawns[1]);

      /* Find passed pawns: any pawns that can never be captured or
       * stopped by an enemy pawn.
       */
      free_pawns[0] = open[0] & ~attack_span[1];
      free_pawns[1] = open[1] & ~attack_span[0];
      WRITE_LOG("  Passed pawns:      ");
      WRITE_SQUARES(free_pawns[0]); WRITE_SQUARES(free_pawns[1]); WRITE_LOG("\n");

      /* Span of passed pawns, useful to find connected passed pawns */
      free_pawn_span[0] = fill_north(free_pawns[0]) | fill_south(free_pawns[0]);
      free_pawn_span[0] = ((free_pawn_span[0]&board_minus_h)>>7) | ((free_pawn_span[0]&board_minus_a)>>9);

      free_pawn_span[1] = fill_north(free_pawns[1]) | fill_south(free_pawns[1]);
      free_pawn_span[1] = ((free_pawn_span[1]&board_minus_h)>>7) | ((free_pawn_span[1]&board_minus_a)>>9);

      /* duos */
      duo[0] = (pawns[0]&((pawns[0]&board_minus_a)<<1)) | (pawns[0]&((pawns[0]&board_minus_h)>>1));
      duo[1] = (pawns[1]&((pawns[1]&board_minus_a)<<1)) | (pawns[1]&((pawns[1]&board_minus_h)>>1));

      /* Candidates: a pawn on a semi-open file that is not free with more
       * defenders than opposing pawns.
       * Get potential candidates.
       */
      candidates[0] = open[0] & ~free_pawns[0];
      candidates[1] = open[1] & ~free_pawns[1];

      /* Check potential candidates: it is only a true candidate if there
       * are pawns behind it *and* it has more defenders than attackers on
       * its stop square.
       */
      stop = fill_north(candidates[0]) & attack[1];
      while (stop) {
         int square = bitscan64(stop);
         stop ^= make_bitboard_square(square);
         int attackers = popcount64(north_pawn_attack[square] & pawns[1]);
         int defenders = popcount64(south_pawn_attack[square] & safe_span[0]);
         if (defenders < attackers) {  /* Not a candidate... unless we are dealing with a hidden passer */
            candidates[0] &= ~board_file[unpack_file(square)];
         }
      }
      stop = fill_south(candidates[1]) & attack[0];
      while (stop) {
         int square = bitscan64(stop);
         stop ^= make_bitboard_square(square);
         int attackers = popcount64(south_pawn_attack[square] & pawns[0]);
         int defenders = popcount64(north_pawn_attack[square] & safe_span[1]);
         if (defenders < attackers) {  /* Not a candidate... unless we are dealing with a hidden passer */
            candidates[1] &= ~board_file[unpack_file(square)];
         }
      }

      WRITE_LOG("  Candidate pawns:   ");
      WRITE_SQUARES(candidates[0]); WRITE_SQUARES(candidates[1]); WRITE_LOG("\n");

      /* Find isolated pawns: pawns that have no friendly pawns on
       * neighbouring files.
       * As calculated here, this includes backward pawns: pawns that have no
       * friendly pawns behind them, but have friendly pawns ahead.
       * We need to consider pawns next to eachother explicitly, otherwise a
       * duo (say) will be classified as a connected pair of isolated pawns
       * (!)
       * The last pawn on a doubled pawn is never backward.
       */
      isolated_pawns[0] = pawns[0] & ~(attack_span[0]|attack_span[0]>>8 | doubled_pawns[0]);
      isolated_pawns[1] = pawns[1] & ~(attack_span[1]|attack_span[1]<<8 | doubled_pawns[1]);

      /* Backward pawns */
      backward_pawns[0] = isolated_pawns[0] & back_attack_span[0];
      backward_pawns[1] = isolated_pawns[1] & back_attack_span[1];

      /* Backward pawns are not isolated, although they are also weak */
      isolated_pawns[0] ^= backward_pawns[0];
      isolated_pawns[1] ^= backward_pawns[1];
      WRITE_LOG("  Isolated pawns:    ");
      WRITE_SQUARES(isolated_pawns[0]); WRITE_SQUARES(isolated_pawns[1]); WRITE_LOG("\n");

      /* Backward pawns are only considered weak if they cannot advance to
       * resolve their backwardness.
       * They *could* still get a smaller penatly though.
       */
      stop = (~attack_span[0]) & attack[1];
      backward_pawns[0] &= fill_south(stop);
      stop = (~attack_span[1]) & attack[0];
      backward_pawns[1] &= fill_south(stop);
      WRITE_LOG("  Backward pawns:    ");
      WRITE_SQUARES(backward_pawns[0]); WRITE_SQUARES(backward_pawns[1]); WRITE_LOG("\n");

      /* Connected pawns */
      connected_pawns[0] = pawns[0] ^ isolated_pawns[0];
      connected_pawns[1] = pawns[1] ^ isolated_pawns[1];

      /* Pawns that have advanced too far (but are not passed) are also weak
       * ("doorgeschoven pionnen")
       * These are like backward pawns in that they can no longer be
       * defended by a friendly pawn.
       */
      advanced_pawns[0] = pawns[0] & ~defend_span[0] & ~(backward_pawns[0]|isolated_pawns[0]);
      advanced_pawns[0] &= ~(shift_bitboard_file_left(pawns[0])|shift_bitboard_file_right(pawns[0]));
      advanced_pawns[1] = pawns[1] & ~defend_span[1] & ~(backward_pawns[1]|isolated_pawns[1]);
      advanced_pawns[1] &= ~(shift_bitboard_file_left(pawns[1])|shift_bitboard_file_right(pawns[1]));
      WRITE_LOG("  Advanced pawns:    ");
      WRITE_SQUARES(advanced_pawns[0]); WRITE_SQUARES(advanced_pawns[1]); WRITE_LOG("\n");

      duo[0] &= ~(span[0]^pawns[0]);
      duo[1] &= ~(span[1]^pawns[1]);

      /* A "duo" is a special case of two connected pawns on the same row.
       * "Hanging pawns" is an isolated duo.
       */

#ifdef PRINT_PAWN_EVALUATION_BITBOARDS
      printf("White: "); print_uint8_t(pawn_files[0]); printf("\n");
      printf("Black: "); print_uint8_t(pawn_files[1]); printf("\n");
      printf("Open:  "); print_uint8_t(open_files); printf("\n");
      printf("Half/W:"); print_uint8_t(half_open_files[0]); printf("\n");
      printf("Half/B:"); print_uint8_t(half_open_files[1]); printf("\n");

      printf("Doubled pawns:\n");
      printf_bitboard(doubled_pawns[0]|doubled_pawns[1]);

      printf("Passed pawns:\n");
      printf_bitboard(free_pawns[0]|free_pawns[1]);

      printf("Isolated pawns:\n");
      printf_bitboard(isolated_pawns[0]|isolated_pawns[1]);

      printf("Connected pawns:\n");
      printf_bitboard(connected_pawns[0]|connected_pawns[1]);

      printf("Backward pawns:\n");
      printf_bitboard(backward_pawns[0]|backward_pawns[1]);

      printf("Duos\n");
      printf_bitboard(duo[0]|duo[1]);

      printf("Connected free pawns:\n");
      printf_bitboard((free_pawn_span[1])&free_pawns[1]);
#endif

      /* Pawn islands */
      n = sparse_popcount64(pawn_files[0] & (pawn_files[0]^(pawn_files[0]>>1)));
      WRITE_LOG("  # islands (white)                     % 6d\n", n);
      if(n) ev -= (n-1)*PAWN_ISLAND_PENALTY;
      n = sparse_popcount64(pawn_files[1] & (pawn_files[1]^(pawn_files[1]>>1)));
      WRITE_LOG("  # islands (black)                     % 6d\n", n);
      if(n) ev += (n-1)*PAWN_ISLAND_PENALTY;

      WRITE_LOG("  (pawns) centre                        % 6d\n", centre_ev);
      WRITE_LOG("  (pawns) wing                          % 6d\n", wing_ev);
      WRITE_LOG("  (pawns) islands:                      % 6d\n", ev);

      /* Pawns that have advanced onto the opponents half of the board */
      ev += popcount64(pawns[0] & board_rank[3]) * 1;
      ev += popcount64(pawns[0] & board_rank[4]) * 2;
      ev += popcount64(pawns[0] & board_rank[5]) * 8;
      ev += popcount64(pawns[0] & board_rank[6]) * 24;

      ev -= popcount64(pawns[1] & board_rank[4]) * 1;
      ev -= popcount64(pawns[1] & board_rank[3]) * 2;
      ev -= popcount64(pawns[1] & board_rank[2]) * 8;
      ev -= popcount64(pawns[1] & board_rank[1]) * 24;
      WRITE_LOG("  (pawns) advanced:                     % 6d\n", ev);

      /* Pawn chains/defended pawns */
      ev += popcount64(attack[0] & pawns[0]);
      ev -= popcount64(attack[1] & pawns[1]);
      WRITE_LOG("  (pawns) defended:                     % 6d\n", ev);

      /* Passed pawns */
      n =  sparse_popcount64(free_pawns[0]);
      free_pawns_mg += n*FREE_PAWN_BONUS_MG;
      free_pawns_eg += n*FREE_PAWN_BONUS_EG;
      n =  sparse_popcount64(free_pawns[1]);
      free_pawns_mg -= n*FREE_PAWN_BONUS_MG;
      free_pawns_eg -= n*FREE_PAWN_BONUS_EG;

      /* Free pawns that are close to promotion are worth more */
      /* FIXME: free pawns become more valuable in the END GAME */
      bb = free_pawns[0];
      while (bb) {
         int p = bitscan64(bb);
         int dist;

         unset_bitboard(&bb, p);
         dist = unpack_rank(p);

         /* FIXME: should correct for the other king being in the "square"
          * (ie, having a shorter Manhatten-distance to the promotion square.
          */
         free_pawns_mg += dist*FREE_PAWN_ADVANCE_BONUS_MG;
         free_pawns_eg += (dist + (dist == 6)) * FREE_PAWN_ADVANCE_BONUS_EG;
      }

      /* Free pawns that are close to promotion are worth more */
      bb = free_pawns[1];
      while (bb) {
         int p = bitscan64(bb);
         int dist;

         unset_bitboard(&bb, p);
         dist = 7 - unpack_rank(p);

         /* FIXME: should correct for the other king being in the "square"
          * (ie, having a shorter Manhatten-distance to the promotion square.
          */
         free_pawns_mg -= dist*FREE_PAWN_ADVANCE_BONUS_MG;
         free_pawns_eg -= (dist + (dist == 6)) * FREE_PAWN_ADVANCE_BONUS_EG;
      }

      /* Extra bonus: connected free pawns */
      //ev += sparse_popcount64(connected_pawns[0]&free_pawns[0])*50;
      //ev -= sparse_popcount64(connected_pawns[1]&free_pawns[1])*50;

      /* Candidate passers get a bonus */
      /* TODO */
      bb = candidates[0];
      while(bb) {
         int square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         ev += candidate_bonus[unpack_rank(square)];
      }
      bb = candidates[1];
      while(bb) {
         int square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         ev -= candidate_bonus[7-unpack_rank(square)];
      }
      WRITE_LOG("  (pawns) candidate                     % 6d\n", ev);

      /* Weak pawns: doubled pawns */
      //ev -= sparse_popcount64(doubled_pawns[0])*12;
      //ev += sparse_popcount64(doubled_pawns[1])*12;
      //ev += sparse_popcount64(doubled_pawns[0]&free_pawns[0])*10;
      //ev -= sparse_popcount64(doubled_pawns[1]&free_pawns[1])*10;

      /* FIXME: for weak doubled pawns, only award these penalties once */
      /* Weak pawns: isolated pawns
       * These are progressively weaker if they are on semi-open files and
       * unable to advance because their stop square is under attack.
       */
      index = sparse_popcount64(isolated_pawns[0]&~free_pawns[0]);
      //if (isolated_pawns[0]&~free_pawns[0]&semi_open_bb) index++;
      ev -= isolated_penalty[index];

      index = sparse_popcount64(isolated_pawns[1]&~free_pawns[1]);
      //if (isolated_pawns[1]&~free_pawns[1]&semi_open_bb) index++;
      ev += isolated_penalty[index];
      WRITE_LOG("  (pawns) isolated                      % 6d\n", ev);

      /* Weak pawns: backward pawns
       * These leave a gaping hole in the position; not so bad if there's a
       * bishop protecting it, but we're better off giving a big penalty
       * here and then compensating for that when a bishop is present.
       * We should only allow the penalty for the forward-most pawn on each
       * line, however (in case of double pawns).
       */
      back_double_span[0] = doubled_pawns[0]; 
      back_double_span[0] |= back_double_span[0]>>8; 
      back_double_span[0] |= back_double_span[0]>>16; 
      back_double_span[0] |= back_double_span[0]>>32; 
      back_double_span[0] <<= 8;

      back_double_span[1] = doubled_pawns[1]; 
      back_double_span[1] |= back_double_span[1]<<8; 
      back_double_span[1] |= back_double_span[1]<<16; 
      back_double_span[1] |= back_double_span[1]<<32; 
      back_double_span[1] >>= 8;

      index = sparse_popcount64(backward_pawns[0] & ~back_double_span[0]);
      //if (backward_pawns[0] & ~back_double_span[0] & semi_open_bb) index++;
      //index = sparse_popcount64(backward_pawns[0] & ~back_double_span[0] & semi_open_bb);
      ev -= backward_penalty[index];

      index = sparse_popcount64(backward_pawns[1] & ~back_double_span[1]);
      //if (backward_pawns[1] & ~back_double_span[1] & semi_open_bb) index++;
      //index = sparse_popcount64(backward_pawns[1] & ~back_double_span[1] & semi_open_bb);
      ev += backward_penalty[index];
      WRITE_LOG("  (pawns) backward                      % 6d\n", ev);

      /* Pawns that have advanced too much. These are the opposite of
       * "backward", but they get the same type of penalty, since they are
       * weak for similar reasons (and also because this is the smaller of
       * the two penalties).
       * We discard rook pawns from this, since they're already weak and we
       * don't want to count double for them (also because there's one side
       * less from which they can be attacked, and this seems to be better
       * in practice).
       */
      index = sparse_popcount64(advanced_pawns[0] & ~(board_afile|board_hfile));
      ev -= backward_penalty[index];
      index = sparse_popcount64(advanced_pawns[1] & ~(board_afile|board_hfile));
      ev += backward_penalty[index];
      WRITE_LOG("  (pawns) advanced                      % 6d\n", ev);

      store_pawn_structure();
   }

   return pawn_structure;
}

