/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include "pieces.h"
#include "movegen.h"
#include "board.h"
#include "see.h"
#include "bits.h"

#undef PRINT_SEE_DEBUG_MESSAGES

static inline int max(int x, int y)
{
   return (x>y)?x:y;
}

static inline int find_least_valued_piece(const board_t *board, bitboard_t own)
{
   int n;

   for (n=0; n<5; n++)
      if (board->bbp[n] & own)
         return bitscan64(board->bbp[n] & own);

   /* Final option: king */
   return bitscan64(board->bbp[KING] & own);
}

/* Perform static exchange evaluation for a given capture move.
 * Uses the "swap" algorithm
 * (http://chessprogramming.wikispaces.com/SEE+-+The+Swap+Algorithm).
 * returns the evaluation in centipawns.
 * The move doesn't need to be a capture; if it isn't then this function
 * can be used to find out if it puts a piece "en-prise"
 */
int static_exchange_evaluation(const board_t *board, const move_t move)
{
   int score[32];
   int depth;
   int side = get_move_player(move);
   int piece = get_move_piece(move);
   int square = get_move_destination(move);
   int origin = get_move_origin(move);
   int last_piece = piece;
   bitboard_t attackers;
   bitboard_t xray_update;
   bitboard_t own;
   bitboard_t mask = board_all;

   /* Initial gain: the piece on the target square */
   depth = 0;
   score[depth] = 0;
   if (is_capture_move(move)) {
      int ptaken = get_captured_piece(move);
      score[depth] = piece_value(ptaken);

      /* Easy: if the piece we use to capture the enemy piece is worth less
       * than the piece we captured, it's always a winning capture, so return a
       * rough estimate for the SEE score.
       */
      if (score[depth] > piece_value(piece))
         return score[depth] - piece_value(piece)/2;
   }

   /* We need to update the list of attackers if the moving piece is a
    * slider or a pawn.
    * This is only necessary if there is a slider attacking one of
    * these squares, but that test by itself is more expensive than simply
    * updating the board as necessary.
    */
   xray_update = board->bbp[PAWN]|board->bbp[BISHOP]|board->bbp[ROOK]|board->bbp[QUEEN];

   if (is_capture_move(move)) {
      /* Perform the capture on the board.
       * In this case we need to explicitly clear the capture square because
       * the first move might be an en-passant capture.
       */
      mask &= ~make_bitboard_square(get_capture_square(move));
   }

#ifdef PRINT_SEE_DEBUG_MESSAGES
   printf("SEE: %d %s %02x\n", depth, side?"BLACK":"White", piece);
#endif

   /* Get a new list of all attackers/defenders */
   mask &= ~make_bitboard_square(origin);
   attackers = get_all_attackers(board, mask, square);

   do {
      /* Consider next move - one ply deeper, flip side */
      depth++;
      side ^= BLACK;
      /* Update score (speculative) */
      score[depth] = piece_value(last_piece) - score[depth-1];
      /* Perform LVA capture */
      own = attackers&board->bbc[side>>7];
      if (own) {
         /* Find the least valued attacker */
         origin = find_least_valued_piece(board, own);
         piece = get_piece(board, origin);
         last_piece = piece;

#ifdef PRINT_SEE_DEBUG_MESSAGES
         printf("SEE: %d %s %02x %d\n", depth, side?"Black":"White", piece&PIECE, piece_value(piece));
#endif

         /* Update the list of attackers/defenders */
         attackers &= ~make_bitboard_square(origin);
         mask &= ~make_bitboard_square(origin);
         if (get_bitboard(&xray_update, origin))
            attackers = get_all_attackers(board, mask, square);

         /* If the piece is a king, the list of attackers should be empty
          * now, otherwise we just performed an illegal capture (putting
          * the king in check), meaning we should terminate the evaluation
          * at this point. Since the king is always the last piece we check
          * for each side, any remaining attackers belong to the
          * opposing side.
          */
         if ( ((piece & PIECE) == KING) && attackers ) {
            break;
         }
      }
   } while(own);

   /* Compute final score */
   while (--depth) {
      score[depth-1] = -max(-score[depth-1], score[depth]);
   }
   return score[depth];
}

