/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>

#include "jazz.h"

#ifndef SVNVERSION
#define SVNVERSION "unknown revision "
#endif
#define VERSIONSTR SVNVERSION

static int get_move_index(movelist_t *movelist, char *s)
{
   int n;
   for (n=0; n<movelist->num_moves; n++) {
      char *move = short_move_string(movelist->move[n], movelist, NULL);
      if (strstr(s, move)) return n;
   }
   return -1;
}

int main(int argc, char **argv)
{
   gamestate_t *game = NULL;
   movelist_t movelist;
   char *str;
   const char *filename = "data/epd/wac.epd";
   FILE *file;
   int n;
   char *move;
   char *bm;
   int correct;
   int time_remaining;
   int number_of_positions;
   int score;
   int time_per_move = 1000;

   printf("Welcome to Jazz (version %s)\n", SVNVERSION);
   printf("Memory sizes:\n"
          " * move_t     : %lu bytes\n"
          " * bitboard_t : %lu bytes\n"
          " * hash_table_: %lu bytes\n"
          " * board_t    : %lu bytes\n",
          sizeof(move_t),
          sizeof(bitboard_t),
          sizeof(hash_table_entry_t),
          sizeof(board_t));
   printf("Size of transposition table: %lu bytes\n",
         sizeof(hash_table_entry_t) * HASH_TABLE_SIZE);

   printf("Initialising Jazz engine\n");
   initialise_jazz();

   for (int n = 1; n<argc; n++) {
      if (strstr(argv[n], "-st")) {
         if (n+1 < argc && argv[n+1][0] != '-') {
            sscanf(argv[n+1], "%d", &time_per_move);
            n++;
         }
      } else if (argv[n][0] == '-') {
         fprintf(stderr, "Unknown option: %s\n", argv[n]);
         exit(0);
      } else {
         filename = argv[n];
      }
   }

   if (!filename) {
      printf("ERROR: no test-suite specified.\n");
      return 0;
   }


   file = fopen(filename, "r");
   //file = fopen("data/epd/wac.epd", "r");
   //file = fopen("data/epd/nolot.epd", "r");
   //file = fopen("data/epd/ecmgcp.epd", "r");
   if (!file) {
      printf("ERROR: cannot open file %s\n", filename);
      return 0;
   }

   printf("Running test suite %s\n", filename);

   str = malloc(65536);

   correct = 0;
   time_remaining = 0;
   number_of_positions = 0;
   score = 0;
   for (n=0; ; n++) {
      int result;
      int move_score;
      bool eof = false;
      char *s;

      eof = !fgets(str, 65536, file);
      s = strstr(str, "\n");
      if (s) *s = '\0';
      if (eof || !strlen(str))
         break;

      game = load_fen_position(str);
      if (!game) {
         printf("could not load file\n");
         exit(0);
      }
      generate_moves(&movelist, game->board, game->side_to_move);

      /* Initialise partial scores for moves */
      int move_points[movelist.num_moves];
      memset(move_points, 0, sizeof move_points);

      /* Best move */
      bm = strstr(str, "bm");
      char bestmove[256] = { 0 };
      if (bm) {
         bm += 3;
         int n = 0;
         while (bm[n] != ';' && bm[n]) n++;
         char s[n+1];
         memcpy(s, bm, n);
         s[n] = 0;
         n--;
         memcpy(bestmove, s, strlen(s));
         while(n >0) {
            if (s[n] == ';') {
               s[n] = 0;
               n--;
               continue;
            }
            if (s[n] == ' ') {
               s[n] = 0;
               n--;
               continue;
            }
            break;
         }
         char *p = s;
         while (*p) {
            char move[20];
            for (int n = 0; p[n] && p[n] != ' ' && n<19; n++) {
               move[n] = p[n];
               move[n+1] = '\0';
            }
            int index = get_move_index(&movelist, move);
            printf("%s %s %d\n", p, move, index);
            if (index >= 0) {
               move_points[index] = 10;
            }

            while (*p && *p != ' ') p++;
            while (*p && *p == ' ') p++;
         }
      }

      /* Partial scores */
      bm = strstr(str, "c0 \"");
      if (bm) {
         bm += 4;
         int n = 0;
         while (bm[n] != '"' && bm[n]) n++;
         char s[n+1];
         memcpy(s, bm, n);
         s[n] = 0;

         move = s;
         while (move && move[0]) {
            while (*move == ' ') move++;
            while (*move == ',') move++;
            while (*move == ' ') move++;
            char *eq = strstr(move, "=");
            if (!eq) break;
            *eq = 0;
            eq++;
            int score;
            sscanf(eq, "%d", &score);
            int index = get_move_index(&movelist, move);
            if (index >= 0) {
               //printf("%d %s %d\n", index, move, score);
               move_points[index] = score;
            }
            move = strstr(eq, ",");
         }
      }

      printf("\n--------------------------------------------------------\n");
      printf("Test position %d\n%s\n", n+1, str);

      /* Interrupt the computer after 10 seconds */
      set_time_per_move(game, time_per_move);
      computer_play(game, 40);
      /* cancel the alarm if it hasn't gone off yet */
      time_remaining += peek_timer(game);

      move = short_move_string(game->move_list[0], &movelist, NULL);

      /* FIXME: replace with a properly parsed list for all moves, as well
       * as their partial score.
       */
      result = 0;
      move_score = 0;
      for (int k = 0; k<movelist.num_moves; k++) {
         if (movelist.move[k] == game->move_list[0]) {
            move_score = move_points[k];
            result = (move_score == 10);
            break;
         }
      }

      score += move_score;
      correct+=result;
      number_of_positions++;

      if (bm) {
         char *p;
         p = strstr(bm, "+");
         if (p) *p =' ';
         p = strstr(bm, ";");
         if (p) *p ='\0';
      } else {
         bm = "???";
      }

      printf("Position %d, selected move %s, best move %s (%s (%d/10), %d/%d %d)\n",
            n+1, move, bestmove, result?"correct":(move_score?"sub-optimal":"wrong"), move_score, correct, number_of_positions, score);
      end_game(game);
   }
   printf("All done, %d milliseconds remaining.\n", time_remaining);

   return 0;
}
