/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <ctype.h>
#include <time.h>
#include <inttypes.h>

#include "jazz.h"
#include "keypressed.h"

#ifdef __APPLE__
#define __unix__
#endif
#ifdef __unix__
#include "signal.h"
#endif

#if defined __LP64__ || defined _WIN64
#define ARCHSTR "(x86_64)"
#elif defined __i386__ || defined _WIN32
#define ARCHSTR "(i386)"
#elif defined POWERPC
#define ARCHSTR "(powerpc)"
#else
#define ARCHSTR "(unknown)"
#endif 

#ifndef SVNVERSION
#define SVNVERSION "unknown revision "
#endif
#define VERSIONSTR SVNVERSION

#define TIME_BUFFER  100

typedef struct help_topic_t {
   char *topic;
   char *cmd;
   char *text;
} help_topic_t;

typedef struct position_signature_t {
   char *fen;
   int depth;
   uint64_t nodes;
} position_signature_t;

static position_signature_t perftests[] = {
   // Martin Sedlak's test positions
   // avoid illegal ep
   { "3k4/3p4/8/K1P4r/8/8/8/8 b - - 0 1",         6, 1134888 },
   { "8/8/8/8/k1p4R/8/3P4/3K4 w - - 0 1",         6, 1134888 },
   { "8/8/4k3/8/2p5/8/B2P2K1/8 w - - 0 1",         6, 1015133 },
   { "8/b2p2k1/8/2P5/8/4K3/8/8 b - - 0 1",         6, 1015133 },
   // en passant capture checks opponent: 
   { "8/8/1k6/2b5/2pP4/8/5K2/8 b - d3 0 1",         6, 1440467 },
   { "8/5k2/8/2Pp4/2B5/1K6/8/8 w - d6 0 1",         6, 1440467 },
   // short castling gives check: 
   { "5k2/8/8/8/8/8/8/4K2R w K - 0 1",            6, 661072 },
   { "4k2r/8/8/8/8/8/8/5K2 b k - 0 1",            6, 661072 },
   // long castling gives check: 
   { "3k4/8/8/8/8/8/8/R3K3 w Q - 0 1",            6, 803711 },
   { "r3k3/8/8/8/8/8/8/3K4 b q - 0 1",            6, 803711 },
   // castling (including losing cr due to rook capture): 
   { "r3k2r/1b4bq/8/8/8/8/7B/R3K2R w KQkq - 0 1",   4, 1274206 },
   { "r3k2r/7b/8/8/8/8/1B4BQ/R3K2R b KQkq - 0 1",    4, 1274206 },
   // castling prevented: 
   { "r3k2r/8/3Q4/8/8/5q2/8/R3K2R b KQkq - 0 1",   4, 1720476 },
   { "r3k2r/8/5Q2/8/8/3q4/8/R3K2R w KQkq - 0 1",   4, 1720476 },
   // promote out of check: 
   { "2K2r2/4P3/8/8/8/8/8/3k4 w - - 0 1",         6, 3821001 },
   { "3K4/8/8/8/8/8/4p3/2k2R2 b - - 0 1",         6, 3821001 },
   // discovered check: 
   { "8/8/1P2K3/8/2n5/1q6/8/5k2 b - - 0 1",         5, 1004658 },
   { "5K2/8/1Q6/2N5/8/1p2k3/8/8 w - - 0 1",         5, 1004658 },
   // promote to give check: 
   { "4k3/1P6/8/8/8/8/K7/8 w - - 0 1",            6, 217342 },
   { "8/k7/8/8/8/8/1p6/4K3 b - - 0 1",            6, 217342 },
   // underpromote to check: 
   { "8/P1k5/K7/8/8/8/8/8 w - - 0 1",            6, 92683 },
   { "8/8/8/8/8/k7/p1K5/8 b - - 0 1",            6, 92683 },
   // self stalemate: 
   { "K1k5/8/P7/8/8/8/8/8 w - - 0 1",            6, 2217 },
   { "8/8/8/8/8/p7/8/k1K5 b - - 0 1",            6, 2217 },
   // stalemate/checkmate: 
   { "8/k1P5/8/1K6/8/8/8/8 w - - 0 1",            7, 567584 },
   { "8/8/8/8/1k6/8/K1p5/8 b - - 0 1",            7, 567584 },
   // double check: 
   { "8/8/2k5/5q2/5n2/8/5K2/8 b - - 0 1",         4, 23527 },
   { "8/5k2/8/5N2/5Q2/2K5/8/8 w - - 0 1",         4, 23527 },

   // short castling impossible although the rook never moved away from its corner 
   { "1k6/1b6/8/8/7R/8/8/4K2R b K - 0 1", 5, 1063513 },
   { "4k2r/8/8/7r/8/8/1B6/1K6 w k - 0 1", 5, 1063513 },

   // long castling impossible although the rook never moved away from its corner 
   { "1k6/8/8/8/R7/1n6/8/R3K3 b Q - 0 1", 5, 346695 },
   { "r3k3/8/1N6/r7/8/8/8/1K6 w q - 0 1", 5, 346695 },

   // From the Wiki
   { "r3k2r/p1ppqpb1/bn2pnp1/3PN3/1p2P3/2N2Q1p/PPPBBPPP/R3K2R w KQkq -", 4, 4085603 },
   { "rnbqkb1r/pp1p1ppp/2p5/4P3/2B5/8/PPP1NnPP/RNBQK2R w KQkq - 0 6", 3, 53392 },

   // John Merlino's test positions, some of these take a long time, only do them
   // in debug mode.
#ifdef DEBUGMODE
   { "r3k2r/8/8/8/3pPp2/8/8/R3K1RR b KQkq e3 0 1", 6, 485647607 },
   { "r3k2r/Pppp1ppp/1b3nbN/nP6/BBP1P3/q4N2/Pp1P2PP/R2Q1RK1 w kq - 0 1", 6, 706045033 },
   { "8/7p/p5pb/4k3/P1pPn3/8/P5PP/1rB2RK1 b - d3 0 28", 6, 38633283 },
   { "8/3K4/2p5/p2b2r1/5k2/8/8/1q6 b - - 1 67", 7, 493407574 },
   { "rnbqkb1r/ppppp1pp/7n/4Pp2/8/8/PPPP1PPP/RNBQKBNR w KQkq f6 0 3", 6, 244063299 },
   { "r3k2r/p1ppqpb1/bn2pnp1/3PN3/1p2P3/2N2Q1p/PPPBBPPP/R3K2R w KQkq -", 5, 193690690 },
   { "8/p7/8/1P6/K1k3p1/6P1/7P/8 w - -", 8, 8103790 },
   { "n1n5/PPPk4/8/8/8/8/4Kppp/5N1N b - -", 6, 71179139 },
   { "r3k2r/p6p/8/B7/1pp1p3/3b4/P6P/R3K2R w KQkq -", 6, 77054993 },
#endif
   { "8/2p5/3p4/KP5r/1R3p1k/8/4P1P1/8 w - -", 7, 178633661 },
   { "8/5p2/8/2k3P1/p3K3/8/1P6/8 b - -", 8, 64451405 },
   { "r3k2r/pb3p2/5npp/n2p4/1p1PPB2/6P1/P2N1PBP/R3K2R w KQkq -", 5, 29179893 },

   { NULL }
};

/* Benchmark positions. For now we just use the Stockfish ones */
static position_signature_t benchtests[] = {
  { "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1" },
  { "r3k2r/p1ppqpb1/bn2pnp1/3PN3/1p2P3/2N2Q1p/PPPBBPPP/R3K2R w KQkq -" },
  { "8/2p5/3p4/KP5r/1R3p1k/8/4P1P1/8 w - -" },
  { "4rrk1/pp1n3p/3q2pQ/2p1pb2/2PP4/2P3N1/P2B2PP/4RRK1 b - - 7 19" },
  { "rq3rk1/ppp2ppp/1bnpb3/3N2B1/3NP3/7P/PPPQ1PP1/2KR3R w - - 7 14" },
  { "r1bq1r1k/1pp1n1pp/1p1p4/4p2Q/4Pp2/1BNP4/PPP2PPP/3R1RK1 w - - 2 14" },
  { "r3r1k1/2p2ppp/p1p1bn2/8/1q2P3/2NPQN2/PPP3PP/R4RK1 b - - 2 15" },
  { "r1bbk1nr/pp3p1p/2n5/1N4p1/2Np1B2/8/PPP2PPP/2KR1B1R w kq - 0 13" },
  { "r1bq1rk1/ppp1nppp/4n3/3p3Q/3P4/1BP1B3/PP1N2PP/R4RK1 w - - 1 16" },
  { "4r1k1/r1q2ppp/ppp2n2/4P3/5Rb1/1N1BQ3/PPP3PP/R5K1 w - - 1 17" },
  { "2rqkb1r/ppp2p2/2npb1p1/1N1Nn2p/2P1PP2/8/PP2B1PP/R1BQK2R b KQ - 0 11" },
  { "r1bq1r1k/b1p1npp1/p2p3p/1p6/3PP3/1B2NN2/PP3PPP/R2Q1RK1 w - - 1 16" },
  { "3r1rk1/p5pp/bpp1pp2/8/q1PP1P2/b3P3/P2NQRPP/1R2B1K1 b - - 6 22" },
  { "r1q2rk1/2p1bppp/2Pp4/p6b/Q1PNp3/4B3/PP1R1PPP/2K4R w - - 2 18" },
  { "4k2r/1pb2ppp/1p2p3/1R1p4/3P4/2r1PN2/P4PPP/1R4K1 b - - 3 22" },
  { "3q2k1/pb3p1p/4pbp1/2r5/PpN2N2/1P2P2P/5PP1/Q2R2K1 b - - 4 26" },

  { NULL }
};

static help_topic_t help_topic[] = {
   /* .........|.........|.........|.........|.........|.........|.........|.........| */
   { "all", NULL, NULL },

   { "analyse", "analyse, analyze",
     "  Analyse the current position.\n" },

   { "bench", "test benchmark [depth]",
     "  Perform a benchmark test to the specified depth. The returned node-count\n"
     "  can be used as a validation that the program is working correctly while\n"
     "  the reported time can be used as a performance measure on the current\n"
     "  system.\n" },

   { "board", NULL,
     "  Print the current board position.\n" },

   { "book", "book [filename|off]",
     "  Set the name of the (polyglot) opening book file to use.\n"
     "  'book off' switches off the opening book.\n" },

   { "fen", NULL,
     "  Print the FEN representation of the current board position.\n" },

   { "force", NULL,
     "  Switch to 'force' mode: Jazz will play neither side in the game until\n"
     "  it receives a 'go' command\n" },

   { "go", NULL,
     "  Switch off force mode, tell Jazz to start playing from the current position.\n" },

   { "help", "help [topic]",
     "  Display a list of help topics, or detailed information about a particular\n"
     "  topic.\n" },

   { "hint", NULL,
     "  Print the move that Jazz is pondering on. If there is no ponder move, no\n"
     "  hint is displayed.\n" },

   { "level", "level moves time inc",
     "  Set time control options: number of moves per session, time per session\n"
     "  (either in minutes or minutes:seconds) and time increment (in seconds)\n" },

   { "memory", "memory [MB]",
     "  Set the (approximate) amount of memory the program can use, in MB. The\n"
     "  actual amount of memory used will be different from this and may be\n"
     "  slightly larger or smaller.\n" },

   { "new", NULL,
     "  Start a new game from the starting position.\n" },

   { "perft", "perft [depth] [print depth]",
     "  Perform a 'perft' (performance test) on the current position: count all\n"
     "  positions that can result from this position to the specified depth.\n"
     "  If 'print depth' is specified then the total will be sub-divided per move\n"
     "  upto 'print depth'\n" },

   { "ponder", "ponder [on|off]",
     "  Switches ponder mode on or off (Jazz will think while it is not on move)\n" }, 

   { "post", "(no)post",
     "  Display thinking output (post) or not (nopost)\n" },

   { "prompt", "prompt [on|off]",
     "  Switch the prompt on or off\n" },

   { "quit", "quit, exit, bye",
     "  Exit Jazz\n" },

   { "readepd", "readepd filename",
     "  Read (and setup) a position from an EPD file.\n" },

   { "setboard", "setboard FEN",
     "  Setup a position on the board from a FEN string.\n" },

   { "sd", "sd depth",
     "  Specify the maximum search depth\n" },

   { "shell", "!command",
     "  Run a shell command.\n" },

   { "st", "st time",
     "  Specify the maximum time (in seconds) to think on a single move\n" }, 

   { "takeback", "takeback, remove",
     "  Reverses the last two moves in the game, if any.\n" }, 

   { "test", "test [movegen|benchmark [depth]]",
     "  Perform tests on the move generator, or the search.\n" },

   { "time", "time csec",
     "  Set the remaining time on the engine's clock, in centi-seconds.\n" },

   { "undo", NULL,
     "  Unmakes the last move in the game, if any. If it is Jazz' turn, it will\n"
     "  start thinking again immediately.\n" },

   { "unicode", "unicode on|off",
     "  Switches use of unicode characters to represent the chess pieces on the\n"
     "  printed board on or off. Useful for terminals that cannot display unicode\n"
     "  or that don't have the required glyphs.\n" }, 

   { "xboard", "xboard [on|off]",
     "  Switch to xboard mode: prompt off, don't display board, don't print SAN\n"
     "  moves, don't trap interrupts (ctrl-C).\n"
     "  'xboard' is equivalent to 'xboard off'\n" },

   { "MOVE", NULL,
     "  Play a move on the board. Moves can be entered as 'long algebraic'\n"
     "  (e2e4, g1f3) or SAN (e4, Nf3).\n" },

   { NULL }
};

static bool prompt = true;
static bool show_board = true;
static bool san = true;
static bool trapint = true;

#ifdef __APPLE__
#define __unix__
#endif
#ifdef __unix__
static sig_t old_signal_handler;

void interrupt_computer(int i)
{
   abort_search = true;
   if (old_signal_handler)
      old_signal_handler(i);
}
#endif


char *chomp(char *s)
{
   char *p;
   p = strstr(s, "\n");
   if (p) *p = '\0';
   return s;
}

static bool streq(const char *s1, const char *s2)
{
   return strcmp(s1, s2) == 0;
}

static FILE *f = NULL;
static char *buf;

static bool may_ponder = false;

static void log_engine_output(const char *msg, ...) __attribute__((unused));
static void log_engine_output(const char *msg, ...)
{
   va_list ap;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   if (f) {
      fprintf(f, "  %s", buf);
      fflush(f);
   }
}

static void log_xboard_output(const char *msg, ...)
{
   va_list ap;
   static bool newline = true;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   if (f) {
      if (newline)
         fprintf(f, "> ");
      newline = false;
      fprintf(f, "%s", buf);
      if (strstr(buf, "\n"))
         newline = true;
   }

   printf("%s", buf);
}

int num_games = 0;

/* Play a sequence of moves from the initial position */
bool input_move(gamestate_t *game, char *move_str)
{
   char *s;

   if (!game)
      return false;

   s = move_str;
   if(s && *s) {
      while (*s == ' ') s++;
      move_t move = move_string_to_move(game, s);
      if (move == 0) {
         fprintf(stderr, "\n");
         if (f) {
            printf("Report illegal move %s in position:\n", move_str);
            print_bitboards_file(f, game->board);
         }
         return false;
      }
      playmove(game, move);
   } else {
      return false;
   }

   //write_epd_file(game, "xb_next.epd");
   return true;
}

static char ponder_input[65536];
static bool keyboard_input_on_move(gamestate_t *game)
{
   if (keyboard_input_waiting() && fgets(ponder_input, sizeof ponder_input, stdin)) {
      if (strstr(ponder_input, "random")) {
         sgenrand(time(NULL));
      } else if (strstr(ponder_input, "easy")) {
         may_ponder = false;
      } else if (strstr(ponder_input, "hard")) {
         may_ponder = true;
      } else if (strstr(ponder_input, "post")) {
         set_xboard_output_function(log_xboard_output);
      } else if (strstr(ponder_input, "nopost")) {
         set_xboard_output_function(NULL);
      } else if (strstr(ponder_input, "?") == ponder_input) {
         return true;
      } else if (strstr(ponder_input, "otim")) {
      } else if (strstr(ponder_input, "ping")) {
         log_xboard_output("pong %s\n", ponder_input+5);
      } else if (strstr(ponder_input, "result")) {
      } else if (strstr(ponder_input, "time")) {
      } else if (streq(ponder_input, "quit")) {
         if (game)
            end_game(game);
         free(buf);
         exit(0);
      }
   }
   return false;
}

static bool keyboard_input_analyse(gamestate_t *game)
{
   bool restart = false;
   if (keyboard_input_waiting() && fgets(ponder_input, sizeof ponder_input, stdin)) {
      /* FIXME: for some of these the search should be at the root
       * position, for others the search should continue uninterrupted.
       * This means we need to push the command on a stack and set a flag
       * to interrupt the current search.
       * At the root and in the main analysis function we should then pop
       * commands off the stack before deciding whether to continue or not.
       */
      if (strstr(ponder_input, "undo")) {
         takeback(game);
         restart = true;
      } else if (strstr(ponder_input, "new")) {
         start_new_game(game);
         restart = true;
      } else if (strstr(ponder_input, "exit")) {
         game->analysing = false;
         restart = true;
      } else if (strstr(ponder_input, "setboard")) {
         char *p = strstr(ponder_input, " ");
         while (p && *p && p[0]==' ') p++;
         setup_fen_position(game, p);
         restart = true;
      } else if (strstr(ponder_input, ".") == ponder_input) {
      } else if (strstr(ponder_input, "hint")) {
         if (game && game->ponder_move)
            log_xboard_output("Hint: %s\n", move_string(game->ponder_move, NULL));
      } else if (strstr(ponder_input, "bk")) {
      } else if (streq(ponder_input, "quit")) {
         if (game)
            end_game(game);
         free(buf);
         exit(0);
      } else {
         if (ponder_input[0] && !input_move(game, ponder_input))
            log_xboard_output("Illegal move: %s\n", ponder_input);
         else
            restart = true;
      }
   }
   return restart;
}

static uint64_t perft(gamestate_t *game, int depth, int root)
{
   movelist_t movelist;
   int me = game->side_to_move;
   uint64_t nodes = 0;
   bool check = game->board->in_check;
   int n;

   if (abort_search) return 0;

   if (depth == 0) return 1;

   generate_moves(&movelist, game->board, game->side_to_move);
   for (n=0; n<movelist.num_moves; n++) {
      uint64_t count = 0;

      playmove(game, movelist.move[n]);
      if (check || !move_leaves_player_in_check(game, movelist.move[n], me))  /* Don't count illegal moves */
         count = perft(game, depth-1, root - 1);
      nodes += count;
      if (root > 0)
         printf("%8s %10"PRIu64" %10"PRIu64"\n", move_string(movelist.move[n], NULL), count, nodes);
      takeback_no_check(game);
   }
   return nodes;
}

static void test_movegen(void)
{
   gamestate_t *game = NULL;
   int n = 0;
   while (perftests[n].fen) {
      uint64_t nodes;
      game = create_game();
      start_new_game(game);

      setup_fen_position(game, perftests[n].fen);
      printf(".");
      fflush(stdout);
      nodes = perft(game, perftests[n].depth, 0);
      if (nodes != perftests[n].nodes) {
         printf("\n");
         printf("*** Failed at position %d (%s):\n", n, perftests[n].fen);
         printf("    Expected %"PRIu64" nodes, got %"PRIu64" nodes\n", perftests[n].nodes, nodes);
         return;
      }

      end_game(game);
      n++;
   }
   printf("\nOk.\n");
}

static void test_benchmark(int depth)
{
   gamestate_t *game = NULL;
   int n = 0;
   uint64_t nodes = 0;
   uint64_t t = get_timer();
   while (benchtests[n].fen) {
      game = create_game();
      start_new_game(game);
      game->random_ok = false;
      game->output_iteration = NULL;
      game->uci_output = NULL;
      game->xboard_output = NULL;

      //printf("%s\n", benchtests[n].fen);
      setup_fen_position(game, benchtests[n].fen);
      printf(".");
      fflush(stdout);

#ifdef __unix__
      if (trapint) old_signal_handler = signal(SIGINT, interrupt_computer);
#endif
      computer_play(game, depth);
#ifdef __unix__
      if (trapint) signal(SIGINT, old_signal_handler);
#endif
      end_game(game);

      if (abort_search) {
         printf("\n*** Aborted");
         break;
      }

      //printf("%"PRIu64"\n", nodes);
      nodes += moves_searched;
      n++;
   }

   uint64_t tt = get_timer();

   printf("\n");
   printf("%"PRIu64" nodes searched\n", nodes);
   printf("Elapsed time %"PRIu64" ms\n", (tt - t) / 1000);
   printf("%g nodes / s\n", 1.0e6*nodes / (tt - t));

}

static void print_help(const char *topic)
{
   if (!topic) topic = "help";
   while (isspace(*topic)) topic++;
   if(!strlen(topic)) topic = "help";

   int n = 0;
   if (streq(topic, "all")) {
   while (help_topic[n].topic) {
      if (help_topic[n].text) {
         char *s = help_topic[n].cmd;
         if (!s) s = help_topic[n].topic;
         printf("%s\n%s\n", s, help_topic[n].text);
      }
      n++;
   }
      return;
   }
   while (help_topic[n].topic) {
      if (streq(help_topic[n].topic, topic)) {
         char *s = help_topic[n].cmd;
         if (!s) s = help_topic[n].topic;
         printf("%s\n%s\n", s, help_topic[n].text);
         if (streq(topic, "help")) break;
         return;
      }
      n++;
   }

   printf("Help topics: \n");
   n = 0;
   while (help_topic[n].topic) {
      if (n) printf(", ");
      printf("%s", help_topic[n].topic);
      n++;
   }
   printf("\n");
}

int main(int argc, char **argv)
{
   bool in_play = true;
   int my_colour = BLACK;
   gamestate_t *game = NULL;
   char input[65536];
   const char *lower_piece_str[] = {
      "p", "n", "b", "r", "q", "k"
   };
   int depth = 60;
   size_t hash_size = HASH_TABLE_SIZE;
   char *pgbook_file = NULL;

   buf = malloc(65536);

   printf("Jazz version %s, built %s\n", VERSIONSTR " " ARCHSTR, __DATE__);
   printf("Type 'help' for a list of commands and help topics\n");
   initialise_jazz();

   for (int n = 1; n<argc; n++) {
      if (strstr(argv[n], "-book")) {
         if (n+1 >= argc) {
            fprintf(stderr, "error: no book specified\n");
            exit(0);
         }
         n++;
         pgbook_file = strdup(argv[n]);
         printf("Using opening book %s\n", pgbook_file);
         //pgbook_file = strdup("bigbook.bin");
      } else if (strstr(argv[n], "-log")) {
         char *logfile = "jazz.log";
         if (n+1 < argc && argv[n+1][0] != '-') {
            logfile = argv[n+1];
            n++;
         }
         if (!f) f = fopen(logfile, "a");
      } else {
         fprintf(stderr, "Unknown option: %s\n", argv[n]);
         exit(0);
      }
   }

   set_default_output_function(NULL);
   //set_default_output_function(log_engine_output);
   set_xboard_output_function(log_xboard_output);

   /* Turn off buffering for stdout and stdin */
   setvbuf(stdout, NULL, _IONBF, 0);
   setvbuf(stdin, NULL, _IONBF, 0);

   /* Write log output (for debugging purposes) */
   //if (!f) f = fopen("xb.log", "a");

   while (true) {
      input[0] = '\0';
      if (show_board && game)
         print_board(game->board);
      if (prompt) {
         printf("#");
         if (game) {
            printf("%d", (int)game->moves_played);
            printf("%s", (game->side_to_move == WHITE)?"w":"b");
            if (!in_play) printf(" (f)");
            if (player_in_check(game, game->side_to_move)) printf("+");
         } else {
            printf("[-]");
         }
         printf(">");
      }
      if (!(may_ponder && game && game->ponder_move)|| keyboard_input_waiting()) {
         if (!fgets(input, sizeof input, stdin))
            break;
      }
      chomp(input);
      if (f) {
         fprintf(f, "< %s\n", input);
         fflush(f);
      }
      if (streq(input, "xboard") || streq(input, "xboard on")) {
         prompt = false;
         show_board = false;
         san = false;
         trapint = false;
         log_xboard_output("\n");
      } else if (streq(input, "xboard off")) {
         prompt = true;
         show_board = true;
         san = true;
         trapint = true;
         log_xboard_output("\n");
      } else if (strstr(input, "help")) {
         print_help(input+4);
      } else if (strstr(input, "protover")) {
         log_xboard_output("feature setboard=1"
                                      " time=1"
                                    " sigint=0"
                                    " colors=0"
                                      " ping=1"
                                    " memory=1"
                                   " analyze=1"
                                    " myname=\"Jazz %s\""
                                 " myversion=\"[%s]\"\n", VERSIONSTR, VERSIONSTR " " __DATE__ " " ARCHSTR);
         log_xboard_output("feature option=\"Opening book (polyglot) -file %s\"\n", pgbook_file?pgbook_file:"");
         log_xboard_output("feature done=1\n");
      } else if (streq(input, "new")) {
         if (game) {
            end_game(game);
         }
         game = create_game();
         game->book = open_opening_book(pgbook_file);
         set_transposition_table_size(game, hash_size);
         start_new_game(game);
         my_colour = BLACK;
         depth = 60;
         num_games++;
         if (f) fprintf(f, "# game %d\n", num_games);
      } else if (strstr(input, "accepted")) {
      } else if (strstr(input, "rejected")) {
      } else if (strstr(input, "option")) {
         log_xboard_output("# %s\n", input);
         if (strstr(input+7, "Opening book (polyglot)")) {
            char *s = strstr(input, "=");
            char *eol = input + strlen(input)-1;
            if (s) s++;

            /* Strip leading and trailing spaces */
            while (*s && isspace(*s)) s++;
            while (isspace(*eol)) { *eol='\0'; eol--; }

            log_xboard_output("# %s\n", s);

            free(pgbook_file);
            pgbook_file = strdup(s);

            if (game) {
               close_opening_book(game->book);
               game->book = open_opening_book(pgbook_file);
            }
         }
      } else if (strstr(input, "book") == input) {
            char *s = input+4;
            char *eol = input + strlen(input)-1;
            if (s) s++;

            /* Strip leading and trailing spaces */
            while (*s && isspace(*s)) s++;
            while (isspace(*eol)) { *eol='\0'; eol--; }

            if (streq(s, "off")) {
               free(pgbook_file);
               pgbook_file = NULL;
               if (game) {
                  close_opening_book(game->book);
                  game->book = NULL;
               }
            } else {
               free(pgbook_file);
               pgbook_file = strdup(s);

               if (game) {
                  close_opening_book(game->book);
                  game->book = open_opening_book(pgbook_file);
               }
            }
      } else if (strstr(input, "memory") == input) {
         unsigned long int memory_size;
         sscanf(input+7, "%lu", &memory_size);

         /* Convert to bytes */
         memory_size <<= 20;

         /* Reserve default */
         if (memory_size > 10<<20)
            memory_size -= 5<<20;

         hash_size = memory_size / sizeof(hash_table_entry_t);
      } else if (strstr(input, "analyze") || strstr(input, "analyse")) {
         if (game) {
            in_play = false;
            set_xboard_output_function(log_xboard_output);
            game->analysing = true;
         }
      } else if (strstr(input, "force")) {
         in_play = false;
      } else if (strstr(input, "undo")) {
         takeback(game);
      } else if (strstr(input, "remove") || strstr(input, "takeback")) {
         takeback(game);
         takeback(game);
      } else if (strstr(input, "setboard")) {
         char *p = strstr(input, " ");
         while (p && *p && p[0]==' ') p++;

         if (!game) {
            game = create_game();
            game->book = open_opening_book(pgbook_file);
            start_new_game(game);
         }

         setup_fen_position(game, p);
      } else if (strstr(input, "seteval") == input) {
         /* Set evaluation parameter, for parameter tuning */
         char *name = input+8;
         while (*name && isspace(*name)) name++;
         char *s;
         s = strstr(name, "=");
         *s = '\0'; s++;
         while (*s && isspace(*s)) s++;
         int value = 0;
         sscanf(s, "%d", &value);
         if (strstr(name, "PAWN_BASE")) {
            PAWN_BASE = value;
         } else if (strstr(name, "MINOR_BASE")) {
            MINOR_BASE = value;
         } else if (strstr(name, "ROOK_BASE")) {
            ROOK_BASE = value;
         } else if (strstr(name, "QUEEN_BASE")) {
            QUEEN_BASE = value;
         } else if (strstr(name, "EXCHANGE_BASE")) {
            EXCHANGE_BASE = value;
         } else if (strstr(name, "MEXCHANGE_BASE")) {
            MEXCHANGE_BASE = value;
         } else if (strstr(name, "EVAL_BISHOP_PAIR")) {
            EVAL_BISHOP_PAIR = value;
         } else if (strstr(name, "ROOK_ADVANTAGE_BASE")) {
            ROOK_ADVANTAGE_BASE = value;
         } else if (strstr(name, "MINOR_ADVANTAGE_BASE")) {
            MINOR_ADVANTAGE_BASE = value;
         } else if (strstr(name, "KNIGHT_VALUE_SCALE")) {
            KNIGHT_VALUE_SCALE = value;
         } else if (strstr(name, "ROOK_VALUE_SCALE")) {
            ROOK_VALUE_SCALE = value;
         } else if (strstr(name, "RAZOR_MARGIN_UNIT")) {
            RAZOR_MARGIN_UNIT = value;
         } else if (strstr(name, "FUTILITY1")) {
            FUTILITY1 = value;
         } else if (strstr(name, "FUTILITY2")) {
            FUTILITY2 = value;
         } else if (strstr(name, "FUTILITY3")) {
            FUTILITY3 = value;
         } else if (strstr(name, "FUTILITY4")) {
            FUTILITY4 = value;
         } else if (strstr(name, "FUTILITY5")) {
            FUTILITY5 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST0")) {
            PAWN_COUNT_VALUE_ADJUST0 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST1")) {
            PAWN_COUNT_VALUE_ADJUST1 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST2")) {
            PAWN_COUNT_VALUE_ADJUST2 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST3")) {
            PAWN_COUNT_VALUE_ADJUST3 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST4")) {
            PAWN_COUNT_VALUE_ADJUST4 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST5")) {
            PAWN_COUNT_VALUE_ADJUST5 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST6")) {
            PAWN_COUNT_VALUE_ADJUST6 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST7")) {
            PAWN_COUNT_VALUE_ADJUST7 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST8")) {
            PAWN_COUNT_VALUE_ADJUST8 = value;
         }

         initialise_evaluation();
      } else if (strstr(input, "perft")) {
         if (game) {
            int depth = 6;
            int root = 0;
            char *s = input + 5;
            while (*s && isspace(*s)) s++;
            if (*s) {
               sscanf(s, "%d", &depth);
               while(*s && isdigit(*s)) s++;
            }
            while (*s && isspace(*s)) s++;
            if (*s) {
               sscanf(s, "%d", &root);
            }
#ifdef __unix__
            if (trapint) old_signal_handler = signal(SIGINT, interrupt_computer);
#endif
            abort_search = false;
            uint64_t t = get_timer();
            for (int n = 1; n<depth+1; n++) {
               uint64_t nodes = perft(game, n, root);
               uint64_t tt = get_timer();

               if (tt == t) tt++;

               if (abort_search) break;
               printf("%2d %10lld %5.2f %12.2fnps\n", n, (long long int)nodes,
                     (double)(tt - t)/1000000.0,nodes*1.0e6/(tt-t));

               t = tt;
            }
#ifdef __unix__
            if (trapint) signal(SIGINT, old_signal_handler);
#endif
         }
      } else if (strstr(input, "test movegen")) {
         test_movegen();
      } else if (strstr(input, "test benchmark")) {
         int depth = 10;
         char *s = input + 15;
         while (*s && isspace(*s)) s++;
         if (*s) sscanf(s, "%d", &depth);
         //printf("Benchmark %d\n", depth);
         test_benchmark(depth);
      } else if (strstr(input, "sd")) {
         sscanf(input, "sd %d", &depth);
      } else if (strstr(input, "go")) {
         my_colour = game->side_to_move;
         in_play = true;

         /* Correctly set the number of moves to go */
         if (game->movestotc) {
            size_t moves_played = game->moves_played / 2;
            if (my_colour == WHITE)
               moves_played += (game->moves_played % 2);
            moves_played %= game->movestotc;
            game->movestogo = game->movestotc - moves_played;
         }
      } else if (strstr(input, "st ")) {
         float time_per_move = 5;
         sscanf(input+3, "%g", &time_per_move);
         set_time_per_move(game, time_per_move*1000);
      } else if (strstr(input, "computer")) {
      } else if (strstr(input, "random")) {
         sgenrand(time(NULL));
      } else if (strstr(input, "easy") || streq(input, "ponder off")) {
         may_ponder = false;
      } else if (strstr(input, "hard") || streq(input, "ponder on")) {
         may_ponder = true;
      } else if (strstr(input, "post")) {
         set_xboard_output_function(log_xboard_output);
      } else if (strstr(input, "nopost")) {
         set_xboard_output_function(NULL);
      } else if (strstr(input, "?") == input) {
      } else if (strstr(input, "hint")) {
         if (game && game->ponder_move) {
            log_xboard_output("Hint: %s\n", move_string(game->ponder_move, NULL));
         }
      } else if (strstr(input, "otim")) {
      } else if (strstr(input, "ping")) {
         log_xboard_output("pong %s\n", input+5);
      } else if (strstr(input, "result")) {
      } else if (strstr(input, "time")) {
         float milliseconds;
         sscanf(input+5, "%g", &milliseconds);
         milliseconds *= 10;

         /* Reserve a bit of time for when we reach the time control, so we don't get penalised by a time
          * loss. We don't need this for a Fischer clock.
          */
         if (milliseconds > TIME_BUFFER)
            milliseconds -= TIME_BUFFER;

         game->time_left[0] = game->time_left[1] = milliseconds;
         set_time_for_game(game);
      } else if (strstr(input, "level")) {
         int moves, minutes, seconds, milliseconds;
         float inc;
         /* Set defaults */
         set_time_per_move(game, 5000);
         game->movestotc = 0;
         game->movestogo = 0;
         game->time_inc[0] = game->time_inc[1] = 0;

         if (strstr(input, ":")) {
            sscanf(input+6, "%d %d:%d %g", &moves, &minutes, &seconds, &inc);
         } else {
            sscanf(input+6, "%d %d %g", &moves, &minutes, &inc);
            seconds = 0;
         }
         seconds += minutes*60;
         milliseconds = seconds * 1000;

         /* Reserve a bit of time for when we reach the time control, so we don't get penalised by a time
          * loss. We don't need this for a Fischer clock.
          */
         if (inc == 0 && milliseconds > TIME_BUFFER)
            milliseconds -= TIME_BUFFER;

         game->movestotc = moves;
         game->movestogo = moves;
         game->time_inc[0] = game->time_inc[1] = inc*1000;
         game->time_left[0] = game->time_left[1] = milliseconds;
         set_time_for_game(game);
      } else if (streq(input, "quit") || streq(input, "exit") || streq(input, "bye")) {
         if (game)
            end_game(game);
         free(buf);
         exit(0);
      } else if (streq(input, "eval")) {
         if (game) {
            game->root_board = game->board;
            printf("Static evaluation: %d\n", static_evaluation(game, game->side_to_move, -CHECKMATE, CHECKMATE));
         }
      } else if (strstr(input, "prompt on") == input) {
         prompt = true;
      } else if (strstr(input, "prompt off") == input) {
         prompt = false;
      } else if (strstr(input, "unicode on") == input) {
         unicode_board = true;
      } else if (strstr(input, "unicode off") == input) {
         unicode_board = false;
      } else if (streq(input, "board")) {
         print_bitboards(game->board);
         print_board(game->board);
      } else if (streq(input, "fen")) {
         if (game)
         printf("%s\n", make_fen_string(game, NULL));
      } else if (streq(input, "moves")) {
         if (game) {
            movelist_t movelist;
            generate_moves(&movelist, game->board, game->side_to_move);
            int k;
            printf("%d moves\n", movelist.num_moves);
            for (k=0; k<movelist.num_moves; k++)
               printf("%s ", move_string(movelist.move[k], NULL));
         }
         printf("\n");
      } else if (strstr(input, "readfen") || strstr(input, "readepd")) {
         FILE *f = fopen(input+8, "r");
         if (f) {
            char s[4096];
            fgets(s, sizeof s, f);
            fclose(f);

            if (!game) {
               game = create_game();
               game->book = open_opening_book(pgbook_file);
               start_new_game(game);
            }

            setup_fen_position(game, s);
            print_board(game->board);
         } else {
            printf("Can't open file: %s\n", input+8);
         }
      } else if (prompt && input[0] == '!') {
         system(input+1);
      } else {
         if (input[0] && !input_move(game, input)) {
            log_xboard_output("Illegal move: %s\n", input);
         }
      }

      if (game && in_play) {
         game->check_keyboard = keyboard_input_on_move;
         if (game->side_to_move == my_colour) {
            const char *p = "";
            move_t move;
            bool played;

            game->pondering = false;
            log_engine_output("Searching position %s\n", make_fen_string(game, NULL));

#ifdef __unix__
            if (trapint) old_signal_handler = signal(SIGINT, interrupt_computer);
#endif
            played = computer_play(game, depth);
#ifdef __unix__
            if (trapint) signal(SIGINT, old_signal_handler);
#endif
            if (played) {
               move = game->move_list[game->moves_played-1];
               if (is_promotion_move(move)) {
                  p = lower_piece_str[get_promotion_piece(move) & PIECE];
               }
               if (san) {
                  movelist_t movelist;
                  generate_legal_moves(&movelist, game->board, game->side_to_move);
                  log_xboard_output("move %s\n", short_move_string(move, &movelist, NULL));
               } else
                  log_xboard_output("move %s%s%s\n", square_str[get_move_origin(move)],
                        square_str[get_move_destination(move)], p);
               if (game->movestogo) {
                  game->movestogo--;
                  if (!game->movestogo)
                     game->movestogo = game->movestotc;
               }
            } else {
               if (root_is_mate) {
                  log_xboard_output("%s\n", game->side_to_move ? "1-0 {White mates}" : "0-1 {Black mates}");
               } else {
                  if (game->fifty_counter[game->moves_played] >= 100) {
                     log_xboard_output("1/2-1/2 {Draw by 50 move rule}\n");
                  } else if (draw_by_repetition(game)) {
                     log_xboard_output("1/2-1/2 {Draw by repetition rule}\n");
                  } else {
                     log_xboard_output("1/2-1/2 {Stalemate}\n");
                  }
               }
            }
         } else {    /* Ponder ? */
            if (may_ponder && ponder_ok && game->ponder_move)
               computer_ponder(game);
         }
      }

      if (game && game->analysing) {
         game->check_keyboard = keyboard_input_analyse;
         computer_analyse(game);
      }
   }

   if (prompt)
      printf("\n");

   return 0;
}

