/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdint.h>
#include <stdio.h>
#include "pg_book.h"
#include "pg_hash.h"
#include "board.h"
#include "squares.h"

uint64_t *pg_key_piece        = pg_key;
uint64_t *pg_key_castle       = pg_key+768;
uint64_t *pg_key_enpassant    = pg_key+772;
uint64_t *pg_key_side_to_move = pg_key+780;

static void int_to_file(FILE *f, int l, uint64_t r){
   int i,c;
   for(i=0;i<l;i++) {
      c=(r>>8*(l-i-1))&0xff;
      fputc(c,f);
   }
}

void entry_to_file(FILE *f, polyglot_position_t *entry)
{
   int_to_file(f,8,entry->key);
   int_to_file(f,2,entry->move);
   int_to_file(f,2,entry->weight);
   int_to_file(f,4,entry->learn);
}


bool int_from_file(FILE *f, int l, uint64_t *r)
{
   int i,c;
   *r = 0;
   for(i=0;i<l;i++){
      c=fgetc(f);
      if(c==EOF){
         return false;
      }
      (*r) = ((*r)<<8) | c;
   }
   return true;
}

bool entry_from_file(FILE *f, polyglot_position_t *entry)
{
   bool ret;
   uint64_t r;

   ret=int_from_file(f,8,&r);
   if(!ret) return false;
   entry->key=r;
   ret=int_from_file(f,2,&r);
   if(!ret) return false;
   entry->move=r;
   ret=int_from_file(f,2,&r);
   if(!ret) return false;
   entry->weight=r;
   ret=int_from_file(f,4,&r);
   if(!ret) return false;
   entry->learn=r;

   return true;
}

static int find_key(FILE *f, uint64_t key, polyglot_position_t *entry)
{
   int first, last, middle;
   polyglot_position_t first_entry, last_entry,middle_entry;
   first_entry.key = 0;
   first_entry.move = 0;
   first_entry.weight = 0;
   first=-1;
   if(fseek(f,-16,SEEK_END)){
      return -1;
   }
   last=ftell(f)/16;
   entry_from_file(f,&last_entry);
   while(1){
      if(last-first==1){
         *entry=last_entry;
         return last;
      }
      middle=(first+last)/2;
      fseek(f,16*middle,SEEK_SET);
      entry_from_file(f,&middle_entry);
      if(key<=middle_entry.key){
         last=middle;
         last_entry=middle_entry;
      }else{
         first=middle;
         first_entry=middle_entry;
      }
   }
}

/* Open a polyglot opening book */
polyglot_book_t *open_polyglot_opening_book(const char *filename)
{
   FILE *f;
   polyglot_book_t *book = NULL;

   f = fopen(filename, "r");
   if (!f)
      return NULL;

   book = malloc(sizeof *book);
   book->f = f;
   book->last_key = 0;
   book->offset = 0;
   return book;
}

uint64_t get_polyglot_key(board_t *board, int side_to_move)
{
   uint64_t key = 0;
   
   bitboard_t occ = get_occupied(board);
   while (occ) {
      int square = bitscan64(occ);
      int piece = get_piece(board, square);
      int pg_piece;
      int index;

      occ ^= make_bitboard_square(square);
      pg_piece = 2*decode_piece_type(piece) + (decode_piece_colour(piece) == WHITE); 

      index = 64 * pg_piece + square;
      key ^= pg_key_piece[index];
   }

   /* Castling rights */
   if (board->init & (E1_MASK|H1_MASK))
      key ^= pg_key_castle[0];
   if (board->init & (E1_MASK|A1_MASK))
      key ^= pg_key_castle[1];
   if (board->init & (E8_MASK|H8_MASK))
      key ^= pg_key_castle[2];
   if (board->init & (E8_MASK|A8_MASK))
      key ^= pg_key_castle[3];

   /* En-passant */
   if (board->ep_square)
      key ^= pg_key_enpassant[unpack_file(board->ep_square)];

   /* Side to move */
   if (side_to_move == WHITE)
      key ^= pg_key_side_to_move[0];

   return key;
}

/* Look up the current position in the opening book */
polyglot_position_t *get_polyglot_book_move(polyglot_book_t *book, board_t *board, int side_to_move)
{
   uint64_t key;
   polyglot_position_t *entry;

   if (!book)
      return NULL;

   key = get_polyglot_key(board, side_to_move);

   if (book->last_key != key) {
      book->last_key = key;
      entry = malloc(sizeof *entry);
      book->offset = find_key(book->f, key, entry);

      if (book->offset == -1 || entry->key != key) {
         book->last_key = 0;
         book->offset = -1;
         free(entry);
         return NULL;
      }

      return entry;
   }
   
   if (book->offset >= 0) {
      return get_next_polyglot_book_move(book);
   }

   return NULL;
}

void free_polyglot_book_move(polyglot_position_t *entry)
{
   free(entry);
}

/* Find successive entries for the current position */
polyglot_position_t *get_next_polyglot_book_move(polyglot_book_t *book)
{
   int pos;
   polyglot_position_t *entry;

   if (!book)
      return NULL;

   entry = malloc(sizeof *entry);
   book->offset++;
   pos = book->offset;
   fseek(book->f, 16*pos, SEEK_SET);

   entry_from_file(book->f, entry);
   if (entry->key == book->last_key)
      return entry;

   free(entry);
   book->offset = -1;
   return NULL;
}

void close_polyglot_opening_book(polyglot_book_t *book)
{
   if (book)
      fclose(book->f);
   free(book);
}


