/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "bitboard.h"
#include "squares.h"

const bitboard_t square_bitboards[64] = {
   A1_MASK, B1_MASK, C1_MASK, D1_MASK, E1_MASK, F1_MASK, G1_MASK, H1_MASK,
   A2_MASK, B2_MASK, C2_MASK, D2_MASK, E2_MASK, F2_MASK, G2_MASK, H2_MASK,
   A3_MASK, B3_MASK, C3_MASK, D3_MASK, E3_MASK, F3_MASK, G3_MASK, H3_MASK,
   A4_MASK, B4_MASK, C4_MASK, D4_MASK, E4_MASK, F4_MASK, G4_MASK, H4_MASK,
   A5_MASK, B5_MASK, C5_MASK, D5_MASK, E5_MASK, F5_MASK, G5_MASK, H5_MASK,
   A6_MASK, B6_MASK, C6_MASK, D6_MASK, E6_MASK, F6_MASK, G6_MASK, H6_MASK,
   A7_MASK, B7_MASK, C7_MASK, D7_MASK, E7_MASK, F7_MASK, G7_MASK, H7_MASK,
   A8_MASK, B8_MASK, C8_MASK, D8_MASK, E8_MASK, F8_MASK, G8_MASK, H8_MASK
};

bitboard_t white_pawn_square[64];
bitboard_t black_pawn_square[64];
bitboard_t north_pawn_attack[64];
bitboard_t south_pawn_attack[64];
bitboard_t north_pawn_attack_span[64];
bitboard_t south_pawn_attack_span[64];

static inline int max(int x, int y) { return (x>y)?x:y; }

static int square_distance(int square1, int square2)
{
   int row1, file1;
   int row2, file2;

   row1 = unpack_rank(square1);
   file1 = unpack_file(square1);
   row2 = unpack_rank(square2);
   file2 = unpack_file(square2);

   return max(abs(row1 - row2), abs(file1-file2));
}

static inline bitboard_t fill_north(bitboard_t bb)
{
   bb |= bb<<8;
   bb |= bb<<16;
   bb |= bb<<32;
   return bb;
}

static inline bitboard_t fill_south(bitboard_t bb)
{
   bb |= bb>>8;
   bb |= bb>>16;
   bb |= bb>>32;
   return bb;
}

void compute_pawn_bitboard_masks(void)
{
   int square, square2;

   /* Bitboards for quick rule-of-the-square evaluation */
   /* Tables, for white */
   for (square = A3; square < A8; square++) {
      white_pawn_square[square] = 0;
      for (square2 = square & ~7; square2<=H8; square2++) {
         if (square_distance(square2, square | (7<<3)) <=
             square_distance(square, square | (7<<3)) ) {
            white_pawn_square[square] |= make_bitboard_square(square2);
         }
      }
   }
   for (square = A2; square < A3; square++)
      white_pawn_square[square] = white_pawn_square[square+8];

   /* For black */
   for (square = A3; square < A8; square++) {
      black_pawn_square[square] = 0;
      for (square2 = A1; square2<=(square | (7<<3)); square2++) {
         if (square_distance(square2, square & 7) <=
             square_distance(square, square & 7) ) {
            black_pawn_square[square] |= make_bitboard_square(square2);
         }
      }
   }
   for (square = A7; square < A8; square++)
      black_pawn_square[square] = black_pawn_square[square-8];

   /* Bitboards for evaluating candidate pawns: north and south attack spans */
   for (square = A2; square < A8; square++) {
      bitboard_t bb = make_bitboard_square(square);

      north_pawn_attack[square] = (shift_bitboard_right_up(bb)   | shift_bitboard_left_up(bb));
      south_pawn_attack[square] = (shift_bitboard_right_down(bb) | shift_bitboard_left_down(bb));
      north_pawn_attack_span[square] = fill_north(shift_bitboard_right_up(bb)   | shift_bitboard_left_up(bb));
      south_pawn_attack_span[square] = fill_south(shift_bitboard_right_down(bb) | shift_bitboard_left_down(bb));
   }
}

static void print_uint8_t(uint8_t b)
{
   int c;
   
   for (c=0; c<8; c++) {
      printf ("%d", (b>>c)&1);
   }
}

/* This function prints a bitboard in such a way that 'A1' ends up on the
 * lower left corner and 'H8' ends up at the upper right corner.
 */
void printf_bitboard(bitboard_t b)
{
   int c;

   printf("+--------+\n");
   for (c=7; c>=0; c--) {
      printf("|");
      print_uint8_t(get_bitboard_row(b, c));
      printf("|\n");
   }
   printf("+--------+\n");
}

void print_bitboard_squares(bitboard_t b)
{
   int square;

   //printf("Bitboard squares: ");
   while(b) {
      square = bitscan64(b);
      b ^= make_bitboard_square(square);
      printf("%c%c ", unpack_file(square)+'a', unpack_rank(square)+'1');
   }
   //printf("\n");
}

