/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include "config.h"
#include "square.h"
#include "board.h"
#include "game.h"
#include "hashtable.h"
#include "pawn_table.h"
#include "evalhash.h"
#include "fen.h"

#define PIECESTRING   " NBRQK"

static void (*default_iteration_output)(const char *, ...) = NULL;
static void (*default_uci_output)(const char *, ...) = NULL;
static void (*default_xboard_output)(const char *, ...) = NULL;

void playgame_init(void)
{
   default_iteration_output = (void *)printf;
   default_uci_output = NULL;
}

void playgame_shutdown(void)
{
}

void set_default_output_function(void (*func)(const char *, ...))
{
   default_iteration_output = func;
}

void set_uci_output_function(void (*func)(const char *, ...))
{
   default_uci_output = func;
}

void set_xboard_output_function(void (*func)(const char *, ...))
{
   default_xboard_output = func;
}

void set_transposition_table_size(gamestate_t *game, uint64_t nelem)
{
   if (!game)
      return;

   /* Round to the next-lowest power of 2 */
   nelem |= (nelem >> 1);
   nelem |= (nelem >> 2);
   nelem |= (nelem >> 4);
   nelem |= (nelem >> 8);
   nelem |= (nelem >> 16);
   nelem |= (nelem >> 32);
   nelem >>= 1;
   nelem++;

   game->default_hash_size = nelem;

   /* Reallocate table if it was already allocated */
   if (game->default_hash_size) {
      destroy_hash_table(game->transposition_table);
      game->transposition_table = create_hash_table(game->default_hash_size);
   }
}

gamestate_t *create_game(void)
{
   return calloc(1, sizeof(gamestate_t));
}

/* Initialize the game datastructure to correspond to a new game */
void start_new_game(gamestate_t *game)
{
   /* Load the starting position */
   if (game->default_hash_size == 0) game->default_hash_size = HASH_TABLE_SIZE;
   game->max_moves = 1000;
   game->score = calloc(game->max_moves, sizeof *game->score);
   game->fifty_counter = calloc(game->max_moves, sizeof(*game->fifty_counter));
   game->move_list = calloc(game->max_moves, sizeof *game->move_list);
#ifdef UNMAKE_MOVE
   game->status = calloc(game->max_moves, sizeof *game->status);
#endif
   game->board_list = calloc(game->max_moves, sizeof *game->board_list);
   game->board = game->board_list;

   game->transposition_table = create_hash_table(game->default_hash_size);
   game->pawn_structure = create_pawn_hash_table(PAWN_TABLE_SIZE);
   game->queen_psq = create_piece_hash_table(PIECE_TABLE_SIZE);
   game->rook_psq = create_piece_hash_table(PIECE_TABLE_SIZE);
   game->knight_psq = create_piece_hash_table(PIECE_TABLE_SIZE);
   game->bishop_psq = create_piece_hash_table(PIECE_TABLE_SIZE);
   game->king_psq = create_piece_hash_table(PIECE_TABLE_SIZE);
   game->eval_hash = create_eval_hash_table(game->default_hash_size);

   setup_fen_position(game, "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1");

   
   game->moves_played = 0;
   game->last_move = 0;

   game->analysing = false;
   game->pondering = false;
   game->random_ok = true;
   game->ponder_move = 0;

   game->name_white = NULL;
   game->name_black = NULL;
   game->out_book = false;

   game->output_iteration = default_iteration_output;
   game->uci_output = default_uci_output;
   game->xboard_output = default_xboard_output;
}

/* Frees the memory used by a gamestate */
void end_game(gamestate_t *game)
{
   if (game) {
      close_opening_book(game->book);

      free(game->name_white);
      free(game->name_black);
      free(game->move_list);
      free(game->board_list);
      free(game->score);
#ifdef UNMAKE_MOVE
      free(game->status);
#endif
      free(game->fifty_counter);
      destroy_hash_table(game->transposition_table);
      destroy_hash_table(game->qs_table);
      destroy_eval_hash_table(game->eval_hash);
      destroy_pawn_hash_table(game->pawn_structure);
      destroy_piece_hash_table(game->queen_psq);
      destroy_piece_hash_table(game->rook_psq);
      destroy_piece_hash_table(game->knight_psq);
      destroy_piece_hash_table(game->bishop_psq);
      destroy_piece_hash_table(game->king_psq);
      free(game);
   }
}
