/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdarg.h>
#include "config.h"
#include "bits.h"
#include "alphabeta.h"
#include "board.h"
#include "movegen.h"
#include "game.h"
#include "inline/game.h"
#include "evaluate.h"
#include "see.h"
#include "history.h"
#include "goodcap.h"
#include "killer.h"
#include "scoremove.h"
#include "fen.h"

#undef TEST_MOVES_BEFORE_SEARCH

#undef QSORT_MOVES

#define min(x,y) ( ((x)<(y))?(x):(y) )
#define max(x,y) ( ((x)>(y))?(x):(y) )

/* When taking back a move here, we don't have to be afraid of going back
 * beyond the first move in the game - so it's safe to skip the test for
 * that in the takeback() function.
 */
#define takeback takeback_no_check

/* Function call to check how much time we have left
 * We do the check every time clock_positions nodes have been searched.
 * Don't do this during QS - there aren't that many QS nodes anyway.
 */
static const int clock_positions = 0x00007FFF;
#define check_clock() \
   if (game->check_clock && ((positions_evaluated & clock_positions)==0))\
      abort_search |= game->check_clock(game)

volatile bool abort_search = 0;
unsigned long long int positions_evaluated;
unsigned long long int moves_searched;
unsigned long long int positions_in_hashtable;
unsigned long long int branches_pruned;
unsigned long long int branches_pruned_1st;

#undef DEBUG_SEARCH_TREE
#ifdef DEBUG_SEARCH_TREE

#define LOG_MOVE_BEGIN(alpha,beta,msg) print_move_score(game, move, me, draft, extension, new_score, alpha, beta, msg, true);
#define LOG_MOVE_END(alpha,beta,msg) print_move_score(game, move, me, draft, extension, new_score, alpha, beta, msg, false);
#define LOG_CHECK_STATE() if (game->board->in_check) print_log_msg(game, "  (check)");
#define WRITE_LOG(...) print_log_msg(game, __VA_ARGS__);

#else

#define LOG_MOVE_BEGIN(alpha,beta,msg) (void)0
#define LOG_MOVE_END(alpha,beta,msg) (void)0
#define LOG_CHECK_STATE() (void)0
#define WRITE_LOG(...) (void)0

#endif

#ifdef DEBUG_SEARCH_TREE
static void print_move_score(const gamestate_t *game, move_t move, int player, int horizon, int extension, int score, int alpha, int beta, char *msg, bool open)
{
   int e, r;
   int c;
   if (!game->trace_ok) return;
   for (c=game->root_moves_played; c<=game->moves_played; c++)
      printf("  ");
   
   if (open)
      printf("[ ");
   else
      printf("] ");

   r = 0;
   e = extension / PLY;
   if (e<0) {
      r = -e;
      e = 0;
   }
   printf("d=%lu h=%d ext=%d red=%d %ctm", game->moves_played-game->root_moves_played,
      horizon/PLY, e, r, player?'b':'w');

   printf(" %s ", move_string(move, NULL));
   if (open)
      printf("    ");
   else
      printf("% d", score);
   if (!msg) msg = "";
   printf(" [% d % d] %s\n", alpha, beta, msg);
}

static void print_log_msg(const gamestate_t *game, const char *msg, ...)
{
   va_list ap;
   static char buf[4096];

   if (!game->trace_ok) return;

   va_start(ap, msg);
   vsnprintf(buf, 4095, msg, ap);
   va_end(ap);

   int c;
   for (c=game->root_moves_played; c<=game->moves_played; c++) printf("  ");
   printf("%s\n", buf);
}
#endif

#ifdef COPY_MOVE
#define in_check(n) game->board[(n)].in_check
#else
#define in_check(n) ((game->status[game->moves_played+(n)] & 0x80)>>7)
#endif

#ifdef QSORT_MOVES
static const int *qsort_score_ptr;

/* quicksort (qsort) callback function */
static int movelist_sorter_qsort(const void *mi1, const void *mi2)
{
   int *score = (int*)qsort_score_ptr;
   int i1 = *(int*)mi1;
   int i2 = *(int*)mi2;

   return score[i2] - score[i1];
}
#endif

void sort_moves(int *move_perm, int num_moves, const int *score)
{
#ifdef QSORT_MOVES
   qsort_score_ptr = score;
   qsort(move_perm, num_moves, sizeof *move_perm, movelist_sorter_qsort);
#else
   /* Insertion sort */
   int n, m;
   for (n=1; n<num_moves; n++) {
      for (m=0; m<=n; m++) {
         if (score[move_perm[m]] <= score[move_perm[n]]) {
            int k = move_perm[n];
            int c;
            for (c=n; c>m; c--)
               move_perm[c] = move_perm[c-1];
            move_perm[m] = k;
            break;
         }
      }
   }
#endif
}

/* Return the number of repetitions of the current position in the game
 * tree.
 */
int count_repetition(const gamestate_t *game)
{
   int n;
   int count;
   count = 0;

   for (n=game->moves_played-1; n>=0; n--) {
      if (is_capture_move(game->move_list[n]) || ((get_move_piece(game->move_list[n]) & PIECE) == PAWN))
         return count;

      count += (game->board_list[n].hash == game->board->hash);
   }

   return count;
}

/* Check if the current position is a draw by repetition. If so, there are
 * no legal moves in it - the position should be treated as stalemate.
 */
inline bool draw_by_repetition(gamestate_t *game)
{
   return game->repetition_hash_table[game->board->hash&0xFFFF]>=3 && count_repetition(game)>=3;
}

#ifdef DEBUG_SEARCH_TREE
#define extension 0
#endif
/* Quiescence search */
int quiescence(gamestate_t *game, int depth, int draft, int alpha, int beta)
{
   movelist_t movelist;
   int move_perm[MAX_MOVES];
   int score[MAX_MOVES];
   int best_score = alpha;
   int n;
   int me;
   int new_score;
   int static_score = alpha;

   game->length_of_variation[depth+1] = depth+1;
   game->quiescence_depth_of_variation[depth + 1] = -draft / PLY;

   me = game->side_to_move;
   bool check = game->board->in_check;

   /* We're at the search horizon, return static evaluation.
    * This is a hard limit to prevent the program from locking up in a
    * perpetual check. These will eventually repeat, of course, but until
    * we can detect that it's best to have this check in.
    * We also abort if the static evaluation gives a beta cutoff.
    */
   WRITE_LOG("  QS search position %s", make_fen_string(game, NULL));
   if (!check) {
      best_score = static_score = static_evaluation(game, me, alpha, beta);
      WRITE_LOG("  QS static eval = %d", best_score);
      if ( (draft == HARD_HORIZON*PLY) || (static_score >= beta && !in_check(0)) ) {
         WRITE_LOG("  Static pruning: %d >= %d", static_score, beta);
         return static_score;
      }
      if (best_score > alpha) alpha = best_score;
   }

   positions_evaluated++;

   LOG_CHECK_STATE();

   /* Generate moves; if we are in check, then generate all possible
    * check evasion moves.
    */
   generate_quiescence_moves(&movelist, game->board, me);

   /* If there are no captures or promotions, simply return the static
    * score - the position is quiet. Or maybe checkmate.
    */
   if (!movelist.num_moves) {
      if (game->board->in_check) {
         /* Detect forced check mates: if the position is a check mate and all
          * moves leading up to this position have been check evasions, the line
          * was actually forced (since all moves are generated when considering
          * check evasions). There might be a simpler mate in the position, but
          * this is better than not detecting the forced line in the first place.
          * If the line is not based on check evasions at every move, we assume
          * that the line was not forced and simply return the static score.
          * Alternatively, we should maybe flag this branch as "dangerous" and
          * let upstream decide what to do about that (eg, research).
          */
         if (!(abs(draft/PLY)&1)) {
            /* If the side to play encountered checks at all nodes up to this
             * point during QS, then the sequence of moves leading to this
             * position was forced and we can return the mate score.
             */
            for (n=draft/PLY; n<=0; n+=2) {
               if (!in_check(draft/PLY - n)) {
                  return static_score;
               }
            }

            return -CHECKMATE + depth;
         }

      }
      WRITE_LOG("  No tactical moves");
      return static_score;
   }

   /* Sort the moves */
   int best_move_score = -CHECKMATE;
   int max_gain = 0;
   int best = score_moves_qs(game, &movelist, depth, score, move_perm);
   best_move_score = score[best];
   if (best_move_score > 0) max_gain = best_move_score;

#ifdef QFUTILITY_PRUNING
   /* Check if the best move (estimated through SEE) looks like it might
    * win material. If not, just return the static score, since we don't expect
    * to gain much here. This is not "safe", but this is the quiescence
    * search, and this suggests that we can't improve our score by
    * captures.
    */
   if (best_move_score < 0 && !check) {
      WRITE_LOG("  No winning captures, pruning (best score %d for %s)",
            best_move_score, move_string(movelist.move[best], NULL));
      return static_score;
   }

   /* Futility pruning during quiescence: if the expected gain from any
    * move in the current position cannot improve alpha, then this node is
    * poor and we might as well skip the rest of QS.
    */
   if (static_score+max_gain+500 < alpha && !check)
      return static_score+max_gain+500;
#endif

   move_t move = movelist.move[best];
   score[best] = CHECKMATE;   /* Make sure that the "best" move is really sorted to the start of the list */
   playmove(game, move);
   if (!check && move_leaves_player_in_check(game, move, me)) {  /* Illegal move! */
      takeback(game);
   } else {
      assert(!player_in_check(game, me));
      moves_searched++;

      LOG_MOVE_BEGIN(alpha, beta, "(Q)");
      new_score = -quiescence(game, depth + 1, draft-PLY, -beta, -alpha);
      LOG_MOVE_END(alpha, beta, "(Q)");
      takeback(game);

      if (new_score > best_score) {
         best_score = new_score;
         if (best_score > alpha) {
            alpha = best_score;
         }
         backup_principle_variation(game, depth + 1, move);
      }
   }

   if (alpha < beta) {
      sort_moves(move_perm, movelist.num_moves, score);

      assert(best == move_perm[0]);

      /* Now search all moves */
      for (n=1; n<movelist.num_moves; n++) {
         int local_beta = beta;

         move = movelist.move[move_perm[n]];

         local_beta = alpha+1;

#ifdef PRUNE_LOSING_CAPTURES
         /* Don't look at losing captures */
         if (score[move_perm[n]] < 0 && !check)
            break;
#endif
         playmove(game, move);
         if (!check && move_leaves_player_in_check(game, move, me)) {  /* Illegal move! */
            takeback(game);
            continue;
         }
         assert(!player_in_check(game, me));
         moves_searched++;

         LOG_MOVE_BEGIN(alpha, local_beta, "(Q)");
         new_score = -quiescence(game, depth + 1, draft-PLY, -local_beta, -alpha);
         LOG_MOVE_END(alpha, local_beta, "(Q)");
         takeback(game);

         if (new_score > best_score) {
            best_score = new_score;
            if (best_score > alpha) {
               alpha = best_score;
            }
            backup_principle_variation(game, depth + 1, move);

            if (best_score >= beta) {           /* Beta cutoff */
               break;
            }
         }
      }
   }

   /* No move was found to improve alpha; examine checking moves */
#undef CHECKS_IN_QUIESCENCE
#ifdef CHECKS_IN_QUIESCENCE
   if (best_score < beta && draft == 0 && !check) {
      generate_checking_moves(&movelist, game->board, me, false);

      //printf("\n");
      //print_board(game->board);
      //for (n=0; n<movelist.num_moves; n++) {
      //   printf("%-6s ", move_string(movelist.move[n], NULL));
      //};
      //printf("\n");


      for (n=0; n<movelist.num_moves; n++) {
         int local_beta = beta;
         move_t move;
         local_beta = alpha+1;

         move = movelist.move[n];
         playmove(game, move);
         if (!check && move_leaves_player_in_check(game, move, me)) {  /* Illegal move! */
            takeback(game);
            continue;
         }
         moves_searched++;

         LOG_MOVE_BEGIN(alpha, local_beta, "(Q)");
         new_score = -quiescence(game, depth + 1, draft-PLY, -local_beta, -alpha);
         LOG_MOVE_END(alpha, local_beta, "(Q)");
         takeback(game);

         if (new_score > best_score) {
            best_score = new_score;
            if (best_score > alpha) {
               alpha = best_score;
            }

            if (best_score >= beta) {           /* Beta cutoff */
               break;
            }
            backup_principle_variation(game, depth + 1, move);
         }
      }
   }
#endif

   return best_score;
}
#ifdef DEBUG_SEARCH_TREE
#undef extension
#endif

/* Store a "combo move"; this is a depth-independent extension of the
 * killer principle.
 */
static void inline store_combo_move(gamestate_t *game, int depth, move_t prev_move, move_t move)
{
#ifdef USE_COMBINATION_MOVES
   if (in_check(0) || is_promotion_move(move) || //is_capture_move(move) ||
      !prev_move) {
      return;
   }
   int c = get_move_player(move) >> 7;
   game->combo[get_move_origin(prev_move)][get_move_destination(prev_move)][c] = move;
#endif
}

/* Store a "counter move"; this is a depth-independent extension of the
 * killer principle.
 */
static void inline store_counter_move(gamestate_t *game, int depth, move_t prev_move, move_t move)
{
#ifdef USE_COUNTER_MOVES
   if (in_check(0) || is_promotion_move(move) || //is_capture_move(move) ||
      !prev_move) {
      return;
   }
   int c = get_move_player(move) >> 7;
   game->counter[get_move_origin(prev_move)][get_move_destination(prev_move)][c] = move;
#endif
}


static inline void store_pv(gamestate_t *game, int depth, move_t move)
{
   if (move) {
      game->length_of_variation[depth+1] = depth+1;
      game->principle_variation[depth+1][depth+1] = move;
   }
}


/* Reduce the search depth for "uninteresting" moves.
 * Any move with a score < 1000 is a candidate, that is, any move sorted
 * below the winning captures and the killer moves.
 *
 * The code below reduces moves that seem poor by 2 ply, but it will only
 * reduce losing captures by 1 ply. This seems to be a good compromise. In
 * addition, we don't reduce moves where the opponent was in check on the
 * move before quite as much. This means that if this move is a capture
 * following an out-of-check move, we do not reduce at all.
 *
 * If late move pruning is used, then the reductions are applied two plies
 * earlier, because the last two plies of the search are a generalised
 * quiescence search.
 */
static int reduce_late_moves(gamestate_t *game, move_t move, int score, int draft, move_t prev_move, bool mate_threat, bool pv_node)
{
#ifdef LATE_MOVE_REDUCTION
   const int lmr_depth = 2;
   int reduction = 0;

   if (score < REDUCTION_LIMIT &&
       !mate_threat &&
       draft > lmr_depth*PLY &&
       prev_move != 0) {
      reduction = -lmr_depth;
      if (pv_node || is_promotion_move(move) || is_capture_move(move) || in_check(0) || in_check(-1))
         reduction = -1;
      WRITE_LOG("  LMR: %d", reduction);
      return reduction * PLY;
   }
#endif
   return 0;
}

static inline bool position_repeated(const gamestate_t *game)
{
   return game->repetition_hash_table[game->board->hash&0xFFFF]>=2
      && count_repetition(game)>=1;
}

static inline bool material_draw(const board_t *board)
{
   if (board->bbp[PAWN] || board->bbp[ROOK] || board->bbp[QUEEN])
      return false;

   if (popcount64(board->bbp[KNIGHT]|board->bbp[BISHOP]) <= 1)
      return true;

   return false;
}

static inline bool legal_draw(const gamestate_t *game)
{
   return game->fifty_counter[game->moves_played-1] >= 101 ||
          position_repeated(game) ||
          material_draw(game->board);
}

static inline bool pawn_ending(const board_t *board)
{
   return !(board->bbp[KNIGHT] || board->bbp[BISHOP] || board->bbp[ROOK] || board->bbp[QUEEN]);
}

static inline bool piece_ending(const board_t *board)
{
   return !board->bbp[PAWN];
}

static inline bool knight_ending(const board_t *board)
{
   bitboard_t pieces = board->bbp[BISHOP] | board->bbp[ROOK] | board->bbp[QUEEN];
   return !(pieces & board->bbc[0]) || !(pieces & board->bbc[1]);
}

static inline bool no_pieces(const board_t *board)
{
   bitboard_t pieces = board->bbp[KNIGHT] | board->bbp[BISHOP] | board->bbp[ROOK] | board->bbp[QUEEN];
   return (pieces & board->bbc[0]) == 0 || (pieces & board->bbc[1]) == 0;
}

/* Asses whether zugzwang is likely to be an issue in the current position;
 * if it is, we will avoid a null-move search.
 */
static inline bool zugzwang_threat(const board_t *board)
{
   bitboard_t pawns, occupied, white_attacks, black_attacks, blocked;

   /* Pawn endings always have a high chance of zugzwang */
   if (pawn_ending(board))
      return true;

   /* Lone piece (no pawn) endings are a likely candidate as well */
   if (piece_ending(board))
      return true;

   /* Situations where the only piece for either side is a knight (which
    * cannot lose a tempo).
    */
   if (knight_ending(board))
      return true;

   /* Situations where one side has no pieces (only pawns, maybe); this
    * includes endings with a bare king.
    */
   if (no_pieces(board))
      return true;

   /* End games where all pawns of a particular side are blocked are other
    * good candidates.
    * Blocked means "can't advance", which means both physically blocked and
    * enemy pawns attacking the stop square.
    */
   get_pawn_attacks(board, &white_attacks, &black_attacks);
   occupied = board->bbc[0] | board->bbc[1];

   blocked = occupied | black_attacks;
   pawns = (board->bbp[PAWN] & board->bbc[0]) << 8;
   if ( (pawns & ~blocked) == 0)
      return true;

   blocked = occupied | white_attacks;
   pawns = (board->bbp[PAWN] & board->bbc[1]) >> 8;
   if ( (pawns & ~blocked) == 0)
      return true;

   return false;
}


/* Alpha-beta search routine.
 * Input:
 *  depth            - the number of plies since the root
 *  draft            - remaining depth (the number of plies to the horizon)
 *  alpha            - the lower bound for the score of this position
 *  beta             - the upper bound for the score of this position
 */
int alphabeta(gamestate_t *game, int depth, int draft, int alpha, int beta)
{
   int new_score = 0;
   hash_table_entry_t *hash;
   movelist_t movelist;
   move_t hash_move;
   move_t prev_move;
   move_t my_prev_move;
   move_t threat_move;
   int move_perm[MAX_MOVES];
   int score[MAX_MOVES];
   int best_score;
   int legal_moves;
   int n = 0;
   int me;
   int extension = 0;
   int hash_flag = HASH_TYPE_ALPHA;
   bool mate_threat = false;
   bool reduction_candidate = false;
   bool avoid_null = false;
   bool pv_node = (beta-alpha) > 1;
   int null_r = 4;

   /* The very first thing we do is to check if this position has occurred
    * three times in the game tree, or whether no irreversible move has
    * been played in 50 moves (100 ply). Either of these conditions means
    * we should flag a draw and return immediately.
    * During the search, we simply score positions as a draw if they have
    * occurred at least once before. From the top level, we don't claim a
    * draw until three repetitions have been made.
    * There is one caveat: the position must not be checkmate. This is
    * relevant for the 50-move rule
    */
   if (legal_draw(game)) {
      if (game->fifty_counter[game->moves_played-1] >= 101 && game->board->in_check) {
         generate_moves(&movelist, game->board, game->side_to_move);
         if (movelist.num_moves == 0)
            return -CHECKMATE + depth;
      }
      return 0;
   }

   int in_alpha = alpha, in_beta = beta;

   /* Mate-distance pruning: check whether an earlier branch in the tree
    * has given us a mate score; if so, we need to adjust the window
    */
   int mate_value = CHECKMATE - depth - 1;
   if (mate_value < beta)
      beta = mate_value;

   mate_value = -CHECKMATE + depth;
   if (mate_value > alpha)
      alpha = mate_value;

   if (alpha >= beta)
      return alpha;

   game->length_of_variation[depth+1] = depth+1;
   game->hash_hit[depth+1] = false;

   /* Maximum search depth, we can't go deeper. */
   if (depth >= MAX_SEARCH_DEPTH) {
      return static_evaluation(game, game->side_to_move, alpha, beta);
   }

   /* Check whether we still have enough time for this ply */
   check_clock();
   if (abort_search) return 0;

   /* Null-move reduction factor, increasing with depth.
    * TODO: tune the parameters 3 and 4
    */
   null_r = 3 + draft / (4*PLY);

   /* Look up this position in the transposition table */
   hash_move = 0;
   hash = query_table_entry(game->transposition_table, game->board->hash);
   if (hash) {
      hash_move = hash->best_move;
      new_score   = score_from_hashtable(hash->score, depth);
      positions_in_hashtable++;
      WRITE_LOG("  Found in TT (flags = 0x%02x): "
                "score %d at distance %d (now %d) and best move %s",
                hash->flags, new_score, hash->depth, draft/PLY,
                move_string(hash->best_move, NULL));
      /* If we searched this position before from the same depth or better,
       * just return the previous score. Otherwise we can still use the
       * recorded move to improve our move ordering.
       * Mate scores are an exception: they should be safe to return
       * regardless of draft since the game has terminated there. This
       * speeds up the search in positions where mate-threats are
       * important.
       * Don't return scores from the hash table if we're close to a 50-move draw.
       * Don't allow exact cutoffs in PV nodes just below the root. The
       * idea behind this is that we always want to get a response to the
       * root move so that we have a "ponder move".
       */
      //if ((hash->depth*PLY > draft || mate_score(new_score)) && game->fifty_counter[game->moves_played] < 80) {
      if (hash->depth*PLY > draft && game->fifty_counter[game->moves_played-1] < 80) {
         bool exact_ok = (depth > 1) || !pv_node;
         if (hash->flags & HASH_TYPE_EXACT && exact_ok) {
            store_pv(game, depth, hash_move);
            game->hash_hit[depth+1] = true;

            return new_score;
         } else if ((hash->flags & (HASH_TYPE_ALPHA|HASH_TYPE_EXACT)) && (new_score<=alpha)) {
            store_pv(game, depth, hash_move);
            game->hash_hit[depth+1] = true;

            return new_score;
         } else if ((hash->flags & (HASH_TYPE_BETA|HASH_TYPE_EXACT)) && (new_score>=beta)) {
            store_pv(game, depth, hash_move);
            game->hash_hit[depth+1] = true;

            return new_score;
         }
      }
   }



   /* We're at the search horizon, go to quiescence search, unless we're in
    * check. In that case the search will be extended by 1 ply by the check
    * extension (below).
    */
   if (draft < PLY && !in_check(0)) {
      WRITE_LOG("  Enter quiescence search");
      return quiescence(game, depth, 0, alpha, beta);
   }

   LOG_CHECK_STATE();

   me = game->side_to_move;
   best_score = -CHECKMATE-1;
   positions_evaluated++;

   /* Store the previous move: this is used to change move ordering on this
    * move. If there is no previous move, then this move will be null.
    */
   prev_move = 0;
   if (game->moves_played>1)
      prev_move = game->move_list[game->moves_played-1];

   my_prev_move = 0;
   if (game->moves_played>2)
      my_prev_move = game->move_list[game->moves_played-2];

   threat_move = 0;
#ifdef ALLOW_NULL_MOVE
    /* Disallow NULL moves if
     *  - we're in check (obviously)
     *  - the last move was a move out-of-check (too dangerous, we may have
     *    something much better that we could otherwise miss)
     *  - the previous move was a null move (the search would collapse into
     *    nothing and we'd just end up researching the position at a
     *    shallower depth - which we've already done thanks to ID)
     *  - we're too close to the horizon (don't skip to QS from a NULL
     *    move)
     *  - there's a mate treat (the null-move is pointless)
     *  - there's a high risk of zugzwang
     */
    avoid_null = avoid_null ||
                 pv_node || 
                 in_check(0) ||
                 in_check(-1) ||
                 prev_move == 0 ||
                 mate_score(beta) ||
                 mate_score(alpha) ||
                 zugzwang_threat(game->board);
    if (!avoid_null) {
       int reduced_draft = draft-null_r*PLY;
#ifdef DEBUG_SEARCH_TREE
       move_t move = 0;
#endif

       /* play null-move */
       LOG_MOVE_BEGIN(beta-1, beta, "(null-move)");
       play_null_move(game);
       moves_searched++;
       new_score = -alphabeta(game, depth+1, reduced_draft, -beta, 1-beta);
       LOG_MOVE_END(beta-1, beta, "(null-move)");
       takeback(game);
       /* Test whether the null-move causes a cutoff, but don't take the
        * cut if this is a PV node.
        * We can still use the threat information.
        */
       if (!pv_node && new_score >= beta && !mate_score(new_score)) {
          hash_move = 0;
          if (reduced_draft > 0)
             store_table_entry(game->transposition_table, game->board->hash, 
                               reduced_draft/PLY, score_to_hashtable(new_score, depth), HASH_TYPE_ALPHA, hash_move);
          return new_score;
       } else {
          /* Retrieve move, store as null-killer on the next ply */
          threat_move = game->best_move[depth + 1];
          if (!pv_node)
             store_null_killer(game, depth + 1, threat_move);
       }

       /* Mate threat detection */
       if (mate_score(new_score) && draft <=5*PLY) {
          threat_move = game->best_move[depth + 1];
          store_mate_killer(game, threat_move, depth + 1, new_score);
          mate_threat = true;
          if (extension <= 0) extension += PLY;
       }
    }
#endif



#undef USE_IID_REDUCTION
#ifdef USE_IID
   /* We don't have a move now, which is bad... try to get one from
    * internal iterative deepening. This improves the move ordering
    * considerably and makes a large difference in speed at larger
    * depths.
    */
   if (hash_move == 0 && draft > 4*PLY) {
      int iid_draft = draft / 2 + PLY;
#ifdef USE_IID_REDUCTION
      if (beta == alpha + 1) iid_draft = draft / 4 + PLY;
#endif
      WRITE_LOG("  Begin IID");

      game->best_move[depth] = 0;
      alphabeta(game, depth, iid_draft, in_alpha, in_beta);
      hash_move = game->best_move[depth];

      WRITE_LOG("  IID found move %s", move_string(hash_move, NULL));

#ifdef USE_IID_REDUCTION
      if (!in_check(0) && !mate_score(alpha) && extension <= 0) {
         hash = query_table_entry(game->transposition_table, game->board->hash);
         if (hash && !mate_score(hash->score) && score_from_hashtable(hash->score, depth) < alpha-100) {
            if (draft <= 8*PLY)
               return score_from_hashtable(hash->score, depth);
            extension -= PLY;
            if (beta == alpha+1) extension -= PLY;
            if (hash->score < alpha-200) extension -= PLY;
         }
      }
#endif
   }
#endif

   /* Generate moves, then sort them based on their estimated value: the
    * move generator gives a crude guess for this, and we can use the
    * killer tables to refine this.
    * The 0-th element of the killer tables is at the root of the tree,
    * rather than at the leaves. The reason for this is simple: we want to
    * reuse the killers as much as possible between iterative deepening.
    */
   generate_moves(&movelist, game->board, me);
   if (movelist.num_moves == 0) {
      if (in_check(0))
            return -CHECKMATE + depth;
      return 0;
   }
   legal_moves = 0;
   game->num_moves[depth] = movelist.num_moves;


   /* Razoring. Avoid if there was only one legal move on the last ply */
   int static_score = -CHECKMATE;
   if (!in_check(0) && (depth == 0 || game->num_moves[depth-1] > 1) && draft < 4*PLY && !mate_score(beta)) {
      int razor_margins[] = { 0, 320, 640, 960, 1280 };
      int razor_margin = razor_margins[draft / PLY] + pv_node*50;
      static_score = static_evaluation(game, me, alpha, beta);
      if (static_score + razor_margin < alpha) {
         int score = quiescence(game, depth, 0, alpha-razor_margin, beta-razor_margin);

         if (score + razor_margin < alpha)
            return score;
      }
   }



   /* Check if this node is a candidate for futility pruning */
   reduction_candidate = !pv_node && draft <= 4*PLY && depth>=1 && !in_check(0);

   /* Determine order in which the moves should be tried */
   int best = score_moves(game, &movelist, depth, hash_move, prev_move,
                           my_prev_move, threat_move, score, move_perm);
   hash_move = movelist.move[best];
   score[best] = CHECKMATE;   /* Make sure that the "best" move is really sorted to the start of the list */

#ifdef CHECK_EXTENSION
   /* Extend all checks by one ply.  */
   if (in_check(0)) {
      /* Single reply out of check, extra extension.
       * Exception: if the only reply is capturing the checking piece,
       * don't allow the extra extension. The idea behind this is that the
       * last move is likely to be bad and cut.
       * Try: award the extension on a second pass if the capture fails
       * low, the last move may have been a winning sacrifice.
       */
#define SINGLE_REPLY_OUT_OF_CHECK
#ifdef SINGLE_REPLY_OUT_OF_CHECK
      if (movelist.num_moves == 1) {
         bool search_with_extension = true;
         move_t move = movelist.move[0];
         moves_searched++;
         playmove(game, move);
         if (is_capture_move(movelist.move[0])) {
            new_score = -alphabeta(game, depth+1, draft, -beta, -alpha);

            /* We expect this move to fail high. If it didn't, there may be
             * a threat and we extend the search (the previous move fails
             * high instead of low).
             */
            if (new_score < beta)
               search_with_extension = false;
         }
         if (search_with_extension) {
            new_score = -alphabeta(game, depth+1, draft+PLY, -beta, -alpha);
         }
         takeback(game);
         if (new_score > alpha)             /* The line is part of the principle variation */
            backup_principle_variation(game, depth + 1, move);

         return new_score;
      }
#endif
      extension += PLY;
      WRITE_LOG("  Check extension: %d", extension/PLY);
   }
#endif

   /* We kept track of the best move (probably the hash move), search that
    * before calling qsort to sort the other moves. If it causes a cut-off,
    * we never search the other moves at all. This speeds up the search a
    * bit. The duplication of code here is a bit ugly though and this
    * could be cleaned up.
    */
   int local_extension = max(extension, 0);
   move_t move;
   move = movelist.move[best];

   /* Store the in-check flag for local tests.
    * Check evasion moves are always legal, so if we are in check now we
    * don't need to check the moves for legality. This saves the expensive
    * "in check" test.
    */
   bool check = in_check(0);

   int max_nodes = 0;
   int nodes = 0;
   move_t max_nodes_move = 0;

   /* Play move and evaluate the resulting position recursively -
    * assuming the move was legal.
    */
   playmove(game, move);
   if (!check && move_leaves_player_in_check(game, move, me)) {  /* Illegal move! */
      game->num_moves[depth]--;
      takeback(game);
   } else {
      assert(!player_in_check(game, me));
      legal_moves++;
      moves_searched++;

      nodes = moves_searched;
      LOG_MOVE_BEGIN(alpha, beta, "(PV)");
      new_score = -alphabeta(game, depth+1, draft+local_extension-PLY, -beta, -alpha);
      LOG_MOVE_END(alpha, beta, "(PV)");

      /* For forced mates further down the tree, we should adjust the score
       * by one ply: mate one node down is a mate-in-one-ply from this ply
       * and a mate-in-two-ply one node further up.
       * We need two checks: one to see if we're mating and one to see if
       * we're being mated.
       */
      takeback(game);

      nodes = moves_searched - nodes;
      if (nodes > max_nodes) { max_nodes = nodes; max_nodes_move = move; }

      best_score = new_score;

      if (best_score > alpha) {           /* New best line found */
         WRITE_LOG("  New best move (PV), %d > %d", best_score, alpha);
         hash_flag = HASH_TYPE_EXACT;
         alpha = best_score;

         /* Store moves that improve alpha in the counter and
          * cmobination move slots. This seems to perform slightly
          * better than following the same logic as killer moves,
          * probably because this means we still get a slightly better
          * move ordering even if the move does not cause a cut-off.
          */
         store_counter_move(game, depth, prev_move, move);
         store_combo_move(game, depth, my_prev_move, move);

         /* Update and store principle variation */
         backup_principle_variation(game, depth + 1, move);

      }
   }

   /* Search remaining moves if no cut-off yet */
   if (alpha < beta) {
      /* Get the static evaluation score - if we haven't calculated it
       * already and are not in check.
       */
      if (!in_check(0) && static_score == -CHECKMATE)
         static_score = static_evaluation(game, me, alpha, beta);

      sort_moves(move_perm, movelist.num_moves, score);

      assert(best == move_perm[0]);

      for (n = 1; n<movelist.num_moves; n++) {
         int local_extension = extension;
         int local_beta = alpha + 1;
         bool do_normal_search = true;

         move = movelist.move[move_perm[n]];

         if (local_extension == 0) {
            local_extension = reduce_late_moves(game, move, score[move_perm[n]], draft,
                                                                        prev_move, mate_threat, pv_node);
            if (is_capture_move(move) && is_good_capture(game, depth, move))
               local_extension = 0;
         }

         /* Play move and evaluate the resulting position recursively -
          * assuming the move was legal.
          */
         playmove(game, move);
         if (!check && move_leaves_player_in_check(game, move, me)) {  /* Illegal move! */
            game->num_moves[depth]--;
            takeback(game);
            continue;
         }
         assert(!player_in_check(game, me));
         legal_moves++;
         moves_searched++;

         nodes = moves_searched;
 
         /* Check for history reductions. This reduces moves that have
          * historically failed low in the current search. Discussed by Ed
          * Schroeder at http://www.top-5000.nl/authors/rebel/hr.htm
          * We avoid the reduction in the following cases:
          *
          * - This move has already been extended (or reduced).
          * - This move is a capture
          * - This move is a check or an escape-from-check
          * - The move is not "bad enough" that it has reached the threshold yet.
          * - The static score for the current position is not below alpha.
          */
         if (local_extension == 0 && reduce_from_history(game, move, prev_move, my_prev_move, static_score, alpha))
            local_extension = -PLY;

         /* Futility pruning: if the static evaluation is very poor, then try
          * to weed out moves that do not improve the static score, either by
          * going directly to quiescence search, or by reducing the search
          * depth by 1 or 2 ply.
          */
#ifdef FUTILITY_PRUNING
         if (reduction_candidate && !in_check(0) && local_extension==0) {
            static const int new_depth[6] = {   0, 0,   0,   0,   2,    0 };
            int margin = futility_margins[draft / PLY];
            int d = new_depth[draft / PLY]*PLY;

            if ((static_score + margin) < alpha) {
               LOG_MOVE_BEGIN(alpha, local_beta, "(futility)");
               new_score = -alphabeta(game, depth+1, d, -local_beta, -alpha);
               LOG_MOVE_END(alpha, local_beta, "(futility)");

               /* Research moves that were pruned by futility pruning but turn
                * out to fail high after the quiescence search.
                */
               if (new_score > alpha && new_score < beta) {
                  LOG_MOVE_BEGIN(alpha, local_beta, "(futility fail high, research)");
                  new_score = -alphabeta(game, depth+1, draft-PLY, -local_beta, -alpha);
               }
               do_normal_search = false;
            }
         }
#endif

         if (do_normal_search) {
            LOG_MOVE_BEGIN(alpha, local_beta, "");
            new_score = -alphabeta(game, depth+1, draft+local_extension-PLY, -local_beta, -alpha);
            LOG_MOVE_END(alpha, local_beta, "");
         }

         /* We expect the search to have failed low. If this is not the
          * case, do a re-search.
          * There are two possibilities:
          *  1. This is a PV node and a non-PV move that was searched with
          *     a null window needs to be re-searched with an open window.
          *  2. A reduced move does not fail low. A research is necessary to
          *     verify this.
          *
          * We do both of these researches separately, which means we may
          * sometimes redo the search twice. Some experimentation may be
          * needed to work out whether this is a real problem in practice
          * (probably not, thanks to the transposition table). It seems to be
          * important to keep the window open when re-searching extended
          * moves.
          */
         if (pv_node && new_score > alpha) {
            LOG_MOVE_BEGIN(alpha, beta, "(PVS verify)");
            local_extension = -1;      /* TODO: does this make sense? */
            new_score = -alphabeta(game, depth+1, draft+local_extension-PLY, -beta, -alpha);
            LOG_MOVE_END(alpha, beta, "(PVS verify)");
         }

         /* Research moves that were reduced but now fail high (improve alpha)
          * on normal depth (confirm the fail high).
          */
         if (new_score > alpha && local_extension < 0) {
            LOG_MOVE_BEGIN(alpha, beta, "(research after reduction fails high)");
            new_score = -alphabeta(game, depth+1, draft-PLY, -beta, -alpha);
            LOG_MOVE_END(alpha, beta, "(research after reduction fails high)");
         }

         /* For forced mates further down the tree, we should adjust the score
          * by one ply: mate one node down is a mate-in-one-ply from this ply
          * and a mate-in-two-ply one node further up.
          * We need two checks: one to see if we're mating and one to see if
          * we're being mated.
          */
         takeback(game);

         nodes = moves_searched - nodes;
         if (nodes > max_nodes) { max_nodes = nodes; max_nodes_move = move; }

         /* Abort if we've exceeded our allowed time to think */
         if (abort_search) return 0;

         /* Record the result in the history reduction table */
         update_history_reduction_table(game, move, prev_move, my_prev_move, new_score, alpha);

         if (new_score > best_score) {
            best_score = new_score;
            hash_move = movelist.move[move_perm[n]];

            if (best_score > alpha) {           /* New best line found */
               WRITE_LOG("  New best move, %d > %d", new_score, alpha);
               hash_flag = HASH_TYPE_EXACT;
               alpha = best_score;

               /* Store moves that improve alpha in the counter and
                * cmobination move slots. This seems to perform slightly
                * better than following the same logic as killer moves,
                * probably because this means we still get a slightly better
                * move ordering even if the move does not cause a cut-off.
                */
               store_counter_move(game, depth, prev_move, move);
               store_combo_move(game, depth, my_prev_move, move);

               /* Update and store principle variation */
               backup_principle_variation(game, depth + 1, move);

               if (alpha >= beta)
                  break;
            }
         }
      }
   } else {
      branches_pruned_1st++;
   }

   if (alpha >= beta) {       /* Beta cutoff */
      /* Store good moves in the killer slots, but only if:
       *  - we were not in check at the beginning of this move
       *  - the move is not a promotion (already high in the list)
       *  - the move was not a capture (already high in the list)
       */
      store_killer(game, move, depth);
      store_mate_killer(game, move, depth, best_score);
      update_history(game, move, depth*depth);

      /* Store good captures */
      store_good_capture(game, move, score[move_perm[n]], depth);

      WRITE_LOG("  Fail high, %d >= %d", alpha, beta);
      hash_flag = HASH_TYPE_BETA;
      branches_pruned++;
   }

   /* Store killer even if this entry is exact; presumably this is still a good move. */
#ifdef STORE_EXACT_KILLER
   if (hash_flag == HASH_TYPE_EXACT) {
      store_killer(game, hash_move, depth);
      store_mate_killer(game, hash_move, depth, best_score);
      update_history(game, hash_move, depth*depth);
   }
#endif

   /* No legal moves - either checkmate or stalemate
    * Should be stalemate, since we use legal move generation when in
    * check, so "no legal moves" should have been caught earlier.
    */
   if (legal_moves == 0) {
      assert(!in_check(0));
      best_score = 0;
   }

   /* Store best move for retrieval from one level up in the search (for
    * IID)
    */
   game->best_move[depth] = hash_move;

   /* store evaluation in the transposition table */
   if (hash_flag & HASH_TYPE_ALPHA)
      hash_move = max_nodes_move;
   if (legal_moves)
   store_table_entry(game->transposition_table, game->board->hash, 
                     draft/PLY, score_to_hashtable(best_score, depth), hash_flag, hash_move);

   WRITE_LOG("  Store in TT (flags = 0x%02x):"
         "Score %d at distance %d and best move %s",
         hash_flag, best_score, draft/PLY,
         move_string(hash_move, NULL));

   return best_score;
}
