#include "config.h"
#include "game.h"
#include "move.h"
#include "killer.h"

/* Store moves that kill a branch in the killer slots, but only if:
 *  - we were not in check at the beginning of this move
 *  - the move is not a promotion (already high in the tree)
 *  - the move was not a capture (already high in the tree)
 */
void store_killer(gamestate_t *game, move_t move, int depth)
{
   if (game->board->in_check || is_promotion_move(move) || is_capture_move(move)) {
      return;
   }

   assert( (game->killer[0][depth] != game->killer[1][depth]) || (game->killer[0][depth] == 0));

   if (moves_are_equal(move, game->killer[0][depth])) {
      /* The move was the first killer - do nothing */
#ifdef USE_THIRD_KILLER
   } else if (moves_are_equal(move, game->killer[1][depth])) {
      /* The move was the second killer (out of three) - promote it to
       * first killer, leaving the third killer untouched.
       */
      game->killer[1][depth]=game->killer[0][depth];
      game->killer[0][depth]=move;
#endif
   } else {
      /* This was either the last killer (out of 2 or 3), or it's a new
       * move. Either way, Degrade first killer to second killer (etc) and
       * store the new first killer.
       */
#ifdef USE_THIRD_KILLER
      game->killer[2][depth]=game->killer[1][depth];
#endif
      game->killer[1][depth]=game->killer[0][depth];
      game->killer[0][depth]=move;
   }
}

void store_mate_killer(gamestate_t *game, move_t move, int depth, int best_score)
{
#ifdef USE_MATE_KILLER
   if (best_score >= (CHECKMATE-1000))
      game->mate_killer[depth] = move;
#endif
}

void store_null_killer(gamestate_t *game, int depth, move_t move)
{
#ifdef USE_NULL_KILLER
   if (!is_capture_move(move))
      game->null_killer[depth] = move;
#endif
}

bool is_killer(const gamestate_t *game, int depth, move_t prev_move, move_t move)
{
   if (moves_are_equal(game->killer[0][depth], move) ||
       moves_are_equal(game->killer[1][depth], move))
      return true;
   return false;
}

