#include "game.h"
#include "movegen.h"
#include "move.h"
#include "scoremove.h"
#include "history.h"
#include "killer.h"
#include "see.h"

static bool inline is_combo(const gamestate_t *game, int depth, move_t prev_move, move_t move)
{
#ifdef USE_COMBINATION_MOVES
   int c = get_move_player(move) >> 7;
   return game->combo[get_move_origin(prev_move)][get_move_destination(prev_move)][c] == move;
#else
   return false;
#endif
}

static bool inline is_counter(const gamestate_t *game, int depth, move_t prev_move, move_t move)
{
#ifdef USE_COUNTER_MOVES
   int c = get_move_player(move) >> 7;
   return game->counter[get_move_origin(prev_move)][get_move_destination(prev_move)][c] == move;
#else
   return false;
#endif
}

static bool is_good_capture(const gamestate_t *game, int depth, move_t move)
{
#ifdef USE_GOOD_CAPTURE
   if (moves_are_equal(game->good_capture[0][depth], move) ||
       moves_are_equal(game->good_capture[1][depth], move))
      return true;
#endif
   return false;
}

/* Determine the "score" for each move: this is not related to the
 * evaluation of the current position, but rather to the relative ordering
 * of the moves during the search.
 * Returns TRUE if the move from the hashtable was found in the move list.
 */
int score_moves(const gamestate_t *game, const movelist_t *movelist, int depth, const move_t hash_move, const move_t prev_move, const move_t my_prev_move, const move_t threat_move, int *restrict score, int *restrict move_perm)
{
   int n;
   int best_move = 0;
   int best_score = -CHECKMATE;

   for (n=0; n<movelist->num_moves; n++) {
      move_t move = movelist->move[n];
      int piece = get_move_piece(move)&PIECE;
      int colour = get_move_player(move) >> 7;
      int from_row, to_row;

      score[n] = 0;
      move_perm[n] = n;

      /* Use the piece square tables to pre-sort moves */
#ifdef USE_DYNAMIC_PSQ_FOR_MOVE_ORDER
      score[n] += game->dynamic_psq[piece][colour][get_move_destination(move)] -
                  game->dynamic_psq[piece][colour][get_move_origin(move)];
#else
      score[n] += piece_square_table[piece][colour][get_move_destination(move)] -
                  piece_square_table[piece][colour][get_move_origin(move)];
#endif

      /* Give the highest priority to the move from the hash table, if it
       * exists.
       */
      if (moves_are_equal(hash_move, move)) {
         score[n] = SORT_HASHMOVE;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }

      /* TODO: advances of free pawn, advances of candidate pawns */

#ifdef USE_MATE_KILLER
      /* Mate killer: a killer move that produced a mate score */
      if (moves_are_equal(game->mate_killer[depth], move)) {
         score[n] = SORT_MATEKILLER;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }

      /* A mate killer from 2 plies ago, a good move but not quite as good
       * as a normal killer.
       */
      if (SORT_MATEKILLER2 > 0 && depth > 2 && moves_are_equal(game->mate_killer[depth-2], move)) {
         score[n] = SORT_MATEKILLER2;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }
#endif

      /* Promotion moves */
      if (is_promotion_move(move)) {
         int see_score = static_exchange_evaluation(game->board, move);
         int ppiece = get_promotion_piece(move) & PIECE;
         if (see_score >= 0)
            score[n] += SORT_PROMOTION + see_score + piece_value(ppiece) - (ppiece == ROOK)*200;
         else
            score[n] += see_score;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }

      /* Regular captures
       * The score is 1100+the SEE score, which sorts winning captures below
       * promotion moves (with a score >= 1150) and losing captures with a
       * score <=1000 below everything else.
       * TODO: bonus for capturing a just-promoted piece?
       */
      if (is_capture_move(move)) {
         if (is_good_capture(game, depth, move)) {
            score[n] += 500;
         } else {
            int see_score = static_exchange_evaluation(game->board, move);
            if (see_score >= 0)
               score[n] += SORT_WINCAPTURE+see_score;
            else
               score[n] += see_score;
         }
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }

      /* Award killers a higher score, so they are searched first
       * Given the weights here, winning captures are normally searched
       * before killer moves, losing captures are searched after killer
       * moves.
       */
      if (moves_are_equal(game->killer[0][depth], move)) {
         score[n] = SORT_KILLER1;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }
      if (moves_are_equal(game->killer[1][depth], move)) {
         score[n] = SORT_KILLER2;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }
#ifdef USE_THIRD_KILLER
      if (moves_are_equal(game->killer[2][depth], move)) {
         score[n] = SORT_KILLER3;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }
#endif

#if defined HISTORY_REDUCTION
      /* Moves that have historically scored badly get sorted lower down on
       * the list. This is important when using the counter-move heuristic,
       * because at larger search depth it may turn up moves that don't
       * work so well in other parts of the tree.
       */
      if (get_history_reduction_score(game, move) < 75) {
         int s = get_history_reduction_score(game, move);
         score[n] -= (100 - s) * 4;
      }
#endif

      /* Counter moves, a generalisation of the Killer concept.
       * The question is whether these should come before, or after
       * killers...
       * Doesn't seem to make much of a difference, but it sometimes hurts
       * badly if they are sorted before the killer moves, so they should
       * probably go between the killers and the losing captures.
       */
      if (is_counter(game, depth, prev_move, move)) {
         score[n] += SORT_COUNTER;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }

      /* Combo moves, a similar idea to a counter move.
       * The question is whether these should come before, or after
       * killers...
       * Doesn't seem to make much of a difference, but it sometimes hurts
       * badly if they are sorted before the killer moves, so they should
       * probably go between the killers and the losing captures.
       */
      if (is_combo(game, depth, my_prev_move, move)) {
         score[n] += SORT_COMBO;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }

#ifdef USE_NULL_KILLER
      /* Null killer: a move that failed high after a null move */
      if (moves_are_equal(game->null_killer[depth], move)) {
         score[n] += SORT_NULLKILLER;
         //continue;
      }
#endif

      /* Also look at killers from earlier positions. This doesn't make a
       * big difference, but it speeds things up very slightly still and
       * isn't likely to hurt.
       */
      if (depth>2 && moves_are_equal(game->killer[0][depth-2], move)) {
         score[n] += SORT_PREVKILLER1;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }
      if (depth>2 && moves_are_equal(game->killer[1][depth-2], move)) {
         score[n] += SORT_PREVKILLER2;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }

      /* Castling moves: these should come before other king moves */
      if (is_castle_move(move)) {
         score[n] += SORT_CASTLE;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = n;
         }
         continue;
      }

#ifdef MOVE_THREATENED_PIECE
      /* Move away a piece that is captured after a null move */
      if (threat_move && get_move_origin(move) == get_move_destination(threat_move)) {
         score[n] += SORT_MOVETHREAT;
         //continue;
      }
#endif

      /* Static move-ordering heuristics.
       * Move the most advanced piece forward
       */
      if (colour) {
         from_row = 7 - unpack_rank(get_move_origin(move));
         to_row = 7 - unpack_rank(get_move_destination(move));
      } else {
         from_row = unpack_rank(get_move_origin(move));
         to_row = unpack_rank(get_move_destination(move));
      }
      score[n] += ((from_row<<4) + (to_row))>>2;

      /* Finally, add history scores to the move */
#ifdef USE_HISTORY_HEURISTIC
      if (game->max_history && !game->board->in_check && !is_promotion_move(move) && !is_capture_move(move)) {
         int history_score = get_move_history_score(game, move);
         score[n] += 50*history_score/game->max_history;
      }
#endif

      if (score[n] > best_score) {
         best_score = score[n];
         best_move = n;
      }
   }

   return best_move;
}

int score_moves_qs(const gamestate_t *game, const movelist_t *movelist, int depth, int *restrict score, int *restrict move_perm)
{
   int best_move_score = -CHECKMATE;
   int best = 0;
   int n;

   for (n=0; n<movelist->num_moves; n++) {
      move_t move = movelist->move[n];
      move_perm[n] = n;
      score[n] = 0;

      /* Promotion moves */
      if (is_promotion_move(move)) {
         int gain = piece_value(get_promotion_piece(move)) - piece_value(PAWN);
         if (is_capture_move(move))
            gain += piece_value(get_captured_piece(move));
         else
            gain += static_exchange_evaluation(game->board, move);
         score[n] += gain;
         if (score[n] > best_move_score) {
            best = n;
            best_move_score = score[n];
         }
         continue;
      }

      /* Capture; use SEE to estimate relative worth of captures */
      if (is_capture_move(move)) {
         int gain = static_exchange_evaluation(game->board, movelist->move[n]);
         score[n] += gain;
         if (score[n] > best_move_score) {
            best = n;
            best_move_score = score[n];
         }
         continue;
      }

      /* Neither capture nor promotion, must be check evasion */

      if (score[n] > best_move_score) {
         best = n;
         best_move_score = score[n];
      }
   }

   return best;
}
