/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef CONFIG_H
#define CONFIG_H

#ifdef __STDC_VERSION__
#  if __STDC_VERSION__ < 199901L 
#     ifdef __GNUC__
#        define restrict __restrict
#     else
#        define restrict 
#     endif
#  endif
#else
#  ifdef __GNUC__
#     define restrict __restrict
#    else
#     define restrict 
#  endif
#endif

/* Define to print out the entire search tree, useful for debugging
 * purposes.
 */
#undef DEBUG_SEARCH_TREE

#define MAX_SEARCH_DEPTH 60       /* maximum depth of search tree */

#define HARD_HORIZON -20

#define MAX_TOTAL_DEPTH (MAX_SEARCH_DEPTH - HARD_HORIZON)

/* Use or don't use rotated bitboards for rook file attacks and bishop
 * attacks.
 */
#undef USE_ROTATED_BITBOARDS

/* Number of entries in the hash table; one entry is 24 bytes
 * Since we actually use two tables, the number of available entries (and
 * the memory use) is actually twice as big.
 * NB: this should be a power of 2!
 */
#define HASH_TABLE_SIZE (1024*1024)    // ~48 MB

/* Evaluation table size */
#define EVAL_TABLE_SIZE (4*1024)

/* Number of entries in the pawn structure hash table */
#define PAWN_TABLE_SIZE (32*1024)

/* Number of entries in piece-square evaluation hash table */
#define PIECE_TABLE_SIZE (2*32*1024)

/* Allow lazy evaluation: if the score is outside the alpha-beta window
 * already, don't do the more expensive parts of the evaluation.
 * Caveat: this will mis-evaluate some positions in which king safety is
 * very bad but the attacker has sacrificed some material.
 * Doesn't seem to be a problem in practice, but does hurt test scores,
 * obviously.
 */
#undef ALLOW_LAZY_EVALUATION

/* Allow or disallow lazy evaluation in PV nodes. */
#undef ALLOW_LAZY_EVALUATION_IN_PV

/* Check extensions: extend the search by one ply on check (the exact
 * conditions under which to extend the search can be tuned)
 */
#define CHECK_EXTENSION

/* Tune how the check extension works: extend all checks, or just repeated
 * check.
 */
#define EXTEND_ALL_CHECKS

/* Late move reductions. The idea is that moves that are low in the move
 * list are generally less interesting than moves that are higher up on the
 * list, and we don't need to spend as much time on them. If this buys us
 * some extra plies, we will detect whatever interesting moves there are
 * at that point anyway.
 */
#define LATE_MOVE_REDUCTION

 /* History reductions. Reduce the search depth for moves that have
  * historically failed low, if certain conditions are met.
  * Discussed by Ed Schroeder at http://www.top-5000.nl/authors/rebel/hr.htm
  */
#define HISTORY_REDUCTION

/* Futility pruning: at the frontier nodes (horizon distance for the current
 * position == 2), if the static evaluation plus the gain of a minor piece
 * does not improve alpha, reduce the search depth.
 */
#define FUTILITY_PRUNING

/* Futility pruning during the quiescence search.
 * Disabled, because it doesn't seem to improve anything.
 */
#define QFUTILITY_PRUNING

/* Use three killer slots rather than two.
 * Needs more testing!
 */
#define USE_THIRD_KILLER

/* "Mate killer": a killer move that produces a mate score. As the third
 * killer above, this doesn't really seem to do much...
 */
#define USE_MATE_KILLER

/* Store killer even if this entry is exact; presumably this is still a good move. */
#undef STORE_EXACT_KILLER

/* Good captures: these are captures that are nominally bad (based on SEE score)
 * but actually turn out to be ok after trying them (ie, they cause a
 * cut-off). These don't qualify for killer moves, but they should still be
 * sorted slightly higher up in the movelist than other bad captures.
 */
#undef USE_GOOD_CAPTURE

/* "Counter move": an extension to the killer idea, a killer move indexed
 * by the origin and destination for the previous move rather than by ply.
 * Doesn't seem to help much.
 * Should check what difference keeping two slots for this (as for killer
 * moves) makes.
 */
#define USE_COUNTER_MOVES

/* Evasion move: a similar idea, indexed by the thread move from a
 * null-move search.
 */
#undef USE_EVASION_MOVES

/* Similar to counter moves, a "followup move" is a move that was
 * historically good following a move we just played. This is not very
 * likely to improve tactical strength much, since such a move is probably
 * in the killer slot anyway.
 */
#define USE_COMBINATION_MOVES

/* Use or don't use null-moves during the alpha-beta search */
#define ALLOW_NULL_MOVE

/* Store refutation of the null move; in other words, the "threat" move */
#define USE_NULL_KILLER

/* Use the threat move from a NULL move search to improve move ordering: if
 * a piece is hung, move it away.
 */
#undef MOVE_THREATENED_PIECE

/* Use or don't use history heuristics - it helps in some cases and hurts
 * in others.
 */
#define USE_HISTORY_HEURISTIC

/* Prune losing captures (as determined by SEE) during quiescence search.
 * This reduces the size of the search tree enormously.
 */
#define PRUNE_LOSING_CAPTURES

/* Allow cut-offs in search due to known draws.
 * Requires probing a heuristic function, which may slow things down.
 */
#define PROBE_ENDGAME_HEURISTIC

/* Use internal iterative deepening to find a "good move" if there is no
 * move from the hash table.
 */
#define USE_IID

#define USE_DYNAMIC_PSQ_FOR_MOVE_ORDER

#endif
