/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef SMP_H
#define SMP_H

#include <stdbool.h>
#include "game.h"
#include "move.h"

#ifdef SMP

#include <pthread.h>
#define MAX_THREADS 16
#define MAX_SPLIT    8

#ifdef __APPLE__
#define OSX_SPINLOCK
#else
//#define LOCK_PTHREAD_MUTEX
#define LOCK_PTHREAD_SPIN
#endif

/* Test if POSIX spinlocks are available */
#if defined LOCK_PTHREAD_SPIN 
#include <unistd.h>
#if !defined _POSIX_SPIN_LOCKS || _POSIX_SPIN_LOCKS < 0
#warning POSIX spinlocks unavailable, replaced with mutex
#undef LOCK_PTHREAD_SPIN
#define LOCK_PTHREAD_MUTEX
#endif
#endif

#if defined LOCK_PTHREAD_MUTEX
   typedef pthread_mutex_t lock_t;

   #define init_lock(x)  pthread_mutex_init(x, NULL)
   #define acquire_lock(x)    pthread_mutex_lock(x)
   #define release_lock(x)  pthread_mutex_unlock(x)

   #define LOCK_DESCRIPTION "POSIX mutex"

#elif defined LOCK_PTHREAD_SPIN
   typedef pthread_spinlock_t lock_t;

   #define init_lock(x)  pthread_spin_init(x, PTHREAD_PROCESS_PRIVATE)
   #define acquire_lock(x)    pthread_spin_lock(x)
   #define release_lock(x)  pthread_spin_unlock(x)

   #define LOCK_DESCRIPTION "POSIX spinlock"

#elif defined OSX_SPINLOCK

   #include <libkern/OSAtomic.h>

typedef OSSpinLock lock_t;

   #define init_lock(x) (*(x) = 0)
   #define acquire_lock(x) OSSpinLockLock(x)
   #define release_lock(x) OSSpinLockUnlock(x)

   #define LOCK_DESCRIPTION "OS X spinlock"

#else
   typedef volatile int lock_t;

   /* Hand-written assembly spin-lock code.
    * Used by Crafty and Stockfish, apparently from the Linux kernel.
    */

   static void inline acquire_lock(lock_t *lock) {
     int dummy;
     asm __volatile__(
         "1:          movl    $1, %0"   "\n\t"
         "            xchgl   (%1), %0" "\n\t"
         "            testl   %0, %0"   "\n\t"
         "            jz      3f"       "\n\t"
         "2:          pause"            "\n\t"
         "            movl    (%1), %0" "\n\t"
         "            testl   %0, %0"   "\n\t"
         "            jnz     2b"       "\n\t"
         "            jmp     1b"       "\n\t"
         "3:"                           "\n\t"
         :"=&q"(dummy)
         :"q"(lock)
         :"cc", "memory");
   }

   static void inline release_lock(lock_t *lock) {
     int dummy;
     asm __volatile__(
         "movl    $0, (%1)"
         :"=&q"(dummy)
         :"q"(lock));
   }

   #define init_lock(x) (*(x) = 0)

   #define LOCK_DESCRIPTION "Custom spinlock"
#endif


typedef struct split_t {
   struct gamestate_t *game;      /* Split point information */
   movelist_t *movelist;
   int static_score;
   volatile uint32_t workers;
   volatile int alpha;
   volatile int beta;
   volatile int score;
   volatile int draft;
   volatile int depth;
   volatile move_t best_move;
   volatile bool stop;
   struct split_t *parent;
   lock_t lock;
} split_t;

extern int get_number_of_cores(void);

extern void init_threads(int num_threads);
extern void kill_threads(void);
extern void wake_all_threads(void);
extern void sleep_all_threads(void);
extern int get_number_of_threads(void);
extern bool thread_should_stop(int id);
extern void split(gamestate_t *game, movelist_t *movelist, int static_score, int *alpha, int *beta, int depth, int draft, int *score, move_t *best_move);
extern void stop_split_search(split_t *split);

extern void share_static_psq(combined_psq_t *psq);
extern void copy_game_history(const gamestate_t *from);
#else

#define MAX_THREADS  1
#define MAX_SPLIT    1

#define get_number_of_cores() 1
#define share_static_psq(x)  (void)0
#define copy_game_history(x) (void)0
#define wake_all_threads()   (void)0
#define sleep_all_threads()  (void)0
#define split(game, movelist, static_score, alpha, beta, depth, draft, score, move) (void)0

#define init_threads(num_threads)   (void)0
#define kill_threads()              (void)0
#define get_number_of_threads()     1

#endif

#endif
