/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <math.h>
#include "config.h"
#include "bitboard.h"
#include "board.h"
#include "bits.h"
#include "movegen.h"
#include "heuristic.h"
#include "names.h"
 
static bitboard_t get_in_between(int sq1, int sq2)
{
   return connecting_ray[sq1][sq2] ^ (make_bitboard_square(sq1)|make_bitboard_square(sq2));
}

static int max(int x, int y) { return (x>y)?x:y; }

static int separation(int sq1, int sq2)
{
   int rank1 = unpack_rank(sq1);
   int rank2 = unpack_rank(sq2);
   int file1 = unpack_file(sq1);
   int file2 = unpack_file(sq2);

   return max(abs(rank1 - rank2), abs(file1 - file2));
}

int heuritsic_krkb(const board_t *board, int side_to_move)
{
   assert(onebit64(board->bbp[BISHOP]));
   assert(onebit64(board->bbp[ROOK]));
   assert(!board->bbp[PAWN]);
   assert(!board->bbp[KNIGHT]);
   assert(!board->bbp[PAWN]);
   const bitboard_t corner = A1_MASK | H1_MASK | A8_MASK | H8_MASK;
   const bitboard_t xcorner = B2_MASK | G2_MASK | B7_MASK | G7_MASK;
   bitboard_t king[2], rook, minor, occ;
   bitboard_t ka[2], rook_attacks, bishop_attacks, bishop_colour;
   int attacker, defender;
   int dks, aks, rs, bs;

   /* The attacking party is the party with the rook */
   attacker = WHITE;
   if (board->bbp[ROOK] & board->bbc[1]) {
      attacker = 1;
   }
   defender = attacker^1;

   dks = bitscan64(board->bbp[KING] & board->bbc[defender]);
   aks = bitscan64(board->bbp[KING] & board->bbc[attacker]);
   rs  = bitscan64(board->bbp[ROOK]);
   bs  = bitscan64(board->bbp[BISHOP]);

   king[defender] = make_bitboard_square(dks);
   king[attacker] = make_bitboard_square(aks);
   rook = make_bitboard_square(rs);
   minor = make_bitboard_square(bs);

   occ = king[0] | king[1] | rook | minor;

   ka[attacker] = king_attack[aks];
   ka[defender] = king_attack[dks];
   rook_attacks = get_rook_attack_bitboard(board, rs) & ~king[attacker];
   bishop_attacks = get_bishop_attack_bitboard(board, bs) & ~king[defender];

   bool check = false;
   if (side_to_move == attacker) {
      if (bishop_attacks & king[attacker]) check = true;
   } else {
      if (rook_attacks & king[defender]) check = true;
   }

   bitboard_t safe_defender = ~(ka[attacker] & ~ka[defender]);
   safe_defender &= ~(ka[attacker] & rook_attacks);
   safe_defender &= ~(rook_attacks & ~ka[defender]);
   safe_defender &= ~king[defender];

   bitboard_t safe_attacker = ~(bishop_attacks | ka[defender]);
   safe_attacker |= ka[attacker] & ~bishop_attacks;

   bitboard_t aattack = ka[attacker] | rook_attacks;

   bishop_colour = board_light;
   if (minor & board_dark) bishop_colour = board_dark;

   if (side_to_move == defender) {
      /* Hanging rook */
      if ((safe_attacker & rook) == board_empty) return heuristic_draw;

      /* Pinned bishop */
      bool pinned = false;
      if (rook_attacks & minor && orth[rs] & king[defender] &&
            (connecting_ray[dks][rs] & king[attacker]) == board_empty) {
         pinned = true;
         if (ka[attacker] & minor) return heuristic_loss;

         /* Stalemate trap */
         if ( !((ka[defender]&corner) && (king[attacker]&knight_attack[dks]&board_edge)) &&
             (king_attack2[dks]&minor) == board_empty)
            return heuristic_loss;
      }

      if ((minor & safe_defender) == board_empty) {
         /* No safe squares for the bishop, loss unless we can attack the rook */
         if ((bishop_attacks & safe_defender) == board_empty) {
            if ((ka[attacker] & rook) || !(king_attack2[dks]&rook))
               return heuristic_loss;
         }
      }

      /* Mate-threat */
      bool mate_threat = false;
      if (king[defender] & board_edge &&
          (king[defender] & corner || popcount64(ka[attacker])==8) &&
          ka[defender]&ka[attacker] &&
          orth[dks]&king[attacker]) {
         bitboard_t mate_rank = orth[dks] & ~orth[aks];
         if ((rook & mate_rank) == board_empty) {
            bitboard_t mate_squares = mate_rank;
            mate_squares &= ~(ka[defender] | king[attacker]);
            if (rook_attacks & mate_squares) {
               mate_threat = true;
            }
         }
      }

      /* Trapped bishop */
      if (((minor|bishop_attacks) & safe_defender) == board_empty) {
         /* Will we be skewered after the king has moved? */
         bitboard_t move = ka[defender] & king_attack[bs] & ~ka[attacker];
         if (onebit64(move)) {
            int square = bitscan64(move);
            bitboard_t rank = orth[square] & orth[bs] & ~king_attack2[square];
            if (rook_attacks & rank) {
               int square2 = bitscan64(rook_attacks & rank);
               //if (bs == C1 && dks == D2 && aks == B1 && rs == A4) {
               //   print_board(board);
               //   printf_bitboard(rank);
               //}
               if ((king_attack2[dks] & rook) == board_empty)
                  if ((connecting_ray[square][square2] & king[attacker]) == board_empty)
                     return heuristic_loss;
            }
         }
      }

      /* Defending king near "good corner"
       * Watch out for mate threats.
       */
      if (!mate_threat && !pinned) {
         bitboard_t good_corner = corner & board_dark;
         if (minor & board_dark) good_corner = corner & board_light;
         bitboard_t bad_corner = corner ^ good_corner;
         bitboard_t kk = ka[defender] | king[defender];
         if (kk & good_corner && kk & (minor | bishop_attacks)) return heuristic_draw;
      }

      /* Everything is centralised and the bishop is not trapped or pinned */
      if (bishop_attacks & safe_defender && (orth[dks] & orth[rs] & minor)==board_empty) {
         const bitboard_t centre = board_centre | board_xcentre | board_offedge;
         bitboard_t bb = bishop_attacks & ~(aattack|occ);
         bb &= ~(orth[dks] & ~orth[aks]);    /* Avoid skewers */
         if (bb && !onebit64(bb)) {
            bitboard_t bb = (king[defender] | minor);
            if ((centre & bb) == bb) return heuristic_draw;
         }
      }
   }

   if (side_to_move == attacker) {
      /* Pinned rook */
      if ((bishop_attacks & rook) && (diag[bs] & king[attacker])) {
         bitboard_t bb = rook | minor | king[attacker];
         if ((connecting_ray[aks][bs]&occ) == bb) return heuristic_draw;
      }

      /* Mate-at-a-glance */
      if (king[defender] & board_edge &&
          (king[defender] & corner || popcount64(ka[attacker])==8) &&
          ka[defender]&ka[attacker] &&
          orth[dks]&king[attacker]) {
         bitboard_t mate_rank = orth[dks] & ~orth[aks];
         if ((rook & mate_rank) == board_empty) {
            bitboard_t mate_squares = mate_rank;
            mate_squares &= ~(ka[defender] | bishop_attacks | king[attacker]);
            if (rook_attacks & mate_squares) {
               mate_rank &= connecting_ray[bitscan64(rook_attacks&mate_squares)][dks];
               if ((minor & mate_rank & king_attack2[dks]) == board_empty) {
                  if ((diag[bs] & mate_rank & ka[defender]) == board_empty) return heuristic_win;
               }
            }
         }
      }

      /* Hanging bishop */
      if ((safe_defender & minor) == board_empty) {
         /* Can the rook capture the bishop? */
         if (rook_attacks & minor) {
            /* Yes - watch out for stalemates */
            if (ka[defender] & ~(ka[attacker]|orth[bs])) return heuristic_win;
         } else {
            /* No, the king must capture the bishop */
            if (!(ka[defender] & rook) || (king_attack[bs]&rook)) return heuristic_win;
         }
      }

      /* Skewer */
      if (orth[dks] & minor) {
         bitboard_t bb = board_rank[unpack_rank(dks)];
         if (!(bb & minor)) bb = board_file[unpack_file(dks)];

         bitboard_t skewer = bb & rook_attacks & ~king[attacker];

         /* TODO: watch out for stale-mate when the defending king escapes into the
          * corner.
          * There are cases where it can do this when it is *not* defending the
          * bishop.
          */
         if (skewer && !(ka[defender] & corner && knight_attack[aks]&king[defender])) {
            int skewer_square = bitscan64(skewer);
            if (ka[defender] & ~(ka[attacker]|orth[skewer_square])) {
               if (!(connecting_ray[dks][skewer_square] & king[attacker]) && !(connecting_ray[bs][skewer_square] & king[attacker])) {
                  if (separation(dks, skewer_square) > 1) {
                     if (separation(bs, dks) > 2) return heuristic_win;
                     if (ka[attacker] & minor) return heuristic_win;
                  }

                  /* Make sure the *rook* is the piece that can capture the bishop */
                  if (ka[attacker] & skewer && !(ka[attacker] & minor)) {
                     if (separation(bs, dks) > 2) return heuristic_win;
                  }
               }
            }
         }
      } else if (orth[bs] & ka[defender]) {
         /* Skewer-in-two? */
         if (separation(bs, dks) > 3 && (rook_attacks & minor) == board_empty) {
            bitboard_t bf = board_file[unpack_file(dks)];
            bitboard_t bbf = board_file[unpack_file(bs)];
            if (abs(unpack_rank(dks) - unpack_rank(bs)) == 1) {
               bf = board_rank[unpack_rank(dks)];
               bbf = board_rank[unpack_rank(bs)];
            }
            bitboard_t bb = bf;

            bb &= safe_attacker;
            bb &= ~(king_attack2[dks] & ~ka[attacker]);
            if (corner & king[defender]) bb &= ~ka[defender];
            if (rook_attacks & bb && (king[attacker]&bbf) == board_empty) {
               int square = bitscan64(rook_attacks & bb);
               int square2 = bitscan64(rook_attacks & bbf);
               if ( (king[attacker] & connecting_ray[dks][square]) == board_empty &&
                    (king[attacker] & connecting_ray[bs][square2]) == board_empty ) {
                  if ((ka[defender] & ~(bf|orth[bs]|ka[attacker])) == board_empty) {
                     return heuristic_win;
                  }
               }
            }
         }
         //if (board_edge & king[defender] && board_offedge & minor)
      }

      /* Trapped bishop */
      if (king[attacker] & corner &&
          knight_attack[aks] & king[defender] &&
          minor & ka[attacker] & ka[defender])
         return heuristic_win;

      /* Discovered check */
      bitboard_t taboo = bishop_attacks | ka[defender];
      if (orth[rs] & king[defender] &&
          (ka[attacker]&king_attack[bs]&~taboo) &&
          king_attack2[aks] & minor &&
          (minor & ka[defender]) == board_empty &&
          (bishop_attacks & ( (ka[defender] & orth[rs])|rook)) == board_empty
          ) {
         if ((orth[rs] & minor) == board_empty) return heuristic_win;
         if (separation(dks, bs) > 2) return heuristic_win;
      }

      /* TODO: if all escape squares for the defending king are on one file/rank we
       * can probably win the bishop that way.
       */

      /* Defending king near "good corner" */
      bitboard_t good_corner = corner & board_dark;
      if (minor & board_dark) good_corner = corner & board_light;
      bitboard_t bad_corner = corner ^ good_corner;
      bitboard_t kk = ka[defender] | king[defender];
      //if (kk & good_corner && kk & (minor | bishop_attacks)) return heuristic_draw;

      /* Defending king near the "bad" corner */
      if (ka[defender] & bad_corner &&
          knight_attack[rs] & bad_corner &&
          ka[defender] & rook &&
          ka[attacker] & rook &&
          king_attack2[aks] & bad_corner)
         return heuristic_win;

      /* Defending king in the "bad" corner
       * This isn't an "obvious" pattern, but it is a tablebase win.
       * Note that the position of the bishop is irrelevant!
       */
      if (king[defender] & bad_corner &&
          king[defender] & knight_attack[rs]) {
         bitboard_t bb;
         if (abs(unpack_rank(rs) - unpack_rank(dks)) == 1)
            bb = board_file[(unpack_file(rs)+unpack_file(dks))/2];
         else
            bb = board_rank[(unpack_rank(rs)+unpack_rank(dks))/2];

         if (bad_corner & board_light)
            bb &= board_dark;
         else
            bb &= board_light;

         if (bb & king[attacker]) {
            if (knight_attack2[rs] & king[attacker])
               return heuristic_win;
            else
               return heuristic_draw;
         }

      }

      if (king[defender] & bad_corner &&
          orth[dks]&orth[aks] &&
          (diag[dks] & king[attacker]) == board_empty &&
          separation(dks, aks) == 3) {

         if ( (rook & ~bishop_colour) &&
              (bishop_attacks & ka[defender]) == board_empty)
            return heuristic_win;

         if (rook & ~(ka[attacker] | ka[defender]) &&
             (((rook|minor) & diag[dks]) | (rook & bishop_attacks)) == board_empty &&
             (rook & orth[aks] & bishop_attacks) == board_empty  )
            return heuristic_win;
      }

      //return heuristic_draw;
   }

   return heuristic_unknown;
}

