/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include "board.h"
#include "pawn_table.h"
#include "piece_tables.h"
#include "config.h"

#define NUM_BUCKETS 2

static inline size_t map_key_to_index(uint64_t key, size_t nelem)
{
   return key & (nelem - 1);
}

static inline uint64_t map_key_to_lock(uint64_t key)
{
   return key >> 8;
}

static inline uint64_t get_lock(piece_square_table_t *t)
{
   return t->lock & 0x00ffffffffffffffll;
}

piece_square_hash_table_t *create_piece_hash_table(size_t nelem)
{
   piece_square_hash_table_t *table = calloc(1, sizeof *table);

   table->number_of_elements = nelem;
   table->data = calloc(nelem+NUM_BUCKETS, sizeof *table->data);
   return table; 
}

void destroy_piece_hash_table(piece_square_hash_table_t *table)
{
   if (table) {
      free(table->data);
      free(table);
   }
}

piece_square_table_t *query_piece_table_entry(piece_square_hash_table_t *table, uint64_t key)
{
   size_t index, b;
   uint64_t lock;

   if (!table)
      return NULL;

   /* Map the key onto the array index, check if entry is there */
   index = map_key_to_index(key, table->number_of_elements);
   lock = map_key_to_lock(key);

   for (b=0; b<NUM_BUCKETS; b++)
      if (get_lock(&table->data[index + b]) == lock)
         return table->data + index + b;

   return NULL;
}

piece_square_table_t *get_free_piece_table_entry(piece_square_hash_table_t *table, uint64_t key)
{
   size_t index, b;
   uint64_t lock;

   if (!table)
      return NULL;

   /* Map the key onto the array index, check if entry is there */
   index = map_key_to_index(key, table->number_of_elements);
   lock = map_key_to_lock(key);

   for (b = 0; b<NUM_BUCKETS; b++) {
      if (get_lock(&table->data[index+b]) == lock) {
         if (b) {
            piece_square_table_t h;
            memcpy(&h, table->data+index, sizeof *table->data);
            memcpy(table->data+index, table->data+index+b, sizeof *table->data);
            memcpy(table->data+index+b, &h, sizeof *table->data);
         }
         break;
      }
   }

   table->data[index].lock = 0;
   table->data[index].lock |= map_key_to_lock(key);
   return table->data + index;
   //memcpy(table->data + index, data, sizeof *data);
}

void store_piece_table_entry(piece_square_hash_table_t *table, uint64_t key, const piece_square_table_t *data)
{
   assert(table);
   piece_square_table_t *psq = get_free_piece_table_entry(table, key);
   memcpy(&psq->data, &data->data, sizeof data->data);
}


void prefetch_piecetable(piece_square_hash_table_t *table, uint64_t key)
{
      size_t index;

   index = map_key_to_index(key, table->number_of_elements);
   __builtin_prefetch(table->data+index);
}
