/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <math.h>
#include <allegro5/allegro.h>
#include <allegro5/allegro_image.h>
#include <allegro5/allegro_ttf.h>
#include <allegro5/allegro_native_dialog.h>
#include <allegro5/allegro_primitives.h>

#include "jazz.h"
#include "bits.h"
#include "genrand.h"
#include "movegen.h"
#include "names.h"
#include "game.h"
#include "inline/game.h"
#include "fen.h"
#include "computer.h"
#include "alphabeta.h"
#include "see.h"
#include "pawns.h"
#include "timer.h"
#include "pgn.h"
#include "evaluate.h"
#include "pg_book.h"

#ifndef M_PI
#define M_PI 3.14159265
#endif

#define PSQ_WIDTH          60
#define TEXT_BUFFER_SIZE  100
#define TEXT_OUTPUT_LINES  10
#define ITERATION_FONT_SIZE 12

#define PROGRAM_NAME "Jazz"

#define DIALOG_FLAGS_SAVE_EPD    0x0001
#define DIALOG_FLAGS_LOAD_EPD    0x0002
#define DIALOG_FLAGS_SAVE_PGN    0x0004
#define DIALOG_FLAGS_LOAD_PGN    0x0008
#define DIALOG_FLAGS_SAVE        0x0005
#define DIALOG_FLAGS_LOAD        0x000A
#define DIALOG_FLAGS_EPD         0x0003
#define DIALOG_FLAGS_PGN         0x000C
typedef struct {
   ALLEGRO_FILECHOOSER *file_dialog;
   ALLEGRO_EVENT_SOURCE event_source;
   ALLEGRO_THREAD *thread;
   ALLEGRO_DISPLAY *display;
   int flags;
} async_dialog_t;

/* To communicate from a separate thread, we need a user event. */
#define EVENT_FILE_DIALOG_CLOSE   ALLEGRO_GET_EVENT_TYPE('e', 'N', 'F', '1')

#define BOARD_SIZE 400
#define BORDER 16

static async_dialog_t *file_dialog = NULL;
static const char *last_path = NULL;
static char *filename = NULL;
static int file_index;

static ALLEGRO_FONT *figurine_font;
static ALLEGRO_FONT *font;
static ALLEGRO_BITMAP *figurine_sheet;
static ALLEGRO_BITMAP *figurines[2][6];
static ALLEGRO_BITMAP *square[2];
static ALLEGRO_BITMAP *side_icon[2] = {NULL, NULL};
static ALLEGRO_BITMAP *cursor;

static ALLEGRO_DISPLAY *psq_display = NULL;
static ALLEGRO_DISPLAY *moves_display;
static ALLEGRO_DISPLAY *board_display;
static ALLEGRO_DISPLAY *iteration_display;
static ALLEGRO_DISPLAY *key_display;
static ALLEGRO_EVENT_QUEUE *queue = NULL;
static int board_x = 0, board_y = 0;
static int psq_index = 0;

static int player_select = -1;
static int player_cursor = E4;

static int display_timer_counter;

static book_t *book = NULL;

/* Our thread to show the native file dialog. */
static void *async_file_dialog_thread_func(ALLEGRO_THREAD *thread, void *arg)
{
   async_dialog_t *data = arg;
   ALLEGRO_EVENT event;
   (void)thread;

   /* We need to set the current display for this thread becuse
    * al_show_native_file_dialog() shows the dialog on the current window.
    */
   al_set_target_backbuffer(data->display);

   /* The next line is the heart of this example - we display the
    * native file dialog.
    */
   al_show_native_file_dialog(board_display, data->file_dialog);

   /* We emit an event to let the main program now that the thread has
    * finished.
    */
   event.user.type = EVENT_FILE_DIALOG_CLOSE;
   al_emit_user_event(&data->event_source, &event, NULL);

   return NULL;
}

/* Function to start the new thread. */
async_dialog_t *spawn_async_file_dialog(const char *initial_path, int flags)
{
   async_dialog_t *data = malloc(sizeof *data);

   if (flags & DIALOG_FLAGS_LOAD) {
      data->file_dialog = al_create_native_file_dialog(
            initial_path, "Open EPD/FEN/PGN file", ".;*.pgn;*.epd;*.fen",
            ALLEGRO_FILECHOOSER_FILE_MUST_EXIST);
   } else {
      data->file_dialog = al_create_native_file_dialog(
            initial_path, "Save EPD file", ".;*.epd",
            ALLEGRO_FILECHOOSER_SAVE);
   }
   al_init_user_event_source(&data->event_source);
   data->display = al_get_current_display();
   data->thread = al_create_thread(async_file_dialog_thread_func, data);
   data->flags = flags;

   al_start_thread(data->thread);

   return data;
}


void stop_async_dialog(async_dialog_t *data)
{
   if (data) {
      al_destroy_thread(data->thread);
      al_destroy_user_event_source(&data->event_source);
      if (data->file_dialog) al_destroy_native_file_dialog(data->file_dialog);
      free(data);
   }
}

void load_figurine_set(int figurine_set)
{
   int size = 40;
   int x, y;

   al_set_target_backbuffer(board_display);
   if (!figurine_sheet)
      return;

   size = al_get_bitmap_width(figurine_sheet) / 6;
   
   for (y=0; y<2; y++) {
      for (x=0; x<6; x++) {
         /* Allegro 4.9 seems to do something odd when it generates sub
          * bitmaps, because performance is really horrible if we use
          * them directly... this is a work-around until the problem is
          * fixed.
          */
         ALLEGRO_BITMAP *bmp = al_create_sub_bitmap(figurine_sheet, 
               x*size, (y+2*figurine_set)*size, size, size);
         if (figurines[y][x]) al_destroy_bitmap(figurines[y][x]);
         figurines[y][x] = bmp;
      }
   }

   for (y = 0; y<2; y++) {
      al_destroy_bitmap(side_icon[y]);
      side_icon[y] = al_clone_bitmap(figurines[y][1]);
   }
}


static char **line = NULL;
static int cur_line = 0;
static int last_line = 0;
static int delta_line = 0;
static bool new_line = false;
static int text_window_size = TEXT_OUTPUT_LINES;

static void draw_iteration_window(void)
{
   ALLEGRO_STATE allegro_state;
   int n;
   int first;

   if (!line)
      return;

   al_store_state(&allegro_state, ALLEGRO_STATE_ALL);
   al_set_target_backbuffer(iteration_display);
   al_clear_to_color(al_map_rgb_f(1,1,1));
   text_window_size = al_get_display_height(iteration_display)/(ITERATION_FONT_SIZE+2);
   first = last_line - text_window_size;
   if (first < 0) first = 0;
   for (n=0; n < text_window_size; n++) {
      int k = (first + n + TEXT_BUFFER_SIZE - delta_line) % TEXT_BUFFER_SIZE;

      if (line[k])
         al_draw_text(font, al_map_rgb(0,0,0), 1, (ITERATION_FONT_SIZE+2)*n, 0, line[k]);
   }
   al_flip_display();
   al_restore_state(&allegro_state);
}

static void iteration_display_message(const char *msg, ...)
{
   char *s = malloc(4096);
   char *eol;
   va_list ap;
   va_start(ap, msg);
   vsnprintf(s, 4095, msg, ap);
   va_end(ap);

   printf("%s", s);

   if (!line) {
      line = calloc(TEXT_BUFFER_SIZE, sizeof *line);
   }

   /* Clear the line if needed */
   if (new_line && line[cur_line]) {
      line[cur_line][0] = '\0';
   }

   eol = strchr(s, '\n');
   if (eol) *eol = '\0';
   new_line = (eol != NULL);

   /* Append string to existing text */
   if (!line[cur_line]) {
      line[cur_line] = s;
   } else {
      char *ss = strdup(line[cur_line]);
      snprintf(line[cur_line], TEXT_BUFFER_SIZE-1, "%s%s", ss, s);
      line[TEXT_BUFFER_SIZE-1] = '\0';
      free(ss);
   }

   if (new_line) {
      cur_line = (cur_line+1) % TEXT_BUFFER_SIZE;
      last_line++;
      draw_iteration_window();
   }
}

static void draw_psq_board(gamestate_t *game)
{
   static char title[256];
   static char *piece_name = NULL;
   ALLEGRO_COLOR black = al_map_rgba_f(0, 0, 0, 1);
   ALLEGRO_COLOR white = al_map_rgba_f(1, 1, 1, 1);
   ALLEGRO_COLOR red = al_map_rgba_f(1, 0, 0, 1);
   ALLEGRO_COLOR dark_red = al_map_rgba_f(.5, 0, 0, 1);
   ALLEGRO_COLOR yellow = al_map_rgba_f(1, 1, 0, 1);
   ALLEGRO_COLOR dark_yellow = al_map_rgba_f(.25, .25, 0, 1);
   ALLEGRO_COLOR light = al_map_rgba_f(.5, .5, 1, 1);
   ALLEGRO_COLOR dark = al_map_rgba_f(.25, .25, .5, 1);
   //ALLEGRO_COLOR dark = al_map_rgba_f(.125, .125, .325, 1);
   piece_square_table_t *psq = NULL;
   pawn_structure_t * pawn;
   uint64_t key;
   int x, y;

   if (!psq_display)
      return;

   al_set_target_backbuffer(psq_display);
   al_clear_to_color(white);

   pawn = query_pawn_table_entry(game->pawn_structure, game->board->pawn_hash);
   
   switch(psq_index) {
      case 0:     // Knight
         key = get_knight_table_key(game->board, pawn);
         psq = query_piece_table_entry(game->knight_psq, key);
         piece_name = "knight";
         break;

      case 1:     // Bishop
         key = get_bishop_table_key(game->board, pawn);
         psq = query_piece_table_entry(game->bishop_psq, key);
         piece_name = "bishop";
         break;

      case 2:     // Rook
         key = get_rook_table_key(game->board, pawn);
         psq = query_piece_table_entry(game->rook_psq, key);
         piece_name = "rook";
         break;

      case 3:     // Queen
         psq = NULL;
         piece_name = "queen";
         break;

      case 4:     // King
         key = get_king_table_key(game->board, pawn);
         psq = query_piece_table_entry(game->king_psq, key);
         piece_name = "king";
         break;

   }
   snprintf(title, 255, PROGRAM_NAME" Piece square table - %s", piece_name);
   al_set_window_title(psq_display, title);

   bitboard_t occ = game->board->bbc[0] | game->board->bbc[1];

   for (y=0; y<8; y++) {
      for (x=0; x<8; x++) {
         ALLEGRO_COLOR piece_tint = white;
         int where = pack_row_file(7-y, x);
         int piece = get_piece(game->board, where);
         ALLEGRO_COLOR col = light;
         int player = (piece&BLACK) >> 7;

         if ((x^y)&1) col = dark;
         al_draw_filled_rectangle(x*PSQ_WIDTH, y*PSQ_WIDTH,
                                  (x+1)*PSQ_WIDTH, (y+1)*PSQ_WIDTH, col);

         /* Highlight pawn structure features */
         if (pawn) {
            /* Open files */
            if (pawn->half_open_files[0] & (1<<x)) {
               al_draw_rectangle(x*PSQ_WIDTH, 8*PSQ_WIDTH-2,
                              (x+1)*PSQ_WIDTH, 8*PSQ_WIDTH-1, white, 1);
            }
            if (pawn->half_open_files[1] & (1<<x)) {
               al_draw_rectangle(x*PSQ_WIDTH, 0, (x+1)*PSQ_WIDTH, 1, white, 1);
            }
            if (pawn->open_files & (1<<x))
               al_draw_rectangle(x*PSQ_WIDTH, 0,
                     (x+1)*PSQ_WIDTH-1, 8*PSQ_WIDTH-1, white, 2);

            /* Outposts */
            if (pawn->outposts[0] & ((bitboard_t)1<<where) ) {
               al_draw_filled_circle((x+0.5)*PSQ_WIDTH,
                     (y+.5)*PSQ_WIDTH, PSQ_WIDTH/2, white);
            }
            if (pawn->outposts[1] & ((bitboard_t)1<<where) ) {
               al_draw_filled_circle((x+0.5)*PSQ_WIDTH,
                     (y+.5)*PSQ_WIDTH, PSQ_WIDTH/2, black);
            }

            /* Strong square */
            if (pawn->strong_squares[0] & ((bitboard_t)1<<where) ) {
               al_draw_rectangle(x*PSQ_WIDTH, y*PSQ_WIDTH,
                     (x+1)*PSQ_WIDTH-1, (y+1)*PSQ_WIDTH-1, white, 2);
            }
            if (pawn->strong_squares[1] & ((bitboard_t)1<<where) ) {
               al_draw_rectangle(x*PSQ_WIDTH, y*PSQ_WIDTH,
                     (x+1)*PSQ_WIDTH-1, (y+1)*PSQ_WIDTH-1, black, 2);
            }

            /* Isolated/backward pawns */
            if ( pawn->weak & ((bitboard_t)1<<where) ) {
               piece_tint = player?dark_red:red;
               player = 0;
            }

            /* Doubled pawns */
            if ( pawn->doubled_pawns & ((bitboard_t)1<<where) ) {
               al_draw_rectangle(x*PSQ_WIDTH, y*PSQ_WIDTH,
                     (x+1)*PSQ_WIDTH-1, (y+1)*PSQ_WIDTH-1, red, 2);
            }

            /* Duo */
#if 0
            if ( (pawn->duo[0]|pawn->duo[1]) & ((bitboard_t)1<<where) ) {
               al_draw_rectangle(x*PSQ_WIDTH, y*PSQ_WIDTH,
                     (x+1)*PSQ_WIDTH-1, (y+1)*PSQ_WIDTH-1, yellow, 2);
            }
#endif

            /* Free pawns */
            if ( pawn->free & ((bitboard_t)1<<where) ) {
               piece_tint = player?dark_yellow:yellow;
               player = 0;
            }
         }

         /* Draw piece currently on this square */
         if (occ & make_bitboard_square(where)) {
            int draw_x = x*PSQ_WIDTH;
            int draw_y = y*PSQ_WIDTH;
            piece = piece&PIECE;
            al_draw_tinted_scaled_bitmap(figurines[player][piece],
               piece_tint, 0, 0,
               al_get_bitmap_width(figurines[player][piece]),
               al_get_bitmap_height(figurines[player][piece]),
               draw_x, draw_y, PSQ_WIDTH, PSQ_WIDTH, 0);
         }

         /* Show evaluation */
         if (psq){
            al_draw_textf(font, red, (x+1)*PSQ_WIDTH, y*PSQ_WIDTH,
                  ALLEGRO_ALIGN_RIGHT, "%d", psq->data[1][where]);
            al_draw_textf(font, red, x * PSQ_WIDTH, (y+1)*PSQ_WIDTH-14, 0, "%d",
                  psq->data[0][where]);
         }
      }
   }

   al_flip_display();
}

void draw_board(gamestate_t *game)
{
   static char title[256];
   ALLEGRO_COLOR white = al_map_rgba_f(1, 1, 1, 1);
   //ALLEGRO_COLOR light = al_map_rgba_f(.5, .5, 1, 1);
   //ALLEGRO_COLOR dark = al_map_rgba_f(.25, .25, .5, 1);
   //ALLEGRO_COLOR bg = al_map_rgba_f(.125, .125, .25, 1);
   ALLEGRO_COLOR light = al_map_rgba_f(.925, .925, .875, 1);
   ALLEGRO_COLOR dark = al_map_rgba_f(.75, .55, .225, 1);
   ALLEGRO_COLOR bg = al_map_rgba_f(.325, .175, .1125, 1);
   double score;
   int x, y;
   int size;

   al_set_target_backbuffer(board_display);
   al_clear_to_color(bg);

   score = game->score[game->moves_played]/100.0;

   game->root_board = game->board;
   score = static_evaluation(game, game->side_to_move, -CHECKMATE, CHECKMATE)/100.0;

#if 0
   int s1, s2;
   s1 = static_evaluation(game, game->side_to_move, -CHECKMATE, CHECKMATE);
   s2 = static_evaluation(game, game->side_to_move^BLACK, -CHECKMATE, CHECKMATE);
   if (s1+s2 != 0) {
      printf("Scores don't balance!\n");
      printf("White score: %d\n", s1);
      printf("Black score: %d\n", s2);
      //exit(0);
   }
#endif

   if (game->side_to_move == BLACK) {
      snprintf(title, 255, PROGRAM_NAME" [ Black to play %.2f]", score);
   } else {
      snprintf(title, 255, PROGRAM_NAME" [ White to play %.2f]", score);
   }
   al_set_window_title(board_display, title);
   //al_set_display_icon(figurines[game->side_to_move == BLACK][1]);

   size = (al_get_display_width(board_display) - 2*BORDER) / 8;

   bitboard_t occ = game->board->bbc[0] | game->board->bbc[1];

   for (y=0; y<8; y++) {
      for (x=0; x<8; x++) {
         int where = pack_row_file(y, x);
         int piece = get_piece(game->board, where);
         int col = (piece&BLACK) >> 7;
         int draw_x = board_x + x*size + BORDER;
         int draw_y = board_y + (7-y)*size + BORDER;
         bool blink_piece = false;

         /* Blink highlighted pieces */
         if ( (player_select == where) )// && ( (display_timer_counter/4)&1) )
            blink_piece = true;

         /* Draw background tile */
         //al_draw_bitmap(square[1-((x^y)&1)], draw_x, draw_y, 0);

         al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size,
                                  (x^y)&1 ? light : dark);


         /* Highlight cursor */
         if (player_cursor == where) {
            al_draw_scaled_bitmap(cursor, 0, 0,
               al_get_bitmap_width(cursor),
               al_get_bitmap_height(cursor),
               draw_x, draw_y, size, size, 0);
         }

         /* Piece */
         if (occ & make_bitboard_square(where)) {
            ALLEGRO_COLOR piece_tint = white;
            piece = piece&PIECE;
            if (blink_piece) {
               float f = (display_timer_counter&31)/32.;
               f = cos(f*M_PI);
               piece_tint = al_map_rgba_f(1,1,1, f*f);
            }
            al_draw_tinted_scaled_bitmap(figurines[col][piece],
               piece_tint, 0, 0,
               al_get_bitmap_width(figurines[col][piece]),
               al_get_bitmap_height(figurines[col][piece]),
               draw_x, draw_y, size, size, 0);
         }
      }
   }

   /* Draw coordinates along the board */
   for (x=0; x<8; x++) {
      int draw_x = BORDER + x*size + size/2;
      int draw_y = BORDER + 8*size;
      al_draw_textf(font, white, draw_x, draw_y, ALLEGRO_ALIGN_CENTRE, "%c", x+'A');

      draw_x = BORDER/2;
      draw_y = (7 - x) * size + size/2 + ITERATION_FONT_SIZE;
      al_draw_textf(font, white, draw_x, draw_y, ALLEGRO_ALIGN_CENTRE, "%c", x+'1');
   }

   if (game->name_white || game->name_black) {
      int width = al_get_display_width(board_display);
      al_draw_textf(font, white, width/2-16, 0, ALLEGRO_ALIGN_RIGHT, "%s", game->name_white);
      al_draw_textf(font, white, width/2, 0, ALLEGRO_ALIGN_CENTRE, "-");
      al_draw_textf(font, white, width/2+16, 0, ALLEGRO_ALIGN_LEFT, "%s", game->name_black);
   }

   al_flip_display();

   /* Check opening book */
   opening_position_t *entry = get_book_move(book, game->board, game->side_to_move);
   if (entry) {
      iteration_display_message("Book moves: ");
      while (entry) {
         int from, to;
         from = (entry->move >> 6) & 63;
         to = entry->move & 63;
         iteration_display_message("%c%c-%c%c (%d) ",
               unpack_file(from)+'a', unpack_rank(from) + '1',
               unpack_file(to)+'a', unpack_rank(to)+'1',
               entry->weight);
         free_book_move(entry);
         entry = get_next_book_move(book);
      }
      iteration_display_message("\n");
   }
}

void show_move_list(gamestate_t *game)
{
   ALLEGRO_COLOR white = al_map_rgba_f(1, 1, 1, 1);
   ALLEGRO_COLOR black = al_map_rgba_f(0, 0, 0, 1);
   int first_move;
   int x, y=0, dy=0, n;
   int nl, nm, lines_needed;
   int start, stop;

   al_set_target_backbuffer(moves_display);
   al_clear_to_color(white);

   /* Number of displayed moves */
   nl = (al_get_display_height(moves_display) - dy) / 16 - 1;
   nm = 2 * nl;

   if (nl < 2)
      return;

   first_move = get_move_player(game->move_list[0])>>7;

   start = 0;
   stop = game->moves_played;
   lines_needed = (first_move + stop - start)/2;

   if ( lines_needed > nl ) {
      start = (first_move + stop - 2*nl) & ~0x1;
      start += 2;
   }

   for(n=start; n<stop; n++) {
      /* Odd plies should normally have been played by black - unless we
       * changed the player, or loaded a position which had "black to play"
       */
      int w = (n&1) ^ first_move;
      int dx = 0;

      x = w * 80 + 10;
      y = ((n+first_move - start)/2)*16 + dy;
      al_draw_textf(figurine_font, black, 8, y, 0, "% 3d.", (n+first_move)/2+1);
      dx = 32;

      al_draw_textf(figurine_font, black, x+dx, y, 0, "%s", move_string(game->move_list[n], NULL));
   }

   //printf("%d %d\n", player_is_mate(game), player_in_check(game, game->side_to_move));
   if (player_is_mate(game)) {
      y+=16;
      x = 24;
      if (player_in_check(game, game->side_to_move)) {
         al_draw_textf(figurine_font, black, x, y, 0, "#");
         al_set_window_title(moves_display, "Jazz - checkmate");
      } else {
         al_draw_textf(font, black, x, y, 0, "=");
         al_set_window_title(moves_display, "Jazz - stalemate");
      }

   }

   al_flip_display();
}

void open_psq_display(void)
{
   if (!psq_display) {
      psq_display = al_create_display(8*PSQ_WIDTH, 8*PSQ_WIDTH);
      al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)psq_display);
   }
}

int main(int argc, char **argv)
{
   gamestate_t *game = NULL;
   movelist_t movelist;
   ALLEGRO_TIMER *display_timer;
   int n, x, y;
   int max_figurine_set = 0;
   int figurine_set = 0;
   bool redraw = true;
   bool done = false;

   if (!al_init()) {
      fprintf(stderr, "Couldn't initialise Allegro\n");
      return 1;
   }
   al_install_mouse();
   al_install_keyboard();
   al_init_primitives_addon();
   al_init_image_addon();
   al_init_font_addon();
   al_init_ttf_addon();

   /* Seed random number generator */
   sgenrand(time(NULL));

   /* The *main* display is actually the board_display, but we create that
    * last because we also want it to be the key window.
    */

   al_set_new_display_option(ALLEGRO_SAMPLE_BUFFERS, 1, ALLEGRO_SUGGEST);
   al_set_new_display_option(ALLEGRO_SAMPLES, 4, ALLEGRO_SUGGEST);

   /* Create display to display the list of moves played so far */
   al_set_new_display_flags(al_get_new_display_flags() | ALLEGRO_RESIZABLE);
   moves_display = al_create_display(200, BOARD_SIZE + 2*BORDER);
   if (!moves_display) {
      fprintf(stderr, "Can't open display\n");
      return 1;
   }
   al_get_window_position(moves_display, &x, &y);
   al_clear_to_color(al_map_rgb_f(1,1,1));
   al_flip_display();
   al_set_window_title(moves_display, PROGRAM_NAME);

   /* Messages */
   al_set_new_window_position(x-BOARD_SIZE - 2*BORDER - 4, y+BOARD_SIZE + 24+2*BORDER);
   //al_set_new_display_flags(al_get_new_display_flags()^ALLEGRO_NOFRAME);
   iteration_display = al_create_display(200 + 4+ BOARD_SIZE+2*BORDER, (ITERATION_FONT_SIZE+2)*TEXT_OUTPUT_LINES);
   al_set_new_display_flags(al_get_new_display_flags()^ALLEGRO_RESIZABLE);
   //al_set_new_display_flags(al_get_new_display_flags()^ALLEGRO_NOFRAME);
   al_clear_to_color(al_map_rgb_f(1,1,1));
   al_flip_display();
   al_set_window_title(iteration_display, PROGRAM_NAME" - Iterations");


   /* Main (board) display */
   al_set_new_window_position(x-BOARD_SIZE-4-2*BORDER, y);
   board_display = al_create_display(BOARD_SIZE+2*BORDER, BOARD_SIZE+2*BORDER);
   al_set_new_window_position(x+204, y);
   if (!board_display) {
      fprintf(stderr, "Can't open display\n");
      return 1;
   }
   al_set_window_title(board_display, "Jazz");

   key_display = board_display;

   memset(figurines, 0, sizeof figurines);
   square[0] = al_load_bitmap("gfx/ws.png");
   square[1] = al_load_bitmap("gfx/bs.png");
   figurine_sheet = al_load_bitmap("gfx/figurines_big.png");
   cursor = al_load_bitmap("gfx/indicator.png");

   if (!figurine_sheet) {
      printf("Can't load\n");
      figurines[0][0] = al_load_bitmap("gfx/wp.png");
      figurines[0][1] = al_load_bitmap("gfx/wn.png");
      figurines[0][2] = al_load_bitmap("gfx/wb.png");
      figurines[0][3] = al_load_bitmap("gfx/wr.png");
      figurines[0][4] = al_load_bitmap("gfx/wq.png");
      figurines[0][5] = al_load_bitmap("gfx/wk.png");

      figurines[1][0] = al_load_bitmap("gfx/bp.png");
      figurines[1][1] = al_load_bitmap("gfx/bn.png");
      figurines[1][2] = al_load_bitmap("gfx/bb.png");
      figurines[1][3] = al_load_bitmap("gfx/br.png");
      figurines[1][4] = al_load_bitmap("gfx/bq.png");
      figurines[1][5] = al_load_bitmap("gfx/bk.png");

      for (n=0; n<6; n++) {
         if (!figurines[0][n] || !figurines[1][n]) {
            fprintf(stderr, "Can't load figurines\n");
            return 1;
         }
         al_convert_mask_to_alpha(figurines[0][n], al_map_rgb_f(1, 0, 1));
         al_convert_mask_to_alpha(figurines[1][n], al_map_rgb_f(1, 0, 1));
      }
      side_icon[0] = al_clone_bitmap(figurines[0][1]);
      side_icon[1] = al_clone_bitmap(figurines[1][1]);
   } else {
      int width = al_get_bitmap_width(figurine_sheet) / 6;
      max_figurine_set = al_get_bitmap_height(figurine_sheet)/(2*width) - 1;
      if (figurine_set > max_figurine_set)
         figurine_set = max_figurine_set;

      load_figurine_set(figurine_set);
   }
   font = al_load_font("gfx/DejaVuSansMono.ttf", ITERATION_FONT_SIZE, ALLEGRO_TTF_NO_KERNING);
   //font = al_load_font("gfx/monaco.ttf", ITERATION_FONT_SIZE, ALLEGRO_TTF_NO_KERNING);
   if (!font) {
      fprintf(stderr, "Can't load font\n");
      exit(0);
   }

   figurine_font = al_load_bitmap_font("gfx/figurines-font.png");
   if (!figurine_font) {
      fprintf(stderr, "Can't load figurine figurine_font\n");
      exit(0);
   }

   printf("Initialising Jazz engine\n");
   initialise_jazz();

   /* Read opening book */
   printf("Opening opening book...");
   //book = open_polyglot_opening_book("performance.bin");
   book = open_opening_book("bigbook.bin");
   if (book)
      printf("done\n");
   else
      printf("false\n");

   /* Redirect output to our own window rather than the terminal */
   set_default_output_function(iteration_display_message);

   printf("Initialising new game\n");
   game = NULL;

   file_index = 0;
   if (argc>1 && strstr(argv[1], ".epd")) {
      game = load_epd_file(argv[1], file_index);
      filename = strdup(argv[1]);
   } else if (argc>1 && strstr(argv[1], ".fen")) {
      game = load_epd_file(argv[1], file_index); 
      filename = strdup(argv[1]);
   } else if (argc>1 && strstr(argv[1], ".pgn")) {
      game = load_pgn_file(argv[1], file_index);
      filename = strdup(argv[1]);
   }

   if (!game) {
      game = create_game();
      start_new_game(game);
   }
   generate_moves(&movelist, game->board, game->side_to_move);

   display_timer = al_create_timer(1.0 / 40.0);
   queue = al_create_event_queue();
   al_register_event_source(queue, al_get_mouse_event_source());
   al_register_event_source(queue, al_get_keyboard_event_source());
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)board_display);
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)moves_display);
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)iteration_display);
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)display_timer);
   al_start_timer(display_timer);
   display_timer_counter = 0;

   draw_board(game);
   redraw = false;

   al_set_display_icon(board_display, side_icon[game->side_to_move == BLACK]);
   while (!done) {
      ALLEGRO_EVENT event;

      if (redraw && al_event_queue_is_empty(queue)) {
         draw_board(game);
         draw_psq_board(game);
         show_move_list(game);
         draw_iteration_window();
         redraw = false;

         //bitboard_t a, b;
         //get_xrayattack_and_block_bitboard(game->board, WHITE, &a, &b);
         //printf_bitboard(b);
#if 0
         pawn_structure_t *ps = evaluate_pawn_structure(game);
         if (ps) {
            printf("Pawn structure:\n");
            int n;
            for (n=0; n<2; n++) {
               printf("%s\n", n?"Black":"White");
               printf("Outposts:\n");
               printf_bitboard(ps->outposts[n]);
               printf("Strong squares:\n");
               printf_bitboard(ps->strong_squares[n]);
            }
            
         }
#endif

         //printf("0x%016llxll\n", game->board->hash);
         //printf("%d %d\n", count_repetition(game), draw_by_repetition(game));
#if 0
         bitboard_t white_attacks = get_attack_bitboard(game->board, WHITE);
         bitboard_t black_attacks = get_attack_bitboard(game->board, BLACK);
         bitboard_t occupied = game->board->bbc[0]|game->board->bbc[1];
         bitboard_t defended_pieces = occupied&white_attacks&black_attacks;
         bitboard_t attacked_pieces = 0;

         attacked_pieces |= game->board->bbc[0] & black_attacks;
         attacked_pieces |= game->board->bbc[1] & white_attacks;
         printf("White attacks:\n");
         printf_bitboard( white_attacks );
         printf("Black attacks:\n");
         printf_bitboard( black_attacks );
         printf("Defended pieces:\n");
         printf_bitboard( defended_pieces );
         printf("Hanging pieces:\n");
         printf_bitboard( attacked_pieces^defended_pieces );
#endif
#undef SHOW_SEE_CAPTURES
#ifdef SHOW_SEE_CAPTURES
         int n;
         for (n=0; n<movelist.num_moves; n++) {
            if (is_capture_move(movelist.move[n])) {
               printf("%s %d\n", move_string(movelist.move[n], NULL), 
                     static_exchange_evaluation(game->board, movelist.move[n]));
            }
         }
#endif
      }

      al_wait_for_event(queue, &event);
      switch (event.type) {
         case ALLEGRO_EVENT_DISPLAY_RESIZE:
            al_acknowledge_resize(event.display.source);

            /* Make sure the board is still square */
            if (event.display.source == board_display) {
               int w = event.display.width;
               if (event.display.height > w)
                  w = event.display.width;
               al_resize_display(board_display, w, w);
            }
         case ALLEGRO_EVENT_DISPLAY_EXPOSE:
            redraw = true;
            break;

         case ALLEGRO_EVENT_DISPLAY_CLOSE:
            if (event.display.source == psq_display) {
               al_unregister_event_source(queue, (ALLEGRO_EVENT_SOURCE *)psq_display);
               al_destroy_display(psq_display);
               psq_display = NULL;
               redraw = true;
            } else {
               done = true;
            }
            break;

         case ALLEGRO_EVENT_DISPLAY_SWITCH_IN:
            key_display = event.display.source;
            break;

         case ALLEGRO_EVENT_DISPLAY_SWITCH_OUT:
            key_display = NULL;
            break;

         case ALLEGRO_EVENT_MOUSE_AXES:
            if (key_display == iteration_display && event.mouse.dz) {
               int nd = delta_line + event.mouse.dz;
               if (nd <0) nd = 0;
               if (nd >= TEXT_BUFFER_SIZE) nd = TEXT_BUFFER_SIZE-1;
               if (nd > last_line-text_window_size)
                  nd = last_line-text_window_size;
               if (nd != delta_line) {
                  delta_line = nd;
                  redraw = true;
               }
            }
            break;

         case ALLEGRO_EVENT_MOUSE_BUTTON_UP:
            if (key_display == board_display) {
            player_cursor = pack_row_file((8*40-event.mouse.y)/40,
                                             event.mouse.x/40);
            if (player_select == -1) {
               player_select = player_cursor;
               display_timer_counter = 0;
            } else {
               /* Confirm that the player has entered a valid move */
               int n = validate_move(&movelist, player_select, player_cursor);

               if (n == -1)
                  n = validate_move(&movelist, player_cursor, player_select);
               if (n > -1) {
                  playmove_was_ok(game, movelist.move[n]);
                  game->last_move = game->moves_played;
                  generate_moves(&movelist, game->board, game->side_to_move);
                  al_set_display_icon(board_display, side_icon[game->side_to_move == BLACK]);
               }
               player_select = -1;
            }
            redraw = true;
            }
            break;
         
         case ALLEGRO_EVENT_KEY_CHAR:
            switch (event.keyboard.keycode) {
               case ALLEGRO_KEY_ESCAPE:
                  done = true;
                  break;
               case ALLEGRO_KEY_F1:             /* Show PSQ values in window */
                  open_psq_display();
                  psq_index = 0;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F2:             /* Show PSQ values in window */
                  open_psq_display();
                  psq_index = 1;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F3:             /* Show PSQ values in window */
                  open_psq_display();
                  psq_index = 2;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F4:            /* Show PSQ values in window */
                  open_psq_display();
                  psq_index = 3;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F7:            /* Show PSQ values in window */
                  open_psq_display();
                  psq_index = 4;
                  redraw = true;
                  break;

               case ALLEGRO_KEY_F5:             /* Load FEN/PGN/EPD file */
                  if (!file_dialog) {
                     file_dialog = spawn_async_file_dialog(last_path, DIALOG_FLAGS_LOAD);
                     al_register_event_source(queue, &file_dialog->event_source);
                  }
                  break;
               case ALLEGRO_KEY_F6:             /* Load FEN/PGN/EPD file */
                  if (!file_dialog) {
                     file_dialog = spawn_async_file_dialog(last_path, DIALOG_FLAGS_SAVE);
                     al_register_event_source(queue, &file_dialog->event_source);
                  }
                  break;
               case ALLEGRO_KEY_DELETE:         /* New game */
                  /* FIXME: this leaks memory, because the internal
                   * pointers in the game struct are not freed, just
                   * overwritten.
                   */
                  start_new_game(game);
                  generate_moves(&movelist, game->board, game->side_to_move);
                  redraw = true;
                  break;
                  
               case ALLEGRO_KEY_N:
                  file_index++;
                  if (filename) {
                     gamestate_t *new_game = load_epd_file(filename, file_index);
                     if (new_game) {
                        end_game(game);
                        game = new_game;
                        generate_moves(&movelist, game->board, game->side_to_move);
                     }
                     redraw = true;
                  }
                  break;

               case ALLEGRO_KEY_P:
                  if (file_index > 0)
                     file_index--;
                  if (filename) {
                     gamestate_t *new_game = load_epd_file(filename, file_index);
                     if (new_game) {
                        end_game(game);
                        game = new_game;
                        generate_moves(&movelist, game->board, game->side_to_move);
                     }
                     redraw = true;
                  }
                  break;

               case ALLEGRO_KEY_FULLSTOP:       /* Switch figurine set */
                  if (figurine_set < max_figurine_set) {
                     figurine_set++;
                     load_figurine_set(figurine_set);
                     redraw = true;
                  }
                  break;
               case ALLEGRO_KEY_COMMA:
                  if (figurine_set > 0) {
                     figurine_set--;
                     load_figurine_set(figurine_set);
                     redraw = true;
                  }
                  break;
               case ALLEGRO_KEY_LEFT:
                  if ((player_cursor&7) != 0) player_cursor--;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_RIGHT:
                  if ((player_cursor&7) != 7) player_cursor++;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_UP:
                  if (player_cursor < 56) player_cursor += 8;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_DOWN:
                  if (player_cursor>7) player_cursor -= 8;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_ENTER:
                  if (player_select == -1) {
                     player_select = player_cursor;
                     display_timer_counter = 0;
                  } else {
                     /* Confirm that the player has entered a valid move */
                     int n = validate_move(&movelist, player_select, player_cursor);

                     if (n == -1)
                        n = validate_move(&movelist, player_cursor, player_select);
                     if (n > -1) {
                        playmove_was_ok(game, movelist.move[n]);
                        game->last_move = game->moves_played;
                        generate_moves(&movelist, game->board, game->side_to_move);
                        al_set_display_icon(board_display, side_icon[game->side_to_move == BLACK]);
                     }
                     player_select = -1;
                  }
                  redraw = true;
                  break;
               case ALLEGRO_KEY_TAB:
                  redo_last_move(game);
                  generate_moves(&movelist, game->board, game->side_to_move);
                  redraw = true;
                  break;
               case ALLEGRO_KEY_HOME:
                  while (game->moves_played)
                     takeback(game);
                  generate_moves(&movelist, game->board, game->side_to_move);
                  redraw = true;
                  break;
               case ALLEGRO_KEY_BACKSPACE:
                  takeback(game);
                  generate_moves(&movelist, game->board, game->side_to_move);
                  redraw = true;
                  break;
               case ALLEGRO_KEY_SPACE:
                  set_time_per_move(game, 5000);
                  computer_play(game, 60);
                  game->last_move = game->moves_played;
                  generate_moves(&movelist, game->board, game->side_to_move);
                  al_set_display_icon(board_display, side_icon[game->side_to_move == BLACK]);
                  player_select = -1;
                  redraw = true;
                  break;
            }
            break;
         case ALLEGRO_EVENT_TIMER:
            display_timer_counter++;
            if (player_select > -1) redraw = true;
            break;
         case EVENT_FILE_DIALOG_CLOSE:
            al_unregister_event_source(queue, &file_dialog->event_source);

            /* If files were selected, we replace the old files list.
             * Otherwise the dialog was cancelled, and we keep the old results.
             */
            if (al_get_native_file_dialog_count(file_dialog->file_dialog) > 0) {
               ALLEGRO_FILECHOOSER *dlg;
               dlg = file_dialog->file_dialog;
               last_path = al_get_native_file_dialog_path(dlg, 0);
               if (al_get_native_file_dialog_count(dlg)) {
                  const char *path;
                  const char *name;
                  gamestate_t *new_game = NULL;
                  path = al_get_native_file_dialog_path(dlg, 0);
                  name = path;

                  /* Check for PGN file extension */
                  ALLEGRO_PATH *alpath = al_create_path(al_get_native_file_dialog_path(file_dialog->file_dialog, 0));
                  if ( strstr(".pgn", al_get_path_extension(alpath)) ) {
                     file_dialog->flags &= DIALOG_FLAGS_PGN;
                  }
                  al_destroy_path(alpath);

                  if (file_dialog->flags & DIALOG_FLAGS_LOAD) {
                     /* Load file */
                     file_index = 0;
                     free(filename);
                     filename = NULL;
                     if (file_dialog->flags & DIALOG_FLAGS_EPD) {
                        new_game = load_epd_file(name, file_index);
                        filename = strdup(name);
                     } else if (file_dialog->flags & DIALOG_FLAGS_PGN) {
                        new_game = load_pgn_file(name, file_index);
                        filename = strdup(name);
                     }
                     if (new_game) {
                        end_game(game);
                        game = new_game;
                        generate_moves(&movelist, game->board, game->side_to_move);
                     } else {
                        al_show_native_message_box(board_display, "Error", "",
                              "Could not import the selected file... sorry :(",
                              NULL, ALLEGRO_MESSAGEBOX_ERROR);
                     }
                     redraw = true;
                  } else {
                     /* Save file */
                     if (file_dialog->flags & DIALOG_FLAGS_EPD)
                        write_epd_file(game, name);
                  }
               }
            } else {
               stop_async_dialog(file_dialog);
            }
            file_dialog = NULL;
            break;
      }

   }

   end_game(game);

   return 0;
}
