/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include <ctype.h>
#include "board.h"
#include "pieces.h"

#if defined _WIN32 || defined _WIN64
bool unicode_board = false;
#else
bool unicode_board = true;
#endif

void clear_board(board_t *board)
{
   memset(board, 0, sizeof *board);
}

static void print_uint8(FILE *f, uint8_t b)
{
   int c;
   
   for (c=0; c<8; c++) {
      int bit = (b>>c)&1;
      if (bit)
         fprintf(f, "%d", bit);
      else
         fprintf(f, ".");
   }
}

void print_bitboards(const board_t *board)
{
   print_bitboards_file(stdout, board);
}

void print_bitboards_file(FILE *f, const board_t *board)
{
   int c;

   fprintf(f, "White pieces\tBlack pieces\tUnmoved pieces\tFlipped \tRotated+\tRotated-\n");
   for (c=7; c>=0; c--) {
      print_uint8(f, get_bitboard_row(board->bbc[0], c));
      fprintf(f, "\t");
      print_uint8(f, get_bitboard_row(board->bbc[1], c));
      fprintf(f, "\t");
      print_uint8(f, get_bitboard_row(board->init, c));
      fprintf(f, "\t");
#ifdef USE_ROTATED_BITBOARDS
      print_uint8(f, get_bitboard_row(board->bbf, c));
#else
      print_uint8(f, 0);
#endif
      fprintf(f, "\t");
#ifdef USE_ROTATED_BITBOARDS
      print_uint8(f, get_bitboard_row(board->bbr[0], c));
#else
      print_uint8(f, 0);
#endif
      fprintf(f, "\t");
#ifdef USE_ROTATED_BITBOARDS
      print_uint8(f, get_bitboard_row(board->bbr[1], c));
#else
      print_uint8(f, 0);
#endif
      fprintf(f, "\n");
   }

   fprintf(f, "\nPawns     Knights   Bishops   Rooks     Queens    Kings      e.p.\n");
   bitboard_t ep_bb = board_empty;
   if (board->ep_square) ep_bb = make_bitboard_square(board->ep_square);
   for (c=7; c>=0; c--) {
      print_uint8(f, get_bitboard_row(board->bbp[PAWN], c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board->bbp[KNIGHT], c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board->bbp[BISHOP], c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board->bbp[ROOK], c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board->bbp[QUEEN], c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board->bbp[KING], c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(ep_bb, c));
      fprintf(f, "\n");
   }

   /*
   fprintf(f, "\nBoard-a   Board-h   Centre    X-centre  K-half    Q-half\n");
   for (c=7; c>=0; c--) {
      print_uint8(f, get_bitboard_row(board_minus_a, c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board_minus_h, c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board_centre, c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board_xcentre, c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board_kwing, c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board_qwing, c));
      fprintf(f, "\n");
   }

   fprintf(f, "\n1strank   2ndrank   7thrank   8thrank                   \n");
   for (c=7; c>=0; c--) {
      print_uint8(f, get_bitboard_row(board_rank1, c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board_rank2, c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board_rank7, c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(board_rank8, c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(0, c));
      fprintf(f, "  ");
      print_uint8(f, get_bitboard_row(0, c));
      fprintf(f, "\n");
   }
   */
}

#define UNICODE
void print_board(const board_t *board)
{
   print_board_file(stdout, board);
}

void print_board_file(FILE *file, const board_t *board)
{
   int c, n;
   assert(board);
   bitboard_t occ = board->bbc[0] | board->bbc[1];

   fprintf(file, "+--+--+--+--+--+--+--+--+\n");

   for (c=7; c>=0; c--) {
      for (n=0; n<8; n++) {
         int square = pack_row_file(c, n);
         if (!(occ & make_bitboard_square(square))) {
            if ( (c^n)&1 )
               fprintf(file, "|  ");
            else
               fprintf(file, "|+ ");
         } else {
            int piece = get_piece(board, square);
            int piece_type = decode_piece_type(piece);
            int piece_colour = decode_piece_colour(piece);
            if (!unicode_board) {
               const char *piece_str = "pnbrqk";
               char c = piece_str[piece_type];
               if (piece_colour == WHITE)
                  c = toupper(c);
               fprintf(file, "|%c ", c);
            } else {
               const char wcode[6] = { '\224', '\225', '\226', '\227', '\230', '\231' };
               const char bcode[6] = { '\232', '\233', '\234', '\235', '\236', '\237' };
               char c = wcode[5 - piece_type];

               if (piece_colour)
                  c = bcode[5 - piece_type];

               fprintf(file, "|\342\231%c ", c);
            }
         }
      }
      fprintf(file, "|\n");
      fprintf(file, "+--+--+--+--+--+--+--+--+\n");
   }
}

