#include "game.h"
#include "move.h"
#include "history.h"

#define HISTORY_MAX 0x4000

void update_history(gamestate_t *game, move_t move, int score)
{
#ifdef USE_HISTORY_HEURISTIC
   int history_score;

   if (game->board->in_check || is_promotion_move(move) || is_capture_move(move))
      return;

   int side = get_move_player(move)>>7;
   int piece = get_move_piece(move) & PIECE;
   int to = get_move_destination(move);

   game->history[side][piece][to] += score;
   history_score = abs(game->history[side][piece][to]);
   if (history_score > game->max_history[side])
      game->max_history[side] = history_score;

   /* Rescale as needed */
   if (game->max_history[side] > HISTORY_MAX) {
      game->max_history[side] /= 2;
      for (side = 0; side <= 1; side++)
         for (piece=PAWN; piece<=KING; piece++)
            for (to=0; to<64; to++)
               game->history[side][piece][to] /= 2;
   }
#endif
}

int get_move_history_score(const gamestate_t *game, move_t move)
{
#ifdef USE_HISTORY_HEURISTIC
   int side = get_move_player(move)>>7;
   int piece = get_move_piece(move) & PIECE;
   int to = get_move_destination(move);
   return game->history[side][piece][to];
#else
   return 0;
#endif
}

int get_move_history_scale(const gamestate_t *game, move_t move)
{
#ifdef USE_HISTORY_HEURISTIC
   int side = get_move_player(move)>>7;
   return game->max_history[side];
#else
   return 1;
#endif
}

void clear_history(gamestate_t *game)
{
#ifdef USE_HISTORY_HEURISTIC
   memset(game->history, 0, sizeof game->history);
   game->max_history[0] = game->max_history[1] = 0;
#endif

   /* Clear the history reduction table */
#ifdef HISTORY_REDUCTION
   for (int n=0; n<64; n++) {
      for (int c=0; c<6; c++) {
         game->history_reduce[0][c][n] = game->history_reduce[1][c][n] = 100;
      }
   }
#endif
}

void scale_history(gamestate_t *game)
{
#ifdef USE_HISTORY_HEURISTIC
   int n, k;
   /* Scale down history scores */
   //while (game->max_history > MAX_HISTORY) {
      for (n=0; n<6; n++) {
         for (k=0; k<64; k++) {
            game->history[0][n][k] /= 2;
            game->history[1][n][k] /= 2;
         }
      }
      game->max_history[0] /= 2;
      game->max_history[1] /= 2;
   //}
#endif
}



#define in_check(n) game->board[(n)].in_check

void update_history_reduction_table(gamestate_t *game, move_t move, move_t prev_move, move_t my_prev_move, int score, int alpha)
{
   int piece_idx = get_promotion_piece(move)&PIECE;
   int side = game->side_to_move>>7;
   if (prev_move == 0 || my_prev_move == 0) return;
#ifdef HISTORY_REDUCTION
   if (score > alpha) {
      game->history_reduce[side][piece_idx][get_move_destination(move)] += 5;
   } else {
      game->history_reduce[side][piece_idx][get_move_destination(move)]--;
   }
#endif
}

/* Heuristics to decide whether this move is a candidate for reduction,
 * based on how similar moves have behaved in the past.
 */
bool reduce_from_history(gamestate_t *game, move_t move, move_t prev_move, move_t my_prev_move, int static_score, int alpha)
{
   int piece_idx;
   int side;
   if (is_capture_move(move) || in_check(0) || in_check(-1) || static_score >= alpha+25)
      return false;

   piece_idx = get_promotion_piece(move)&PIECE;
   side = get_move_player(move) >> 7;
#ifdef HISTORY_REDUCTION
   if (game->history_reduce[side][piece_idx][get_move_destination(move)] < 50) {
      game->history_reduce[side][piece_idx][get_move_destination(move)] = 50;
      return true;
   }
#endif

   return false;
}

int get_history_reduction_score(const gamestate_t *const game, move_t move)
{
#ifdef HISTORY_REDUCTION
   int piece = get_move_piece(move)&PIECE;
   int side = get_move_player(move) >> 7;

   return game->history_reduce[side][piece][get_move_destination(move)];
#else
   return 100;
#endif
}
