/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef EVALUATE_H
#define EVALUATE_H

#include <stdint.h>
#include "board.h"

#define PENALISE_HANGING_PIECES
#define USE_SLIDER_TERMS

#define CHECKMATE 16000
#define STALEMATE     0
#define ILLEGAL   (CHECKMATE+500)

/* King safety evaluation parameters */
#define KING_SAFETY_PIECE_WEIGHT_NORMALISATION  14    /* Normalisation factor for attacking power of pieces */
#define KING_SAFETY_PIECE_WEIGHT_MAX             4    /* Maximum index due to piece attack pattern */
#define KING_SAFTEY_STRENGTH_SCALE              96    /* Normalisation for army strength */

/* Parameters for king safety evaluation table (adopted from Jazz) */
#define KING_SAFETY_CEIL         262.5       /* Asymptotic value */
#define KING_SAFETY_SLOPE          0.15      /* Slope of the king safety evaluation term */
#define KING_SAFETY_INFLECTION    18         /* Inflection point where the evaluation begins to rise steeply */

/* Trade down/pawn value in end games */
#define PAWN_STEP_ENDGAME           5

/* Knight value adjustment multiplier (4 according to Kaufman) */
#define KNIGHT_VALUE_SCALE    4

/* Rook value adjustment multiplier (3 according to Kaufman) */
#define ROOK_VALUE_SCALE      3

extern const int piece_value[NUM_PIECES];
extern const int8_t centre_table[64];

void initialise_evaluation(void);
int16_t static_evaluation(const board_t *board, sides side_to_move, int alpha, int beta);

static inline bool mate_score(int score)
{
   static const int mate = CHECKMATE - 1000;
   if ( score > mate || score < -mate )
      return true;

   return false;
}

static inline int score_to_hashtable(int score, int depth)
{
   if (mate_score(score)) {
      assert(score + depth <= CHECKMATE);
      assert(score - depth >= -CHECKMATE);
      if (score > 0)
         return score + depth;
      else
         return score - depth;
   } else
      return score;
}

static inline int score_from_hashtable(int score, int depth)
{
   if (mate_score(score)) {
      assert( (score>0 && (score - depth <=  CHECKMATE)) ||
              (score<0 && (score + depth >= -CHECKMATE)) );
      if (score > 0)
         return score - depth;
      else
         return score + depth;
   } else
      return score;
}


#endif

