/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdarg.h>
#include <ctype.h>
#include <math.h>
#include "square.h"
#include "board.h"
#include "move.h"
#include "movegen.h"
#include "game.h"
#include "timer.h"

static inline int min(int x, int y)
{
   return (x<y) ? x : y;
}

static inline int max(int x, int y)
{
   return (x>y) ? x : y;
}

static char buf[65536];
static void printfstderr(const char *msg, ...)
{
   va_list ap;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   fprintf(stderr, "%s", buf);
}

static void (*default_iteration_output)(const char *, ...) = NULL;
static void (*default_uci_output)(const char *, ...) = NULL;
static void (*default_xboard_output)(const char *, ...) = NULL;
static void (*default_error_output)(const char *, ...) = printfstderr;

#define pstderr if (game && game->error_output) game->error_output

void playgame_init(void)
{
   default_iteration_output = (void *)printf;
   default_uci_output = NULL;
}

void playgame_shutdown(void)
{
}

void set_default_output_function(void (*func)(const char *, ...))
{
   default_iteration_output = func;
}

void set_uci_output_function(void (*func)(const char *, ...))
{
   default_uci_output = func;
}

void set_xboard_output_function(void (*func)(const char *, ...))
{
   default_xboard_output = func;
}

void set_error_output_function(void (*func)(const char *, ...))
{
   default_error_output = func;
}

void set_transposition_table_size(game_t *game, uint64_t nelem)
{
   if (!game)
      return;

   /* Round to the next-lowest power of 2 */
   nelem |= (nelem >> 1);
   nelem |= (nelem >> 2);
   nelem |= (nelem >> 4);
   nelem |= (nelem >> 8);
   nelem |= (nelem >> 16);
   nelem |= (nelem >> 32);
   nelem >>= 1;
   nelem++;

   game->default_hash_size = nelem;

   /* Reallocate table if it was already allocated */
   if (game->default_hash_size) {
      destroy_hash_table(game->transposition_table);
      game->transposition_table = create_hash_table(game->default_hash_size);
   }
}


game_t *create_game(void)
{
   game_t *game = calloc(1, sizeof(game_t));
   game->default_hash_size = HASH_TABLE_SIZE;
   game->start_fen = strdup("lgkcckwl/hhhhhhhh/8/8/8/8/PPPPPPPP/RNBQKBNR w KQ -");
   set_time_per_move(game, 100);
   return game;
}

/* Initialize the game datastructure to correspond to a new game */
void start_new_game(game_t *game)
{
   game->max_moves = 1000;
   free(game->score);
   free(game->in_check);
   free(game->did_castle);
   free(game->fifty_counter);
   free(game->ep);
   free(game->ep_capture);
   free(game->hash_key);
   free(game->init);
   free(game->move_list);
   destroy_hash_table(game->transposition_table);
   game->score          = calloc(game->max_moves, sizeof *game->score);
   game->in_check       = calloc(game->max_moves, sizeof *game->in_check);
   game->did_castle     = calloc(game->max_moves, sizeof *game->did_castle);
   game->fifty_counter  = calloc(game->max_moves, sizeof *game->fifty_counter);
   game->ep             = calloc(game->max_moves, sizeof *game->ep);
   game->ep_capture     = calloc(game->max_moves, sizeof *game->ep_capture);
   game->hash_key       = calloc(game->max_moves, sizeof *game->hash_key);
   game->init           = calloc(game->max_moves, sizeof *game->init);
   game->move_list      = calloc(game->max_moves, sizeof *game->move_list);

   game->transposition_table = NULL;

   clear_board(&game->board);
   if (game->start_fen)
      setup_fen_position(game, game->start_fen);

   if (game->default_hash_size == 0) game->default_hash_size = HASH_TABLE_SIZE;
   game->transposition_table = create_hash_table(game->default_hash_size);

   game->analysing = false;
   game->pondering = false;
   game->ponder_move = 0;

   game->name_white = NULL;
   game->name_black = NULL;

   game->output_iteration = default_iteration_output;
   game->uci_output = default_uci_output;
   game->xboard_output = default_xboard_output;
   game->error_output = default_error_output;
}

/* Frees the memory used by a game */
void end_game(game_t *game)
{
   if (game) {
      destroy_hash_table(game->transposition_table);
      free(game->start_fen);
      free(game->xb_setup);
      free(game->name_white);
      free(game->name_black);
      free(game->score);
      free(game->in_check);
      free(game->did_castle);
      free(game->fifty_counter);
      free(game->ep);
      free(game->ep_capture);
      free(game->hash_key);
      free(game->init);
      free(game);
   }
}

int get_piece_index(const game_t *game, const char *symbol)
{
   int n;

   if (!symbol)
      return -1;

   return -1;
}

void setup_fen_position(game_t *game, const char *str)
{
   const char *s = str;
   int n;
   int square = A8;
   int prev_rank = 16;

   assert(game);
   if (!s) return;
   game->moves_played = 0;
   clear_board(&game->board);
   memset(game->repetition_hash_table, 0, sizeof(game->repetition_hash_table));

   /* Parse first record: piece positions */
   while(*s && (*s != ' ') && (*s != '[') && square>=0) {
      switch (*s) {
         case '/':
            square -= prev_rank;
            break;
         case '1':
            if (isdigit(s[1])) {
               square += 10;
               break;
            }
         case '2':
         case '3':
         case '4':
         case '5':
         case '6':
         case '7':
         case '8':
         case '9':
         case '0':
            square += (*s - '1')+1;
            break;

         default:
            {
               char *p = strchr(white_piece_string, *s);
               int colour = WHITE;
               if (p) {
                  n = p - white_piece_string;
               } else {
                  p = strchr(black_piece_string, *s);
                  colour = BLACK;
                  n = p - black_piece_string;
               }
               assert(n >= 0);
               assert(n < NUM_PIECES);
               if (p) {
                  place_piece(&game->board, n, colour, square);
                  square++;
                  break;
               }
            }
            break;
      }
      s++;
   }

   /* Optional: check for holdings */
   while(*s && (*s == ' ')) s++;
   if (*s == '[') {
      s++;
      while (*s != ']') {
         char *p = strchr(white_piece_string, *s);
         sides side = WHITE;
         if (p) {
            n = p - white_piece_string;
         } else {
            p = strchr(black_piece_string, *s);
            side = BLACK;
            n = p - black_piece_string;
         }
         assert(n >= 0);
         assert(n < NUM_PIECES);
         if (p)
            game->board.holdings[side] |= 1 << n;
         s++;
      }
      s++;
   }


   /* Second record: side to move */
   while(*s && (*s == ' ')) s++;
   game->board.side_to_move = WHITE;
   if (*s == 'b') {
      game->board.side_to_move = BLACK;
      game->board.hash ^= side_to_move_key;
   }
   if (*s) s++;

   /* Third record: castling rights. */
   while(*s && (*s == ' ')) s++;
   if (!isdigit(*s)) {
      bitboard_t castle_init = board_empty;

      /* Clear castle flags by default */
      game->board.init &= ~game->board.bbp[KING];

      while(*s && (*s != ' ')) {
         switch (*s) {
            case '-':
               break;
            case 'K':
               castle_init |= E1_MASK;
               castle_init |= H1_MASK;
               break;
            case 'Q':
               castle_init |= E1_MASK;
               castle_init |= A1_MASK;
               break;

            case 'k':
               castle_init |= E8_MASK;
               castle_init |= H8_MASK;
               break;
            case 'q':
               castle_init |= E8_MASK;
               castle_init |= A8_MASK;
               break;

            default:
               break;
         }
         s++;
      }
      if (*s) s++;

      /* Set castle flags.  */
      game->board.init |= castle_init;
   }

   /* Make sure the initial flags are at least somewhat sane by making sure only occupied squares have their
    * init bits set.
    */
   game->board.init &= game->board.bbc[WHITE] | game->board.bbc[BLACK];

   /* Fourth record: En-passant square.
    * If this record is a number, then the game doesn't have en-passant capture and we skip it.
    */
   while(*s && (*s == ' ')) s++;
   if (!isdigit(*s)) {
      if (*s && (*s != '-')) {
         int col = *s - 'a';
         int row = s[1] - '1';
         s+=2;
         /* En-passant move-to square */
         game->board.ep = pack_row_file(row, col);
         /* En-passant capture square, this may be encoded in the FEN */
         if (!isspace(*s)) {
            int col = *s - 'a';
            int row = s[1] - '1';
            s+=2;
            game->board.ep_capture = pack_row_file(row, col);
         } else {
            /* Assume we have normal pawns, in which case we can simply derive it from the move-to square */
            if (game->board.side_to_move == WHITE)
               game->board.ep_capture = game->board.ep - 8;
            else
               game->board.ep_capture = game->board.ep + 8;
         }
      }
      if (*s == '-') s++;
   }

   /* Fifth record: half-move counter (50 move counter) */ 
   while(*s && (*s == ' ')) s++;
   sscanf(s, "%d\n", &n);
   game->board.fifty_counter = game->fifty_counter[0] = n;
   while(*s && (*s != ' ')) s++;

   memset(game->repetition_hash_table, 0, sizeof game->repetition_hash_table);
   game->repetition_hash_table[game->board.hash&0xFFFF] = 1;

   game->board.init &= board_rank1 | board_rank8;

   /* Set the in-check flag */
   game->board.in_check = player_in_check(&game->board, game->board.side_to_move);
   game->in_check[0] = game->board.in_check;
}

char *make_fen_string(const game_t *game, char *buffer)
{
   static char static_buffer[4096];
   bitboard_t occ;
   char *fen = buffer;
   int n = 0;
   int r, f, s;

   if (!fen) fen = static_buffer;
   fen[0] = '\0';

   occ = get_occupied(&game->board);

   /* First record: board position */
   /* Scan all ranks */
   for (r = 8-1; r>=0; r--) {
      int count = 0;
      for (f = 0; f < 8; f++) {
         s = pack_row_file(r, f);

         /* Empty? */
         if ((occ & make_bitboard_square(s)) == 0) {
            count++;
            continue;
         }

         /* Not empty, do we have a count? */
         if (count) n += snprintf(fen+n, 4096 - n, "%d", count);
         count = 0;

         /* Print piece */
         int side = WHITE;
         if (game->board.bbc[BLACK] & make_bitboard_square(s)) {
            side = BLACK;
         }

         int piece = get_piece(&game->board, s);
         if (side == WHITE) {
            n += snprintf(fen+n, 4096-n, "%c", white_piece_string[piece]);
         } else {
            n += snprintf(fen+n, 4096-n, "%c", black_piece_string[piece]);
         }
      }
      if (count) n += snprintf(fen+n, 4096 - n, "%d", count);
      if (r) n += snprintf(fen+n, 4096 - n, "/");
   }

   n += snprintf(fen+n, 4096 - n, " ");

   /* Second record: side to move */
   if (game->board.side_to_move == WHITE)
      n += snprintf(fen+n, 4096 - n, "w ");
   else
      n += snprintf(fen+n, 4096 - n, "b ");

   /* Third record: castling rights
    * FIXME: check for kingside/queenside castling!
    */
   if (may_castle(&game->board, WHITE)) n += snprintf(fen+n, 4096 - n, "KQ");
   if (may_castle(&game->board, BLACK)) n += snprintf(fen+n, 4096 - n, "kq");
   n += snprintf(fen+n, 4096 - n, " ");

   /* Fourth record: en-passant square */
   if (game->board.ep)
      n += snprintf(fen+n, 4096 - n, "%s", square_names[game->board.ep]);
   else
      n += snprintf(fen+n, 4096 - n, "-");
   n += snprintf(fen+n, 4096 - n, " ");

   /* Fifth and sixth record: half-move counter and full-move counter */
   int par = (game->moves_played&1) && next_side[game->board.side_to_move];
   n += snprintf(fen+n, 4096 - n, "%d ", game->fifty_counter[game->moves_played]);
   n += snprintf(fen+n, 4096 - n, "%d", (int)(game->moves_played + par)/2 + 1);

   return fen;
}

static const char *file_names[] = { "a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p" };
static const char *row_names[] = { "1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11","12","13","14","15","16" };
static char static_output_buffer[256];

char *short_move_string(const board_t *board, const move_t move, char *buffer)
{
   movelist_t movelist;
   char *s = buffer;
   const char *piece = "";
   const char *suffix = "";
   const char *token = "";
   const char *origin = "";
   int p = get_move_piece(move);

   if (!s) s = static_output_buffer;

   if (board)
      generate_moves(&movelist, board, board->side_to_move);

   if (move == 0) {
      snprintf(s, 15, "(null)");
      goto done;
   }

   if (is_castle_move(move)) {
      int from = decode_pickup_square(get_move_pickup(move,0));
      int to   = decode_drop_square(get_move_drop(move,0));
      if (to > from)
         snprintf(s, 256, "O-O");
      else
         snprintf(s, 256, "O-O-O");
      goto done;
   }

   if (is_drop_move(move)) {
      uint16_t p = get_move_drop(move, 0);
      char fp = piece_symbol_string[decode_drop_piece(p)];
      int to = decode_drop_square(get_move_drop(move, 0));
      snprintf(s, 256, "%c@%s", fp, square_names[to]);
      goto done;
   }

   int from, to;
   int c, k;
   char fp = '\0', tp = '\0';

   if (is_promotion_move(move))
      tp = piece_symbol_string[decode_drop_piece(get_move_drop(move, 0))];

   from = to = decode_drop_square(get_move_drop(move, 0));
   k = get_move_pickups(move);
   for (c=0; c<k; c++) {
      uint16_t p = get_move_pickup(move, c);
      if (decode_pickup_side(p) == decode_drop_side(get_move_drop(move, 0))) {
         from = decode_pickup_square(p);
         fp = piece_symbol_string[decode_pickup_piece(p)];
         break;
      }
   }

   if (is_capture_move(move)) token = "x";
   if (fp == ' ' && is_capture_move(move)) 
      fp = file_names[unpack_file(get_move_from(move))][0];

   /* Test whether the combination piece/to is unambiguous, otherwise dismbiguate */
   if (board == NULL) goto disambiguous;
   int count = 0;
   int n;
   for (n=0; n<movelist.num_moves; n++) {
      if (get_move_gates(movelist.move[n])) continue;
      if (get_move_piece(move) == get_move_piece(movelist.move[n]) &&
            to == get_move_to(movelist.move[n]) && !is_drop_move(move)) count++;
   }
   if (count <= 1) goto disambiguous;

   /* Try to disambiguate by file */
   count = 0;
   for (n=0; n<movelist.num_moves; n++) {
      if (get_move_gates(movelist.move[n])) continue;
      if (get_move_piece(move) == get_move_piece(movelist.move[n]) &&
            to == get_move_to(movelist.move[n]) &&
            unpack_file(from) == unpack_file(get_move_from(movelist.move[n])))
         count++;
   }
   if (count == 1) {
      origin = file_names[unpack_file(from)];
   } else {
      /* Disambiguate by row */
      origin = row_names[unpack_rank(from)];
   }
disambiguous:

   snprintf(s, 256, "%c%s%s%s%c", fp, origin, token, square_names[to], tp);
done:

   /* Check for gate moves */
   if (get_move_gates(move)) {
      char *p = s + strlen(s);
      int c = get_move_drops(move)-1;
      uint16_t drop = get_move_drop(move, c);
      char fp = piece_symbol_string[decode_drop_piece(drop)];
      int to = decode_drop_square(drop);
      snprintf(p, 200, "/%c%s", fp, square_names[to]);
   }
   return s;
}
