/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <inttypes.h>
#include <limits.h>
#include "genrand.h"
#include "movegen.h"
#include "board.h"
#include "game.h"
#include "play.h"
#include "search.h"
#include "evaluate.h"
#include "timer.h"
#include "keypressed.h"
#include "movelist.h"

#undef DEBUG
#ifdef DEBUG
#define abtrace printf
#else
#define abtrace(s,...)
#endif

#define WINDOW_SIZE 50

#define print_iter if (game->output_iteration) game->output_iteration
#define xb if (game->xboard_output) game->xboard_output

#define check_keyboard(game) ((game->check_keyboard) ?  game->check_keyboard(game) : false)

static const int *qsort_score_ptr;
bool pause_analysis = false;
bool win_or_loss;

static int min(int x, int y)
{
   return (x<y)?x:y;
}

static int max(int x, int y)
{
   return (x>y)?x:y;
}

/* quicksort (qsort) callback function */
static int movelist_sorter(const void *mi1, const void *mi2)
{
   int *score = (int*)qsort_score_ptr;
   int i1 = *(int*)mi1;
   int i2 = *(int*)mi2;

   return score[i2] - score[i1];
}

/* Retrieve principle variation from the hash table */
void retrieve_principle_variation(game_t *game, move_t move)
{
   static int n = 30;
   hash_table_entry_t *hash;

   return;

   if (n == 0) return;
   n--;

   if (count_repetition(game) > 1) return;

   print_iter("%s ", move_string(move, NULL));
   playmove(game, move);

   /* Look up th resulting position in the hash table */
   hash = query_table_entry(game->transposition_table, game->board.hash);

   /* Recursive call to get next move from the table */
   if (hash)
      retrieve_principle_variation(game, hash->best_move);

   takeback(game);
}

static void insert_pv_in_transposition_table(game_t *game, int score)
{
   int n;

   if (score == 0)
      return;

   uint64_t hash_key;
   hash_key = game->board.hash;

   /* Don't inject moves from the quiescence search into the transposition table */
   for (n=0; n<game->length_of_variation[0]; n++) {
      int depth = game->length_of_variation[0] - n;
      move_t move = game->principle_variation[n][0];
      hash_table_entry_t *hash;

      hash = query_table_entry(game->transposition_table, hash_key);

      /* Don't inject captures: they'll be order high up in the movelist anyway, and we don't want to inject
       * moves from the quiescence search and risk getting a cutoff from them.
       */
      if (!hash && !is_capture_move(move))
         //store_table_entry(game->transposition_table, hash_key, 
         //                  depth, score_to_hashtable(score, n), HASH_TYPE_EXACT, move);

      hash_key = update_hash_from_move(move, hash_key);
   }
}

static void extend_pv_from_transposition_table(game_t *game)
{
   static uint8_t repetition_table[0x10000];
   hash_table_entry_t *hash;
   int n;
   uint64_t hash_key;

   /* Set up the repetition table */
   memcpy(repetition_table, game->repetition_hash_table, 0x10000);

   hash_key = game->board.hash;
   for (n=0; n<game->length_of_variation[0]; n++) {
      move_t move = game->principle_variation[n][0];

      hash_key = update_hash_from_move(move, hash_key);
      repetition_table[hash_key&0xFFFF]++;
   }

   /* Now keep pulling moves from the hash table */
   hash = query_table_entry(game->transposition_table, hash_key);
   while (hash && hash->best_move && (hash->flags&HASH_TYPE_EXACT)) {
      if (repetition_table[hash_key&0xFFFF] >= 1) break;

      n = game->length_of_variation[0];
      game->principle_variation[n][0] = hash->best_move;
      game->length_of_variation[0]++;

      hash_key = update_hash_from_move(hash->best_move, hash_key);
      hash = query_table_entry(game->transposition_table, hash_key);
      repetition_table[hash_key&0xFFFF]++;
   }
}

static void print_principle_variation(const game_t *game)
{
   int c;
   int root_move_number = 0;

   print_iter("   ");
   if (game->board.side_to_move) {
      root_move_number = 1;
      print_iter("%d. ... ", (int)(game->moves_played)/2+1);
   }
   for (c=0; c<game->length_of_variation[0]; c++) {
      if (((root_move_number+c) & 1) == 0)
         print_iter("%d. ", (int)(root_move_number+game->moves_played+c)/2+1);
      print_iter("%-5s ", short_move_string(NULL, game->principle_variation[c][0], NULL));
   }
}


static void print_principle_variation_xb(const game_t *game)
{
   int c;
   int root_move_number = 0;

   xb("   ");
   if (game->board.side_to_move) {
      root_move_number = 1;
      xb("%d. ... ", (int)(game->moves_played)/2+1);
   }
   for (c=0; c<game->length_of_variation[0]; c++) {
      if (((root_move_number+c) & 1) == 0)
         xb("%d. ", (int)(root_move_number+game->moves_played+c)/2+1);
      xb("%-5s ", short_move_string(NULL, game->principle_variation[c][0], NULL));
   }
}


/* Returns true while the game has not ended */
bool computer_play(game_t *game, int max_depth)
{
   static const char *output_lines[] = {
      "<%2d>   ---   % 7.2f %8d  %+7.2f -> %+7.2f %2d /%2d ",
      "<%2d>         % 7.2f %8d  %+7.2f -> %+7.2f %2d /%2d ",
      "<%2d>   !!!   % 7.2f %8d  %+7.2f -> %+7.2f %2d /%2d ",
      "[%2d] %+7.2f % 7.2f %8d     (EBR=%5.2f)     %2d /%2d "
   };
   uint64_t start_time, time, prev_time, prev_prev_time;
   sides me;
   movelist_t movelist;
   int perm[MAX_MOVES];
   int score[MAX_MOVES];
   int eval[MAX_MOVES];
   int node_count[MAX_MOVES];
   bool checking[MAX_MOVES];
   int alpha, beta;
   int legal_moves;
   int depth;
   int n;
   int root_move_number = 0;

   game->extra_time = 0;
   game->running_clock = game->board.side_to_move;
   game->root_moves_played = game->moves_played;
   game->ponder_move = 0;

   me = game->board.side_to_move;

   abort_search = false;
   positions_evaluated = 0;
   moves_searched = 0;
   positions_in_hashtable = 0;
   branches_pruned = 0;
   win_or_loss = false;
   pause_analysis = false;

   /* Count a number of obvious things: repetition draw, 50 move rule */
   if (game->fifty_counter[game->moves_played] >= 100 || count_repetition(game)>=3)
      return false;

   /* Start the clock */
   start_time = get_timer();
   start_clock(game);

   /* Prepare the transpostion table for a new search iteration:
    * Resets the write count and increases the generation counter.
    */
   prepare_hashtable_search(game->transposition_table);

   memset(score, 0, sizeof score);
   memset(checking, 0, sizeof checking);

   memset(game->principle_variation, 0, sizeof game->principle_variation);
   game->length_of_variation[0] = 0;

   /* Clear the history table */
   //memset(game->history, 0, sizeof game->history);
   //game->max_history = 0;

   generate_moves(&movelist, &game->board, me);

   if (max_depth>MAX_SEARCH_DEPTH) max_depth = MAX_SEARCH_DEPTH;

   legal_moves = movelist.num_moves;

   if ((game->moves_played & 1) && (me == WHITE)) root_move_number++;

   /* Look up the root position in the hash table, re-search the move from
    * the hash table first
    */
   hash_table_entry_t *hash = query_table_entry(game->transposition_table, game->board.hash);

   /* First step: initial ordering of the moves, weed out illegal moves */
   prev_time = time = get_timer();
   for (n=0; n<movelist.num_moves; n++) {
      move_t move = movelist.move[n];
      bool check;
      perm[n] = n;

      //eval[n] = static_evaluation(game, me, -CHECKMATE, CHECKMATE);

      playmove(game, move);
      check = player_in_check(&game->board, me);
      checking[n] = player_in_check(&game->board, next_side[me]);
      if (!check) eval[n] = -search(game, -CHECKMATE, CHECKMATE, 0, 0);
      score[n] = eval[n];
      takeback(game);
      if (check) {   /* Illegal move */
         legal_moves--;
         score[n] = -ILLEGAL-100;
         node_count[n] = -ILLEGAL-100;
         continue;
      }

      /* Give the highest priority to the move from the hash table, if it
       * exists.
       */
      if (hash && moves_are_equal(hash->best_move, move)) {
         score[n] = eval[n] = hash->score;
         score[n] += CHECKMATE;
         continue;
      }

      /* Castling moves */
      if (is_castle_move(move)) {
         score[n] += 1000;
         continue;
      }

      /* If we're at the beginning of the game, encourage development moves */
      if (game->moves_played < 10) {
      }

      /* Regular captures
       * The score is 950+the SSE score, which sorts winning captures below
       * promotion moves (with a score >= 1050) and losing captures with a
       * score <1000 below everything else.
       */
      if (is_capture_move(move)) {
      }

      if (game->moves_played < 10)
         score[n] += genrandui() & 0x1ff;
   }

   /* Record whether the side to move is in check or not */
   game->in_check[game->moves_played] = player_in_check(&game->board, me);

   /* Hmm... no legal moves? */
   if (legal_moves == 0) {
      win_or_loss = game->in_check[game->moves_played];
      return false;
   }

   /* Sort the move list based on the score */
   qsort_score_ptr = score;
   qsort(perm, movelist.num_moves, sizeof *perm, movelist_sorter);
   if (hash && moves_are_equal(hash->best_move, movelist.move[perm[0]]))
      score[perm[0]] -= CHECKMATE;

   /* If there is only one legal move, then clearly that's the best one.
    * Otherwise use iterative deepening to find the best move.
    */
   if (legal_moves == 1)
      max_depth = 0;

   if (legal_moves == 1) {
      int best_move = perm[0];
      xb("% 3d % 4d %5"PRIu64" %9"PRIu64" ", 1, eval[best_move], (get_timer() - start_time)/10000, (uint64_t)moves_searched);
      move_t move = movelist.move[best_move];
      xb("%s\n", move_string(move, NULL));
      pause_analysis = true;
   }

   int best_move = perm[0];

   /* Iterative deepening loop */
   int mate_depth = 0;
   bool in_check = player_in_check(&game->board, me);
   for (depth=1; depth<max_depth; depth++) {
      int best_score = eval[best_move];
      int c;

      game->length_of_variation[1] = 1;
      backup_principle_variation(game, 0, movelist.move[best_move]);

      if (depth == 1) {
         alpha = -CHECKMATE;
         beta = CHECKMATE;
      } else {
         alpha = best_score - WINDOW_SIZE;
         beta = best_score + WINDOW_SIZE;
      }

      for (n=0; n<legal_moves; n++) {
         int previous_moves_searched = moves_searched;
         int open_alpha_window = 50;
         int open_beta_window = 50;
         move_t move = movelist.move[perm[n]];
         int new_score;
         int r = 0;

         /* Aspiration search */
#if 0
         if (perm[n] == best_move) {
            if (depth == 1) {
               alpha = best_score - WINDOW_SIZE;
               beta = best_score + WINDOW_SIZE;
            } else {
               alpha = best_score - 8;
               beta = best_score + 8;
            }

            if (best_score > CHECKMATE - 1000) {
               beta = best_score+1;
               alpha = best_score;
            }
         } else {
            alpha = best_score;
            beta = best_score + 1;
         }
#endif
         if (n > 0) {
            alpha = best_score;
            beta = best_score + 1;
         }

         /* Late move reductions */
         r = (n > 2 && depth > 2);
         r += (n > 10 && depth > 6);
         if (is_promotion_move(move) || is_capture_move(move) || in_check || checking[perm[n]])
            r = 0;

         playmove(game, move);
         assert(!player_in_check(&game->board, me));
         int counter = 0;
         while (true) {
            abtrace("Searching move %s (%d/%d) with window [%d %d] at %d\n",
                     short_move_string(&game->board, move, NULL), n, legal_moves, -beta, -alpha, depth-r);
            new_score = -search(game, -beta, -alpha, depth - r, 0);
            abtrace("Returned score %d\n", new_score);
            if (abort_search) break;

            /* Fail high/low, adjust window.
             * Non-PV moves are expected to fail low, so only research fail
             * low for PV-nodes.
             */
            if (new_score <= alpha && n==0) {
               //abtrace("alphabeta [%d %d] returned %d\n",alpha,beta,new_score);
               if (new_score > CHECKMATE - 1000) {
                  alpha = new_score - 2;
               } else {
                  alpha = max(new_score - open_alpha_window, -CHECKMATE);
                  open_alpha_window = CHECKMATE;
               }
               r = 0;
               abtrace("\n");
               abtrace("Fail low %d, set alpha to %d\n", new_score, alpha);
               if (n == 0)    /* Best move failed low, allocate extra time */
                  game->extra_time = peek_timer(game) / 2 + game->time_inc[game->board.side_to_move];
               continue;
            }
            if (new_score >= beta && beta < CHECKMATE) {
               abtrace("alphabeta [%d %d] returned %d\n",alpha,beta,new_score);
               if (new_score > CHECKMATE - 1000) {
                  beta = min(new_score + depth, CHECKMATE);
                  //alpha = new_score-2;
                  //alpha = min(beta - depth - 1, new_score - depth);
               } else {
                  beta = min(new_score + open_beta_window, CHECKMATE);
                  open_beta_window = CHECKMATE;
               }
               r = 0;
               abtrace("\n");
               abtrace("Fail high %d, set beta to %d\n", new_score, beta);
               abtrace("%d (%c)\n", game->length_of_variation[1], game->hash_hit[1] ? 'H' : '-');
               continue;
            }
            break;
         }
         takeback(game);

         if (abort_search) break;

         /* Update score for this move */
         score[perm[n]] = new_score;
         eval[perm[n]] = new_score;
         node_count[perm[n]] = moves_searched - previous_moves_searched;

         /* Score of best move decreased substantially? */
         if (n == 0 && new_score<(best_score-50) && depth > 1) {
               int sqdepth = game->quiescence_depth_of_variation[0];
               int sdepth = game->length_of_variation[0];

               backup_principle_variation(game, 0, move);
               print_iter(output_lines[0],
                     depth+1, (get_timer() - start_time)/1000000.0,
                     positions_evaluated,
                     best_score/100.0, new_score/100.0,
                     sdepth, sqdepth);
               print_principle_variation(game);
               if (game->hash_hit[0]) print_iter("<H>");
               print_iter("\n");
         }


         if (new_score > best_score || n == 0) {
            backup_principle_variation(game, 0, move);

            /* New best move, or score improved substantially */
            if (new_score > (best_score+50) && depth>1) {
               int sqdepth = game->quiescence_depth_of_variation[0];
               int sdepth = game->length_of_variation[0];
               print_iter(output_lines[2],
                     depth+1, (get_timer() - start_time)/1000000.0,
                     positions_evaluated,
                     best_score/100.0, new_score/100.0,
                     sdepth, sqdepth);
               print_principle_variation(game);
               if (game->hash_hit[0]) print_iter("<H>");
               print_iter("\n");
            }

            if (n > 0 && new_score <= (best_score+50) && depth>1) {
               int sqdepth = game->quiescence_depth_of_variation[0];
               int sdepth = game->length_of_variation[0];
               print_iter(output_lines[1],
                     depth+1, (get_timer() - start_time)/1000000.0,
                     positions_evaluated,
                     best_score/100.0, new_score/100.0,
                     sdepth, sqdepth);
               print_principle_variation(game);
               if (game->hash_hit[0]) print_iter("<H>");
               print_iter("\n");
            }

            best_move = perm[n];
            best_score = new_score;
         }
      }

      /* Inject principle variation into the transposition table, in case
       * it was lost.
       */
      extend_pv_from_transposition_table(game);
      insert_pv_in_transposition_table(game, eval[best_move]);

      prev_prev_time = prev_time;
      prev_time = time;
      time = get_timer();

      int sqdepth = game->quiescence_depth_of_variation[0];
      int sdepth = game->length_of_variation[0];
      print_iter(output_lines[3], depth+1,
            eval[best_move]/100.0, (time - start_time)/1000000.0,
            positions_evaluated,
            (float)moves_searched/positions_evaluated,
            (float)(time - prev_time)/(prev_time - prev_prev_time),
            sdepth, sqdepth);
      print_principle_variation(game);
      if (game->hash_hit[0]) print_iter("<H>");
      print_iter("\n");

      /* Send output to XBoard */
      xb("% 3d % 4d %5"PRIu64" %9"PRIu64" ", depth+1, eval[best_move], (get_timer() - start_time)/10000, (uint64_t)moves_searched);
      print_principle_variation_xb(game);
      xb("\n");

      /* Re-sort the move list */
      score[best_move]+=CHECKMATE;
      node_count[best_move] = moves_searched + 1;
      qsort_score_ptr = node_count;
      qsort(perm, movelist.num_moves, sizeof *perm, movelist_sorter);
      score[best_move]-=CHECKMATE;
      //print_iter("%s %d %d\n", move_string(movelist.move[best_move], NULL), best_move, perm[0]);

      /* Break out early if we've found a checkmate */
      if (abs(best_score) >= CHECKMATE-1000) {
         int ply = CHECKMATE - abs(best_score);
         if (mate_depth == 0 || ply < mate_depth)
            mate_depth = ply;
         print_iter("Mate in %d (%d ply)\n", ply/2 + 1, ply);

         /* Don't trap the analysis in a closed loop */
         pause_analysis = true;

         /* Break early if we find a forced mate, but only if it is within
          * the horizon.
          */
         if (mate_depth <= depth || game->length_of_variation[0] == mate_depth)
            break;
      }

      if (abort_search) break;

      /* Break out early if we're told to do so */
      if (check_keyboard(game))
         break;

      /* It's possible the search was paused, so we have to adjust the clocks. This is a display issue only. */
      time += game->start_time - start_time;
      prev_time += game->start_time - start_time;
      prev_prev_time += game->start_time - start_time;
      start_time = game->start_time;

      /* Check if we have enough time for the next iteration, assuming an
       * effective branching ratio of 2.
       */
      if ( game->check_clock && peek_timer(game) > get_game_time_for_move(game)/2 )
         break;
   }

   /* Print output */
   print_iter(" --> %s %.2f\n", move_string(movelist.move[best_move], NULL), eval[best_move]/100.0);
   print_iter("     ");
   print_iter("\n");
   retrieve_principle_variation(game, movelist.move[best_move]);
   print_iter("%"PRIu64" nodes visited (%d [%.2f%%] in transposition table), "
              "%g nodes/s\n"
              "%d branches pruned\n"
              "%"PRIu64" moves searched\n",
         (uint64_t)positions_evaluated,
         positions_in_hashtable,
         100.0*(float)positions_in_hashtable/(positions_in_hashtable + positions_evaluated),
         1.0e6*(positions_evaluated+positions_in_hashtable) / (time - start_time),
         branches_pruned,
         (uint64_t)moves_searched);

   /* Sanity check */
   assert( movelist.move[best_move] == game->principle_variation[0][0] || legal_moves == 1);

   /* Done */
   if (!game->analysing)
      playmove(game, movelist.move[best_move]);
   //printf("play %s ", move_string(movelist.move[best_move], NULL));
   //printf("[%s]\n", move_string(game->move_list[game->moves_played-1], NULL));

   /* Store the best reply as the move we would like to ponder on (the hint move). */
   if (!game->pondering && game->length_of_variation[0] > 1)
      game->ponder_move = game->principle_variation[1][0];;

   return true;
}

static bool interrupt_ponder(game_t *game)
{
   return keyboard_input_waiting();
}

bool computer_ponder(game_t *game)
{
   move_t ponder_move = game->ponder_move;
   size_t max_nodes = game->max_nodes;
   game->max_nodes = 0;

   game->pondering = true;
   playmove(game, game->ponder_move);

   /* Search the current position */
   void *old_keyboard_handler = game->check_keyboard;
   void *old_clock_handler = game->check_clock;
   game->check_keyboard = interrupt_ponder;
   set_ponder_timer(game);

   if (computer_play(game, MAX_SEARCH_DEPTH))
      takeback(game);

   game->check_keyboard = old_keyboard_handler;
   game->check_clock = old_clock_handler;

   takeback(game);
   game->pondering = false;

   game->ponder_move = ponder_move;
   game->max_nodes = max_nodes;

   /* Report whether we stopped pondering because of keyboard input, or
    * because the main thinking loop terminated (as in the case of a legal
    * draw or a forced mate).
    */
   if (!keyboard_input_waiting())
      game->ponder_move = 0;
   return true;
}

bool computer_analyse(game_t *game)
{
   game->pondering = true;
   size_t max_nodes = game->max_nodes;
   game->max_nodes = 0;

   /* Search the current position */
   void *old_keyboard_handler = game->check_keyboard;
   void *old_clock_handler = game->check_clock;
   set_ponder_timer(game);

   while (game->analysing) {
      if (!pause_analysis)
         computer_play(game, MAX_SEARCH_DEPTH);
      check_keyboard(game);
   }

   game->check_keyboard = old_keyboard_handler;
   game->check_clock = old_clock_handler;

   game->pondering = false;
   game->ponder_move = 0;
   game->max_nodes = max_nodes;
   return true;
}


