/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "board.h"
#include "movelist.h"
#include "movegen.h"
#include "game.h"
#include "evaluate.h"
#include "search.h"
#include "timer.h"
#include "see.h"

#define HISTORY_HEURISTIC

volatile bool abort_search = 0;
size_t positions_evaluated;
size_t moves_searched;
int positions_in_hashtable;
int branches_pruned;

static inline int min(int x, int y)
{
   return (x<y) ? x : y;
}

static inline int max(int x, int y)
{
   return (x>y) ? x : y;
}

/* Return the number of repetitions of the current position in the game
 * tree.
 */
int count_repetition(const game_t *game)
{
   int n;
   int count;
   count = 0;

   for (n=game->moves_played-1; n>=0; n--) {
      if (is_irreversible_move(game->move_list[n]))
         return count;

      count += (game->hash_key[n] == game->board.hash);
   }

   return count;
}

static inline bool position_repeated_once(const game_t *game)
{
   return game->repetition_hash_table[game->board.hash&0xFFFF]>=2 && count_repetition(game)>=1;
}

static inline bool position_repeated(const game_t *game)
{
   return game->repetition_hash_table[game->board.hash&0xFFFF]>=2 && count_repetition(game)>=2;
}

static inline bool material_draw(const board_t *board)
{
   /* Material draw: only bare kings left */
   if (onebit64(board->bbc[WHITE]) && onebit64(board->bbc[BLACK]))
      return true;

   /* Only a single minor left */
   bitboard_t kings = get_royal(board);
   bitboard_t minors = board->bbp[KNIGHT] | board->bbp[BISHOP] | board->bbp[LIEUTENANT];
   bitboard_t occ = get_occupied(board);
   if ( minors && (minors | kings) == occ && onebit64(minors))
      return true;

   /* Only a single minor left */
   return false;
}

static inline bool legal_draw(const game_t *game)
{
   return game->fifty_counter[game->moves_played] > 101 ||
          material_draw(&game->board) ||
          position_repeated(game);
}

/* Function call to check how much time we have left
 * We do the check every time clock_positions nodes have been searched.
 * Don't do this during QS - there aren't that many QS nodes anyway.
 */
static const int clock_positions = 0x00007FFF;
#define check_clock() \
   if (game->max_nodes && (moves_searched == game->max_nodes)) abort_search |= 1;\
   if (game->check_clock && ((positions_evaluated & clock_positions)==0))\
      abort_search |= game->check_clock(game)

#define in_check(n) (game->in_check[game->moves_played + n])

/* Store a "combo move"; this is a depth-independent extension of the
 * killer principle.
 */
static void inline store_combo_move(game_t *game, int depth, move_t prev_move, move_t move)
{
#ifdef USE_COMBINATION_MOVES
   if (in_check(0) || is_promotion_move(move) || is_capture_move(move) || !prev_move) {
      return;
   }
   int c = get_move_player(move) >> 7;
   game->combo[get_move_from(prev_move)][get_move_to(prev_move)][c] = move;
#endif
}

static bool inline is_combo(const game_t *game, int depth, move_t prev_move, move_t move)
{
#ifdef USE_COMBINATION_MOVES
   int c = get_move_player(move) >> 7;
   return game->combo[get_move_from(prev_move)][get_move_to(prev_move)][c] == move;
#else
   return false;
#endif
}

/* Store a "counter move"; this is a depth-independent extension of the
 * killer principle.
 */
static void inline store_counter_move(game_t *game, int depth, move_t prev_move, move_t move)
{
   if (in_check(0) || is_promotion_move(move) || is_capture_move(move) || !prev_move)
      return;

   int c = get_move_player(move);
   game->counter[get_move_from(prev_move)][get_move_to(prev_move)][c] = move;
}

static bool inline is_counter(const game_t *game, int depth, move_t prev_move, move_t move)
{
   int c = get_move_player(move) >> 7;
   return game->counter[get_move_from(prev_move)][get_move_to(prev_move)][c] == move;
}

static bool is_killer(const game_t *game, int depth, move_t prev_move, move_t move) __attribute__((unused));
static bool is_killer(const game_t *game, int depth, move_t prev_move, move_t move)
{
   if (moves_are_equal(game->killer[0][depth], move) ||
       moves_are_equal(game->killer[1][depth], move))
      return true;
   return false;
}

static bool is_good_capture(const game_t *game, int depth, move_t move)
{
   if (moves_are_equal(game->good_capture[0][depth], move) ||
       moves_are_equal(game->good_capture[1][depth], move))
      return true;
   return false;
}

/* Store good captures. To qualify as a good capture, the move
 *  - must be a capture (obviously)
 *  - have a nominally bad SEE score
 *  - improve on the actual score
 */
static inline void store_good_capture(game_t *game, move_t move, int score, int depth)
{
   if (!is_capture_move(move))
      return;

   if (score > 1100 && !is_good_capture(game, depth, move)) {
      return;
   }
   //if (is_good_capture(game, depth, move))
   //   printf("Used good capture %s at depth %d\n", move_string(move, NULL), depth);

   if (moves_are_equal(move, game->good_capture[0][depth])) {
      /* The move was the first capture - do nothing */
   } else {
      /* This was either the last capture (out of 2 or 3), or it's a new
       * move. Either way, Degrade first capture to second capture (etc) and
       * store the new first capture.
       */
      game->good_capture[1][depth]=game->good_capture[0][depth];
      game->good_capture[0][depth]=move;
      //printf("Stored good capture %s at depth %d\n", move_string(move, NULL), depth);
   }
}

/* Store moves that kill a branch in the killer slots, but only if:
 *  - we were not in check at the beginning of this move
 *  - the move is not a promotion (already high in the tree)
 *  - the move was not a capture (already high in the tree)
 */
static inline void store_killer(game_t *game, move_t move, int depth)
{
   if (in_check(0) || is_promotion_move(move) || is_capture_move(move)) {
      return;
   }

   if (moves_are_equal(move, game->killer[0][depth])) {
      /* The move was the first killer - do nothing */
   } else {
      /* This was either the last killer (out of 2 or 3), or it's a new
       * move. Either way, Degrade first killer to second killer (etc) and
       * store the new first killer.
       */
      game->killer[1][depth]=game->killer[0][depth];
      game->killer[0][depth]=move;
   }
}


static inline void store_mate_killer(game_t *game, move_t move, int depth, int best_score)
{
   if (best_score >= (CHECKMATE-1000))
      game->mate_killer[depth] = move;
}

static void store_null_killer(game_t *game, int depth, move_t move)
{
   if (!is_capture_move(move))
      game->null_killer[depth] = move;
}

static void adjust_history_score(game_t *game, move_t move, int score)
{
#ifdef HISTORY_HEURISTIC
   if (is_capture_move(move) || is_promotion_move(move) || in_check(0))
      return;

   sides side = get_move_player(move);
   int piece = get_move_piece(move);
   int to = get_move_to(move);

   game->history[side][piece][to] += score;
   if (abs(game->history[side][piece][to]) > game->max_history)
      game->max_history = abs(game->history[side][piece][to]);

   /* Rescale as needed */
   if (game->max_history > 16384) {
      game->max_history /= 2;
      for (side = WHITE; side <= BLACK; side++)
         for (piece=WPAWN; piece<NUM_PIECES; piece++)
            for (to=0; to<64; to++)
               game->history[side][piece][to] /= 2;
   }
#endif
}

static int get_move_history(const game_t *game, move_t move)
{
#ifdef HISTORY_HEURISTIC
   if (is_capture_move(move) || is_promotion_move(move) || in_check(0))
      return 0;

   sides side = get_move_player(move);
   int piece = get_move_piece(move);
   int to = get_move_to(move);

   return game->history[side][piece][to];
#else
   return 0;
#endif
}

static inline bool score_moves(const game_t *game, movelist_t *movelist, int depth, const move_t hash_move)
{
   const board_t *board = &game->board;
   bool have_hash_move = false;
   int n;

   int *score = movelist->score;

   for (n=0; n<movelist->num_moves; n++) {
      move_t move = movelist->move[n];
      int from_row, to_row;
      sides colour = game->board.side_to_move;
      int piece = get_move_piece(move);
      int from = get_move_from(move);
      int to = get_move_to(from);

      score[n] = 0;

      assert(is_drop_move(move)||(game->board.bbc[get_move_player(move)] & make_bitboard_square(get_move_from(move))));

      /* Give the highest priority to the move from the hash table, if it
       * exists.
       */
      if (moves_are_equal(hash_move, move)) {
         score[n] += CHECKMATE;
         continue;
      }

      /* TODO: advances of free pawn, advances of candidate pawns */

      /* Mate killer: a killer move that produced a mate score */
      if (moves_are_equal(game->mate_killer[depth], move)) {
         score[n] += CHECKMATE-100;
         continue;
      }

      /* Promotion moves */
      if (is_promotion_move(move)) {
         score[n] += 1350+piece_value[get_move_promotion_piece(move)];
         continue;
      }

      /* Regular captures
       * The score is 1100+the SSE score, which sorts winning captures below
       * promotion moves (with a score >= 1150) and losing captures with a
       * score <=1000 below everything else.
       */
      if (is_capture_move(move)) {
         if (is_good_capture(game, depth, move)) {
            score[n] += 1250;
         } else {
            int see_score = static_exchange_evaluation(&game->board, move);
            if (see_score > 0)
               score[n] += 1300+see_score;
            else
               score[n] += see_score;
            //int atk_value[NUM_PIECES] = { 1, 3, 3, 5, 9, 10,
            //                              1, 3, 3, 8, 9, 10 };
            //int attacker = get_move_piece(move);
            //int victim = get_move_captured_piece(move);
            //score[n] += (atk_value[victim] * 10 + (10 - atk_value[victim]))*20;
         }
         continue;
      }

      /* Award killers a higher score, so they are searched first
       * Given the weights here, winning captures are normally searched
       * before killer moves, losing captures are searched after killer
       * moves.
       */
      if (moves_are_equal(game->killer[0][depth], move)) {
         score[n] += 1300;
         continue;
      }
      if (moves_are_equal(game->killer[1][depth], move)) {
         score[n] += 1290;
         continue;
      }

#if 0
      /* Counter moves, a generalisation of the Killer concept.
       * The question is whether these should come before, or after
       * killers...
       * Doesn't seem to make much of a difference, but it sometimes hurts
       * badly if they are sorted before the killer moves, so they should
       * probably go between the killers and the losing captures.
       */
      if (is_counter(game, depth, prev_move, move)) {
         score[n] += 1000;
         continue;
      }

      /* Combo moves, a similar idea to a counter move.
       * The question is whether these should come before, or after
       * killers...
       * Doesn't seem to make much of a difference, but it sometimes hurts
       * badly if they are sorted before the killer moves, so they should
       * probably go between the killers and the losing captures.
       */
      if (is_combo(game, depth, my_prev_move, move)) {
         score[n] += 1000;
         continue;
      }
#endif

      /* Null killer: a move that failed high after a null move */
      if (moves_are_equal(game->null_killer[depth], move)) {
         score[n] += 1200;
         //continue;
      }

      /* Also look at killers from earlier positions. This doesn't make a
       * big difference, but it speeds things up very slightly still and
       * isn't likely to hurt.
       */
      if (depth>2 && moves_are_equal(game->killer[0][depth-2], move)) {
         score[n] += 1280;
         continue;
      }
      if (depth>2 && moves_are_equal(game->killer[1][depth-2], move)) {
         score[n] += 1270;
         continue;
      }

      /* Castling moves: these should come before other king moves */
      if (is_castle_move(move)) {
         score[n] += 500;
         continue;
      }

      /* Pawn pushes of passed pawns */
      if (get_move_piece(move) == WPAWN) {
         bitboard_t block = board->bbp[BHOPLITE] & board->bbc[BLACK];
         int square = get_move_from(move);
         if ((white_passed_pawn_mask[square] & block) == board_empty) {
            score[n] += 1000 + 10 * unpack_rank(square);
            continue;
         }
      }
      if (get_move_piece(move) == BHOPLITE) {
         bitboard_t block = board->bbp[WPAWN] & board->bbc[WHITE];
         int square = get_move_from(move);
         if ((black_passed_pawn_mask[square] & block) == board_empty) {
            score[n] += 1000 + 10 * (7-unpack_rank(square));
            continue;
         }
      }

      if (game->max_history) {
         score[n] = 100 + 500 * get_move_history(game, move) / game->max_history;
         if (score[n] == 100) score[n] = 0;
      }

      /* If we don't know what else to do, order moves based on the
       * centralisation table.
       */
      if (score[n] == 0)
         score[n] += centre_table[to] - centre_table[from];

      /* Idea: pawn advances are typically bad, so penalise pawn pushes
       * (except maybe to the 7th rank) and sort them lower down.
       */
   }
}


int qsearch(game_t *game, int alpha, int beta, int draft, int depth)
{
   assert(draft >= -64);
   assert(draft <= 0);

   game->quiescence_depth_of_variation[depth+1] = -draft;
   truncate_principle_variation(game, depth);

   /* Check whether our search time for this move has expired */
   check_clock();
   if (abort_search) return 0;

   sides me = game->board.side_to_move;
   bool check = in_check(0);

   /* Do static evaluation, and decide whether we accept the score here as-is, assuming any move we do
    * will do no worse.
    * If the result is actually very bad (much below alpha), then return immediately as well. This is a
    * (dangerous!) form of futility pruning, but it greatly reduces the size of the tree. The idea is
    * discussed in Ed Schroeder's writeup for Rebel.
    */
   int static_score = alpha;
   if (!check) {
      static_score = static_evaluation(&game->board, me, alpha, beta);
      assert(abs(static_score) < CHECKMATE);
      positions_evaluated++;
      if (static_score > alpha) {
         alpha = static_score;
         if (alpha >= beta) {
            return alpha;
         }
      }
   }

   movelist_t movelist;
   if (check) {
      generate_moves(&movelist, &game->board, me);
   } else {
      generate_quiescence_moves(&movelist, &game->board, me);
   }
   int legal_moves = movelist.num_moves;
   if (legal_moves == 0)
      return static_score;

   /* Sort moves */
   score_moves(game, &movelist, depth, 0);

   /* Now search all moves */
   int best_score;
   alpha = best_score = static_score;
   for (int n=0; n<movelist.num_moves; n++) {
      move_t move = get_next_move(&movelist);
      int move_score = get_move_score(&movelist);
      int local_beta = beta;

      /* Prune losing captures */
      //if (!check && is_capture_move(move) && static_exchange_evaluation(&game->board, move) < 0)
      if (!check && is_capture_move(move) && move_score < 1000)
         continue;

      /* Principle variation search: use full window only for first move. */
      if (n>0) local_beta = alpha+1;

      /* Play move and evaluate the resulting position recursively -
       * assuming the move was legal.
       */
      playmove(game, move);
      //if (!check && player_in_check(&game->board, me)) { /* Illegal move! */
      if (!check && move_leaves_player_in_check(&game->board, move, me)) { /* Illegal move! */
         legal_moves--;
         takeback(game);
         continue;
      }
      assert(!player_in_check(&game->board, me));
      moves_searched++;

      int score = -qsearch(game, -local_beta, -alpha, draft-1, depth+1);
      takeback(game);

      if (score > best_score) {
         best_score = score;

         if (best_score > alpha) {           /* New best line found */
            alpha = best_score;

            /* Update and store principle variation */
            game->quiescence_depth_of_variation[depth+1] = -draft+1;
            backup_principle_variation(game, depth + 1, move);

            if (best_score >= beta) {       /* Beta cutoff */
               branches_pruned++;
               break;
            }
         }
      }
   }

   /* Abort if we've exceeded our allowed time to think */
   if (abort_search) return 0;

   assert(abs(best_score) < CHECKMATE);
   return best_score;
}


int search(game_t *game, int alpha, int beta, int draft, int depth)
{
   movelist_t movelist;
   bool check = false;
   int legal_moves = 0;
   int repeat_numerator = 1;
   assert(game);
   assert(beta > alpha);

   /* Truncate the principle variation at this depth here, otherwise we may
    * end up with rubbish at the end of the PV.
    */
   truncate_principle_variation(game, depth);

   /* The very first thing we do is to check if this position has occurred
    * three times in the game tree, or whether no irreversible move has
    * been played in 50 moves (100 ply). Either of these conditions means
    * we should flag a draw and return immediately.
    * During the search, we simply score positions as a draw if they have
    * occurred at least once before. From the top level, we don't claim a
    * draw until three repetitions have been made.
    */
   if (legal_draw(game))
      return 0;

   if (position_repeated_once(game)) repeat_numerator = 2;

   /* Check whether our search time for this move has expired */
   check_clock();
   if (abort_search) return 0;

   /* Check whether an earlier branch in the tree has given us a mate
    * score; if so, we need to adjust the window
    */
   alpha = max(alpha, -CHECKMATE + depth);
   beta  = min(beta,   CHECKMATE - depth - 1);
   if (alpha >= beta)
      return alpha;

   /* Look up this position in the transposition table */
   hash_table_entry_t *hash;
   move_t hash_move = 0;
   hash = query_table_entry(game->transposition_table, game->board.hash);
   if (hash) {
      int new_score;
      hash_move = hash->best_move;
      new_score = score_from_hashtable(hash->score, depth);
      positions_in_hashtable++;

      assert(hash->score <= CHECKMATE);
      assert(hash->score >= -CHECKMATE);
      assert(new_score < CHECKMATE);
      assert(new_score > -CHECKMATE);

      if (!mate_score(new_score)) new_score /= repeat_numerator;
      /* If we searched this position before from the same depth or better,
       * just return the previous score. Otherwise we can still use the
       * recorded move to improve our move ordering.
       * Don't return scores from the hash table if we're close to a 50-move draw.
       */
      if (hash->depth > draft && game->fifty_counter[game->moves_played] < 80) {
         if (hash->flags & HASH_TYPE_EXACT) {
            game->hash_hit[depth+1] = true;
            backup_principle_variation(game, depth + 1, hash_move);

            return new_score;
         } else if ((hash->flags & HASH_TYPE_ALPHA) && (new_score<=alpha)) {
            game->hash_hit[depth+1] = true;

            return new_score;
         } else if ((hash->flags & HASH_TYPE_BETA) && (new_score>=beta)) {
            game->hash_hit[depth+1] = true;

            return new_score;
         }
      }

      if (hash->flags & HASH_TYPE_ALPHA)
         hash_move = 0;
   }

   sides me = game->board.side_to_move;

   /* Record whether the side to move is in check or not */
   check = in_check(0);

   /* Check extension: if we're in check, extend the search by one ply */
   if (check)
      draft++;

   if (draft <= 0)
      return qsearch(game, alpha, beta, 0, depth)/repeat_numerator;

   int static_score = 0;
   if (!check)
      static_score = static_evaluation(&game->board, me, alpha, beta);

   /* Store the previous move: this is used to change move ordering on this
    * move. If there is no previous move, then this move will be null.
    */
   move_t prev_move = 0;
   if (game->moves_played>0)
      prev_move = game->move_list[game->moves_played-1];

   move_t my_prev_move = 0;
   if (game->moves_played>1)
      my_prev_move = game->move_list[game->moves_played-2];

   /* Try to get a cut-off from a null-move */
   bool avoid_null = beta > alpha+1 ||
                     in_check(0) ||
                     in_check(-1) ||
                     prev_move == 0 ||
                     draft < 2 ||
                     mate_score(beta);
   if (!avoid_null && static_score>=beta) {
      int r = 2;

      /* play null-move */
      play_null_move(game);
      int new_score = -search(game, -beta, 1-beta, draft-r, depth+1);
      takeback(game);

      if (new_score >= beta && !mate_score(new_score)) {
         hash_move = 0;
         store_table_entry(game->transposition_table, game->board.hash, 
               draft, new_score, HASH_TYPE_ALPHA, hash_move);
         return new_score;
      }
   }

   /* Razoring */
   if (draft < 4 && beta==alpha+1 && !mate_score(beta) && !check) {
      int margin = 200 + draft*50;
      if (static_score < beta - margin) {
         int score = qsearch(game, alpha, beta, 0, depth);
         if (score < beta - margin)
            return score;
      }

      margin = 300*draft;

      if (static_score-margin >= beta)
         return beta;
   }

   move_t best_move = 0;
   int hash_flag = HASH_TYPE_ALPHA;
   int best_score = -2*CHECKMATE;

   /* Generate all moves, search in order. */
   positions_evaluated++;
   generate_moves(&movelist, &game->board, me);
   legal_moves += movelist.num_moves;
   if (legal_moves == 0)
      return check ? (-CHECKMATE + depth) : STALEMATE;

   /* Score the moves */
   score_moves(game, &movelist, depth, hash_move);
   int score = 0;
   bool ok_to_prune = false;
   for (int n = 0; n<movelist.num_moves; n++) {
      move_t move = get_next_move(&movelist);
      int move_score = get_move_score(&movelist);
      int local_beta = beta;

      if (n > 0) {
         local_beta = alpha + 1;

         /* Forward pruning */
         if (ok_to_prune && draft < 4) {
            //if (!check && !mate_score(beta) && static_score < alpha - 75 * draft*(draft-1) &&
            //      (!is_capture_move(move) || static_exchange_evaluation(&game->board, move) < 0))
            //   continue;
            if (!check && !mate_score(beta) && static_score < alpha - 75 * draft*(draft-1) && move_score < 1000)
               continue;
         }
      }

      assert(check == in_check(0));

      /* Play move and evaluate the resulting position recursively -
       * assuming the move was legal.
       */
      playmove(game, move);
      //if (!check && player_in_check(&game->board, me)) { /* Illegal move! */
      if (!check && move_leaves_player_in_check(&game->board, move, me)) { /* Illegal move! */
         legal_moves--;
         takeback(game);
         continue;
      }
      assert(!player_in_check(&game->board, me));
      ok_to_prune = true;
      moves_searched++;

      int r = 0;
      if (!check) {
         /*
         if (draft > 3) {
            if (n > 3 && is_capture_move(move) &&
               static_exchange_evaluation(&game->board, move) < 0) r = 1;
         }
         */
         if (draft > 3 && move_score < 1000) r = 1;
      }

      score = -search(game, -local_beta, -alpha, draft-1 - r, depth+1);

      if (score > best_score && local_beta != beta)
         score = -search(game, -beta, -alpha, draft-1, depth+1);

      takeback(game);

      if (score > best_score) {
         best_score = score;

         if (best_score > alpha) {           /* New best line found */
            hash_flag = HASH_TYPE_EXACT;
            alpha = best_score;
            best_move = move;

            adjust_history_score(game, move, draft * draft);

            /* Store moves that improve alpha in the counter and
             * cmobination move slots. This seems to perform slightly
             * better than following the same logic as killer moves,
             * probably because this means we still get a slightly better
             * move ordering even if the move does not cause a cut-off.
             */
            store_counter_move(game, depth, prev_move, move);
            store_combo_move(game, depth, my_prev_move, move);

            /* Update and store principle variation */
            backup_principle_variation(game, depth + 1, move);

            if (best_score >= beta) {        /* Beta cutoff */
               /* Store good moves in the killer slots, but only if:
                *  - we were not in check at the beginning of this move
                *  - the move is not a promotion (already high in the list)
                *  - the move was not a capture (already high in the list)
                */
               store_killer(game, move, depth);
               store_mate_killer(game, move, depth, best_score);

               /* Store good captures */
               //store_good_capture(game, move, move_score[perm[n]], depth);
               hash_flag = HASH_TYPE_BETA;
               branches_pruned++;
               break;
            }
         } else {
            adjust_history_score(game, move, -draft * draft);
         }
      }
   }

done:
   /* Abort if we've exceeded our allowed time to think */
   if (abort_search) return 0;

   /* If there are no legal moves, the game is over */
   if (legal_moves == 0) {
      hash_flag = HASH_TYPE_EXACT;
      best_move = 0;
      best_score = check ? (-CHECKMATE + depth) : STALEMATE;
   }

   /* The best score should now have been adjusted */
   assert(best_score != -2*CHECKMATE);

   /* Store best move for retrieval from one level up in the search (for IID) */
   game->best_move[depth] = best_move;

   if (!mate_score(best_score)) best_score /= repeat_numerator;

   /* store evaluation in the transposition table */
   if (legal_moves && !abort_search) {
      int score = score_to_hashtable(best_score, depth);
      assert(score >= -CHECKMATE);
      assert(score <=  CHECKMATE);
      store_table_entry(game->transposition_table, game->board.hash, draft,
                        score, hash_flag, best_move);
   }


   return best_score;
}

