/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <ctype.h>
#include <time.h>

#include "leonidas.h"
#include "keypressed.h"

#ifdef __LP64__
#define ARCHSTR "(x86_64)"
#elif defined __i386__
#define ARCHSTR "(i386)"
#elif defined POWERPC
#define ARCHSTR "(powerpc)"
#else
#define ARCHSTR "(unknown)"
#endif 

#ifndef SVNVERSION
#define SVNVERSION "?" //"$Rev: 469 $"
#endif
#define VERSIONSTR "[" SVNVERSION " " __DATE__ "]"

#define TIME_BUFFER  200

static movelist_t movelist;

static game_t *(*new_variant_game)(void) = create_standard_game;

static bool prompt = true;

static const char *standard_variants[] = {
   /*  8x8  */
   "normal",
   "spartan",
   "seirawan",
   NULL
};

char *chomp(char *s)
{
   char *p;
   p = strstr(s, "\n");
   if (p) {
      *p = '\0';
      p--;
      while (isspace(*p)) { *p = '\0'; p--; }
   }
   return s;
}

static bool streq(const char *s1, const char *s2)
{
   return strcmp(s1, s2) == 0;
}

static FILE *f = NULL;
static char *buf;

static bool may_ponder = false;

static void log_engine_output(const char *msg, ...) __attribute__((unused));
static void log_engine_output(const char *msg, ...)
{
   va_list ap;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   if (f) {
      fprintf(f, "  %s", buf);
      fflush(f);
   }
}

static void log_xboard_output(const char *msg, ...)
{
   va_list ap;
   static bool newline = true;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   if (f) {
      if (newline)
         fprintf(f, "> ");
      newline = false;
      fprintf(f, "%s", buf);
      if (strstr(buf, "\n"))
         newline = true;
   }

   printf("%s", buf);
}

static void log_xboard_error(const char *msg, ...)
{
   va_list ap;
   static bool newline = true;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   if (f) {
      if (newline)
         fprintf(f, "> ");
      newline = false;
      fprintf(f, "%s", buf);
      if (strstr(buf, "\n"))
         newline = true;
   }

   printf("telluser %s", buf);
}

int num_games = 0;

/* Play a sequence of moves from the initial position */
bool input_move(game_t *game, char *move)
{
   movelist_t movelist;
   int from, to, n;
   char *s;

   s = move;
   if(s && *s) {
      generate_moves(&movelist, &game->board, game->board.side_to_move);
      while (*s == ' ') s++;
      if (*s) {
         n = -1;
         if (s[1] == '@') {   /* Drop */
            int col = s[2] - 'a';
            int row = s[3] - '1';
            to = pack_row_file(row, col);

            const char *p = strchr(piece_symbol_string, s[0]);
            if (p == NULL)
               p = strchr(piece_symbol_string, toupper(s[0]));

            int drop_piece = p - piece_symbol_string;

            int k;
            for (k=0; k<movelist.num_moves; k++) {
               int p = get_move_drop(movelist.move[k], 0);
               int square = decode_drop_square(p);
               int piece = decode_drop_piece(p);
               
               if (piece == drop_piece && square == to) {
                  n = k;
                  break;
               }
            }
         } else if (s[0] == 'O' && strstr(s, "O-O") == s) {
            /* FIXME: handle gating moves */
            if (streq(s, "O-O-O")) {
               int k;
               for (k=0; k<movelist.num_moves; k++) {
                  if (is_castle_move(movelist.move[k])) {
                     from = get_move_from(movelist.move[k]);
                     to = get_move_to(movelist.move[k]);
                     if (unpack_file(to) < 8/2) {
                        n = k;
                        break;
                     }
                  }
               }
            } else {
               int k;
               for (k=0; k<movelist.num_moves; k++) {
                  if (is_castle_move(movelist.move[k])) {
                     from = get_move_from(movelist.move[k]);
                     to = get_move_to(movelist.move[k]);
                     if (unpack_file(to) >= 8/2) {
                        n = k;
                        break;
                     }
                  }
               }
            }
         } else {
            int col = *s - 'a'; s++;
            int row = *s - '1'; s++;
            from = pack_row_file(row, col);
            col = *s - 'a'; s++;
            row = *s - '1'; s++;
            to = pack_row_file(row, col);

            //printf("Read move from %d (%s) to %d (%s), validating\n", from, square_names[from], to, square_names[to]);
            n = validate_move(&movelist, from, to);
            //printf("Move validated as number %d ('%s')\n", n, move_string(movelist.move[n], NULL));

            /* Check for gating moves, with and without castling */
            if (*s) {
               int gate_sq = from;
               int real_from = from, real_to = to;
               if (n<0) {  /* Probably a gate move: castling with drop on rook square */
                  if (from > to) {
                     real_from = to;
                     real_to = from - 1;
                  } else {
                     real_from = to;
                     real_to = from + 2;
                  }
                  n = validate_move(&movelist, real_from, real_to);
               }
               if (n>=0 && !is_promotion_move(movelist.move[n])) {
                  /* Find the move that gates in the correct piece type */
                  const char *p = strchr(piece_symbol_string, *s);
                  if (p == NULL)
                     p = strchr(piece_symbol_string, toupper(*s));
                  int gated_piece = p - piece_symbol_string;

                  for (n=0 ; n<movelist.num_moves; n++) {
                     if (!is_gate_move(movelist.move[n]))
                        continue;

                     uint16_t gate = get_move_holding(movelist.move[n]);
                     int drop = decode_drop_square(gate);
                     int piece  = decode_holding_piece(gate);
                     int from = get_move_from(movelist.move[n]);
                     int to = get_move_to(movelist.move[n]);
                     if ((piece == gated_piece) && (drop == gate_sq) && (from == real_from) && (to == real_to))
                        break;
                  }

                  if (n == movelist.num_moves)  /* Promotion to unknown piece type */
                     return false;
               }
            }
         }
         if (n<0) {
            int k;
            fprintf(stderr, "# Legal moves: ");
            for (k=0; k<movelist.num_moves; k++)
               fprintf(stderr, "%s ", move_string(movelist.move[k], NULL));
            fprintf(stderr, "\n");
            return false;
         }
         /* FIXME: handle gating moves */
         if (is_promotion_move(movelist.move[n])) {
            /* Find the move that promotes to the correct piece type */
            const char *p = strchr(piece_symbol_string, *s);
            if (p == NULL)
               p = strchr(piece_symbol_string, toupper(*s));

            int promotion_piece = p - piece_symbol_string;
            if (promotion_piece == KING) promotion_piece = BASILEUS;
            for ( ; n<movelist.num_moves; n++)
               if (get_move_promotion_piece(movelist.move[n]) == promotion_piece)
                  break;

            if (n == movelist.num_moves)  /* Promotion to unknown piece type */
               return false;

            s++;
         }
         playmove(game, movelist.move[n]);
         //print_bitboards(&game->board);
      }
   } else {
      return false;
   }

   return true;
}

static char ponder_input[65536];
static bool keyboard_input_on_move(game_t *game)
{
   if (keyboard_input_waiting() && fgets(ponder_input, sizeof ponder_input, stdin)) {
      if (strstr(ponder_input, "random")) {
         sgenrand(time(NULL));
      } else if (strstr(ponder_input, "easy")) {
         may_ponder = false;
      } else if (strstr(ponder_input, "hard")) {
         may_ponder = true;
      } else if (strstr(ponder_input, "post")) {
         set_xboard_output_function(log_xboard_output);
      } else if (strstr(ponder_input, "nopost")) {
         set_xboard_output_function(NULL);
      } else if (strstr(ponder_input, "?") == ponder_input) {
         return true;
      } else if (strstr(ponder_input, "otim")) {
      } else if (strstr(ponder_input, "ping")) {
         log_xboard_output("pong %s\n", ponder_input+5);
      } else if (strstr(ponder_input, "result")) {
      } else if (strstr(ponder_input, "time")) {
      } else if (strstr(ponder_input, "pause")) {
         uint64_t start_pause = get_timer();
         /* Sleep until keyboard input */
         while(!fgets(ponder_input, sizeof ponder_input, stdin) || !strstr(ponder_input, "resume"));
         uint64_t stop_pause = get_timer();
         game->start_time += stop_pause - start_pause;
      } else if (streq(ponder_input, "quit")) {
         if (game)
            end_game(game);
         free(buf);
         exit(0);
      }
   }
   return false;
}

static bool keyboard_input_analyse(game_t *game)
{
   bool restart = false;
   if (keyboard_input_waiting() && fgets(ponder_input, sizeof ponder_input, stdin)) {
      if (strstr(ponder_input, "undo")) {
         takeback(game);
         restart = true;
      } else if (strstr(ponder_input, "new")) {
         start_new_game(game);
         restart = true;
      } else if (strstr(ponder_input, "exit")) {
         game->analysing = false;
         restart = true;
      } else if (strstr(ponder_input, "setboard")) {
         char *p = strstr(ponder_input, " ");
         while (p && *p && p[0]==' ') p++;
         setup_fen_position(game, p);
         restart = true;
      } else if (strstr(ponder_input, ".") == ponder_input) {
      } else if (strstr(ponder_input, "hint")) {
         if (game && game->ponder_move)
            log_xboard_output("Hint: %s\n", move_string(game->ponder_move, NULL));
      } else if (strstr(ponder_input, "bk")) {
      } else if (streq(ponder_input, "quit")) {
         if (game)
            end_game(game);
         free(buf);
         exit(0);
      } else {
         if (ponder_input[0] && !input_move(game, ponder_input))
            log_xboard_output("Illegal move: %s\n", ponder_input);
         else
            restart = true;
      }
      pause_analysis = false;
   }
   return restart;
}

void send_legal_move_targets(game_t *game, int from)
{
   char fen[4096];

   if (!game) return;

   bitboard_t bb = board_empty;
   bitboard_t bc = board_empty;
   int n;
   for (n=0; n<movelist.num_moves; n++)
      if (get_move_from(movelist.move[n]) == from) {
         bb |= square_bitboards[get_move_to(movelist.move[n])];
         if (is_capture_move(movelist.move[n]))
            bc |= square_bitboards[get_move_to(movelist.move[n])];
      }
   int r, f;
   n = 0;
   for (r = 8-1; r>=0; r--) {
      int count = 0;
      for (f = 0; f < 8; f++) {
         int s = pack_row_file(r, f);

         /* Empty? */
         if ((bb & square_bitboards[s]) == 0) {
            count++;
            continue;
         }

         /* Not empty, do we have a count? */
         if (count) n += snprintf(fen+n, 4096 - n, "%d", count);
         count = 0;

         if ((bc & square_bitboards[s]) == 0)
            n += snprintf(fen+n, 4096-n, "Y");
         else
            n += snprintf(fen+n, 4096-n, "R");
      }
      if (count) n += snprintf(fen+n, 4096 - n, "%d", count);
      if (r) n += snprintf(fen+n, 4096 - n, "/");
   }

   log_xboard_output("highlight %s\n", fen);
}

int main(int argc, char **argv)
{
   bool update_options_without_reset = false;
   bool in_play = true;
   sides my_colour = BLACK;
   game_t *game = NULL;
   char buffer[65536];
   char *input;
   int depth = 60;
   size_t hash_size = HASH_TABLE_SIZE;
   int nps = -1;
   int time_per_move = 0;

   buf = malloc(65536);

   initialise_leonidas();

   set_default_output_function(NULL);
   //set_default_output_function(log_engine_output);
   set_xboard_output_function(log_xboard_output);
   set_error_output_function(log_xboard_error);

   /* Turn off buffering for stdout */
   setvbuf(stdout, NULL, _IONBF, 0);
   setvbuf(stdin, NULL, _IONBF, 0);

   /* Write log output (for debugging purposes) */
   //f = fopen("xb.log", "a");

   while (true) {
      movelist.num_moves = 0;
      if (game) generate_moves(&movelist, &game->board, game->board.side_to_move);
      input = buffer;
      buffer[0] = '\0';
      if (prompt) {
         printf("#");
         if (game) {
            printf("[%s] %d", game->name, (int)game->moves_played);
            printf("%s", (game->board.side_to_move == WHITE)?"w":"b");
            if (player_in_check(&game->board, game->board.side_to_move)) printf("+");
         } else {
            printf("[-]");
         }
         printf(">");
      }
      if (!(may_ponder && game && game->ponder_move) || keyboard_input_waiting()) {
         if (!fgets(buffer, sizeof buffer, stdin))
            break;
      }
      chomp(buffer);
      while (input[0] && isspace(input[0])) input++;
      if (f) {
         fprintf(f, "< %s\n", input);
         fflush(f);
      }
      if (streq(input, "xboard")) {
         prompt = false;
         log_xboard_output("\n");
      } else if (strstr(input, "protover")) {
         int n;
         log_xboard_output("feature setboard=1"
                                      " time=1"
                                    " sigint=0"
                                    " colors=0"
                                 " highlight=1"
                                   " analyze=1"
                                      " ping=1"
                                    " memory=1"
                                     " pause=1"
                                       " nps=1"
                                    " myname=\"Leonidas %s\""
                                 " myversion=\"%s %s\"\n", SVNVERSION, VERSIONSTR, ARCHSTR);
         log_xboard_output("feature variants=\"");
         for (n=0; standard_variants[n]; n++) log_xboard_output("%s%s", n?",":"",standard_variants[n]);
         log_xboard_output("\"\n");
         log_xboard_output("feature done=1\n");
      } else if (strstr(input, "accepted")) {
      } else if (strstr(input, "rejected")) {
      } else if (strstr(input, "option")) {
      } else if (strstr(input, "variants")) {
         int n;
         for (n=0; standard_variants[n]; n++) printf("%s\n", standard_variants[n]);
      } else if (strstr(input, "variant")) {
         if (strstr(input, "normal")) {
            new_variant_game = create_standard_game;
         } else if (strstr(input, "fischerandom")) {
            //new_variant_game = create_standard_game;
         } else if (strstr(input, "berolina")) {
            new_variant_game = create_berolina_game;
         } else if (strstr(input, "pocketknight")) {
            //new_variant_game = create_pocketknight_game;
         } else if (strstr(input, "spartan")) {
            new_variant_game = create_spartan_game;
         } else if (strstr(input, "seirawan")) {
            new_variant_game = create_seirawan_game;
         }
         if (game) {
            end_game(game);
         }
         game = new_variant_game();
         set_transposition_table_size(game, hash_size);
         start_new_game(game);
         if (game->xb_setup && game->start_fen)
            log_xboard_output("setup %s %s\n", game->xb_setup, game->start_fen);
      } else if (streq(input, "new")) {
         if (game) {
            end_game(game);
         }
         game = new_variant_game();
         if (game == NULL) {
            log_xboard_output("telluser Could not initialise game!");
            continue;
         }
         set_transposition_table_size(game, hash_size);
         start_new_game(game);
         in_play = true;
         my_colour = BLACK;
         depth = 60;
         nps = -1;
         num_games++;
         if (f) fprintf(f, "# game %d\n", num_games);
      } else if (strstr(input, "memory") == input) {
         unsigned long int memory_size;
         sscanf(input+7, "%lu", &memory_size);

         /* Convert to bytes */
         memory_size <<= 20;

         hash_size = memory_size / sizeof(hash_table_entry_t);
         set_transposition_table_size(game, hash_size);
      } else if (strstr(input, "analyze") || strstr(input, "analyse")) {
         if (game) {
            in_play = false;
            set_xboard_output_function(log_xboard_output);
            game->analysing = true;
            pause_analysis = false;
         }
      } else if (strstr(input, "force")) {
         in_play = false;
      } else if (strstr(input, "undo")) {
         takeback(game);
      } else if (strstr(input, "remove")) {
         takeback(game);
         takeback(game);
      } else if (strstr(input, "setboard")) {
         char *p = strstr(input, " ");
         while (p && *p && p[0]==' ') p++;

         if (!game) {
            game = new_variant_game();
            set_transposition_table_size(game, hash_size);
            start_new_game(game);
         }

         setup_fen_position(game, p);
      } else if (strstr(input, "eval")) {
         if (game) {
            printf("%d\n", static_evaluation(&game->board, -CHECKMATE, CHECKMATE, game->board.side_to_move));
         }
      } else if (strstr(input, "readfen") || strstr(input, "readepd")) {
         FILE *f = fopen(input+8, "r");
         if (f) {
            char s[4096];
            fgets(s, sizeof s, f);
            fclose(f);

            if (!game) {
               game = new_variant_game();
               set_transposition_table_size(game, hash_size);
               start_new_game(game);
            }

            setup_fen_position(game, s);
         }
      } else if (strstr(input, "sd ")) {
         sscanf(input, "sd %d", &depth);
      } else if (strstr(input, "go")) {
         my_colour = game->board.side_to_move;
         in_play = true;

         /* Correctly set the number of moves to go */
         if (game->movestotc) {
            size_t moves_played = game->moves_played / 2;
            if (my_colour == WHITE)
               moves_played += (game->moves_played % 2);
            moves_played %= game->movestotc;
            game->movestogo = game->movestotc - moves_played;
         }
      } else if (strstr(input, "st ")) {
         sscanf(input+3, "%d", &time_per_move);
         set_time_per_move(game, time_per_move*1000);
         if (nps > 0) game->max_nodes = nps * time_per_move;
      } else if (strstr(input, "computer")) {
      } else if (strstr(input, "prompt on") == input) {
         prompt = true;
      } else if (strstr(input, "prompt off") == input) {
         prompt = false;
      } else if (strstr(input, "lift")) {
         int square = 0;
         for (square = 0; square < 64; square++) {
            if (strstr(input+5, square_names[square])) {
                send_legal_move_targets(game, square);
                break;
            }
         }
      } else if (strstr(input, "put")) {
      } else if (strstr(input, "random")) {
         sgenrand(time(NULL));
      } else if (strstr(input, "easy") || strstr(input, "ponder off")) {
         may_ponder = false;
      } else if (strstr(input, "hard") || strstr(input, "ponder on")) {
         may_ponder = true;
      } else if (strstr(input, "post")) {
         set_xboard_output_function(log_xboard_output);
      } else if (strstr(input, "nopost")) {
         set_xboard_output_function(NULL);
      } else if (strstr(input, "?") == input) {
      } else if (strstr(ponder_input, "hint")) {
         if (game && game->ponder_move)
            log_xboard_output("Hint: %s\n", move_string(game->ponder_move, NULL));
      } else if (strstr(input, "otim")) {
      } else if (strstr(input, "ping")) {
         log_xboard_output("pong %s\n", input+5);
      } else if (strstr(input, "result")) {
      } else if (strstr(input, "time")) {
         int milliseconds;
         sscanf(input+5, "%d", &milliseconds);
         milliseconds *= 10;

         /* Reserve a bit of time for when we reach the time control, so we don't get penalised by a time
          * loss. We don't need this for a Fischer clock.
          */
         if (milliseconds > TIME_BUFFER) {
            milliseconds -= TIME_BUFFER;
         } else if (milliseconds > TIME_BUFFER/2) {
               milliseconds -= TIME_BUFFER/2;
         }

         game->time_left[0] = game->time_left[1] = milliseconds;
         set_time_for_game(game);
      } else if (strstr(input, "nps")) {
         /* This is a convoluted mess... */
         sscanf(input+4, "%d", &nps);
         /* If we have a fixed time per move, then we can set a maximum number of nodes.
          * Otherwise... things get complicated.
          */
         game->max_nodes = 0;
         if (time_per_move > 0 && nps > 0) game->max_nodes = nps * time_per_move;
      } else if (strstr(input, "maxnodes")) {
         unsigned int nodes = 0;
         sscanf(input+9, "%u", &nodes);
         game->max_nodes = nodes;
      } else if (strstr(input, "level")) {
         int moves, minutes, seconds, inc, milliseconds;
         /* Set defaults */
         set_time_per_move(game, 5000);
         game->movestotc = 0;
         game->movestogo = 0;
         game->time_inc[0] = game->time_inc[1] = 0;

         if (strstr(input, ":")) {
            sscanf(input+6, "%d %d:%d %d", &moves, &minutes, &seconds, &inc);
         } else {
            sscanf(input+6, "%d %d %d", &moves, &minutes, &inc);
            seconds = 0;
         }
         seconds += minutes*60;
         milliseconds = seconds * 1000;

         /* Reserve a bit of time for when we reach the time control, so we don't get penalised by a time
          * loss. We don't need this for a Fischer clock.
          */
         if (inc == 0 && milliseconds > TIME_BUFFER)
            milliseconds -= TIME_BUFFER;

         game->movestotc = moves;
         game->movestogo = moves;
         game->time_inc[0] = game->time_inc[1] = inc;
         game->time_left[0] = game->time_left[1] = milliseconds;
         set_time_for_game(game);
      } else if (streq(input, "quit")) {
         exit(0);
      } else if (streq(input, "board")) {
         print_bitboards(&game->board);
      } else if (streq(input, "moves")) {
         int k;
         if (game) {
            printf("%d moves\n", movelist.num_moves);
            for (k=0; k<movelist.num_moves; k++)
               printf("%s ", move_string(movelist.move[k], NULL));
         }
         printf("\n");
      } else {
         if (!game || (!input_move(game, input) && strlen(input))) {
            log_xboard_output("Illegal move: %s\n", input);
         }
      }

      if (game && in_play) {
         game->check_keyboard = keyboard_input_on_move;
         if (game->board.side_to_move == my_colour) {
            char p = ' ';
            move_t move;

            /* Check whether NPS time control is sane. */
            if (nps>-1 && !time_per_move) {
               log_xboard_output("telluser Warning: specifying NPS without fixed time per move is not implemented\n");
               game->max_nodes = nps;
            }

            game->pondering = false;

            bool played = computer_play(game, depth);
            if (played) {
               move = game->move_list[game->moves_played-1];
               if (is_promotion_move(move)) {
                  p = tolower(piece_symbol_string[get_move_promotion_piece(move)]);
               }
               if (is_gate_move(move)) {
                  if (is_castle_move(move)) {
                     uint16_t gate = get_move_holding(move);
                     int piece  = decode_holding_piece(gate);
                     int drop = decode_drop_square(gate);
                     int from = get_move_from(move);
                     int to = get_move_to(move);

                     if (drop != from) {
                        /* Drop on the rook square */
                        if (from < to) {  /* King-side */
                           from = to+1;
                           to = from-3;
                        } else {          /* Queen-side */
                           from = to - 2;
                           to = from + 4;
                        }
                     }

                     p = tolower(piece_symbol_string[piece]);
                     log_xboard_output("move %s%s%c\n", square_names[from], square_names[to], p);
                  } else {
                     uint16_t gate = get_move_holding(move);
                     int piece  = decode_holding_piece(gate);
                     int from = get_move_from(move);
                     int to = get_move_to(move);
                     p = tolower(piece_symbol_string[piece]);
                     log_xboard_output("move %s%s%c\n", square_names[from], square_names[to], p);
                  }
               } else {
                  if (is_drop_move(move) || is_castle_move(move))
                     log_xboard_output("move %s\n", move_string(move, NULL));
                  else
                     log_xboard_output("move %s%s%c\n", square_names[get_move_from(move)],
                           square_names[get_move_to(move)], p);
               }
               if (game->movestogo) {
                  game->movestogo--;
                  if (!game->movestogo)
                     game->movestogo = game->movestotc;
               }
            } else {
               /* Game is over, find out why */
               if (win_or_loss) {
                  log_xboard_output("%s\n", game->board.side_to_move ? "1-0 {White mates}" : "0-1 {Black mates}");
               } else {
                  if (game->fifty_counter[game->moves_played] >= 100) {
                     log_xboard_output("1/2-1/2 {Draw by 50 move rule}\n");
                  } else if (count_repetition(game)>=3) {
                     log_xboard_output("1/2-1/2 {Draw by repetition rule}\n");
                  } else {
                     log_xboard_output("1/2-1/2 {Stalemate}\n");
                  }
               }
            }
         } else {    /* Ponder ? */
            if (may_ponder && ponder_ok && game->ponder_move)
               computer_ponder(game);
         }
      }

      if (game && game->analysing) {
         game->check_keyboard = keyboard_input_analyse;
         computer_analyse(game);
      }
   }

   end_game(game);
   return 0;
}

