/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef GAME_H
#define GAME_H

#include <string.h>
#include <stdbool.h>
#include <stdint.h>
#include "assert.h"

#include "pieces.h"
#include "board.h"
#include "move.h"
#include "hashtable.h"
#include "movegen.h"

#define MAX_SEARCH_DEPTH 60       /* maximum depth of search tree */

#define HARD_HORIZON -20

#define MAX_TOTAL_DEPTH (MAX_SEARCH_DEPTH - HARD_HORIZON)

#define HASH_TABLE_SIZE (1024*1024)

typedef struct game_t {
   board_t board;      /* pointer to the current board position */

   char *start_fen;     /* FEN string encoding the starting position of this game */
   char *xb_setup;      /* setup string that has to be sent to XBoard (startup position will be attached) */
   
   size_t moves_played; /* Number of moves played to current position */
   size_t last_move;    /* Number of the last move played in the game; useful when we take back a move */

   size_t max_moves;    /* Maximum number of moves that can be stored */
   int16_t *score;      /* Evaluation score associated with each position */
   move_t *move_list;   /* list of moves played in the game */

   /* State information */
   int8_t *fifty_counter;  /* Counter for the 50-moves rule */ 
   int8_t *ep;          /* En-passant square */ 
   int8_t *ep_capture;  /* Piece that is captured en-passant */ 
   uint64_t *hash_key;  /* hash keys */
   bitboard_t *init;
   bool *in_check;
   bool *did_castle;

   /* Killer moves, storage space requirements must come from the search
    * function.
    */
   move_t killer[2][MAX_TOTAL_DEPTH];
   move_t mate_killer[MAX_TOTAL_DEPTH];

   /* Good captures: these are captures that are nominally bad (based on SEE score)
    * but actually turn out to be ok after trying them. These don't qualify for killer moves,
    * but they should still be sorted slightly higher up in the movelist than other bad captures.
    */
   move_t good_capture[2][MAX_TOTAL_DEPTH];

   /* Botwinnik-Markov extension.
    * This is like a killer move or a refutation move, but for NULL moves.
    * When encountering it on consecutive plies, extend the search (or at least sort the move higher in the
    * list).
    */
   move_t null_killer[MAX_TOTAL_DEPTH];

   /* Counter moves, for use with the counter move heuristic.
    * Indexed by from- and to square and side to move.
    */
   move_t counter[64][64][NUM_SIDES];

   /* Combination moves, similar to counter moves except indexed by our own
    * previous moves.
    * Indexed by from- and to square and side to move.
    */
   move_t combo[64][64][NUM_SIDES];

   int history[NUM_SIDES][MAX_PIECE_TYPES][64];
   int max_history;

   /* Data structure for retrieving the principle variation. At each depth,
    * there are two branches for the tree: the principle variation and the
    * "current" branch. If the current branch turns out to be better than
    * the PV, it becomes the new PV.
    * A binary tree would be enough to store this information (and be easy
    * to manipulate), but for now we use a plain NxN array (of which half
    * the space is wasted, obviously)
    */
   move_t principle_variation[MAX_TOTAL_DEPTH][MAX_TOTAL_DEPTH];
   int length_of_variation[MAX_TOTAL_DEPTH];
   int quiescence_depth_of_variation[MAX_TOTAL_DEPTH];
   bool hash_hit[MAX_TOTAL_DEPTH];
   move_t best_move[MAX_TOTAL_DEPTH];

   /* Transposition table */
   size_t default_hash_size;
   hash_table_t *transposition_table;

   /* Hash table for repetition detection */
   int8_t repetition_hash_table[0xFFFF+1];

   /* various things having to do with time management */
   int time_left[2];       /* Total time left on the clock, in msec */
   int time_inc[2];        /* Time increment per move, in msec */
   int time_per_move;      /* Time per move, in msec */
   uint64_t start_time;    /* Timer value when the search started */
   int movestogo;
   int movestotc;
   int extra_time;
   int running_clock;
   int root_moves_played;
   size_t max_nodes;       /* Maximum number of nodes to search */

   /* Whether the engine is currently pondering or not, if it is, disable
    * time control.
    */
   bool pondering;
   move_t ponder_move;

   bool analysing;

   /* Meta-data */
   char *name;
   char *name_white;
   char *name_black;

   /* Various function pointers, so we can easily customise things and
    * adjust to different UIs.
    */
   void (*output_iteration)(const char *, ...) __attribute__((format(printf, 1, 2)));
   void (*uci_output)(const char *, ...) __attribute__((format(printf, 1, 2)));
   void (*xboard_output)(const char *, ...) __attribute__((format(printf, 1, 2)));
   void (*error_output)(const char *, ...) __attribute__((format(printf, 1, 2)));

   bool (*check_clock)(const struct game_t *game);
   bool (*check_keyboard)(struct game_t *game);
} game_t;

extern void playgame_init(void);
extern void playgame_shutdown(void);

/* Setters for virtual methods, so we can customise the engine output
 * easily
 */
extern void set_default_output_function(void (*func)(const char *, ...));
extern void set_uci_output_function(void (*func)(const char *, ...));
extern void set_xboard_output_function(void (*func)(const char *, ...));
extern void set_error_output_function(void (*func)(const char *, ...));
extern void set_transposition_table_size(game_t *game, uint64_t nelem);

extern game_t *create_game(void);
extern void start_new_game(game_t *game);
extern void end_game(game_t *game);

extern void setup_fen_position(game_t *game, const char *str);
extern char *make_fen_string(const game_t *game, char *buffer);
extern char *short_move_string(const board_t *board, const move_t move, char *buffer);


/**********************************************************************
 *                      static inline functions                       *
 **********************************************************************/
static inline void playmove(game_t *game, move_t move)
{
   assert(game->board.in_check == player_in_check(&game->board, game->board.side_to_move));
   assert(game->board.in_check == game->in_check[game->moves_played]);
   game->hash_key[game->moves_played] = game->board.hash;
   game->init[game->moves_played] = game->board.init;
   game->ep[game->moves_played] = game->board.ep;
   game->ep_capture[game->moves_played] = game->board.ep_capture;
   //game->did_castle[game->moves_played] = game->board.did_castle[game->board.side_to_move];

   makemove(&game->board, move);

   game->move_list[game->moves_played] = move;

   game->repetition_hash_table[game->board.hash&0xFFFF]++;

   game->moves_played++;

   /* Test whether the move is a checking move */
   //bool check = player_in_check(&game->board, game->board.side_to_move);
   bool check = move_checked_player(&game->board, move, game->board.side_to_move);
   game->board.in_check = check;
   game->in_check[game->moves_played] = check;
   assert(check == player_in_check(&game->board, game->board.side_to_move));

   /* Advance the 50-move counter if this move was reversible, reset if it
    * wasn't.
    * FIXME: need to factor in moves by non-reversible pieces (pawns).
    */
   game->board.fifty_counter = game->fifty_counter[game->moves_played] =
      is_irreversible_move(move) ? 0 : (game->fifty_counter[game->moves_played-1] + 1);

   /* Set en-passant square */
   game->board.ep = 0;
   game->board.ep_capture = 0;
   if (move & MOVE_SET_ENPASSANT) {
      game->board.ep = (get_move_from(move) + get_move_to(move))/2;
      game->board.ep_capture = get_move_to(move);
   }
}

static inline void play_null_move(game_t *game)
{
   assert(game->board.in_check == player_in_check(&game->board, game->board.side_to_move));
   assert(!game->board.in_check);
   game->hash_key[game->moves_played] = game->board.hash;
   game->init[game->moves_played] = game->board.init;
   game->ep[game->moves_played] = game->board.ep;
   game->ep_capture[game->moves_played] = game->board.ep_capture;
   //game->did_castle[game->moves_played] = game->board.did_castle[game->board.side_to_move];
   makemove(&game->board, 0);
   game->repetition_hash_table[game->board.hash&0xFFFF]++;

   game->move_list[game->moves_played] = 0;
   game->moves_played++;
   game->board.ep = 0;
   game->board.ep_capture = 0;
   game->in_check[game->moves_played] = game->board.in_check = false;
}

static inline void takeback(game_t *game)
{
   game->moves_played--;
   game->repetition_hash_table[game->board.hash&0xFFFF]--;
   unmakemove(&game->board, game->move_list[game->moves_played]);
   game->board.init = game->init[game->moves_played];

   game->board.ep = game->ep[game->moves_played];
   game->board.ep_capture = game->ep_capture[game->moves_played];
   game->board.fifty_counter = game->fifty_counter[game->moves_played];
   game->board.in_check = game->in_check[game->moves_played];
   //game->board.did_castle[game->board.side_to_move] = game->did_castle[game->moves_played];
   assert(game->board.in_check == player_in_check(&game->board, game->board.side_to_move));
}

#if 0
/* Redo the last move: if a move has been reverted, but no new move has
 * been played yet.
 * Mainly for player interaction.
 */
static inline void redo_last_move(game_t *game)
{
   if (game->moves_played < game->last_move) {
      move_t move = game->move_list[game->moves_played];

      game->hash_key[game->moves_played] = game->board.hash;
      if (game->board.large_board)
         game->init[game->moves_played].lbb = game->board.large_init;
      else
         game->init[game->moves_played].bb = game->board.init;
      game->ep[game->moves_played] = game->board.ep;
      game->ep_capture[game->moves_played] = game->board.ep_capture;
      if (!game->large_board)
         makemove(&game->board, move);
      else
         large_makemove(&game->board, move);

      game->repetition_hash_table[game->board.hash&0xFFFF]++;

      game->moves_played++;

      game->board.fifty_counter = game->fifty_counter[game->moves_played] =
         is_irreversible_move(move) ? 0 : (game->fifty_counter[game->moves_played-1] + 1);

      /* Set en-passant square */
      game->board.ep = 0;
      if (move & MOVE_SET_ENPASSANT) {
         game->board.ep = (get_move_from(move) + get_move_to(move))/2;
         game->board.ep_capture = get_move_to(move);
      }
   }
}


static inline bool player_in_check_for_board(game_t *game, board_t *board, sides side)
{
#if !defined USE_ATTACK_TABLE
   if (!game->large_board) {
      bitboard_t royal = board->royal & board->bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (!royal) return false;

      bitboard_t mask             = get_super_attacks_for_squares(royal);
      bitboard_t attacked_squares = generate_attack_bitboard(board, 0, mask, next_side[side]);
      return (attacked_squares & royal) == royal;
   } else {
      large_bitboard_t royal = board->large_royal & board->large_bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (is_zero128(royal)) return false;

      large_bitboard_t mask             = get_large_super_attacks_for_squares(royal);
      large_bitboard_t attacked_squares = generate_large_attack_bitboard(board, large_board_empty, mask, next_side[side]);
      if (expect(board->rule_flags & RF_KING_TABOO, false)) {
         large_bitboard_t other_king = board->large_royal & board->large_bbc[next_side[side]] & mask;
         if (!is_zero128(other_king)) {
            int square = bitscan128(other_king);
            large_bitboard_t occ = board->large_bbc[WHITE]|board->large_bbc[BLACK];

            attacked_squares |= get_large_file_attacks(large_vmslider, occ, square);
         }
      }
      return is_equal128(attacked_squares & royal, royal);
   }
#else
   if (!game->large_board) {
      bitboard_t royal = board->royal & board->bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (!royal) return false;

      return (board->attacks[next_side[side]] & royal) == royal;
   } else {
      large_bitboard_t royal = board->large_royal & board->large_bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (is_zero128(royal)) return false;

      return is_equal128(board->large_attacks[next_side[side]] & royal, royal);
   }
#endif
   return false;
}


/* Test whether a move left the player in check, assuming he was not in check before. */
static inline bool player_in_check_after_move(game_t *game, move_t move, sides side)
{
   if (!game->large_board) {
      bitboard_t royal = game->board.royal & game->board.bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (!royal) return false;

      /* If the last move did not affect any squares in the superpiece attack set, then it cannot have been a
       * checking move.
       */
      bitboard_t mask      = get_super_attacks_for_squares(royal);
      bitboard_t move_mask = get_move_mask(move);
      if ((mask & move_mask) == 0) return false;

      /* Generate attack bitboard */
      bitboard_t attacked_squares = generate_attack_bitboard(&game->board, 0, mask, next_side[side]);
      return (attacked_squares & royal) == royal;
   } else {
      large_bitboard_t royal = game->board.large_royal & game->board.large_bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (is_zero128(royal)) return false;

      /* If the last move did not affect any squares in the superpiece attack set, then it cannot have been a
       * checking move.
       */
      large_bitboard_t mask      = get_large_super_attacks_for_squares(royal);
      large_bitboard_t move_mask = get_move_large_mask(move);
      if (is_zero128(mask & move_mask)) return false;

      /* Generate attack bitboard */
      large_bitboard_t attacked_squares = generate_large_attack_bitboard(&game->board, large_board_empty, mask, next_side[side]);
      return is_equal128(attacked_squares & royal, royal);
   }
}
#endif



/************************************* 
 * Propagate the principle variation *
 *************************************/
static inline void backup_principle_variation(game_t *game, int depth, move_t move)
{
   int c;

   /* Store move */
   if (move) {
      game->length_of_variation[depth] = depth;
      game->principle_variation[depth][depth] = move;
      //printf("Store at %d %s", depth, move_string(move, NULL));
   }

   /* Copy tree below last stored move */
   if (game->length_of_variation[depth+1] > depth) {
      //printf(" [%d", game->length_of_variation[depth+1]);
      for (c=depth+1; c<game->length_of_variation[depth+1]; c++) {
         //printf(" %s", move_string(game->principle_variation[c][depth+1], NULL));
         game->principle_variation[c][depth] = game->principle_variation[c][depth+1];
      }
      game->length_of_variation[depth] = game->length_of_variation[depth+1];
      game->quiescence_depth_of_variation[depth] = game->quiescence_depth_of_variation[depth+1];
      game->hash_hit[depth] = game->hash_hit[depth+1];
   }

   //printf("\n");
}

static inline void truncate_principle_variation(game_t *game, int depth)
{
   /* Truncate the principle variation at this depth */
   game->length_of_variation[depth+1] = depth;
   game->hash_hit[depth+1] = false;
}


#endif
