/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef MOVE_H
#define MOVE_H

#include <stdbool.h>
#include <stdint.h>
#include "pieces.h"
#include "assert.h"
#include "bitboard.h"

/* Define structure to hold a move.
 * Because of the specific requirements of variants or other games, this has to be a fairly flexible data
 * structure. Each move can consist of (in that order):
 *  N pickups (N = 0, 1, 2, 3)
 *  N drops   (N = 0, 1, 2, 3)
 *  N store/retrieve (N = 0, 1)
 * A "pickup" stores a square and a piece to be picked up (6+4+1 = 11 bits).
 * A "drop" stores a square and a piece to be dropped there (6+4+1 = 11 bits).
 * A "store" stores a piece in the holdings (1+4+1 = 6 bits), a "retrieve" does the opposite
 * The extra bits for each piece are used to encode the colour of the piece.
 * We need 5 bits to store the number of pickups, drops and stores. If we restrict the number of pickups,
 * drops and stores such that the number of bits they take up is no more than 55, that leaves us 4 bits of
 * state-changing information.
 * One of these should be to switch to the next player.
 *
 * The order in which the components should be resolved is:
 *  1. pickups (so all destination squares are empty)
 *  2. drops (which may happen on the square that was just cleared by the move)
 *  3. store/retrieve (could be any order really)
 * To encode a promotion, then, we don't encode the pawn move: we pickup the pawn, and drop the promoted piece
 * on its destination square.
 *
 * Experiment: we could also have dedicated move performing functions for special types of moves, accessed
 * through a vtable. Might be slow though.
 *
 * Move bits:
 *  1-2  Number of pieces picked up
 *  3-4  Number of pieces dropped
 *  5    Number of pieces taken from/stored in holdings
 *  6-16 First pickup/drop/store/retrieve
 * 17-27 Second pickup/drop/store/retrieve
 * 28-38 Third pickup/drop/store/retrieve
 * 39-49 Fourth pickup/drop/store/retrieve
 * 50-60 Fifth pickup/drop/store/retrieve
 *  61   Unused
 *  62   Reset half-move counter (50 move counter)
 *  63   Don't change the side-to-move
 *  64   Set the en-passant target square
 *
 * If needed to encode moves on a board with up to 128 squares, we could use 12 bits for pickups/drops,
 * restrict their number to 4 in total, and use the remaining 7 bits for adjusting holdings, which is enough
 * given that only 6 are really needed.
 */

/* Examples of different move types:
 *  Normal move:        1 pickup, 1 drop
 *  Normal capture:     2 pickup, 1 drop
 *  Normal castle:      2 pickup, 2 drop
 *  Normal promotion:   1 pickup, 1 drop
 *  Capture promotion:  2 pickup, 1 drop
 *  pickup >= drop
 *
 *  Drop:               0 pickup, 1 drop
 *  Gate move:          1 pickup, 2 drop
 *  Gate capture:       2 pickup, 2 drop
 *  Gate castle:        2 pickup, 3 drop
 *  drop >= pickup
 *
 *  capture: #pickups = 2, #drops = 1 + #gates
 *  castle:  #pickups = 2, #drops = 2 + #gates
 *  
 *
 * Disambiguation is needed for "gate capture" and "normal castle".
 * Perhaps the easiest is that in the case of a castle, both pickups are
 * for the same side.
 * Easiest (fastest?) way to deal with gates is to add a "gate" flag to the move...
 *
 * FIXME: adjust capture and castle classification functions to work with
 * gates.
 */
typedef uint64_t move_t;

/* Offsets and test masks */
#define MOVE_SQUARE_MASK      0x3f
#define MOVE_PIECE_MASK       0x0f
#define MOVE_DROP_MASK        0xfff
#define MOVE_PICKUP_MASK      0xfff
#define MOVE_HOLDING_MASK     0x3f

#define MOVE_SET_ENPASSANT    0x8000000000000000ll
#define MOVE_KEEP_TURN        0x4000000000000000ll
#define MOVE_RESET50          0x2000000000000000ll

#define MOVE_HEADER_SIZE      5
#define MOVE_PICKUP_SIZE      11                // bits
#define MOVE_DROP_SIZE        MOVE_PICKUP_SIZE  // bits

#define MOVE_SLOT1            (MOVE_HEADER_SIZE)
#define MOVE_SLOT2            (MOVE_SLOT1 + MOVE_PICKUP_SIZE)
#define MOVE_SLOT3            (MOVE_SLOT2 + MOVE_PICKUP_SIZE)
#define MOVE_SLOT4            (MOVE_SLOT3 + MOVE_PICKUP_SIZE)
#define MOVE_SLOT5            (MOVE_SLOT4 + MOVE_PICKUP_SIZE)
#define MOVE_SLOT(n)          (MOVE_SLOT1 + (n)*MOVE_PICKUP_SIZE)

const char *move_string(move_t move, char *buffer);
const char *move_stack_string(move_t move, char *buffer);

/**********************************************************************
 *                      static inline functions                       *
 **********************************************************************/
static inline int get_move_pickups(move_t move)
{
   return move & 0x03;
}

static inline int get_move_drops(move_t move)
{
   return (move >> 2) & 0x03;
}

static inline int get_move_holdings(move_t move)
{
   return (move >> 4) & 0x01;
}
#define get_move_gates get_move_holdings

/* Get the first, second or third pickup */
static inline int get_move_pickup(move_t move, int n)
{
   assert (n<=get_move_pickups(move));

   return (move >> (MOVE_SLOT1 + n * MOVE_PICKUP_SIZE)) & MOVE_PICKUP_MASK;
}

/* Get the first, second or third drop */
static inline int get_move_drop(move_t move, int n)
{
   assert (n<=get_move_drops(move));

   return (move >> (MOVE_SLOT1 + MOVE_PICKUP_SIZE*(n+get_move_pickups(move)))) & MOVE_DROP_MASK;
}

static inline int get_move_holding(move_t move)
{
   return (move >> (MOVE_SLOT(get_move_drops(move)+get_move_pickups(move)-1))) & MOVE_DROP_MASK;
}

/* Decode drop and pickup piece/squares */
static inline int decode_drop_square(int drop)
{
   return (drop >> 5) & MOVE_SQUARE_MASK;
}

static inline int decode_drop_piece(int drop)
{
   return drop & MOVE_PIECE_MASK;
}

static inline int decode_drop_side(int drop)
{
   return (drop>>4) & 0x1;
}

/* The code for pickups is the same as for drops */
#define decode_pickup_square decode_drop_square
#define decode_pickup_piece decode_drop_piece
#define decode_pickup_side decode_drop_side

/* The code for holdings is almost the same as for drops */
#define decode_holding_piece decode_drop_piece
#define decode_holding_side decode_drop_side


/* Encode a normal (non-capture) move */
static inline move_t encode_normal_move(uint8_t piece, uint8_t from, uint8_t to)
{
   return 0x1 | (0x1 << 2) | (piece | from<<5) << MOVE_SLOT1 | (piece | to<<5) << MOVE_SLOT2;
}

/* Encode a normal capture */
static inline move_t encode_normal_capture(uint8_t piece, uint8_t from, uint8_t to, uint8_t ptaken)
{
   return 0x2 | (0x1 << 2) |
            (piece  | from<<5) << MOVE_SLOT1 |
            (ptaken | to  <<5) << MOVE_SLOT2 | (uint64_t)
            (piece  | to  <<5) << MOVE_SLOT3;
}

/* Encode an en-passant capture, where the pickup square is different from the destination square */
static inline move_t encode_en_passant_capture(uint8_t piece, uint8_t from, uint8_t to, uint8_t ptaken, uint8_t ep)
{
   return 0x2 | (0x1 << 2) |
            (piece  | from<<5) << MOVE_SLOT1 |
            (ptaken | ep  <<5) << MOVE_SLOT2 | (uint64_t)
            (piece  | to  <<5) << MOVE_SLOT3;
}

/* Encode a normal (non-capture) promotion */
static inline move_t encode_normal_promotion(uint8_t piece, uint8_t from, uint8_t to, uint8_t tpiece)
{
   return 0x1 | (0x1 << 2) |
            (piece  | from<<5) << MOVE_SLOT1 |
            (tpiece | to  <<5) << MOVE_SLOT2;
}

/* Encode a capture promotion */
static inline move_t encode_capture_promotion(uint8_t piece, uint8_t from, uint8_t to, uint8_t ptaken, uint8_t tpiece)
{
   return 0x2 | (0x1 << 2) |
            (piece  | from<<5) << MOVE_SLOT1 |
            (ptaken | to  <<5) << MOVE_SLOT2 | (uint64_t)
            (tpiece | to  <<5) << MOVE_SLOT3;
}

/* Encode a castling move */
static inline move_t encode_castle_move(uint8_t piece, uint8_t from, uint8_t to, uint8_t p2, uint8_t f2, uint8_t t2) 
{
   return 0x2 | (0x2 << 2) |
            (piece | from<<5) << MOVE_SLOT1 |
            (p2    | f2  <<5) << MOVE_SLOT2 | (uint64_t)
            (piece | to  <<5) << MOVE_SLOT3 | (uint64_t)
            (p2    | t2  <<5) << MOVE_SLOT4 |
            MOVE_RESET50;
}

/* Encode a drop */
static inline move_t encode_drop_move(uint8_t piece, uint8_t to)
{
   return 0x0 | (0x1 << 2) | (piece | to<<5) << MOVE_SLOT1;
}

/* Add gate information to a move: one extra drop, plus removing from
 * holdings.
 */
static inline move_t add_move_gate(move_t move, uint8_t piece, uint8_t to)
{
   int drops = get_move_drops(move);
   int pickups = get_move_pickups(move);
   int n = drops + pickups;

   move &= ~(0x03 << 2);
   move |= (drops + 1) << 2 | (0x1 << 4) | (uint64_t) (piece | to << 5) << MOVE_SLOT(n);
   return move;
}

/* Add store/retrieval information to a move */
static inline move_t add_move_store(move_t move, uint8_t piece, int8_t count)
{
   int shift;
   assert(count < 2);
   assert(count > -2);
   assert((get_move_pickups(move) + get_move_drops(move)) < 5);

   count += 2;
   shift = MOVE_SLOT1 + MOVE_PICKUP_SIZE * (get_move_pickups(move) + get_move_drops(move));

   return move | (0x1 << 4) | (uint64_t)(piece | count<<5) << shift;
}
#define add_move_retrieve(move, piece, count) add_move_store((move), (piece), -(count))

/* Query move types:
 * A normal move is one pickup, one drop (by the same piece)
 * A capture is two pickups (from two different sides), one drop
 * A promotion is a drop by a different piece type than a capture
 * A drop has no pickups
 * A "shot" is a pickup without a drop
 * Castling is two pickups and two drops
 */

static inline bool is_normal_move(move_t move)
{
   return (get_move_pickups(move) == 1) && (get_move_drops(move) == 1) &&
            (decode_pickup_piece(get_move_pickup(move, 0)) == decode_drop_piece(get_move_drop(move, 0)));
}

static inline bool is_capture_move(move_t move)
{
   /* FIXME: if the move contains a gating, a capture will have drops ==
    * pickups (like a normal castling move).
    * A better disambiguation is (pickup colour 1) != (pickup colour 2),
    * which is more work.
    */
   return (get_move_pickups(move) == 2) && (get_move_drops(move) == 1 + get_move_gates(move));
   //return ((get_move_pickups(move) == 2 && get_move_drops(move)==1)) ||
   //       (decode_pickup_piece(get_move_pickup(move, 0)) != decode_pickup_piece(get_move_pickup(move, 1)));
}

static inline bool is_enpassant_move(move_t move)
{
   uint16_t p1 = get_move_pickup(move, 1);
   uint16_t p2 = get_move_drop(move, 0);

   return is_capture_move(move) && (decode_pickup_piece(p1)!=decode_drop_square(p2));
}

static inline bool is_promotion_move(move_t move)
{
   if (get_move_drops(move) > 1 || get_move_pickups(move) < 1)
      return false;
   int drop_piece = decode_drop_piece(get_move_drop(move, 0));
   if (get_move_pickups(move) == 1)
      return drop_piece != decode_pickup_piece(get_move_pickup(move, 0));
   else
      return drop_piece != decode_pickup_piece(get_move_pickup(move, 0)) &&
             drop_piece != decode_pickup_piece(get_move_pickup(move, 1));
}

static inline bool is_irreversible_move(move_t move)
{
   return (move & MOVE_RESET50) || is_capture_move(move) || is_promotion_move(move);
}

static inline bool is_drop_move(move_t move)
{
   return (get_move_pickups(move) == 0) && (get_move_drops(move) == 1);
}

static inline bool is_castle_move(move_t move)
{
   return (get_move_pickups(move) == 2) && (get_move_drops(move) == 2 + get_move_gates(move));
}

#define is_gate_move get_move_gates

static inline bool moves_are_equal(move_t m1, move_t m2)
{
   return m1 == m2;
}

/* Convenience macros: the piece type of the first pickup and the first pickup and drop squares */
static inline int get_move_piece(move_t move)
{
   uint16_t p = get_move_pickup(move, 0);

   return decode_pickup_piece(p);
}

static inline int get_move_captured_piece(move_t move)
{
   uint16_t p = get_move_pickup(move, 1);

   return decode_pickup_piece(p);
}

static inline int get_move_promotion_piece(move_t move)
{
   uint16_t p = get_move_drop(move, 0);

   return decode_drop_piece(p);
}

static inline int get_move_from(move_t move)
{
   uint16_t p = get_move_pickup(move, 0);

   return decode_pickup_square(p);
}

static inline int get_move_to(move_t move)
{
   uint16_t p = get_move_drop(move, 0);

   return decode_drop_square(p);
}

static inline int get_move_capture_square(move_t move)
{
   uint16_t p = get_move_pickup(move, 1);

   return decode_pickup_square(p);
}

static inline int get_move_player(move_t move)
{
   uint16_t p = get_move_pickup(move, 0);

   return decode_pickup_side(p);
}

#endif


