/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef MOVEGEN_H
#define MOVEGEN_H

#include "bitboard.h"
#include "board.h"
#include "move.h"
#include "movelist.h"

#define NORMAL_PAWN     0
#define BEROLINA_PAWN   1
#define SPARTAN_PAWN    2

extern int white_pawn_type, black_pawn_type;

extern bitboard_t king_attack[64];
extern bitboard_t king_attack2[64];
extern bitboard_t ferz_attack[64];
extern bitboard_t knight_attack[64];
extern bitboard_t pawn_attack[64];
extern bitboard_t pawn_attacker[64];
extern bitboard_t captain_attack[64];
extern bitboard_t lieutenant_attack[64];
extern bitboard_t hoplite_attack[64];
extern bitboard_t hoplite_attacker[64];
extern bitboard_t super[64];
extern bitboard_t rook_empty[64];
extern bitboard_t bishop_empty[64];
extern bitboard_t mslider[8][64];
extern bitboard_t connecting_ray[64][64];
extern bitboard_t joined_ray[64][64];

extern bitboard_t eastwest[64];
extern bitboard_t white_passed_pawn_mask[64];
extern bitboard_t black_passed_pawn_mask[64];

extern bitboard_t mrook[64][64][64];
extern bitboard_t mbish[64][64][64];

void initialise_movement_tables(int white_pawn, int black_pawn);

extern bool verify_move(movelist_t *movelist, const board_t *board, sides side_to_move, move_t move);
extern void generate_evasion_moves(movelist_t *movelist, const board_t *board, sides side_to_move);
extern void generate_moves(movelist_t *movelist, const board_t *board, sides side_to_move);
extern void generate_quiescence_moves(movelist_t *movelist, const board_t *board, sides side_to_move);
extern int validate_move(const movelist_t *movelist, int from, int to);

/**********************************************************************
 *                         Helper functions                           *
 **********************************************************************/
typedef bitboard_t (*movegen_func_t)(int square, bitboard_t occ);

static inline bitboard_t get_file_attacks(bitboard_t occ, int square)
{
   int file = unpack_file(square);
   int rank = unpack_rank(square);
   uint8_t index = ( (board_afile & (occ >> file)) * 0x8040201008040201) >> 57;
   bitboard_t bb;

   bb = mslider[7-rank][index & 63] & board_rank1;
   return ((bb * board_a1h8_diagonal[7]) & board_hfile) >> (7-file);
}

static inline bitboard_t get_rank_attacks(bitboard_t occ, int square)
{
   int file = unpack_file(square);
   int rank = unpack_rank(square);
   uint8_t index = occ >> (8*rank + 1);

   return mslider[file][index & 63] & board_rank[rank];
}

static inline bitboard_t get_diagonal_attacks(bitboard_t occ, int square)
{
   int file = unpack_file(square);
   int diag_nr = get_a1h8_diagonal(square);
   bitboard_t mask = board_a1h8_diagonal[diag_nr];
   uint8_t index = ((occ & mask) * board_afile) >> 57;

   return mslider[file][index & 63] & mask;
}

static inline bitboard_t get_antidiagonal_attacks(bitboard_t occ, int square)
{
   int file = unpack_file(square);
   int diag_nr = get_a8h1_diagonal(square);
   bitboard_t mask = board_a8h1_diagonal[diag_nr];
   uint8_t index = ((occ & mask) * board_afile) >> 57;

   return mslider[file][index & 63] & mask;
}



#undef FULL_ROOK_BISHOP_TABLE
/**********************************************************************
 *    Attack bitboards for specific locations for a given occupancy   *
 **********************************************************************/
static inline bitboard_t get_rook_attacks(int where, bitboard_t occ)
{
#ifndef FULL_ROOK_BISHOP_TABLE
   return get_file_attacks(occ, where) | get_rank_attacks(occ, where);
#else
   int file = unpack_file(where);
   int rank = unpack_rank(where);
   uint8_t index1 = (occ >> (8*rank + 1)) & 63;
   uint8_t index2 = (( (board_afile & (occ >> file)) * 0x8040201008040201) >> 57) & 63;

#if 0
   bitboard_t bb = get_file_attacks(occ, where) | get_rank_attacks(occ, where);
   if (bb != mrook[where][index1][index2]) {
      printf("Rook\n");
      printf_bitboard(bb);
      printf_bitboard(mrook[where][index1][index2]);
      exit(0);
   }
#endif

   return mrook[where][index1][index2];
#endif
}

static inline bitboard_t get_bishop_attacks(int where, bitboard_t occ)
{
#ifndef FULL_ROOK_BISHOP_TABLE
   return get_diagonal_attacks(occ, where) |
          get_antidiagonal_attacks(occ, where);
#else
   int file = unpack_file(where);
   int diag_nr = get_a1h8_diagonal(where);
   int adiag_nr = get_a8h1_diagonal(where);
   uint8_t index1 = (((occ & board_a1h8_diagonal[diag_nr])  * board_afile) >> 57) & 63;
   uint8_t index2 = (((occ & board_a8h1_diagonal[adiag_nr]) * board_afile) >> 57) & 63;

#if 0
   bitboard_t bb = get_diagonal_attacks(occ, where) | get_antidiagonal_attacks(occ, where);
   if (bb != mbish[where][index1][index2]) {
      printf("Bishop\n");
      printf_bitboard(bb);
      printf_bitboard(mbish[where][index1][index2]);
      exit(0);
   }
#endif

   return mbish[where][index1][index2];
#endif
}

static inline bitboard_t get_queen_attacks(int where, bitboard_t occ)
{
   return get_rook_attacks(where, occ) | get_bishop_attacks(where, occ);
}

static inline bitboard_t get_king_attacks(int where, bitboard_t occ)
{
   return king_attack[where];
}

static inline bitboard_t get_knight_attacks(int where, bitboard_t occ)
{
   return knight_attack[where];
}

static inline bitboard_t get_marshal_attacks(int where, bitboard_t occ)
{
   return get_rook_attacks(where, occ) | get_knight_attacks(where, occ);
}

static inline bitboard_t get_wpawn_attacks(int where, bitboard_t occ)
{
   return pawn_attack[where];
}



static inline bitboard_t get_general_attacks(int where, bitboard_t occ)
{
   return get_rook_attacks(where, occ) | get_king_attacks(where, occ);
}

static inline bitboard_t get_warlord_attacks(int where, bitboard_t occ)
{
   return get_bishop_attacks(where, occ) | get_knight_attacks(where, occ);
}

static inline bitboard_t get_captain_attacks(int where, bitboard_t occ)
{
   return captain_attack[where];
}

static inline bitboard_t get_lieutenant_attacks(int where, bitboard_t occ)
{
   return lieutenant_attack[where];
}

static inline bitboard_t get_bhoplite_attacks(int where, bitboard_t occ)
{
   return hoplite_attack[where];
}

static inline bitboard_t get_dummy_attacks(int where, bitboard_t occ)
{
   return board_empty;
}

static inline bitboard_t get_super_attacks_for_squares(bitboard_t squares)
{
   bitboard_t attacks = 0;

   while (squares) {
      int square = bitscan64(squares);
      squares ^= make_bitboard_square(square);
      attacks |= super[square];
   }

   return attacks;
}

static inline bitboard_t get_all_attackers(const board_t *board, int square)
{
   static const movegen_func_t atk[NUM_PIECES] = {
      get_wpawn_attacks, get_knight_attacks, get_bishop_attacks,
      get_rook_attacks, get_queen_attacks, get_king_attacks,
      get_bhoplite_attacks, get_lieutenant_attacks, get_captain_attacks,
      get_general_attacks, get_warlord_attacks, get_king_attacks,
      get_marshal_attacks
   };
   bitboard_t own, enemy, possible_attackers;
   bitboard_t occupied;
   bitboard_t attacked;
   bitboard_t attacker;
   int n;

   occupied = get_occupied(board);
   possible_attackers = occupied & super[square];
   occupied |= make_bitboard_square(square);

   attacked = board_empty;
   attacker = board_empty;

   /* Pieces */
   while (possible_attackers) {
      int first_attacker = bitscan64(possible_attackers);
      n = get_piece(board, first_attacker);
      assert(possible_attackers & board->bbp[n]);

      bitboard_t bb = possible_attackers & board->bbp[n];
      possible_attackers &= ~board->bbp[n];
      while (bb) {
         int from = bitscan64(bb);
         bb ^= make_bitboard_square(from);
         attacked = atk[n](from, occupied);
         if (attacked & make_bitboard_square(square))
            attacker |= make_bitboard_square(from);
      }
   }

   return attacker;
}

static inline bitboard_t get_pinned_pieces(const board_t *board, bitboard_t kings, sides side)
{
   bitboard_t pinned = board_empty;
   bitboard_t occ = get_occupied(board);
   bitboard_t bb;

   bb = kings;
   while (bb) {
      int ksq = bitscan64(bb);
      bb ^= make_bitboard_square(ksq);

      /* Potential pinners are on rays centred on the king square */
      bitboard_t pinners = ((rook_empty[ksq]   & get_rook_movers(board)  ) |
                            (bishop_empty[ksq] & get_bishop_movers(board))) & board->bbc[next_side[side]];

      while (pinners) {
         int psq = bitscan64(pinners);
         pinners ^= make_bitboard_square(psq);
         bitboard_t m = ~(make_bitboard_square(psq) | make_bitboard_square(ksq));

         /* Get pieces on the ray connecting the king and the piece */
         m &= connecting_ray[ksq][psq] & occ;

         /* If there is exactly one, it is pinned (or waiting to deliver a
          * discovered check).
          */
         if (onebit64(m))
            pinned |= m;
      }
   }

   return pinned;
}

static inline bool masked_square_is_attacked_by_side(const board_t *board, int square, bitboard_t mask, sides side)
{
   static const movegen_func_t atk[NUM_PIECES] = {
      get_wpawn_attacks, get_knight_attacks, get_bishop_attacks,
      get_rook_attacks, get_queen_attacks, get_king_attacks,
      get_bhoplite_attacks, get_lieutenant_attacks, get_captain_attacks,
      get_general_attacks, get_warlord_attacks, get_king_attacks,
      get_marshal_attacks
   };
   bitboard_t possible_attackers;
   bitboard_t occupied;
   int n;

   occupied = (get_occupied(board) | make_bitboard_square(square)) & mask;
   //possible_attackers = board->bbc[side] & super[square];
   possible_attackers = ((rook_empty[square] & get_rook_movers(board)) |
                         (bishop_empty[square] & get_bishop_movers(board)) |
                         (knight_attack[square] & get_knight_movers(board)) |
                         king_attack2[square]
                         ) & board->bbc[side];

   /* Pieces */
   while (possible_attackers) {
      int first_attacker = bitscan64(possible_attackers);
      n = get_piece(board, first_attacker);
      assert(possible_attackers & board->bbp[n]);

      bitboard_t bb = possible_attackers & board->bbp[n];
      possible_attackers &= ~board->bbp[n];
      while (bb) {
         int from = bitscan64(bb);
         bb ^= make_bitboard_square(from);
         bitboard_t attacked = atk[n](from, occupied);
         if (attacked & make_bitboard_square(square))
            return true;
      }
   }

   return false;
}


static inline bool square_is_attacked_by_side(const board_t *board, int square, sides side)
{
   return masked_square_is_attacked_by_side(board, square, board_all, side);
#if 0
   static const movegen_func_t atk[NUM_PIECES] = {
      get_wpawn_attacks, get_knight_attacks, get_bishop_attacks,
      get_rook_attacks, get_queen_attacks, get_king_attacks,
      get_bhoplite_attacks, get_lieutenant_attacks, get_captain_attacks,
      get_general_attacks, get_warlord_attacks, get_king_attacks
   };
   bitboard_t possible_attackers;
   bitboard_t occupied;
   int n;

   occupied = board->bbc[WHITE] | board->bbc[BLACK] | make_bitboard_square(square);
   possible_attackers = board->bbc[side] & super[square];

   /* Pieces */
   while (possible_attackers) {
      int first_attacker = bitscan64(possible_attackers);
      n = get_piece(board, first_attacker);
      assert(possible_attackers & board->bbp[n]);

      bitboard_t bb = possible_attackers & board->bbp[n];
      possible_attackers &= ~board->bbp[n];
      while (bb) {
         int from = bitscan64(bb);
         bb ^= make_bitboard_square(from);
         bitboard_t attacked = atk[n](from, occupied);
         if (attacked & make_bitboard_square(square))
            return true;
      }
   }

   return false;
#endif
}


static inline bitboard_t get_all_attackers_for_side(const board_t *board, int square, sides side)
{
   static const movegen_func_t atk[NUM_PIECES] = {
      get_wpawn_attacks, get_knight_attacks, get_bishop_attacks,
      get_rook_attacks, get_queen_attacks, get_king_attacks,
      get_bhoplite_attacks, get_lieutenant_attacks, get_captain_attacks,
      get_general_attacks, get_warlord_attacks, get_king_attacks,
      get_marshal_attacks
   };
   bitboard_t own, enemy, possible_attackers;
   bitboard_t occupied;
   bitboard_t attacked;
   bitboard_t attacker;
   int n;

   occupied = get_occupied(board) | make_bitboard_square(square);
   possible_attackers = board->bbc[side] & super[square];

   attacked = board_empty;
   attacker = board_empty;

   /* Pieces */
   while (possible_attackers) {
      int first_attacker = bitscan64(possible_attackers);
      n = get_piece(board, first_attacker);
      assert(possible_attackers & board->bbp[n]);

      bitboard_t bb = possible_attackers & board->bbp[n];
      possible_attackers &= ~board->bbp[n];
      while (bb) {
         int from = bitscan64(bb);
         bb ^= make_bitboard_square(from);
         attacked = atk[n](from, occupied);
         if (attacked & make_bitboard_square(square))
            attacker |= make_bitboard_square(from);
      }
   }

   return attacker;
}

static inline bitboard_t generate_attack_bitboard_mask(const board_t *board, bitboard_t test_squares, bitboard_t occ_mask, bitboard_t source_mask, sides side_to_move)
{
   static const movegen_func_t atk[NUM_PIECES] = {
      get_wpawn_attacks, get_knight_attacks, get_bishop_attacks,
      get_rook_attacks, get_queen_attacks, get_king_attacks,
      get_bhoplite_attacks, get_lieutenant_attacks, get_captain_attacks,
      get_general_attacks, get_warlord_attacks, get_king_attacks,
      get_marshal_attacks
   };
   bitboard_t own, enemy, own_movers;
   bitboard_t occupied;
   bitboard_t attacked;
   int n;

   own = board->bbc[side_to_move];
   enemy = board->bbc[next_side[side_to_move]];

   occupied = own | enemy | test_squares;
   occupied &= occ_mask;

   own_movers = own & source_mask;

   attacked = board_empty;

   bitboard_t possible_attackers = own_movers;

   /* Pieces */
   while (possible_attackers) {
      int first_attacker = bitscan64(possible_attackers);
      n = get_piece(board, first_attacker);
      assert(possible_attackers & board->bbp[n]);
      possible_attackers &= ~board->bbp[n];

      bitboard_t bb = own_movers & board->bbp[n];
      while (bb) {
         int from = bitscan64(bb);
         bb ^= make_bitboard_square(from);
         attacked |= atk[n](from, occupied);
      }
   }

   return attacked;
}

static inline bitboard_t generate_attack_bitboard(const board_t *board, bitboard_t test_squares, bitboard_t source_mask, sides side_to_move)
{
   return generate_attack_bitboard_mask(board, test_squares, board_all, source_mask, side_to_move);
}

static inline bool player_in_check(const board_t *board, sides side)
{
   bitboard_t royal = (board->bbp[KING]|board->bbp[BASILEUS]) & board->bbc[side];
   bitboard_t own = board->bbc[side];
   bitboard_t enemy = board->bbc[next_side[side]];
   bitboard_t occ = own | enemy;
   assert(royal);

#if 1
   bool check = false;
   int square = bitscan64(royal);
   royal ^= make_bitboard_square(square);

   /* Test whether the (first) king is under attack */
   if (super[square] & enemy) {
      check = check
         ||  (knight_attack[square] & enemy & get_knight_movers(board))
         ||  (king_attack[square] & enemy & (board->bbp[KING]|board->bbp[BASILEUS]))
         ||  (lieutenant_attack[square] & enemy & board->bbp[LIEUTENANT])
         ||  (ferz_attack[square] & enemy & board->bbp[GENERAL])
         ||  (captain_attack[square] & enemy & board->bbp[CAPTAIN])
         ||  (pawn_attacker[square] & enemy & board->bbp[WPAWN])
         ||  (hoplite_attacker[square] & enemy & board->bbp[BHOPLITE]);
      if (!check) {
      bitboard_t rooks = get_rook_movers(board);
      bitboard_t bishops = get_bishop_movers(board);
      if ((rooks | bishops) & super[square] & enemy) {
         check = check ||  (get_rook_attacks(square, occ) & enemy & rooks)
                       ||  (get_bishop_attacks(square, occ) & enemy & bishops);
      }
      }

   }

   if (check && royal) {
      square = bitscan64(royal);
      check = false;
      if (super[square] & enemy) {
         bitboard_t attacked_squares = generate_attack_bitboard(board, 0, super[square], next_side[side]);
         check = (attacked_squares & royal) == royal;
      }
   }

   return check;
#else

   /* TODO: optimise this... */
   bitboard_t mask             = get_super_attacks_for_squares(royal);
   bitboard_t attacked_squares = generate_attack_bitboard(board, 0, mask, next_side[side]);
   return (attacked_squares & royal) == royal;
#endif
}

/* We can check whether the move was a checking move without needing to do
 * the (slightly expensive) attack lookup: if neither the target nor the
 * destination square are in the 'superpiece' attack sphere
 */
static inline bool move_checked_player(const board_t *board, move_t move, sides side)
{
   bitboard_t king_bb = get_royal(board) & board->bbc[side];
   bitboard_t bb;
   int from = get_move_from(move);
   int to = get_move_to(move);

   /* If the move didn't originate on a square connecting to the king, then
    * it can't possibly have been a checking move (exceptions: castling, en-passant).
    * Break out early in this case.
    */
   if (__builtin_expect(is_enpassant_move(move) || is_castle_move(move), false))
      return player_in_check(board, side);

   bb = super[from] | super[to];
   if ((bb & king_bb) == board_empty)
      return false;

   /* Test for discovered checks */
   bool possible_discovered = false;
   bitboard_t enemy = board->bbc[next_side[side]];
   bitboard_t occ = get_occupied(board);
   if (super[from] & king_bb && super[from] & get_sliders(board) & enemy) {
      bb = make_bitboard_square(from);
      possible_discovered = true;
      if (onebit64(king_bb)) {
         int square = bitscan64(king_bb);
         if (get_rook_attacks(square, occ) & get_rook_movers(board) & enemy)
            return true;
         if (get_bishop_attacks(square, occ) & get_bishop_movers(board) & enemy)
            return true;
         possible_discovered = false;
      }
   }

   /* Test for direct checks */
   bool possible_direct = false;
   if (super[to] & king_bb) {
      bb = make_bitboard_square(to);
      if (bb & get_knight_movers(board) && knight_attack[to] & king_bb)
         possible_direct = true;

      else if (bb & board->bbp[LIEUTENANT] && lieutenant_attack[to] & king_bb)
         possible_direct = true;

      else if (bb & board->bbp[CAPTAIN] && captain_attack[to] & king_bb)
         possible_direct = true;

      /* Spartan kings or generals.
       * We include normal kings in the test despite the fact they can
       * never give a direct check (they'd be in check themselves) because
       * if we don't do that here, we will trigger an assert when testing
       * for consistency in the in-check flag after such moves.
       */
      else if (bb & (get_royal(board)|board->bbp[GENERAL]) && king_attack[to] & king_bb)
         possible_direct = true;

      /* Rooks */
      else if (bb & get_rook_movers(board) && get_rook_attacks(to, occ) & king_bb)
         possible_direct = true;

      /* Bishops */
      else if (bb & get_bishop_movers(board) && get_bishop_attacks(to, occ) & king_bb)
         possible_direct = true;

      /* White pawns */
      else if (bb & board->bbp[WPAWN] && pawn_attack[to] & king_bb)
         possible_direct = true;

      /* Black pawns */
      else if (bb & board->bbp[BHOPLITE] && hoplite_attack[to] & king_bb)
         possible_direct = true;
   }

   if (!possible_discovered && !possible_direct)
      return false;

   if (onebit64(king_bb)) {
      if (possible_direct || square_is_attacked_by_side(board, bitscan64(king_bb), next_side[side]))
         return true;
      return false;
   }
   int s1 = bitscan64(king_bb);
   int s2 = bitscan64(king_bb ^ make_bitboard_square(s1));

   if (square_is_attacked_by_side(board, s1, next_side[side]) &&
       square_is_attacked_by_side(board, s2, next_side[side]))
      return true;

   return false;
}

/* Test whether a move leaves the player in check.
 * We can assume that the player was not in check prior to the move (since
 * we generate only legal evasions) so we only need to test whether the
 * piece was pinned, or is a king move into check.
 */
static inline bool move_leaves_player_in_check(const board_t *board, move_t move, sides side)
{
   if (get_move_piece(move) == KING || get_move_piece(move) == BASILEUS)
      return player_in_check(board, side);

   bitboard_t king = get_royal(board) & board->bbc[side];
   bitboard_t bb = rook_empty[get_move_from(move)];
   if (bb & king) {
      bitboard_t rooks = get_rook_movers(board) & board->bbc[next_side[side]];
      if (bb & rooks)
         return player_in_check(board, side);
   }

   bb = bishop_empty[get_move_from(move)];
   if (bb & king) {
      bitboard_t bishops = get_bishop_movers(board) & board->bbc[next_side[side]];
      if (bb & bishops)
         return player_in_check(board, side);
   }

   assert(!player_in_check(board, side));
   return false;
}

#endif

