/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include <math.h>
#include "evaluate.h"
#include "board.h"
#include "movegen.h"
#include "pieces.h"

/* Mapping of the piece-square tables for either side */
static int psq_map[NUM_SIDES][64] = {
 { A1, B1, C1, D1, E1, F1, G1, H1,
   A2, B2, C2, D2, E2, F2, G2, H2,
   A3, B3, C3, D3, E3, F3, G3, H3,
   A4, B4, C4, D4, E4, F4, G4, H4,
   A5, B5, C5, D5, E5, F5, G5, H5,
   A6, B6, C6, D6, E6, F6, G6, H6,
   A7, B7, C7, D7, E7, F7, G7, H7,
   A8, B8, C8, D8, E8, F8, G8, H8 },
 { A8, B8, C8, D8, E8, F8, G8, H8,
   A7, B7, C7, D7, E7, F7, G7, H7,
   A6, B6, C6, D6, E6, F6, G6, H6,
   A5, B5, C5, D5, E5, F5, G5, H5,
   A4, B4, C4, D4, E4, F4, G4, H4,
   A3, B3, C3, D3, E3, F3, G3, H3,
   A2, B2, C2, D2, E2, F2, G2, H2,
   A1, B1, C1, D1, E1, F1, G1, H1 },
};

/* Base centralisation score for each square */
const int8_t centre_table[64] = {
   0, 2,  4,  6,  6,  4, 2, 0,
   2, 4,  6,  8,  8,  6, 4, 2,
   4, 6,  8, 10, 10,  8, 6, 4,
   6, 8, 10, 12, 12, 10, 8, 6,
   6, 8, 10, 12, 12, 10, 8, 6,
   4, 6,  8, 10, 10,  8, 6, 4,
   2, 4,  6,  8,  8,  6, 4, 2,
   0, 2,  4,  6,  6,  4, 2, 0
};

static int8_t piece_table[NUM_PIECES][64];

/* Base advancement score for each square */
static int advance_table[64];

/* Extra advancement score for proximity to promotion zone, for pawns */
static int promo_table[64];

/* End game bonus for driving the defending king to the edge */
static int lone_king_table[64];

/* Bonus for passed pawns */
static int passed_table[64];

/* Evaluation table for KNBK ending, assuming a dark-squared bishop.
 * For a light-squared bishop, flip the board.
 * This table gives the score from the perspective of the attacker, so the
 * corner of the bishop colour gets a higher score.
 */
static int8_t knbk_table[64];

const int piece_value[NUM_PIECES] = {
   100, 300, 350, 500, 950, 475,
   100, 300, 300, 725, 825, 475,
   900
   //100, 400, 400, 600, 1200, 500,
   //100, 400, 400, 900, 1050, 500
};

const int mobility_weight[NUM_PIECES] = {
   8, 12, 12,12, 2, 4,
   8,  8,  8, 4, 4, 4,
   4
};

const int mobility_scale = 8;

int min_moves[NUM_PIECES];
int max_moves[NUM_PIECES];
int avg_moves[NUM_PIECES];
int king_safety_weight[NUM_PIECES];

/* The value of a passed pawn should be
 *    (value of least valuable piece opponent can sacrifice) - pawn - epsilon,
 * or
 *    (value of most valuable promotion piece) - pawn - epsilon,
 * whichever is smaller.
 */
const int passer_value = 150;

/* Progressive king safety evaluation table. */
static int16_t king_safety_penalty[65] = { 0 -2, -5, -8, -12, -16, -20, -25, -31, -37, -44, -51, -59, -67, -76, -85, -95, -104, -114, -124, -134, -143, -152, -161, -170, -177, -185, -191, -198, -203, -208, -213, -217, -220, -224, -226, -229, -231, -233, -235, -236, -237, -238, -239, -240, -241, -242, -242, -243, -243, -243, -244, -244, -244, -244, -244, -245, -245, -245, -245, -245, -245, -245, -245, -245 };

typedef struct attack_t {
   bitboard_t attacks[NUM_SIDES][NUM_PIECES];
} attack_t;

static int min(int x, int y)
{
   return (x<y)?x:y;
}

static int max(int x, int y)
{
   return (x>y)?x:y;
}

/* Manhatten distance between two squares (for evaluating lone king endings) */
static int king_distance(int square1, int square2)
{
   int row1, file1;
   int row2, file2;

   row1 = unpack_rank(square1);
   file1 = unpack_file(square1);
   row2 = unpack_rank(square2);
   file2 = unpack_file(square2);

   return max(abs(row1 - row2), abs(file1-file2));
}

/* Get mobility parameters and king safety weight for the different piece
 * types.
 */
static void get_mobility_statistics(void)
{
   board_t board;
   int board_size = 64;

   /* Loop over all piece types */
   int n, ps;
   for (n=0; n<NUM_PIECES; n++) {
      sides side = WHITE;

      clear_board(&board);

      /* Weight of this piece for king attacks */
      king_safety_weight[n] = 1;

      /* Loop over all squares */
      int sum_m = 0;
      int max_m = 0;
      int min_m = board_size;
      for (ps = 0; ps < board_size; ps++) {
         int moves = 0;

         clear_board(&board);

         /* Place the piece on the board */
         bitboard_t pbb = make_bitboard_square(ps);
         board.bbc[side] |= pbb;
         board.bbp[n] |= pbb;
         board.piece[ps] = n;

         pbb = generate_attack_bitboard(&board, 0, board.bbp[n], side);

         moves = popcount64(pbb);

         sum_m += moves;
         if (moves > max_m) max_m = moves;
         if (moves < min_m) min_m = moves;
      }

      max_moves[n] = max_m;
      min_moves[n] = min_m;
      avg_moves[n] = sum_m / board_size;

      if (n == WPAWN || n == BHOPLITE) continue;

      /* Determine evaluation weight for king safety evaluation. This is a measure of a piece's
       * short-range attack potential, as measured on a 5x5 board and a 3x3 board.
       * This gives a total initial attack strength for the normal chess array of 96 (assigning 1 for the
       * pawns), 104 for the Spartans in Spartan chess and 138 for the normal array in Capablanca chess.
       * If we scale the king safety evaluation with the attack strength, then this results in a slightly
       * higher weight for king safety for white in Spartan chess and a much higher weight for king safety in
       * Capablanca chess.
       * This seems reasonable.
       * A reasonable first guess for when to reduce the weight of the king safety to 0 is at 2/3 of the
       * initial strength (64 for normal chess).
       */
      ps = E4;

      /* If this piece is a leaper, it may have a constraint and therefore we should search the place on the
       * board where it attacks the most squares.
       */

      clear_board(&board);

      /* Create a 5x5 mask */
      bitboard_t mask5 = C4_MASK | D4_MASK | E4_MASK | F4_MASK | G4_MASK;
      mask5 |= mask5 >> 8;
      mask5 |= mask5 >> 8;
      mask5 |= mask5 << 8;
      mask5 |= mask5 << 8;

      /* Create a 3x3 mask5 */
      bitboard_t mask3 = D4_MASK | E4_MASK | F4_MASK;
      mask3 |= mask3 >> 8;
      mask3 |= mask3 << 8;

      /* Place the piece on the board */
      bitboard_t pbb = make_bitboard_square(ps);
      board.bbc[WHITE] |= pbb;
      board.bbp[n] |= pbb;
      board.piece[ps] = n;

      pbb = generate_attack_bitboard(&board, 0, board.bbp[n], WHITE);

      /* Get evaluation weight.
       * Penalise short-range pieces for which the 3x3 and 5x5 attack sets coincide.
       * Penalise colour-bound pieces.
       */
      bool colour_bound = (n == BISHOP) || (n == LIEUTENANT);
      int count5 = popcount64(pbb & mask5);
      int count3 = popcount64(pbb & mask3);
      if (count3 == count5) count3 = 0;
      if (colour_bound) count3 = 0;
      if (n == CAPTAIN || n == LIEUTENANT) count3 += 4;
      king_safety_weight[n] = count5 + count3;
   }
}

static void compute_piece_square_tables(void)
{
   int r, f;

   for (r = 0; r<8; r++) {
      for (f = 0; f<8; f++) {
         int s = pack_row_file(r, f);

         /* Advance table, a progressive piece advancement */
         advance_table[s] = r;

         //centre_table[s] = ((8-1)&~0x1) - (abs(2*r - 8 + 1)/2 + abs(2*f - 8 + 1)/2);

         promo_table[s] = 0;
         //if (r == large_board_ranks-1) promo_table[s] = 99;
         //if (r == large_board_ranks-2) promo_table[s] = 90;
         //if (r == large_board_ranks-3) promo_table[s] = 50;
         //if (r == large_board_ranks-4) promo_table[s] = 10;

         //knbk_table[s] = abs(large_anti_diagonal_nr[s]-large_anti_diagonal_nr[8-1])*5;

         lone_king_table[s] = 6*(abs(2*r - 8 + 1) + abs(2*f - 8 + 1));

         //passed_table[s] = sqrt(PAWN_PASSED_BONUS*PAWN_PASSED_BONUS * r*r*r);

         psq_map[WHITE][s] = s;
         psq_map[BLACK][s] = pack_row_file(8-1 - r, f);
         ///printf("[%2d %2d]  ", lone_king_table[s], 48 -6*centre_table[s]);
      }
      //printf("\n");
   }

   memset(piece_table, 0, sizeof piece_table);
   for (int s=0; s<64; s++) {
      piece_table[WPAWN][s] = centre_table[s];
      piece_table[KNIGHT][s] = centre_table[s];
      piece_table[BISHOP][s] = centre_table[s];
      piece_table[ROOK][s] = centre_table[s] + 5*(unpack_rank(s) == 6);
      piece_table[QUEEN][s] = centre_table[s];
      piece_table[LIEUTENANT][s] = centre_table[s];
      piece_table[CAPTAIN][s] = centre_table[s];
      piece_table[WARLORD][s] = centre_table[s];
      piece_table[GENERAL][s] = centre_table[s];
      piece_table[BHOPLITE][s] = advance_table[s] + 5 * unpack_rank(s);

      if (black_pawn_type == NORMAL_PAWN)
         piece_table[BHOPLITE][s] = centre_table[s];
   }
   for (r = 0; r<8; r++) {
      for (f = 0; f<8; f+=7) {
         int s = pack_row_file(r, f);
         piece_table[WPAWN][s] -= 30;
         if (black_pawn_type == NORMAL_PAWN)
            piece_table[BHOPLITE][s] -= 30;
      }
   }
}

/* Initialise parameter-driven evaluation tables */
void initialise_evaluation(void)
{
   double x;
   int n;

#define FIXED_KING_SAFETY     // Seems needed on Windows???
#ifndef FIXED_KING_SAFETY
   /* King safety */
   double scale = -KING_SAFETY_CEIL;
   x = scale / (1.0 + exp(KING_SAFETY_SLOPE*KING_SAFETY_INFLECTION));
   king_safety_penalty[0] = 0;
   for (n=1; n<65; n++) {
      king_safety_penalty[n] = scale / (1.0 + exp(-KING_SAFETY_SLOPE*(n - KING_SAFETY_INFLECTION))) - x;
   }
#endif

   get_mobility_statistics();

   compute_piece_square_tables();
}


/* Dedicated evaluation function for endings with a lone king */
static bool lone_king_evaluation(const board_t *board, sides defender, int16_t *ev)
{
   sides attacker = next_side[defender];
   bitboard_t royal;
   int defender_king_square;
   int safe_zone = 0;
   uint16_t eval = 0;

   assert(board);
   royal = get_royal(board);
   bitboard_t lone_king = board->bbc[defender] & royal;

   /* Confirm that this is a lone king ending (as opposed to multiple kings or no king) */
   assert(lone_king && onebit64(lone_king));
   defender_king_square = bitscan64(lone_king);

   /* Where the defender should go depends on the material the attacker has. If the attacker has one colour
    * bound piece, the defender wants to go to the opposite corner. Otherwise, close to the centre is good
    * enough.
    */
   bitboard_t safe = board_all;
   bitboard_t bb = board->bbc[attacker];
   while (bb) {
      int n = get_piece(board, bitscan64(bb));
      bb &= ~board->bbp[n];

      if (n == BISHOP) {
         if (board->bbp[n] & board_light)
            safe &= board_light;
         if (board->bbp[n] & board_dark)
            safe &= board_dark;
      }
   }
   if (safe) safe = ~safe;

   if (safe == board_light) {
      safe_zone = 1;
   } else if (safe == board_dark) {
      safe_zone = 2;
   }

   /* Bring the attacking king closer to the defending king */
   bb = royal ^ lone_king;
   while (bb) {
      int attacker_king_square = bitscan64(bb);
      bb ^= make_bitboard_square(attacker_king_square);
      eval -= 8*king_distance(attacker_king_square, defender_king_square);
   }

   /* Evaluate position of the defending king, from the point of view of the attacker */
   if (safe_zone == 1) {
      eval += 10 * knbk_table[defender_king_square];
   } else if (safe_zone == 2) {
      eval += 10 * knbk_table[flip_horizontal(defender_king_square)];
   }
   eval += lone_king_table[defender_king_square];

   *ev = (attacker == WHITE) ? eval : -eval;
   return true;
}

/* Evaluate pawns */
static int16_t evaluate_white_pawns(const board_t *board)
{
   bitboard_t pawns = board->bbp[WPAWN] & board->bbc[WHITE];
   bitboard_t block = board->bbp[BHOPLITE] & board->bbc[BLACK];
   int16_t ev = 0;

   bitboard_t bb = pawns;
   while(bb) {
      int square = bitscan64(bb);
      bb ^= make_bitboard_square(square);

      /* Test of this pawn is a passer */
      if ((white_passed_pawn_mask[square] & block) == board_empty) {
         int rank = unpack_rank(square);
         int val = 0;
         if (rank > 3)
            val = (rank - 3) * passer_value / 3;

         if (pawn_attacker[square] & pawns) val += passer_value / 6;

         if (white_passed_pawn_mask[square] & board->bbc[BLACK]) val /= 4;
         if (white_passed_pawn_mask[square] & board->bbc[WHITE]) val /= 2;

         ev += val;
      }

      /* Bonus if a pawn protects another pawn, or is next to it. */
      ev += ((pawn_attack[square] | eastwest[square]) & pawns) != board_empty;
   }

   return ev;
}


static int16_t evaluate_black_pawns(const board_t *board)
{
   bitboard_t pawns = board->bbp[BHOPLITE] & board->bbc[BLACK];
   bitboard_t block = board->bbp[WPAWN] & board->bbc[WHITE];
   int16_t ev = 0;

   bitboard_t bb = pawns;
   while(bb) {
      int square = bitscan64(bb);
      bb ^= make_bitboard_square(square);

      /* Test of this pawn is a passer
       * TODO: hoplites can choose multiple paths; they clearly become more
       * valuable as more paths open up, but they should already receive
       * their greatest jump by opening up one of them.
       * TODO: avoid creating too many open files, which generates passed
       * pawns for white. Treat this as a penalty for black rather than a
       * bonus for white. Perhaps give a penalty for black if a white pawn
       * can only be intercepted by 1 hoplite? This should keep the
       * position more closed.
       */
      if ((black_passed_pawn_mask[square] & block) == board_empty) {
         int rank = 7 - unpack_rank(square);
         int val = 0;
         if (rank > 3)
            val = (rank - 3) * passer_value / 3;

         if (hoplite_attacker[square] & pawns) val += passer_value / 6;

         if (black_passed_pawn_mask[square] & board->bbc[WHITE]) val /= 4;
         if (black_passed_pawn_mask[square] & board->bbc[BLACK]) val /= 2;

         ev += val;
      }

      /* Bonus if a hoplite protects another hoplite, or is next to it */
      ev += ((hoplite_attack[square] | eastwest[square]) & pawns) != board_empty;
   }

   return ev;
}

/* Evaluate the king safety score for a king on a specified location.
 * This may not be the location of the king, however, since we may test for
 * castling.
 */
static int16_t evaluate_king_safety(const board_t *board, int king_square, sides side, attack_t *a)
{
   /* Construct 'A', 'B' and 'C' type squares */
   bitboard_t k, sq;
   bitboard_t kingab, kingb, kingabc;
   k = make_bitboard_square(king_square);
   sq = k | shift_bitboard_file_left(k) | shift_bitboard_file_right(k);
   if (side == WHITE) {
      kingab   = (sq >> 8 | sq) ^ k;
      kingb    = sq << 8;
      kingabc  = sq << 16;
      kingab  |= kingb;
      kingabc |= kingab;
   } else {
      kingab   = (sq << 8 | sq) ^ k;
      kingb    = sq >> 8;
      kingabc  = sq >> 16;
      kingab  |= kingb;
      kingabc |= kingab;
   }

   int index = 0;
   int pattern = 0;
   int n;

   /* Loop over all pieces, find out what the attack pattern on the enemy
    * king looks like.
    */
   bitboard_t atk = 0;
   bitboard_t def = 0;
   for (n=0; n<NUM_PIECES; n++) {
      if (a->attacks[next_side[side]][n] & kingabc) {
         atk |= a->attacks[next_side[side]][n];
         pattern += king_safety_weight[n];
      }

      def |= a->attacks[side][n];
   }

   index = min(KING_SAFETY_PIECE_WEIGHT_MAX, pattern / KING_SAFETY_PIECE_WEIGHT_NORMALISATION);

   /* Number of attackers on "A" and "B" squares; includes king */
   index += popcount64(atk & kingab);

   /* Penalise more if there are no defenders for "A" and "B" type squares
    * (other than the king itself)
    */
   index += popcount64(atk & ~def & kingab);

   /* Penalise missing pieces on "B" type squares */
   index += popcount64(kingb) - popcount64(kingb & board->bbc[side]);

   /* Penalise missing pawns on "B" type squares */
   index += popcount64(kingb) - popcount64(kingb & get_pawns(board) & board->bbc[side]);

   /* Adjust the counter for the position of the king */
   index += advance_table[psq_map[side][king_square]];
   index = min(40, index);

   //int as2 = army_attack_strength[BLACK]*army_attack_strength[BLACK];
   //int kss = KING_SAFTEY_STRENGTH_SCALE*KING_SAFTEY_STRENGTH_SCALE;
   //ks -= king_safety_penalty[king_counter[WHITE]] * as2/kss;
   return king_safety_penalty[index];
}

int16_t static_evaluation(const board_t *board, sides side_to_move, int alpha, int beta)
{
   attack_t a;
   int piece_count[NUM_SIDES][NUM_PIECES];
   int army_attack_strength[NUM_SIDES] = { 0, 0 };
   int piece_types[NUM_SIDES] = { 0, 0 };
   int num_pieces[NUM_SIDES] = { 0, 0 };
   int num_pawns[NUM_SIDES] = { 0, 0 };
   int mate_potential[NUM_SIDES] = { 0, 0 };
   bool can_win[NUM_SIDES] = { true, true };
   uint32_t pt;
   int16_t pev = 0;
   int16_t mat = 0;
   int16_t psq = 0;
   int16_t mob = 0;
   int16_t ks = 0;
   int16_t ev = 0;

   bitboard_t bb, mb;

   memset(&a, 0, sizeof a);
   memset(&piece_count, 0, sizeof piece_count);

   //masked_mobility[BLACK][WPAWN] |= generate_attack_bitboard(board, 0, board->bbp[WPAWN]&board->bbc[WHITE], WHITE);

   if (board->holdings[WHITE]) {
      int count = popcount64(board->init & board_rank1);
      int weight = 8;
      if (count > 4) count = 8;
      if (count == 2) count = 1;
      uint16_t tpm = board->holdings[WHITE];
      while (tpm) {
         int t = bitscan16(tpm);
         tpm ^= 1<<t;
         mat += count * piece_value[t] / weight;
      }
   }

   if (board->holdings[BLACK]) {
      int count = popcount64(board->init & board_rank8);
      int weight = 8;
      if (count > 4) count = 8;
      if (count == 2) count = 1;
      uint16_t tpm = board->holdings[BLACK];
      while (tpm) {
         int t = bitscan16(tpm);
         tpm ^= 1<<t;
         mat -= count * piece_value[t] / weight;
      }
   }

   pt = 0;
   bb = board->bbc[WHITE];
   while (bb) {
      int square = bitscan64(bb);
      int piece = get_piece(board, square);
      bb ^= make_bitboard_square(square);

      piece_count[WHITE][piece]++;
      
      mb = generate_attack_bitboard(board, 0, make_bitboard_square(square), WHITE);
      a.attacks[WHITE][piece] |= mb;
      mob += mobility_weight[piece] * popcount64(mb);
      mat += piece_value[piece];
      psq += piece_table[piece][psq_map[WHITE][square]];
      army_attack_strength[WHITE] += king_safety_weight[piece];
      num_pieces[WHITE]++;
      if (!(pt & (1<<piece))) {
         piece_types[WHITE]++;
         pt |= 1<<piece;
      }
   }

   pt = 0;
   bb = board->bbc[BLACK];
   while (bb) {
      int square = bitscan64(bb);
      int piece = get_piece(board, square);
      bb ^= make_bitboard_square(square);
      
      piece_count[BLACK][piece]++;
      
      mb = generate_attack_bitboard(board, 0, make_bitboard_square(square), BLACK);
      a.attacks[BLACK][piece] |= mb;
      mob -= mobility_weight[piece] * popcount64(mb);
      mat -= piece_value[piece];
      psq -= piece_table[piece][psq_map[BLACK][square]];
      army_attack_strength[BLACK] += king_safety_weight[piece];
      num_pieces[BLACK]++;
      if (!(pt & (1<<piece))) {
         piece_types[BLACK]++;
         pt |= 1<<piece;
      }
   }
   pev -= evaluate_black_pawns(board);

   mob /= mobility_scale;

   /* Rooks on open files */
   bb = get_pawns(board);
   bb |= bb << 8;
   bb |= bb << 16;
   bb |= bb << 32;
   bb |= bb >> 32;
   bb = ~bb;
   psq += 5*popcount64(bb & get_rook_movers(board) & board->bbc[WHITE]);
   psq -= 5*popcount64(bb & get_rook_movers(board) & board->bbc[BLACK]);

   /* Bishop pair */
   mat += 25 * (piece_count[WHITE][BISHOP] > 1);
   mat -= 25 * (piece_count[BLACK][BISHOP] > 1);

   /* Number of pawns */
   num_pawns[WHITE] = piece_count[WHITE][WPAWN];
   num_pawns[BLACK] = piece_count[BLACK][BHOPLITE];

   /* Material value scaling for Rook and Knight (following Kaufman).
    * Adjust the material score: rooks become stronger as pawns disappear,
    * knights become a bit weaker.
    * NB: for rooks, this scaling may be redundant with bonuses for open
    * files!
    * Do these hold for Spartan?
    */
   mat += piece_count[WHITE][ROOK] * (5-num_pawns[WHITE]) * ROOK_VALUE_SCALE;
   mat -= piece_count[BLACK][ROOK] * (5-num_pawns[BLACK]) * ROOK_VALUE_SCALE;

   /* only apply the knight bonus for the first knight. Kauffman says
    * knight pairs should be penalised, which in itself seems odd to me. I
    * think it makes more sense to apply the bonus only to the first
    * knight.
    */
   mat -= ( piece_count[WHITE][KNIGHT] != 0) * (5-num_pawns[WHITE]) * KNIGHT_VALUE_SCALE;
   mat += ( piece_count[BLACK][KNIGHT] != 0) * (5-num_pawns[BLACK]) * KNIGHT_VALUE_SCALE;

   /* The Spartan Captain becomes relatively stronger when heavy material
    * (rooks and queens) are exchanged and weaker as hoplites disappear.
    */
   int major = piece_count[WHITE][ROOK] + 2*piece_count[WHITE][QUEEN];
   mat -= piece_count[BLACK][CAPTAIN] * ((4 - major) * 10 - (8-piece_count[BLACK][BHOPLITE])*3);
   mat += piece_count[BLACK][LIEUTENANT] * (8-piece_count[BLACK][BHOPLITE])*2;

   /* The Spartan hoplite becomes stronger when (normal) pawns are
    * exchanged. The reason is that hoplites are much more mobile than
    * pawns, so they become stronger as the board is cleared.
    */
   if (black_pawn_type == SPARTAN_PAWN)
      mat -= piece_count[BLACK][BHOPLITE] * (8-piece_count[WHITE][WPAWN]) * 5;

   /* Assess mate potential */
   mate_potential[WHITE] = popcount64(get_mate_potential(board) & board->bbc[WHITE]);
   mate_potential[BLACK] = popcount64(get_mate_potential(board) & board->bbc[BLACK]);

   /* Trade down bonus: if ahead in material, trade pieces. Otherwise, trade pawns. */
   int pieces = num_pieces[WHITE] + num_pieces[BLACK] - num_pawns[WHITE] - num_pawns[BLACK];
   if (pieces < 6) {
      if (mat > 0) {
         mat -= pieces * 2;
         mat += (num_pawns[WHITE]  + num_pawns[BLACK])  * 4;
      } else {
         mat += pieces * 2;
         mat -= (num_pawns[WHITE]  + num_pawns[BLACK])  * 4;
      }
   }

   /* Without pawns it's harder to win, so penalise for having fewer pawns */
   if (num_pawns[WHITE] < 4) mat -= (4 - num_pawns[WHITE] + (num_pawns[WHITE] == 0)) * PAWN_STEP_ENDGAME;
   if (num_pawns[BLACK] < 4) mat += (4 - num_pawns[BLACK] + (num_pawns[BLACK] == 0)) * PAWN_STEP_ENDGAME;

   /* Special case: multiple pieces facing two Spartan kings, drawish. */
   if (num_pieces[BLACK] == 2 && piece_types[BLACK] == 1) {
      int ks1 = bitscan64(get_royal(board) & board->bbc[BLACK]);
      int ks2 = bitscan64(get_royal(board) & board->bbc[BLACK] ^ make_bitboard_square(ks1));
      int d = king_distance(ks1, ks2);

      if (d == 1) {
         if (num_pieces[WHITE] == 2) can_win[WHITE] = false;
         if (num_pieces[WHITE] == 3 && !(board->bbp[ROOK]|board->bbp[QUEEN])) can_win[WHITE] = false;

         /* Reduce score as the two kings come closer together */
         mat = (mat * (d-1)) / 8;
      }
   }

   /* Rook vs minor, draw-ish */
   if (piece_types[WHITE] == 2 && piece_count[WHITE][ROOK] &&
         piece_types[BLACK] == 2 && piece_count[BLACK][LIEUTENANT])
      mat /= 4;

   if (piece_types[WHITE] == 2 && piece_count[WHITE][ROOK] &&
         piece_types[BLACK] == 2 && piece_count[BLACK][CAPTAIN])
      mat /= 4;

   /* Two knights cannot force mate */
   if (piece_types[BLACK] == 1 && piece_types[WHITE] == 2 && board->bbp[KNIGHT])
      can_win[WHITE] = false;

   if (piece_types[WHITE] == 1 && piece_types[BLACK] == 2 && board->bbp[KNIGHT])
      can_win[BLACK] = false;

   /* Lack of mating potential */
   bitboard_t minors = board->bbp[LIEUTENANT]|board->bbp[KNIGHT]|board->bbp[BISHOP];
   if (num_pieces[WHITE] == 2 && (minors&board->bbc[WHITE]))
      can_win[WHITE] = false;

   if (num_pieces[BLACK] == 2 && (minors&board->bbc[BLACK]))
      can_win[BLACK] = false;

   /* Lone king? */
   if (num_pieces[WHITE] == 1 || num_pieces[BLACK] == 1) {
      sides defender = (piece_types[WHITE] == 1) ? WHITE : BLACK;

      /* Call special evaluation to mate a lone king */
      if (lone_king_evaluation(board, defender, &psq)) {
         //mob = 0;
         pev = 0;

         goto done;
      }
   }

   /* King safety */
   /* White */
   int wks1, wks2, wksc;
   int ksq = bitscan64(board->bbp[KING]&board->bbc[WHITE]);
   int wks = 1.1*evaluate_king_safety(board, ksq, WHITE, &a);
   int as2 = army_attack_strength[BLACK]*army_attack_strength[BLACK];
   int kss = KING_SAFTEY_STRENGTH_SCALE*KING_SAFTEY_STRENGTH_SCALE;
   /* Castling rights? */
   wks2 = wks1 = -CHECKMATE;
   if ((board->init & short_castle_mask[WHITE]) == short_castle_mask[WHITE])
      wks1 = evaluate_king_safety(board, short_castle_king_dest[WHITE], WHITE, &a);
   if ((board->init & long_castle_mask[WHITE]) == long_castle_mask[WHITE])
      wks2 = evaluate_king_safety(board, long_castle_king_dest[WHITE], WHITE, &a);
   wksc = max(wks1, wks2);
   if (wks1 > -CHECKMATE && wks2 > -CHECKMATE) wksc *= 1.1;
   wks = (wks + max(wks, wksc)) / 2;
   ks += wks * as2 / kss;
   psq += (centre_table[ksq] * max(KING_SAFTEY_STRENGTH_SCALE-army_attack_strength[BLACK], 0)) / KING_SAFTEY_STRENGTH_SCALE;

   /* Black */
   if (board->bbp[BASILEUS] & board->bbc[BLACK]) {
      if (onebit64(board->bbp[BASILEUS] & board->bbc[BLACK])) {
         int bks = evaluate_king_safety(board, bitscan64(board->bbp[BASILEUS]&board->bbc[BLACK]), BLACK, &a);
         int as2 = army_attack_strength[WHITE]*army_attack_strength[WHITE];
         ks -= bks * as2 / kss;
      } else {
         int k;
         k = bitscan64(board->bbp[BASILEUS]&board->bbc[BLACK]);
         int bks1 = evaluate_king_safety(board, k, BLACK, &a);
         k = bitscan64((board->bbp[BASILEUS]&board->bbc[BLACK]) ^ make_bitboard_square(k));
         int bks2 = evaluate_king_safety(board, k, BLACK, &a);
         int as2 = army_attack_strength[WHITE]*army_attack_strength[WHITE];
         int bks;
         //bks = max(bks1, bks2) + min(bks1, bks2) / 4;
         bks = (bks1 + bks2) / 4;
         ks -= bks * as2 / kss;
      }
   } else {
      /* Black */
      int wks1, wks2, wksc;
      int ksq = bitscan64(board->bbp[KING]&board->bbc[BLACK]);
      int wks = 1.1*evaluate_king_safety(board, ksq, BLACK, &a);
      int as2 = army_attack_strength[WHITE]*army_attack_strength[WHITE];
      int kss = KING_SAFTEY_STRENGTH_SCALE*KING_SAFTEY_STRENGTH_SCALE;
      /* Castling rights? */
      wks2 = wks1 = -CHECKMATE;
      if ((board->init & short_castle_mask[BLACK]) == short_castle_mask[BLACK])
         wks1 = evaluate_king_safety(board, short_castle_king_dest[BLACK], BLACK, &a);
      if ((board->init & long_castle_mask[BLACK]) == long_castle_mask[BLACK])
         wks2 = evaluate_king_safety(board, long_castle_king_dest[BLACK], BLACK, &a);
      wksc = max(wks1, wks2);
      if (wks1 > -CHECKMATE && wks2 > -CHECKMATE) wksc *= 1.1;
      wks = (wks + max(wks, wksc)) / 2;
      ks -= wks * as2 / kss;
      psq -= (centre_table[ksq] * max(KING_SAFTEY_STRENGTH_SCALE-army_attack_strength[WHITE], 0)) / KING_SAFTEY_STRENGTH_SCALE;
   }

done:
   ev = mat + psq + mob + pev + ks;

   /* Adjust the score: if the side that is nominally ahead can't win, drop the score to 0(ish) */
   if (ev > 0 && !can_win[WHITE]) ev = psq;
   if (ev < 0 && !can_win[BLACK]) ev = psq;

   /* If we're only up against pieces and have only one piece with mate potential, things become more
    * difficult because we can't just exchange one piece.
    */
   if (ev > 0 && num_pawns[WHITE] == 0 && num_pawns[BLACK] == 0 && piece_types[BLACK] > 1 && mate_potential[WHITE] < 2) ev /= 2;
   if (ev < 0 && num_pawns[WHITE] == 0 && num_pawns[BLACK] == 0 && piece_types[WHITE] > 1 && mate_potential[BLACK] < 2) ev /= 2;

   /* Tapered evaluation when we're about to hit the 50 move counter */
   if (board->fifty_counter > 80)
      ev = ev * (101 - board->fifty_counter) / 20;

   return (side_to_move == WHITE) ? ev : -ev;
}

