/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include "move.h"

const char *move_string(move_t move, char *buffer)
{
   static char static_buffer[256];
   char *s = static_buffer;
   char dash = '-';
   if (buffer)
      s = buffer;
   s[0] = '\0';
   int n = 0;

   int c, k;
   
   /* In normal chess (and Capablanca), "O-O" is king-side castling, "O-O-O" is queen-side.
    * This holds true in FRC and CRC games, but it is not true in Janus chess.
    * This is a problem, because simply testing whether the king starts out on the left-side of the board
    * would break FRC/CRC variants.
    */
   if (is_castle_move(move)) {
      int from = get_move_from(move);
      int to   = get_move_to(move);
      if (to > from)
         snprintf(s, 256, "O-O");
      else
         snprintf(s, 256, "O-O-O");
      goto done;
   }

   if (is_drop_move(move)) {
      uint16_t p = get_move_drop(move, 0);
      char fp = piece_symbol_string[decode_drop_piece(p)];
      int to = decode_drop_square(get_move_drop(move, 0));
      snprintf(s, 256, "%c@%s", fp, square_names[to]);
      goto done;
   }

   /* Normal move or capture */
   if (is_capture_move(move))
      dash = 'x';

   int from, to;
   char fp = '\0', tp = '\0';

   if (is_promotion_move(move))
      tp = piece_symbol_string[decode_drop_piece(get_move_drop(move, 0))];

   from = to = decode_drop_square(get_move_drop(move, 0));
   k = get_move_pickups(move);
   for (c=0; c<k; c++) {
      uint16_t p = get_move_pickup(move, c);
      if (decode_pickup_side(p) == decode_drop_side(get_move_drop(move, 0))) {
         from = decode_pickup_square(p);
         fp = piece_symbol_string[decode_pickup_piece(p)];
         break;
      }
   }

   snprintf(s, 256, "%c%s%c%s%c", fp, square_names[from], dash, square_names[to], tp);
done:

   /* Check for gate moves */
   if (get_move_gates(move)) {
      char *p = s + strlen(s);
      int c = get_move_drops(move)-1;
      uint16_t drop = get_move_drop(move, c);
      char fp = piece_symbol_string[decode_drop_piece(drop)];
      int to = decode_drop_square(drop);
      snprintf(p, 200, "/%c%s", fp, square_names[to]);
   }
   return s;
}

const char *move_stack_string(move_t move, char *buffer)
{
   static char static_buffer[256];
   char *s = static_buffer;
   char dash = '-';
   if (buffer)
      s = buffer;
   s[0] = '\0';
   int n = 0;

   int c, k;
   k = get_move_pickups(move);
   for (c=0; c<k; c++) {
      uint16_t p = get_move_pickup(move, c);
      snprintf(s+n, 256-n, "%s", square_names[decode_pickup_square(p)]);
      n+=2;
   }
   snprintf(s+n, 256-n, "%c", dash);
   n++;
   k = get_move_drops(move);
   for (c=0; c<k; c++) {
      uint16_t p = get_move_drop(move, c);
      snprintf(s+n, 256-n, "%s", square_names[decode_drop_square(p)]);
      n+=2;
   }

   return s;
}
