/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef BOARD_H
#define BOARD_H

#include <stdbool.h>
#include <stdint.h>
#include "assert.h"
#include "square.h"
#include "bitboard.h"
#include "squares.h"
#include "pieces.h"
#include "move.h"
#include "hashkey.h"
#include "psq.h"
#include "config.h"

#define expect(x,y) __builtin_expect((x), (y))

struct board_t {
   /* Board occupancy bitboards
    *  Index 0 is for white pieces
    *  Index 1 is for black pieces 
    */
   union {
      bitboard_t       bbc[NUM_SIDES];
      large_bitboard_t large_bbc[NUM_SIDES];
   };

   /* Location of royal pieces.
    *  In a sense, this information is redundant, since the king is redundant. However, having a separate
    *  bitboard to track the position of the royal pieces gives us a lot more flexibility in defining chess
    *  variants where there can be multiple royal pieces.
    */
   union {
      bitboard_t       royal;
      large_bitboard_t large_royal;
   };

   /* Bitboard encoding initial positions, used in move generation for
    * castling moves and "special" pawn moves.
    */
   union {
      bitboard_t       init;
      large_bitboard_t large_init;
   };

   /* Bitboards for all piece types, up to 16 per variant */
   union {
      bitboard_t       bbp[MAX_PIECE_TYPES];
      large_bitboard_t large_bbp[MAX_PIECE_TYPES];
   };

   /* Pseudo-attack tables.
    * These are initialised at startup for all piece types for both players and contain all squares that
    * are attacked by a particular side. For an empty square, this means that a piece on that square could
    * be captured.
    * For sliders, this is typically the union of the move and attack set, for leapers it's just the possible
    * destinations and similar for steppers. A distinction comes in when dealing with lame leapers and
    * hoppers, and this is where pseudo comes in. Rather than the actual attack set, we store the squares that
    * affect the attack set in addition to the attack set itself (for normal pieces, these sets coincide). If
    * a square is not in the pseudo-attack set, then it is not attacked. If it is, then we nede to do a check
    * by generating attacks for that piece type only.
    * This will be much faster than the alternative of generating all possible attacks to a target square.
    *
    * The attack set needs to be updated whenever a move touches one of its squares, or captures a piece of
    * that type. The main use of the attack tables is in speeding up check detection.
    */
   union {
      bitboard_t attack_table[MAX_PIECE_TYPES][NUM_SIDES];
      large_bitboard_t large_attack_table[MAX_PIECE_TYPES][NUM_SIDES];
   };
   union {
      bitboard_t attacks[NUM_SIDES];
      large_bitboard_t large_attacks[NUM_SIDES];
   };

   /* Drop zone, for both players */
   union {
      bitboard_t       drop_zone[NUM_SIDES];
      large_bitboard_t large_drop_zone[NUM_SIDES];
   };

   bool large_board;
   /* Bitboard representation for the chess board.
    * We have bitboards for pieces of both colours and for all piece types.
    * There are two special bit boards: one with the position of the royal pieces ("kings") and one with
    * pieces that are still on their initial square.
    */

   /* Byte board holding the type of piece at each location.
    * NB: the contents of empty squares is not defined!
    */
   int8_t piece[128];

   /* En-passant target square and capture location. */
   int8_t ep;
   int8_t ep_capture;

   /* Piece holdings.
    * These are indexed by [piece type][side to move]
    * We actually have a bool flag to specify whether we're interested in these or not, so we can skip a chunk
    * of code for variants where we're not.
    */
   int8_t holdings[MAX_PIECE_TYPES][NUM_SIDES];
   bool use_holdings;

   /* Description of all piece types */
   piece_description_t *piece_types;

#ifndef PIECE_VALUES_IN_PSQ
   int material[NUM_SIDES];
#endif
   int psq;

   /* Hash key */
   uint64_t hash;
   uint64_t pawn_hash;

   /* Whether the current side is in check or not */
   bool in_check;

   /* Whether a side has castled or not */
   bool did_castle[NUM_SIDES];

   /* Side to move */
   sides side_to_move;

   /* Rule flags, to change the behaviour of the move generator or the evaluation function */
   uint32_t rule_flags;
} __attribute__((aligned(16)));
typedef struct board_t board_t;

#define RF_FORCE_CAPTURE   0x00000001
#define RF_MULTI_CAPTURE   0x00000002
#define RF_FORCE_DROPS     0x00000004
#define RF_KING_TABOO      0x00000008     /* Kings cannot face eachother along a ray */
#define RF_KING_TRAPPED    0x00000010     /* Kings are trapped in a "palace" */
#define RF_KING_CORNERDRIVE 0x00000020    /* Kings can drive eachother from the corners */
#define RF_CAPTURE_ANY_FLAG 0x00000100    /* There is a "capture the flag" victory condition. */
#define RF_CAPTURE_ALL_FLAG 0x00000200    /* There is a "capture the flag" victory condition. */
#define RF_CAPTURE_THE_FLAG 0x00000300    /* There is a "capture the flag" victory condition. */

void clear_board(board_t *board);
void print_uint8(uint8_t b);
void print_bitboards(const board_t *board);
void print_board(const board_t *board);

/**********************************************************************
 *                      static inline functions                       *
 **********************************************************************/
static inline void place_piece(board_t *board, sides side, int piece, int square)
{
   if (board->large_board) {
      large_bitboard_t bb = large_square_bitboards[square];
      board->large_bbc[side] |= bb;
      board->large_bbp[piece] |= bb;
      if (board->piece_types->piece_flags[piece]&PF_ROYAL) board->large_royal |= bb;
      board->piece[square] = piece;
   } else {
      bitboard_t bb = square_bitboards[square];
      board->bbc[side] |= bb;
      board->bbp[piece] |= bb;
      if (board->piece_types->piece_flags[piece]&PF_ROYAL) board->royal |= bb;
      board->piece[square] = piece;
   }
}

static inline void remove_piece(board_t *board, int square)
{
   int piece = board->piece[square];
   if (board->large_board) {
      large_bitboard_t bb = ~large_square_bitboards[square];
      board->large_bbc[WHITE] &= bb;
      board->large_bbc[BLACK] &= bb;
      board->large_bbp[piece] &= bb;
      if (board->piece_types->piece_flags[piece]&PF_ROYAL) board->large_royal &= bb;
   } else {
      bitboard_t bb = ~square_bitboards[square];
      board->bbc[WHITE] &= bb;
      board->bbc[BLACK] &= bb;
      board->bbp[piece] &= bb;
      if (board->piece_types->piece_flags[piece]&PF_ROYAL) board->royal &= bb;
   }
}

static inline bool square_is_occupied(board_t *board, int square)
{
   int piece = board->piece[square];
   if (board->large_board) {
      large_bitboard_t bb = large_square_bitboards[square];
      if (!is_zero128(bb & (board->large_bbc[WHITE]|board->large_bbc[BLACK]))) return true;
   } else {
      bitboard_t bb = square_bitboards[square];
      if (bb & (board->bbc[WHITE]|board->bbc[BLACK])) return true;
   }

   return false;
}



static inline void assert_move(board_t *board, move_t move)
{
#ifdef DEBUGMODE
    int n, c;
    assert(board->large_board == false);

    /* First: check all pickups */
    n = get_move_pickups(move);
    for (c=0; c<n; c++) {
       uint16_t p = get_move_pickup(move, c);
       int square = decode_pickup_square(p);
       int piece  = decode_pickup_piece(p);
       sides side = decode_pickup_side(p);
       bitboard_t bb = make_bitboard_square(square);

       if (!(board->bbc[side]&bb)){
          print_bitboards(board);
          printf("%s\n", move_string(move, NULL));
       }
       if (!(board->bbp[piece]&bb)){
          print_bitboards(board);
          printf("%s\n", move_string(move, NULL));
       }
       assert(board->bbc[side]&bb);
       assert(board->bbp[piece]&bb);
       assert(board->piece[square] == piece);
    }

    /* Second: check all drops */
    n = get_move_drops(move);
    for (c=0; c<n; c++) {
       uint16_t p = get_move_drop(move, c);
       int square = decode_drop_square(p);
       int piece  = decode_drop_piece(p);
       sides side = decode_drop_side(p);
       bitboard_t bb = make_bitboard_square(square);

       assert((board->bbc[side]&bb) == 0);
    }

    assert((board->bbc[0] & board->bbc[1]) == 0);
    assert( (board->bbc[0]  | board->bbc[1]) ==
          (board->bbp[0]  | board->bbp[1]  | board->bbp[2]  | board->bbp[3]  | board->bbp[4]  |
           board->bbp[5]  | board->bbp[6]  | board->bbp[7]  | board->bbp[8]  | board->bbp[9]  |
           board->bbp[10] | board->bbp[11] | board->bbp[12] | board->bbp[13] | board->bbp[14] |
           board->bbp[15]) );
#endif
}



static inline void assert_move_large(board_t *board, move_t move)
{
#ifdef DEBUGMODE
   int n, c;
   assert(board->large_board == true);

   /* First: check all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      large_bitboard_t bb = large_square_bitboards[square];

      if (is_zero128(board->large_bbc[side]&bb)){
         print_bitboards(board);
         printf("%s\n", move_string(move, NULL));
      }
      if (is_zero128(board->large_bbp[piece]&bb)){
         print_bitboards(board);
         printf("%s\n", move_string(move, NULL));
      }
      assert(!is_zero128(board->large_bbc[side]&bb));
      assert(!is_zero128(board->large_bbp[piece]&bb));
      assert(board->piece[square] == piece);
   }

   /* Second: check all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      large_bitboard_t bb = large_square_bitboards[square];

      if (!is_zero128(board->large_bbc[side]&bb)){
         print_bitboards(board);
         printf("%s\n", move_string(move, NULL));
         printf("Pickups: ");
         int c;
         n = get_move_pickups(move);
         for (c=0; c<n; c++) {
            uint16_t p = get_move_pickup(move, c);
            printf("%s ", square_names[decode_pickup_square(p)]);
         }
         printf("\nDrops: ");
         n = get_move_pickups(move);
         for (c=0; c<n; c++) {
            uint16_t p = get_move_drop(move, c);
            printf("%s ", square_names[decode_drop_square(p)]);
         }
         printf("\n");
      }
      assert(is_zero128(board->large_bbc[side]&bb));
   }

   assert(is_zero128(board->large_bbc[0] & board->large_bbc[1]));
   assert( is_equal128(board->large_bbc[0] | board->large_bbc[1],
            board->large_bbp[0] |board->large_bbp[1] |board->large_bbp[2] |board->large_bbp[3] |board->large_bbp[4]  |
            board->large_bbp[5] |board->large_bbp[6] |board->large_bbp[7] |board->large_bbp[8] |board->large_bbp[9]  |
            board->large_bbp[10]|board->large_bbp[11]|board->large_bbp[12]|board->large_bbp[13]|board->large_bbp[14] |
            board->large_bbp[15]) );
#endif
}



static inline void makemove(board_t *board, move_t move)
{
   int n, c;

   /* Do some basic sanity checks up-front */
   assert(board->large_board == false);
   assert_move(board, move);

   /* First: resolve all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->bbp[piece] ^= bb;
      board->royal &= ~bb;
      board->init &= ~bb;
      board->hash ^= piece_key[piece][side][square];
      if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      board->psq -= psq[piece][side][square];
#ifndef PIECE_VALUES_IN_PSQ
      board->material[side] -= board->piece_types->piece_value[piece];
#endif

      assert((board->bbc[side]&bb) == 0);
      assert((board->bbp[piece]&bb) == 0);
   }

   /* Second: resolve all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->bbp[piece] ^= bb;
      board->royal |= (bitboard_t)(board->piece_types->piece_flags[piece]&PF_ROYAL) << square;
      board->piece[square] = piece;
      board->hash ^= piece_key[piece][side][square];
      if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      board->psq += psq[piece][side][square];
#ifndef PIECE_VALUES_IN_PSQ
      board->material[side] += board->piece_types->piece_value[piece];
#endif

      assert(board->bbc[side]&bb);
      assert(board->bbp[piece]&bb);
   }

   if (expect(is_castle_move(move), false))
      board->did_castle[board->side_to_move] = true;

   /* Third: update holdings */
   if (expect(board->use_holdings && get_move_holdings(move), false)) {
      uint16_t p = get_move_holding(move);
      int count = decode_holding_count(p);
      int piece  = decode_holding_piece(p);
      sides side = decode_holding_side(p);
      board->holdings[piece][side] += count;
   }

   /* Fourth: flip side to move */
   if (expect((move & MOVE_KEEP_TURN) == 0, true)) {
      board->side_to_move = next_side[board->side_to_move];
      board->hash ^= side_to_move_key;
   }

#ifdef DEBUGMODE
   /* Sanity check: make sure all drops and pickups are balances so there is no "vacuum" */
   for (n=0; n<MAX_PIECE_TYPES; n++)
      for (c=0; c<NUM_SIDES; c++)
         assert(board->holdings[n][c] >= 0);
#endif

   /* Assume we're not in check */
   board->in_check = false;

   assert((board->bbc[0] & board->bbc[1]) == 0);
   assert( (board->bbc[0]  | board->bbc[1]) ==
           (board->bbp[0]  | board->bbp[1]  | board->bbp[2]  | board->bbp[3]  | board->bbp[4]  |
            board->bbp[5]  | board->bbp[6]  | board->bbp[7]  | board->bbp[8]  | board->bbp[9]  |
            board->bbp[10] | board->bbp[11] | board->bbp[12] | board->bbp[13] | board->bbp[14] |
            board->bbp[15]) );
}

static inline void large_makemove(board_t *board, move_t move)
{
   int n, c;

   /* Do some basic sanity checks up-front */
   assert(board->large_board == true);
   assert_move_large(board, move);

   /* First: resolve all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      large_bitboard_t bb = large_square_bitboards[square];
      board->large_bbc[side] ^= bb;
      board->large_bbp[piece] ^= bb;
      board->large_royal &= ~bb;
      board->large_init &= ~bb;
      board->hash ^= piece_key[piece][side][square];
      if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      board->psq -= psq[piece][side][square];
#ifndef PIECE_VALUES_IN_PSQ
      board->material[side] -= board->piece_types->piece_value[piece];
#endif

      assert(is_zero128(board->large_bbc[side]&bb));
      assert(is_zero128(board->large_bbp[piece]&bb));
   }

   /* Second: resolve all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      large_bitboard_t bb = large_square_bitboards[square];
      board->large_bbc[side] ^= bb;
      board->large_bbp[piece] ^= bb;
      if (board->piece_types->piece_flags[piece]&PF_ROYAL) board->large_royal |= bb;
      board->piece[square] = piece;
      board->hash ^= piece_key[piece][side][square];
      if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      board->psq += psq[piece][side][square];
#ifndef PIECE_VALUES_IN_PSQ
      board->material[side] += board->piece_types->piece_value[piece];
#endif

      assert(!is_zero128(board->large_bbc[side]&bb));
      assert(!is_zero128(board->large_bbp[piece]&bb));
   }

   if (expect(is_castle_move(move), false))
      board->did_castle[board->side_to_move] = true;

   /* Third: update holdings */
   if (expect(board->use_holdings && get_move_holdings(move), false)) {
      uint16_t p = get_move_holding(move);
      int count = decode_holding_count(p);
      int piece  = decode_holding_piece(p);
      sides side = decode_holding_side(p);
      board->holdings[piece][side] += count;
   }

   /* Fourth: flip side to move */
   if (expect((move & MOVE_KEEP_TURN) == 0, true)) {
      board->side_to_move = next_side[board->side_to_move];
      board->hash ^= side_to_move_key;
   }

#ifdef DEBUGMODE
   /* Sanity check: make sure all drops and pickups are balances so there is no "vacuum" */
   for (n=0; n<MAX_PIECE_TYPES; n++)
      for (c=0; c<NUM_SIDES; c++)
         assert(board->holdings[n][c] >= 0);
#endif

   /* Assume we're not in check */
   board->in_check = false;

   assert(is_zero128(board->large_bbc[0] & board->large_bbc[1]));
   assert(is_equal128(board->large_bbc[0]  | board->large_bbc[1],
            board->large_bbp[0]  | board->large_bbp[1]  | board->large_bbp[2]  | board->large_bbp[3]  | board->large_bbp[4]  |
            board->large_bbp[5]  | board->large_bbp[6]  | board->large_bbp[7]  | board->large_bbp[8]  | board->large_bbp[9]  |
            board->large_bbp[10] | board->large_bbp[11] | board->large_bbp[12] | board->large_bbp[13] | board->large_bbp[14] |
            board->large_bbp[15]) );
}

static inline void unmakemove(board_t *board, move_t move)
{
   int n, c;

   /* Sanity checks */
   assert(board->large_board == false);

   /* First: flip side to move */
   if (expect((move & MOVE_KEEP_TURN) == 0, true)) {
      board->side_to_move = next_side[board->side_to_move];
      board->hash ^= side_to_move_key;
   }

   /* Second: reverse all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->bbp[piece] ^= bb;
      board->royal &= ~bb;
      board->hash ^= piece_key[piece][side][square];
      if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      board->psq -= psq[piece][side][square];
#ifndef PIECE_VALUES_IN_PSQ
      board->material[side] -= board->piece_types->piece_value[piece];
#endif

      assert((board->bbc[side]&bb) == 0);
      assert((board->bbp[piece]&bb) == 0);
   }

   /* Third: reverse all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->bbp[piece] ^= bb;
      board->royal |= (bitboard_t)(board->piece_types->piece_flags[piece]&PF_ROYAL) << square;
      board->piece[square] = piece;
      board->hash ^= piece_key[piece][side][square];
      if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      board->psq += psq[piece][side][square];
#ifndef PIECE_VALUES_IN_PSQ
      board->material[side] += board->piece_types->piece_value[piece];
#endif

      assert(board->bbc[side]&bb);
      assert(board->bbp[piece]&bb);
   }

   /* Fourth: reverse store/retrieve */
   if (expect(board->use_holdings && get_move_holdings(move), false)) {
      uint16_t p = get_move_holding(move);
      int count = decode_holding_count(p);
      int piece  = decode_holding_piece(p);
      sides side = decode_holding_side(p);
      board->holdings[piece][side] -= count;
      assert(board->holdings[piece][side] >= 0);
   }

   assert((board->bbc[0] & board->bbc[1]) == 0);
   assert( (board->bbc[0]  | board->bbc[1]) ==
           (board->bbp[0]  | board->bbp[1]  | board->bbp[2]  | board->bbp[3]  | board->bbp[4]  |
            board->bbp[5]  | board->bbp[6]  | board->bbp[7]  | board->bbp[8]  | board->bbp[9]  |
            board->bbp[10] | board->bbp[11] | board->bbp[12] | board->bbp[13] | board->bbp[14] |
            board->bbp[15]) );
}


static inline void large_unmakemove(board_t *board, move_t move)
{
   int n, c;

   /* Sanity checks */
   assert(board->large_board == true);

   /* First: flip side to move */
   if (expect((move & MOVE_KEEP_TURN) == 0, true)) {
      board->side_to_move = next_side[board->side_to_move];
      board->hash ^= side_to_move_key;
   }

   /* Second: reverse all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      large_bitboard_t bb = large_square_bitboards[square];
      board->large_bbc[side] ^= bb;
      board->large_bbp[piece] ^= bb;
      board->large_royal &= ~bb;
      board->hash ^= piece_key[piece][side][square];
      if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      board->psq -= psq[piece][side][square];
#ifndef PIECE_VALUES_IN_PSQ
      board->material[side] -= board->piece_types->piece_value[piece];
#endif

      assert(is_zero128(board->large_bbc[side]&bb));
      assert(is_zero128(board->large_bbp[piece]&bb));
   }

   /* Third: reverse all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      large_bitboard_t bb = large_square_bitboards[square];
      board->large_bbc[side] ^= bb;
      board->large_bbp[piece] ^= bb;
      if (board->piece_types->piece_flags[piece]&PF_ROYAL) board->large_royal |= bb;
      board->piece[square] = piece;
      board->hash ^= piece_key[piece][side][square];
      if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      board->psq += psq[piece][side][square];
#ifndef PIECE_VALUES_IN_PSQ
      board->material[side] += board->piece_types->piece_value[piece];
#endif

      assert(!is_zero128(board->large_bbc[side]&bb));
      assert(!is_zero128(board->large_bbp[piece]&bb));
   }

   /* Fourth: reverse store/retrieve */
   if (expect(board->use_holdings && get_move_holdings(move), false)) {
      uint16_t p = get_move_holding(move);
      int count = decode_holding_count(p);
      int piece  = decode_holding_piece(p);
      sides side = decode_holding_side(p);
      board->holdings[piece][side] -= count;
      assert(board->holdings[piece][side] >= 0);
   }

   assert(is_zero128(board->large_bbc[0] & board->large_bbc[1]));
   assert(is_equal128(board->large_bbc[0]  | board->large_bbc[1],
            board->large_bbp[0]  | board->large_bbp[1]  | board->large_bbp[2]  | board->large_bbp[3]  | board->large_bbp[4]  |
            board->large_bbp[5]  | board->large_bbp[6]  | board->large_bbp[7]  | board->large_bbp[8]  | board->large_bbp[9]  |
            board->large_bbp[10] | board->large_bbp[11] | board->large_bbp[12] | board->large_bbp[13] | board->large_bbp[14] |
            board->large_bbp[15]) );
}

static inline void large_makemove_see(board_t *board, move_t move)
{
   int n, c;

   /* Sanity checks */
   assert(board->large_board == true);

   /* First: resolve all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      large_bitboard_t bb = large_square_bitboards[square];
      board->large_bbc[side] ^= bb;
      board->large_bbp[piece] ^= bb;
      board->large_royal &= ~bb;

      assert(is_zero128(board->large_bbc[side]&bb));
      assert(is_zero128(board->large_bbp[piece]&bb));
   }

   /* Second: resolve all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      large_bitboard_t bb = large_square_bitboards[square];
      board->large_bbc[side] ^= bb;
      board->large_bbp[piece] ^= bb;
      if (board->piece_types->piece_flags[piece]&PF_ROYAL) board->large_royal |= bb;
      board->piece[square] = piece;

      assert(!is_zero128(board->large_bbc[side]&bb));
      assert(!is_zero128(board->large_bbp[piece]&bb));
   }

   /* Next player */
   if (expect((move & MOVE_KEEP_TURN) == 0, true))
      board->side_to_move = next_side[board->side_to_move];
}

static inline void makemove_see(board_t *board, move_t move)
{
   int n, c;

   if (board->large_board) {
      large_makemove_see(board, move);
      return;
   }

   /* Sanity checks */
   assert(board->large_board == false);

   /* First: resolve all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->bbp[piece] ^= bb;
      board->royal &= ~bb;

      assert((board->bbc[side]&bb) == 0);
      assert((board->bbp[piece]&bb) == 0);
   }

   /* Second: resolve all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->bbp[piece] ^= bb;
      board->royal |= (bitboard_t)(board->piece_types->piece_flags[piece]&PF_ROYAL) << square;
      board->piece[square] = piece;

      assert(board->bbc[side]&bb);
      assert(board->bbp[piece]&bb);
   }

   /* Next player */
   if (expect((move & MOVE_KEEP_TURN) == 0, true))
      board->side_to_move = next_side[board->side_to_move];
}

static inline uint64_t update_hash_from_move(move_t move, uint64_t hash)
{
   int n, c;

   /* First: resolve all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      hash ^= piece_key[piece][side][square];
   }

   /* Second: resolve all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      hash ^= piece_key[piece][side][square];
   }

   /* Third: update holdings */
   if (expect(get_move_holdings(move), 0)) {
      uint16_t p = get_move_holding(move);
      int count = decode_holding_count(p);
      int piece  = decode_holding_piece(p);
      sides side = decode_holding_side(p);
   }

   /* Fourth: flip side to move */
   if (expect((move & MOVE_KEEP_TURN) == 0, true)) {
      hash ^= side_to_move_key;
   }

   return hash;
}


static inline uint8_t get_row_occupancy_number(const board_t *board, const int square)
{
   bitboard_t bb;
   assert(board);

   bb = or_bitboards(board->bbc[0], board->bbc[1]);
   return get_bitboard_row(bb, unpack_rank(square));
}

static inline uint8_t get_file_occupancy_number(const board_t *board, const int square)
{
   bitboard_t bb;
   uint8_t occ;
   int file = unpack_file(square);
   assert(board);

   bb = or_bitboards(board->bbc[0], board->bbc[1]);
   bb = board_afile & (bb >> file);
   occ = (bb * 0x0102040810204080ll) >> 56;
   return occ;
}

/* Get the occupancy number for the diagonal parallel to h1-a8 that passes
 * through the indicated square.
 * The number of bits used by the occupancy number are given by the length
 * of the diagonal, which is 1+(diag_nr&7) (diag_nr starting from 0).
 * Any extra high bits are *not* cleared, however. If this is considered
 * necessary, the caller should do this explicitly.
 */
static inline uint8_t get_diagonal_a8h1_occupancy_number(const board_t *board, const int square)
{
   assert(board);

   bitboard_t bb;
   bb = or_bitboards(board->bbc[0], board->bbc[1]);

   return ( (bb & board_a8h1_diagonal[get_a8h1_diagonal(square)]) * board_afile) >> 56;
}

/* Get the occupancy number for the diagonal parallel to a1-a8 that passes
 * through the indicated square.
 * The number of bits used by the occupancy number are given by the length
 * of the diagonal, which is 1+(diag_nr&7) (diag_nr starting from 0).
 * Any extra high bits are *not* cleared, however. If this is considered
 * necessary, the caller should do this explicitly.
 */
static inline uint8_t get_diagonal_a1h8_occupancy_number(const board_t *board, const int square)
{
   assert(board);

   bitboard_t bb;
   bb = or_bitboards(board->bbc[0], board->bbc[1]);

   return ( (bb & board_a1h8_diagonal[get_a1h8_diagonal(square)]) * board_afile) >> 56;
}

static inline uint8_t get_piece(const board_t *board, const int where)
{
   return board->piece[where];
}

static inline sides get_piece_colour(const board_t *board, const int where)
{
   if (board->large_board) {
      if (!is_zero128(board->large_bbc[WHITE] & large_square_bitboards[where]))
         return WHITE;

      if (!is_zero128(board->large_bbc[BLACK] & large_square_bitboards[where]))
         return BLACK;
   } else {
      if (board->bbc[WHITE] & make_bitboard_square(where))
         return WHITE;

      if (board->bbc[BLACK] & make_bitboard_square(where))
         return BLACK;
   }

   return NONE;
}

/* Returns TRUE or FALSE depending on whether to given side may still
 * castle or not in the given position.
 */
static inline bool may_castle(const board_t *board, sides side)
{
   if (board->did_castle[side])
      return false;

   if (board->large_board) {
      if (is_zero128(board->large_init & board->large_royal & board->large_bbc[side]))
         return false;

      if (is_zero128(board->large_init & large_short_castle_mask[side] & board->large_bbc[side]))
         return false;

      if (is_zero128(board->large_init & large_long_castle_mask[side] & board->large_bbc[side]))
         return false;
   } else {
      if (!(board->init & board->royal & board->bbc[side]))
         return false;

      if (!(board->init & (A1_MASK|H1_MASK|A8_MASK|H8_MASK) & board->bbc[side]))
         return false;
   }

   return true;
}



static int count_piece_types(const board_t *board, sides side)
{
   int count = 0;
   if (board->large_board) {
      large_bitboard_t own = board->large_bbc[side];
      int n;

      for (n=0; n<board->piece_types->num_piece_types; n++)
         count += !is_zero128(board->large_bbp[n] & own);
   } else {
      bitboard_t own = board->bbc[side];
      int n;

      for (n=0; n<board->piece_types->num_piece_types; n++)
         count += (board->bbp[n] & own) != 0;
   }

   return count;
}
#endif
