/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef EVALUATE_H
#define EVALUATE_H

#include <stdint.h>
#include "game.h"

#define PENALISE_HANGING_PIECES
#define USE_SLIDER_TERMS

#define CHECKMATE 16000
#define STALEMATE     0
#define ILLEGAL   (CHECKMATE+500)

/* Safety margin for lazy evaluation. The sum of all terms apart from material should not exceed this value. */
#define LAZY_MARGIN 500
/* Tuning versions (against 295 and 304), error bar is +/- 7 elo
 * Version                    ELO      LOS vs. 304M2
 * 295                        2082      0%
 * 304   CHECKMATE (disabled) 2098      0%
 * 304M  300                  2109     49%
 * 304M2 500                  2109      -
 * 304M3 1000                 2101      5%
 */


/* Penalty when castling rights have been lost since the root of the search */
#define CASTLE_LOST_PENALTY   15

/* Material balance/imbalance parameters */
#define PAIR_BONUS 30

/* King safety evaluation parameters */
#define KING_SAFETY_PIECE_WEIGHT_NORMALISATION  14    /* Normalisation factor for attacking power of pieces */
#define KING_SAFETY_PIECE_WEIGHT_MAX             4    /* Maximum index due to piece attack pattern */
#define KING_SAFTEY_STRENGTH_SCALE              96    /* Normalisation for army strength */

/* Parameters for king safety evaluation table (adopted from Jazz) */
#define KING_SAFETY_CEIL         262.5       /* Asymptotic value */
#define KING_SAFETY_SLOPE          0.15      /* Slope of the king safety evaluation term */
#define KING_SAFETY_INFLECTION    18         /* Inflection point where the evaluation begins to rise steeply */
/* Parameter settings:
 * (match against 228):
 * 236-M1         625  0.15  18
 * 236-M2         500  0.15  18
 * 236-M3         700  0.15  18
 * 236-M4         625  0.10  18
 * 236-M5         625  0.20  18
 * 236-M6         625  0.15  16
 *
 * 246-M1         400
 * 246-M2         550
 *
 * 255M           525
 *
 * Rank Name              Elo    +    - games score oppo. draws 
 *    1 Sjaak 236M2      2115   15   15  1592   55%  2076   25% 
 *    2 Sjaak 245:246M2  2114   15   15  1598   55%  2076   26% 
 *    3 Sjaak 236M3      2112   15   15  1590   55%  2076   26% 
 *    4 Sjaak 236M       2107   10   10  3192   53%  2087   28% 
 *    5 Sjaak 236M4      2099   15   15  1600   53%  2076   25% 
 *    6 Sjaak 255M       2097   10   10  3190   51%  2092   27% 
 *    7 Sjaak 236M5      2096   15   15  1598   53%  2076   25% 
 *    8 Sjaak 236M6      2094   15   15  1594   53%  2076   27% 
 *    9 Sjaak 245:246M   2090   15   15  1590   52%  2076   25% 
 *   10 Sjaak 228        2076    5    5 14348   46%  2103   26% 
 *
 * Retuning for new scaling (with king_safety_ceil -> /2)
 * TODO: complete run for these parameters for Capablanca, to test across variants.
 * 296         262.5
 * 296M        300
 * 296M2       200
 */

/* Generic mobility scores */
#define MOBILITY_CEIL         5.0
#define MOBILITY_SLOPE        1.0
#define MOBILITY_INFLECTION   0.0

/* Parameter settings:
 * (match results against 185 (2027) / 204 (2103)):
 * 212/223 M1     5.0  1.0  0.0 (2125)
 * 212/223 M2     5.0  0.5  0.0 (2115)
 * 212/223 M3     5.0  1.5  0.0 (2118)
 * 212/223 M4     5.0  1.0  0.2 (2111)
 *
 * 203 M5         4.0  1.0  0.0
 * 203 M6         5.0  1.0  0.4
 */

/* Specific mobility scores for sliders */
#define MOBILITY_SLIDER_CEIL         5.0
#define MOBILITY_SLIDER_SLOPE        1.0
#define MOBILITY_SLIDER_INFLECTION   0.0
/*                                  Normal   Spartan
 * 465            5.0  1.0  0.0     2099     2099
 * 465 M1         5.0  0.5  0.0     2103     2098
 * 465 M2         5.0  1.5  0.0     2102     2106
 * 465 M3         5.0  1.0  0.2     2094     2098
 * 465 M4         4.0  1.0  0.0     2103     2099
 *
 * Error bar on ELO +/- 14
 */

/* Ditto for leapers */
#define MOBILITY_LEAPER_CEIL         5.0
#define MOBILITY_LEAPER_SLOPE        1.0
#define MOBILITY_LEAPER_INFLECTION   0.0

/* Slider terms */
#define OPEN_FILE_BONUS            15
#define SEMI_OPEN_FILE_BONUS        5
#define SEVENTH_RANK_BONUS         15
#define FILE_BATTERY                5
#define DIAGONAL_BATTERY            2
/* 346M:    25 10 30 5 2
 * 346M2:   15  5 15 5 2
 * 352M3:   15 10 15 5 2
 *
 * Rank Name          Elo    +    - games score oppo. draws 
 *    1 Sjaak 346M2  2116    9    9  3984   53%  2093   22% 
 *    2 Sjaak 352M3  2108   10   10  3976   52%  2093   20% 
 *    3 Sjaak 342    2093    5    5 15940   49%  2102   21% 
 *    4 Sjaak 346M   2090   10   10  3992   49%  2093   19% 
 */

/* Pawn structure evaluation terms */
#define PAWN_EDGE_PENALTY          15
#define PAWN_WEAK_PENALTY           2
#define PAWN_DOUBLED_PENALTY        5
#define PAWN_OPEN_BONUS             2
#define PAWN_PASSED_BONUS           3
#define PAWN_PROTECT_PASSED_BONUS   5

/* Passed pawn bonus (vs 263M6, 2083):
 *  284        3  (2109)
 *  284M       4  (2108)
 */

/* Trade down/pawn value in end games */
#define PAWN_STEP_ENDGAME           5

/* Bonus for side-to-move (contempt factor) */
#define SIDE_TO_MOVE_BONUS          5

void initialise_evaluation(const game_t *game);
int16_t static_evaluation(game_t *game, sides side_to_move, int alpha, int beta);

static inline bool mate_score(int score)
{
   static const int mate = CHECKMATE - 1000;
   if ( score > mate || score < -mate )
      return true;

   return false;
}

static inline bool illegal_score(int score)
{
   if (score > CHECKMATE || score < -CHECKMATE)
      return true;
   return false;
}

static inline int score_to_hashtable(int score, int depth)
{
#ifdef ABSOLUTE_MATE_SCORES
   if (mate_score(score)) {
      if (score > 0)
         return score + depth;
      else
         return score - depth;
   } else
#endif
      return score;
}

static inline int score_from_hashtable(int score, int depth)
{
#ifdef ABSOLUTE_MATE_SCORES
   if (mate_score(score)) {
      if (score > 0)
         return score - depth;
      else
         return score + depth;
   } else
#endif
      return score;
}


#endif
