/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef GAME_H
#define GAME_H

#include <string.h>
#include <stdbool.h>
#include <stdint.h>
#include "assert.h"

#include "pieces.h"
#include "board.h"
#include "move.h"
#include "movegen.h"
#include "hashtable.h"
#include "attack_table.h"
#include "move_bitboard.h"
#include "pawn_table.h"
#include "eval_table.h"

#define MAX_SEARCH_DEPTH 60       /* maximum depth of search tree */

#define HARD_HORIZON -20

#define MAX_TOTAL_DEPTH (MAX_SEARCH_DEPTH - HARD_HORIZON)

#define HASH_TABLE_SIZE (1024*1024)

#undef USE_HISTORY_HEURISTIC

struct game_t {
   board_t board;      /* pointer to the current board position */
   piece_description_t pt;
   int16_t mate_score;  /* The score returned for mate, normally -CHECKMATE */
   int16_t stale_score; /* The score returned for stale mate, normally -STALEMATE */
   int16_t rep_score;   /* The score returned for a repetition, normally -STALEMATE */
   int16_t bare_king_score;   /* The score penalty for a lone king. Normally 0, but -CHECKMATE for shatranj */
   int16_t no_piece_score;/* The score returned for having no pieces, normally -CHECKMATE */
   int16_t flag_score;  /* The score returned when flags are captured, normally -CHECKMATE */
   ubitboard_t flag[2]; /* Flag bitboard, for "capture the flag" */
   board_t root_board; /* The board position at the root of the search */

   char *start_fen;     /* FEN string encoding the starting position of this game */
   char *xb_setup;      /* setup string that has to be sent to XBoard (startup position will be attached) */
   
   size_t moves_played; /* Number of moves played to current position */
   size_t last_move;    /* Number of the last move played in the game; useful when we take back a move */

   size_t max_moves;    /* Maximum number of moves that can be stored */
   int16_t *score;      /* Evaluation score associated with each position */
   move_t *move_list;   /* list of moves played in the game */

   /* State information */
   int8_t *fifty_counter;  /* Counter for the 50-moves rule */ 
   int8_t *ep;          /* En-passant square */ 
   int8_t *ep_capture;  /* Piece that is captured en-passant */ 
   uint64_t *hash_key;  /* hash keys */
   ubitboard_t *init;   /* Unmoved pieces, for castling rights */
   bool *in_check;
   bool *did_castle;

   /* Killer moves, storage space requirements must come from the search
    * function.
    */
   move_t killer[2][MAX_TOTAL_DEPTH];
   move_t mate_killer[MAX_TOTAL_DEPTH];

   /* Good captures: these are captures that are nominally bad (based on SEE score)
    * but actually turn out to be ok after trying them. These don't qualify for killer moves,
    * but they should still be sorted slightly higher up in the movelist than other bad captures.
    */
   move_t good_capture[2][MAX_TOTAL_DEPTH];

   /* Botwinnik-Markov extension.
    * This is like a killer move or a refutation move, but for NULL moves.
    * When encountering it on consecutive plies, extend the search (or at least sort the move higher in the
    * list).
    */
   move_t null_killer[MAX_TOTAL_DEPTH];

   /* Counter moves, for use with the counter move heuristic.
    * Indexed by from- and to square and side to move.
    */
   move_t counter[128][128][NUM_SIDES];

   /* Combination moves, similar to counter moves except indexed by our own
    * previous moves.
    * Indexed by from- and to square and side to move.
    */
   move_t combo[128][128][2];

   /* History reductions, as discussed by Ed Schroeder.
    * The idea is to reduce the depth of moves that have failed low in the
    * past. This requires a history table.
    * This is indexed by colour, piece type and destination square.
    * NB: I tried colour/from square/destination square, and it seems to be
    * worse, at least with the parameters given by Ed Schroeder. Probably
    * because using the piece type means we can catch a bad move more
    * easily even if we moved the piece first (Rf5 exf5 is bad independent
    * of whether the rook was on f8 or f7, this situation catches both).
    */
   int history_reduce[MAX_PIECE_TYPES][NUM_SIDES][128];

#ifdef USE_HISTORY_HEURISTIC
   /* History heuristic */
   int history_score[NUM_SIDES][MAX_PIECE_TYPES][128];
   int max_history;
#endif

   /* Data structure for retrieving the principle variation. At each depth,
    * there are two branches for the tree: the principle variation and the
    * "current" branch. If the current branch turns out to be better than
    * the PV, it becomes the new PV.
    * A binary tree would be enough to store this information (and be easy
    * to manipulate), but for now we use a plain NxN array (of which half
    * the space is wasted, obviously)
    */
   move_t principle_variation[MAX_TOTAL_DEPTH][MAX_TOTAL_DEPTH];
   int length_of_variation[MAX_TOTAL_DEPTH];
   int quiescence_depth_of_variation[MAX_TOTAL_DEPTH];
   bool hash_hit[MAX_TOTAL_DEPTH];
   move_t best_move[MAX_TOTAL_DEPTH];

   /* Transposition table */
   size_t default_hash_size;
   hash_table_t *transposition_table;

   /* Pawn table */
   pawn_hash_table_t *pawn_table;

   /* Evaluation table */
   eval_hash_table_t *eval_table;

   /* Hash table for repetition detection */
   int8_t repetition_hash_table[0xFFFF+1];

   /* various things having to do with time management */
   int time_left[2];       /* Total time left on the clock, in msec */
   int time_inc[2];        /* Time increment per move, in msec */
   int time_per_move;      /* Time per move, in msec */
   uint64_t start_time;    /* Timer value when the search started */
   int movestogo;
   int movestotc;
   int extra_time;
   int running_clock;
   int root_moves_played;
   size_t max_nodes;       /* Maximum number of nodes to search */

   /* Whether the engine is currently pondering or not, if it is, disable
    * time control.
    */
   bool pondering;
   move_t ponder_move;

   bool analysing;

   /* Large board? */
   bool large_board;

   /* Meta-data */
   char *name;
   char *name_white;
   char *name_black;

   /* Various function pointers, so we can easily customise things and
    * adjust to different UIs.
    */
   void (*output_iteration)(const char *, ...) __attribute__((format(printf, 1, 2)));
   void (*uci_output)(const char *, ...) __attribute__((format(printf, 1, 2)));
   void (*xboard_output)(const char *, ...) __attribute__((format(printf, 1, 2)));
   void (*error_output)(const char *, ...) __attribute__((format(printf, 1, 2)));

   bool (*check_clock)(const struct game_t *game);
   bool (*check_keyboard)(struct game_t *game);

   void *real_base;
} __attribute__((aligned(16)));
typedef struct game_t game_t;
extern void set_board_size(game_t *game, int files, int ranks);
extern void remove_square(game_t *game, int square);
extern void place_flag(game_t *game, sides side, int square);
extern void playgame_init(void);
extern void playgame_shutdown(void);

/* Setters for virtual methods, so we can customise the engine output
 * easily
 */
extern void set_default_output_function(void (*func)(const char *, ...));
extern void set_uci_output_function(void (*func)(const char *, ...));
extern void set_xboard_output_function(void (*func)(const char *, ...));
extern void set_error_output_function(void (*func)(const char *, ...));
extern void set_transposition_table_size(game_t *game, uint64_t nelem);

extern game_t *create_game(void);
extern void start_new_game(game_t *game);
extern void end_game(game_t *game);

extern bool add_piece_type(game_t *game, move_flag_t move_flags, move_flag_t capture_flags, uint8_t piece_flags, ubitboard_t promotion_zone, const char *promotion_string, const char *name, const char *symbol, const char *notation);
extern void set_maximum_number_of_kings(game_t *game, sides side, int count);
extern void set_maximum_number_of_pieces(game_t *game, sides side, const char *symbol, int count);
extern void add_rule_flag(game_t *game, move_flag_t rule_flag);
extern void remove_rule_flag(game_t *game, move_flag_t rule_flag);
extern bool add_special_move(game_t *game, const char *symbol, ubitboard_t zone, move_flag_t move_flags);
extern bool set_piece_drop_zone(game_t *game, const char *symbol, ubitboard_t zone);
extern void set_player_drop_zone(game_t *game, sides side, ubitboard_t zone);
extern bool set_piece_value(game_t *game, const char *symbol, uint16_t value);
extern bool limit_piece_movement_region(game_t *game, const char *symbol, ubitboard_t region);
extern void load_piece_evaluation_terms(game_t *game);
extern bool set_evaluation_terms(game_t *game, const char *symbol, int advance, int shield, int centre);
extern void sort_piece_values(game_t *game);
extern void asses_mating_potential(game_t *game);
extern void asses_mating_pairs(game_t *game);
extern void print_piece_types(const game_t *game);
extern void identify_castle_partner(game_t *game);
extern void get_mobility_statistics(game_t *game);

extern int get_piece_index(const game_t *game, const char *symbol);

extern void setup_fen_position(game_t *game, const char *str);
extern char *make_fen_string(const game_t *game, char *buffer);
extern char *short_move_string(const game_t *game, const move_t move, char *buffer);

/**********************************************************************
 *                      static inline functions                       *
 **********************************************************************/
static inline void playmove(game_t *game, move_t move)
{
   game->hash_key[game->moves_played] = game->board.hash;
   if (game->board.large_board)
      game->init[game->moves_played].lbb = game->board.large_init;
   else
      game->init[game->moves_played].bb = game->board.init;
   game->ep[game->moves_played] = game->board.ep;
   game->ep_capture[game->moves_played] = game->board.ep_capture;
   game->did_castle[game->moves_played] = game->board.did_castle[game->board.side_to_move];

   if (!game->large_board)
      makemove(&game->board, move);
   else
      large_makemove(&game->board, move);

   update_attack_table(&game->board, move);

   game->move_list[game->moves_played] = move;

   game->repetition_hash_table[game->board.hash&0xFFFF]++;

   game->moves_played++;

   /* Advance the 50-move counter if this move was reversible, reset if it
    * wasn't.
    * FIXME: need to factor in moves by non-reversible pieces (pawns).
    */
   game->fifty_counter[game->moves_played] =
      is_irreversible_move(move) ? 0 : (game->fifty_counter[game->moves_played-1] + 1);

   /* Set en-passant square */
   game->board.ep = 0;
   if (move & MOVE_SET_ENPASSANT) {
      game->board.ep = (get_move_from(move) + get_move_to(move))/2;
      game->board.ep_capture = get_move_to(move);
   }
}

static inline void play_null_move(game_t *game)
{
   game->hash_key[game->moves_played] = game->board.hash;
   if (game->board.large_board)
      game->init[game->moves_played].lbb = game->board.large_init;
   else
      game->init[game->moves_played].bb = game->board.init;
   game->ep[game->moves_played] = game->board.ep;
   game->ep_capture[game->moves_played] = game->board.ep_capture;
   game->did_castle[game->moves_played] = game->board.did_castle[game->board.side_to_move];
   if (!game->large_board)
      makemove(&game->board, 0);
   else
      large_makemove(&game->board, 0);
   game->repetition_hash_table[game->board.hash&0xFFFF]++;

   game->move_list[game->moves_played] = 0;
   game->moves_played++;
   game->board.ep = 0;
}

static inline void takeback(game_t *game)
{
   game->moves_played--;
   game->repetition_hash_table[game->board.hash&0xFFFF]--;
   if (!game->large_board)
      unmakemove(&game->board, game->move_list[game->moves_played]);
   else
      large_unmakemove(&game->board, game->move_list[game->moves_played]);
   if (game->board.large_board)
      game->board.large_init = game->init[game->moves_played].lbb;
   else
      game->board.init = game->init[game->moves_played].bb;

   update_attack_table(&game->board, game->move_list[game->moves_played]);

   game->board.ep = game->ep[game->moves_played];
   game->board.ep_capture = game->ep_capture[game->moves_played];
   game->board.did_castle[game->board.side_to_move] = game->did_castle[game->moves_played];
}

/* Redo the last move: if a move has been reverted, but no new move has
 * been played yet.
 * Mainly for player interaction.
 */
static inline void redo_last_move(game_t *game)
{
   if (game->moves_played < game->last_move) {
      move_t move = game->move_list[game->moves_played];

      game->hash_key[game->moves_played] = game->board.hash;
      if (game->board.large_board)
         game->init[game->moves_played].lbb = game->board.large_init;
      else
         game->init[game->moves_played].bb = game->board.init;
      game->ep[game->moves_played] = game->board.ep;
      game->ep_capture[game->moves_played] = game->board.ep_capture;
      if (!game->large_board)
         makemove(&game->board, move);
      else
         large_makemove(&game->board, move);

      game->repetition_hash_table[game->board.hash&0xFFFF]++;

      game->moves_played++;

      game->fifty_counter[game->moves_played] =
         is_irreversible_move(move) ? 0 : (game->fifty_counter[game->moves_played-1] + 1);

      /* Set en-passant square */
      game->board.ep = 0;
      if (move & MOVE_SET_ENPASSANT) {
         game->board.ep = (get_move_from(move) + get_move_to(move))/2;
         game->board.ep_capture = get_move_to(move);
      }
   }
}


static inline bool player_in_check_for_board(game_t *game, board_t *board, sides side)
{
#if !defined USE_ATTACK_TABLE
   if (!game->large_board) {
      bitboard_t royal = board->royal & board->bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (!royal) return false;

      bitboard_t mask             = get_super_attacks_for_squares(royal);
      bitboard_t attacked_squares = generate_attack_bitboard(board, 0, mask, next_side[side]);
      return (attacked_squares & royal) == royal;
   } else {
      large_bitboard_t royal = board->large_royal & board->large_bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (is_zero128(royal)) return false;

      large_bitboard_t mask             = get_large_super_attacks_for_squares(royal);
      large_bitboard_t attacked_squares = generate_large_attack_bitboard(board, large_board_empty, mask, next_side[side]);
      if (expect(board->rule_flags & RF_KING_TABOO, false)) {
         large_bitboard_t other_king = board->large_royal & board->large_bbc[next_side[side]] & mask;
         if (!is_zero128(other_king)) {
            int square = bitscan128(other_king);
            large_bitboard_t occ = board->large_bbc[WHITE]|board->large_bbc[BLACK];

            attacked_squares |= get_large_file_attacks(large_vmslider, occ, square);
         }
      }
      return is_equal128(attacked_squares & royal, royal);
   }
#else
   if (!game->large_board) {
      bitboard_t royal = board->royal & board->bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (!royal) return false;

      return (board->attacks[next_side[side]] & royal) == royal;
   } else {
      large_bitboard_t royal = board->large_royal & board->large_bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (is_zero128(royal)) return false;

      return is_equal128(board->large_attacks[next_side[side]] & royal, royal);
   }
#endif
   return false;
}



static inline bool player_in_check(game_t *game, sides side)
{
#if !defined USE_ATTACK_TABLE
   if (!game->large_board) {
      bitboard_t royal = game->board.royal & game->board.bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (!royal) return false;

      bitboard_t mask             = get_super_attacks_for_squares(royal);
      bitboard_t attacked_squares = generate_attack_bitboard(&game->board, 0, mask, next_side[side]);
      return (attacked_squares & royal) == royal;
   } else {
      large_bitboard_t royal = game->board.large_royal & game->board.large_bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (is_zero128(royal)) return false;

      large_bitboard_t mask             = get_large_super_attacks_for_squares(royal);
      large_bitboard_t attacked_squares = generate_large_attack_bitboard(&game->board, large_board_empty, mask, next_side[side]);
      if (expect(game->board.rule_flags & RF_KING_TABOO, false)) {
         large_bitboard_t other_king = game->board.large_royal & game->board.large_bbc[next_side[side]] & mask;
         if (!is_zero128(other_king)) {
            int square = bitscan128(other_king);
            large_bitboard_t occ = game->board.large_bbc[WHITE]|game->board.large_bbc[BLACK];

            attacked_squares |= get_large_file_attacks(large_vmslider, occ, square);
         }
      }
      return is_equal128(attacked_squares & royal, royal);
   }
#else
   if (!game->large_board) {
      bitboard_t royal = game->board.royal & game->board.bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (!royal) return false;

      return (game->board.attacks[next_side[side]] & royal) == royal;
   } else {
      large_bitboard_t royal = game->board.large_royal & game->board.large_bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (is_zero128(royal)) return false;

      return is_equal128(game->board.large_attacks[next_side[side]] & royal, royal);
   }
#endif
   return false;
}

/* Test whether a move left the player in check, assuming he was not in check before. */
static inline bool player_in_check_after_move(game_t *game, move_t move, sides side)
{
   if (!game->large_board) {
      bitboard_t royal = game->board.royal & game->board.bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (!royal) return false;

      /* If the last move did not affect any squares in the superpiece attack set, then it cannot have been a
       * checking move.
       */
      bitboard_t mask      = get_super_attacks_for_squares(royal);
      bitboard_t move_mask = get_move_mask(move);
      if ((mask & move_mask) == 0) return false;

      /* Generate attack bitboard */
      bitboard_t attacked_squares = generate_attack_bitboard(&game->board, 0, mask, next_side[side]);
      return (attacked_squares & royal) == royal;
   } else {
      large_bitboard_t royal = game->board.large_royal & game->board.large_bbc[side];

      /* If there are no royal pieces, then there is no check */
      if (is_zero128(royal)) return false;

      /* If the last move did not affect any squares in the superpiece attack set, then it cannot have been a
       * checking move.
       */
      large_bitboard_t mask      = get_large_super_attacks_for_squares(royal);
      large_bitboard_t move_mask = get_move_large_mask(move);
      if (is_zero128(mask & move_mask)) return false;

      /* Generate attack bitboard */
      large_bitboard_t attacked_squares = generate_large_attack_bitboard(&game->board, large_board_empty, mask, next_side[side]);
      return is_equal128(attacked_squares & royal, royal);
   }
}



/************************************* 
 * Propagate the principle variation *
 *************************************/
static inline void backup_principle_variation(game_t *game, int depth, move_t move)
{
   int c;

   /* Store move */
   if (move) {
      game->length_of_variation[depth] = depth;
      game->principle_variation[depth][depth] = move;
      //printf("Store at %d %s", depth, move_string(move, NULL));
   }

   /* Copy tree below last stored move */
   if (game->length_of_variation[depth+1] > depth) {
      //printf(" [%d", game->length_of_variation[depth+1]);
      for (c=depth+1; c<game->length_of_variation[depth+1]; c++) {
         //printf(" %s", move_string(game->principle_variation[c][depth+1], NULL));
         game->principle_variation[c][depth] = game->principle_variation[c][depth+1];
      }
      game->length_of_variation[depth] = game->length_of_variation[depth+1];
      game->quiescence_depth_of_variation[depth] = game->quiescence_depth_of_variation[depth+1];
      game->hash_hit[depth] = game->hash_hit[depth+1];
   }

   //printf("\n");
}

static inline void truncate_principle_variation(game_t *game, int depth)
{
   /* Truncate the principle variation at this depth */
   game->length_of_variation[depth+1] = depth;
   game->hash_hit[depth+1] = false;
}



/************************************* 
 * Game termination tests            *
 *************************************/
static bool player_captured_flags(const game_t *game, sides side)
{
   if (!(game->board.rule_flags & RF_CAPTURE_THE_FLAG)) return false;

   if (game->large_board) {
      if (game->flag[side].bb) {
         large_bitboard_t bb = game->flag[side].lbb & game->board.large_bbc[side];
         if (!is_zero128(bb) && (game->board.rule_flags & RF_CAPTURE_ANY_FLAG)) return true;
         if (is_equal128(bb, game->flag[side].lbb)) return true;
      }
   } else {
      if (game->flag[side].bb) {
         bitboard_t bb = game->flag[side].bb & game->board.bbc[side];
         if (bb && (game->board.rule_flags & RF_CAPTURE_ANY_FLAG)) return true;
         if (bb == game->flag[side].bb) return true;
      }
   }

   return false;
}



/* Returns one of:
 *          -1: continue search, baring rule does not apply
 *           0: draw
 * CHECKMATE+1: win, the other side has a bare king and we don't.
 */
static inline int score_bare_kings(const game_t *game, sides side)
{
   if (!game->large_board) {
      /* If both sides have a bare king: draw */
      bitboard_t occ = game->board.bbc[WHITE] | game->board.bbc[BLACK];
      if (occ == game->board.royal && onebit64(game->board.bbc[WHITE]) && onebit64(game->board.bbc[BLACK]))
         return 0;

      /* If the *other* side has a bare king, then we win. For the side on move, we do nothing since they may
       * still bare our king, which we leave for the search to pickup.
       */
      sides other = next_side[side];
      if (expect(game->bare_king_score < 0, false) && (game->board.bbc[other] & game->board.royal) == game->board.bbc[other])
         return -game->bare_king_score+1;
   } else {
      /* If both sides have a bare king: draw */
      large_bitboard_t occ = game->board.large_bbc[WHITE] | game->board.large_bbc[BLACK];
      if (is_equal128(occ, game->board.large_royal))
         return 0;

      /* If the *other* side has a bare king, then we win. For the side on move, we do nothing since they may
       * still bare our king, which we leave for the search to pickup.
       */
      sides other = next_side[side];
      if (expect(game->bare_king_score, false) && is_equal128(game->board.large_bbc[other] & game->board.large_royal, game->board.large_bbc[other]))
         return -game->bare_king_score+1;
   }
   return -1;
}

#endif
