/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef MOVEGEN_H
#define MOVEGEN_H

#include <stdint.h>
#include "bitboard.h"
#include "board.h"
#include "move.h"
#include "moveflags.h"

#define MAX_LEAPER_TYPES 16
#define MAX_STEPPER_TYPES 16

extern uint8_t leaper_description[MAX_LEAPER_TYPES]; // (n,m) encoded in two nibbles
extern bitboard_t leapers[MAX_LEAPER_TYPES][64];
extern large_bitboard_t large_leapers[MAX_LEAPER_TYPES][128];
extern int number_of_leapers;

extern uint32_t stepper_description[MAX_STEPPER_TYPES]; // 8 directions, with repeat counts (0-7) for each->24 bits
extern bitboard_t stepper_step[MAX_STEPPER_TYPES][64];  // Single step bitmasks
extern large_bitboard_t large_stepper_step[MAX_STEPPER_TYPES][128];  // Single step bitmasks
extern int number_of_steppers;
extern int step_shift[8];
extern int large_step_shift[8];

/* Slider and hopper moves, for one rank/file (need to shift) */
extern bitboard_t hmslider[8][256];    // Only need 6 bits of occ, but keep code simple and general
extern bitboard_t vmslider[8][256];    // Only need 6 bits of occ, but keep code simple and general
extern bitboard_t hmhopper[8][256];    // Need full 8 bits of occ
extern bitboard_t vmhopper[8][256];    // Need full 8 bits of occ

extern large_bitboard_t **large_hmslider;
extern large_bitboard_t **large_vmslider;
extern large_bitboard_t **large_hmhopper;
extern large_bitboard_t **large_vmhopper;

/* Slider and hopper captures */
extern bitboard_t hcslider[8][256];    // Only need 6 bits of occ, but keep code simple and general
extern bitboard_t vcslider[8][256];    // Only need 6 bits of occ, but keep code simple and general
extern bitboard_t hchopper[8][256];    // Need full 8 bits of occ
extern bitboard_t vchopper[8][256];    // Need full 8 bits of occ

extern large_bitboard_t **large_hcslider;
extern large_bitboard_t **large_vcslider;
extern large_bitboard_t **large_hchopper;
extern large_bitboard_t **large_vchopper;


extern bitboard_t super[64];
extern large_bitboard_t large_super[128];

extern bitboard_t king_zone[64];

/* Masks for stepper move generation */
static const bitboard_t step_mask[] = {
   0xFFFFFFFFFFFFFFFFll, 0x7F7F7F7F7F7F7F7Fll, 0x7F7F7F7F7F7F7F7Fll, 0x7F7F7F7F7F7F7F7Fll,
   0xFFFFFFFFFFFFFFFFll, 0xFEFEFEFEFEFEFEFEll, 0xFEFEFEFEFEFEFEFEll, 0xFEFEFEFEFEFEFEFEll
};
extern large_bitboard_t large_step_mask[8];

move_flag_t define_leaper(const char *movestr);
move_flag_t define_slider(const char *movestr);
move_flag_t define_stepper(const char *movestr);
move_flag_t define_piece_move(const char *movestr);

void reset_piece_definitions(void);
void initialise_slider_tables(void);
void initialise_super_tables(void);

void initialise_large_slider_tables(void);
move_flag_t set_leaper_mask(move_flag_t flags, bitboard_t mask);
move_flag_t set_large_leaper_mask(move_flag_t flags, large_bitboard_t mask);
void deduce_castle_flags(bool large_board, sides side, int king_from, int king_to, int rook_from);

bitboard_t get_leaper_moves(move_flag_t flags, bitboard_t occ, int square);
large_bitboard_t get_large_leaper_moves(move_flag_t flags, large_bitboard_t occ, int square);
bitboard_t get_aleaper_moves(move_flag_t flags, bitboard_t occ, int square, sides side);
large_bitboard_t get_large_aleaper_moves(move_flag_t flags, large_bitboard_t occ, int square, sides side);
bitboard_t get_stepper_move_bitboard(move_flag_t flags, bitboard_t occ, bitboard_t steppers);
bitboard_t get_stepper_move_span_bitboard(move_flag_t flags, bitboard_t occ, bitboard_t steppers);
bitboard_t get_stepper_attack_bitboard(move_flag_t flags, bitboard_t occ, bitboard_t steppers);
large_bitboard_t get_large_stepper_move_bitboard(move_flag_t flags, large_bitboard_t occ, large_bitboard_t steppers);
large_bitboard_t get_large_stepper_move_span_bitboard(move_flag_t flags, large_bitboard_t occ, large_bitboard_t steppers);
large_bitboard_t get_large_stepper_attack_bitboard(move_flag_t flags, large_bitboard_t occ, large_bitboard_t steppers);
void generate_moves(movelist_t *movelist, const board_t *board, sides side_to_move);
void generate_quiescence_moves(movelist_t *movelist, const board_t *board, sides side_to_move);
bitboard_t get_move_bitboard(const board_t *board, bitboard_t piece_mask, sides side_to_move);
bitboard_t get_attack_bitboard(const board_t *board, bitboard_t piece_mask, sides side_to_move);

large_bitboard_t get_large_attack_bitboard(const board_t *board, large_bitboard_t piece_mask, sides side_to_move);

int validate_move(const movelist_t *movelist, int from, int to);

/* TODO: special out-of-check move generators */

/**********************************************************************
 *                      static inline functions                       *
 **********************************************************************/
static inline bitboard_t get_file_attacks(bitboard_t table[8][256], bitboard_t occ, int square)
{
   int file = unpack_file(square);
   int rank = unpack_rank(square);
   uint8_t index = ( (board_afile & (occ >> file)) * 0x8040201008040201) >> 56;
   bitboard_t bb;

   bb = table[7-rank][index];
   return ((bb * board_a1h8_diagonal[7]) & board_hfile) >> (7-file);
}

static inline bitboard_t get_rank_attacks(bitboard_t table[8][256], bitboard_t occ, int square)
{
   int file = unpack_file(square);
   int rank = unpack_rank(square);
   uint8_t index = occ >> (8*rank);

   return table[file][index] << (8*rank);
}

static inline bitboard_t get_diagonal_attacks(bitboard_t table[8][256], bitboard_t occ, int square)
{
   int file = unpack_file(square);
   int diag_nr = get_a1h8_diagonal(square);
   bitboard_t mask = board_a1h8_diagonal[diag_nr];
   uint8_t index = ((occ & mask) * board_afile) >> 56;

   return (table[file][index] * board_afile) & mask;
}

static inline bitboard_t get_antidiagonal_attacks(bitboard_t table[8][256], bitboard_t occ, int square)
{
   int file = unpack_file(square);
   int diag_nr = get_a8h1_diagonal(square);
   bitboard_t mask = board_a8h1_diagonal[diag_nr];
   uint8_t index = ((occ & mask) * board_afile) >> 56;

   return (table[file][index] * board_afile) & mask;
}

static inline bitboard_t get_super_attacks_for_squares(bitboard_t squares)
{
   bitboard_t attacks = 0;

   while (squares) {
      int square = bitscan64(squares);
      squares ^= make_bitboard_square(square);
      attacks |= super[square];
   }

   return attacks;
}

static inline large_bitboard_t get_large_super_attacks_for_squares(large_bitboard_t squares)
{
   large_bitboard_t attacks = large_board_empty;

   while (!is_zero128(squares)) {
      int square = bitscan128(squares);
      squares ^= large_square_bitboards[square];
      attacks |= large_super[square];
   }

   return attacks;
}

static inline large_bitboard_t get_large_file_attacks(large_bitboard_t **table, large_bitboard_t occ, int square)
{
   int file = unpack_file(square);
   int rank = unpack_rank(square);
   uint16_t index = get_large_bitboard_file(occ, file);

   return shl128(table[rank][index], file);
}

static inline large_bitboard_t get_large_rank_attacks(large_bitboard_t **table, large_bitboard_t occ, int square)
{
   int file = unpack_file(square);
   int rank = unpack_rank(square);
   uint16_t index = get_large_bitboard_row(occ, rank);

   return shl128(table[file][index], (large_board_files*rank));
}

static inline large_bitboard_t large_fill_north(large_bitboard_t bb)
{
   int n = 1;

   while (n < large_board_ranks) {
      bb |= shl128(bb, n*large_board_files);
      n *= 2;
   }

   return bb;
}

static inline large_bitboard_t get_large_diagonal_attacks(large_bitboard_t **table, large_bitboard_t occ, int square)
{
   assert(table);
   int file = unpack_file(square);
   int diag_nr = large_diagonal_nr[square];
   large_bitboard_t mask = large_board_diagonal[diag_nr];
#if HAVE_UINT128_T
   uint16_t index = shr128(mul128(occ & mask, large_board_west_edge), large_board_files * (large_board_ranks-1));
   index &= large_row_mask;

   return mul128(table[file][index], large_board_west_edge) & mask;
#else
   large_bitboard_t ib = shr128(large_fill_north(occ & mask), large_board_files * (large_board_ranks-1));
   uint16_t index = get_large_bitboard_row(ib, 0);
   index &= large_row_mask;

   return large_fill_north(table[file][index]) & mask;
#endif
}

static inline large_bitboard_t get_large_antidiagonal_attacks(large_bitboard_t **table, large_bitboard_t occ, int square)
{
   assert(table);
   int file = unpack_file(square);
   int diag_nr = large_anti_diagonal_nr[square];
   large_bitboard_t mask = large_board_antidiagonal[diag_nr];
#if HAVE_UINT128_T
   uint16_t index = shr128(mul128(occ & mask, large_board_west_edge), large_board_files * (large_board_ranks-1));
   index &= large_row_mask;

   return mul128(table[file][index], large_board_west_edge) & mask;
#else
   large_bitboard_t ib = shr128(large_fill_north(occ & mask), large_board_files * (large_board_ranks-1));
   uint16_t index = get_large_bitboard_row(ib, 0);
   index &= large_row_mask;

   return large_fill_north(table[file][index]) & mask;
#endif
}




/* Generate an attack bitboard for all attackers within a specified mask */
static inline bitboard_t generate_attack_bitboard(const board_t *board, bitboard_t test_squares, bitboard_t source_mask, sides side_to_move)
{
   /* Duplicate step-shift; this seems to be faster... */
   static int step_shift[8] =           {  8,  9,  1,  -7,  -8,  -9,  -1,  7 };
   piece_description_t *piece_types;
   move_flag_t *piece_capture_flags;
   move_flag_t *piece_move_flags;
   bitboard_t own, enemy, own_movers;
   bitboard_t occupied;
   bitboard_t attacked;
   int n;

   piece_types = board->piece_types;
   piece_capture_flags = piece_types->piece_capture_flags;
   piece_move_flags = piece_types->piece_move_flags;

   /* Bookkeeping: we keep a pointer to the next move in the move list, and
    * update the number of moves in the list at the end of this function
    */
   own = board->bbc[side_to_move];
   enemy = board->bbc[next_side[side_to_move]];

   /* FIXME: doesn't take into account more than two sides yet... */
   occupied = or_bitboards(own, enemy) | test_squares;

   own_movers = own & source_mask;

   attacked = 0;

   bitboard_t possible_attackers = own_movers;

   while (possible_attackers) {
      int first_attacker = bitscan64(possible_attackers);
      n = board->piece[first_attacker];
      possible_attackers &= ~board->bbp[n];

      bitboard_t bb = own_movers & board->bbp[n];

      /* Steppers */
      if (bb && piece_capture_flags[n] & MF_STEPPER) {
         int si = (piece_capture_flags[n] & MF_STEPPER) >> 8;
         int d;
         for (d=0; d<8; d++) {
            int c = (stepper_description[si] >> (d*3)) & 7;
            bitboard_t captures = bb;
            /* We have a repetition count, so we do a number of steps one after the other.
             * This can effectively duplicate a slider.
             */
            for ( ; c>0; c--) {
               //printf("%d %d [%d] %d\n", n, d, step_shift[d], c);
               captures &= step_mask[d];
               captures = sshift64(captures, step_shift[d]);
               attacked |= captures;
               captures &= ~occupied;
            }
         }
      }

      /* Sliders and leapers */
      if (piece_capture_flags[n] & MF_HOPSLIDELEAP)
      while (bb) {
         int from = bitscan64(bb);
         bb ^= make_bitboard_square(from);

         if (piece_capture_flags[n] & MF_SLIDER) {
            /* Get all regular captures */
            if (piece_capture_flags[n] & MF_SLIDER_H) attacked |= get_rank_attacks(hcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_V) attacked |= get_file_attacks(hcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_D) attacked |= get_diagonal_attacks(hcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_A) attacked |= get_antidiagonal_attacks(hcslider, occupied, from);
         }

         if (piece_capture_flags[n] & MF_HOPPER) {
            /* Get all regular captures */
            if (piece_capture_flags[n] & MF_HOPPER_H) attacked |= get_rank_attacks(hchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_V) attacked |= get_file_attacks(hchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_D) attacked |= get_diagonal_attacks(hchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_A) attacked |= get_antidiagonal_attacks(hchopper, occupied, from);
         }

         if (piece_capture_flags[n] & MF_IS_LEAPER) {
            if (piece_capture_flags[n] & MF_LEAPER_ASYMM)
               attacked |= get_aleaper_moves(piece_move_flags[n], occupied, from, side_to_move);
            else
               attacked |= get_leaper_moves(piece_move_flags[n], occupied, from);
         }
      }
   }

   return attacked;
}




/* Generate an attack bitboard for all attackers within a specified mask */
static inline large_bitboard_t generate_large_attack_bitboard(const board_t *board, large_bitboard_t test_squares, large_bitboard_t source_mask, sides side_to_move)
{
   /* Duplicate step-shift; this seems to be faster... */
   piece_description_t *piece_types;
   move_flag_t *piece_capture_flags;
   move_flag_t *piece_move_flags;
   large_bitboard_t own, enemy, own_movers;
   large_bitboard_t occupied;
   large_bitboard_t attacked;
   int n;

   piece_types = board->piece_types;
   piece_capture_flags = piece_types->piece_capture_flags;
   piece_move_flags = piece_types->piece_move_flags;

   /* Bookkeeping: we keep a pointer to the next move in the move list, and
    * update the number of moves in the list at the end of this function
    */
   own = board->large_bbc[side_to_move];
   enemy = board->large_bbc[next_side[side_to_move]];

   occupied = own | enemy | test_squares;

   own_movers = own & source_mask;

   attacked = large_board_empty;

   large_bitboard_t possible_attackers = own_movers;

   while (!is_zero128(possible_attackers)) {
      int first_attacker = bitscan128(possible_attackers);
      n = board->piece[first_attacker];
      possible_attackers &= ~board->large_bbp[n];

      large_bitboard_t bb = own_movers & board->large_bbp[n];

      /* Steppers */
      if (!is_zero128(bb) && piece_capture_flags[n] & MF_STEPPER) {
         int si = (piece_capture_flags[n] & MF_STEPPER) >> 8;
         int d;
         for (d=0; d<8; d++) {
            int c = (stepper_description[si] >> (d*3)) & 7;
            large_bitboard_t captures = bb;
            /* We have a repetition count, so we do a number of steps one after the other.
             * This can effectively duplicate a slider.
             */
            for ( ; c>0; c--) {
               captures &= large_step_mask[d];
               captures = sshift128(captures, large_step_shift[d]);
               attacked |= captures;
               captures &= ~occupied;
            }
         }
      }

      /* Sliders and leapers */
      if (piece_capture_flags[n] & MF_HOPSLIDELEAP)
      while (!is_zero128(bb)) {
         int from = bitscan128(bb);
         bb ^= large_square_bitboards[from];

         /* Get all regular captures */
         if (piece_capture_flags[n] & MF_SLIDER) {
            if (piece_capture_flags[n] & MF_SLIDER_H) attacked |= get_large_rank_attacks(large_hcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_V) attacked |= get_large_file_attacks(large_vcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_D) attacked |= get_large_diagonal_attacks(large_hcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_A) attacked |= get_large_antidiagonal_attacks(large_hcslider, occupied, from);
         }

         if (piece_capture_flags[n] & MF_HOPPER) {
            if (piece_capture_flags[n] & MF_HOPPER_H) attacked |= get_large_rank_attacks(large_hchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_V) attacked |= get_large_file_attacks(large_vchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_D) attacked |= get_large_diagonal_attacks(large_hchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_A) attacked |= get_large_antidiagonal_attacks(large_hchopper, occupied, from);
         }

         if (piece_capture_flags[n] & MF_IS_LEAPER) {
            if (piece_capture_flags[n] & MF_LEAPER_ASYMM)
               attacked |= get_large_aleaper_moves(piece_move_flags[n], occupied, from, side_to_move);
            else
               attacked |= get_large_leaper_moves(piece_move_flags[n], occupied, from);
         }
      }
   }

   return attacked;
}

static inline large_bitboard_t get_all_large_attackers(const board_t *board, int square)
{
   /* Duplicate step-shift; this seems to be faster... */
   piece_description_t *piece_types;
   move_flag_t *piece_capture_flags;
   move_flag_t *piece_move_flags;
   large_bitboard_t own, enemy, own_movers;
   large_bitboard_t occupied;
   large_bitboard_t attacked;
   large_bitboard_t attacker;
   int n;

   assert(board->large_board);

   piece_types = board->piece_types;
   piece_capture_flags = piece_types->piece_capture_flags;
   piece_move_flags = piece_types->piece_move_flags;

   attacked = large_board_empty;
   attacker = large_board_empty;

   occupied = board->large_bbc[WHITE] | board->large_bbc[BLACK] | large_square_bitboards[square];
   large_bitboard_t possible_attackers = (board->large_bbc[WHITE] | board->large_bbc[BLACK]) & large_super[square];

   while (!is_zero128(possible_attackers)) {
      int first_attacker = bitscan128(possible_attackers);
      n = board->piece[first_attacker];
      possible_attackers &= ~board->large_bbp[n];

      large_bitboard_t bb = board->large_bbp[n];

      /* Steppers */
      if (!is_zero128(bb) && piece_capture_flags[n] & MF_STEPPER) {
         int si = (piece_capture_flags[n] & MF_STEPPER) >> 8;
         int d;
         for (d=0; d<8; d++) {
            int c = (stepper_description[si] >> (d*3)) & 7;
            large_bitboard_t captures = bb;
            /* We have a repetition count, so we do a number of steps one after the other.
             * This can effectively duplicate a slider.
             */
            for ( ; c>0; c--) {
               captures &= large_step_mask[d];
               if (large_step_shift[d] > 0) {
                  attacked = shl128(captures, large_step_shift[d]);
                  if (!is_zero128(attacked & large_square_bitboards[square]))
                     attacker |= large_square_bitboards[square - large_step_shift[d]];
               } else {
                  attacked = shr128(captures, -large_step_shift[d]);
                  if (!is_zero128(attacked & large_square_bitboards[square]))
                     attacker |= large_square_bitboards[square + large_step_shift[d]];
               }
            }
         }
      }

      /* Sliders and leapers */
      if (piece_capture_flags[n] & MF_HOPSLIDELEAP)
      while (!is_zero128(bb)) {
         int from = bitscan128(bb);
         bb ^= large_square_bitboards[from];
         attacked = large_board_empty;

         /* Get all regular captures */
         if (piece_capture_flags[n] & MF_SLIDER) {
            if (piece_capture_flags[n] & MF_SLIDER_H) attacked |= get_large_rank_attacks(large_hcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_V) attacked |= get_large_file_attacks(large_vcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_D) attacked |= get_large_diagonal_attacks(large_hcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_A) attacked |= get_large_antidiagonal_attacks(large_hcslider, occupied, from);
         }

         if (piece_capture_flags[n] & MF_HOPPER) {
            if (piece_capture_flags[n] & MF_HOPPER_H) attacked |= get_large_rank_attacks(large_hchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_V) attacked |= get_large_file_attacks(large_vchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_D) attacked |= get_large_diagonal_attacks(large_hchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_A) attacked |= get_large_antidiagonal_attacks(large_hchopper, occupied, from);
         }

         if (piece_capture_flags[n] & MF_IS_LEAPER)
            attacked |= get_large_leaper_moves(piece_move_flags[n], occupied, from);

         if (!is_zero128(attacked & large_square_bitboards[square]))
            attacker |= large_square_bitboards[from];
      }
   }

   return attacker;
}

/* Generate an attack bitboard for all attackers within a specified mask */
static inline ubitboard_t get_all_attackers(const board_t *board, int square)
{
   /* Duplicate step-shift; this seems to be faster... */
   static int step_shift[8] =           {  8,  9,  1,  -7,  -8,  -9,  -1,  7 };
   piece_description_t *piece_types;
   move_flag_t *piece_capture_flags;
   move_flag_t *piece_move_flags;
   bitboard_t own, enemy, own_movers;
   bitboard_t occupied;
   bitboard_t attacked;
   bitboard_t attacker;
   int n;

   if (board->large_board) {
      ubitboard_t ret;
      ret.lbb = get_all_large_attackers(board, square);
      return ret;
   }

   piece_types = board->piece_types;
   piece_capture_flags = piece_types->piece_capture_flags;
   piece_move_flags = piece_types->piece_move_flags;

   attacked = 0;
   attacker = 0;

   occupied = board->bbc[WHITE] | board->bbc[BLACK] | make_bitboard_square(square);
   bitboard_t possible_attackers = (board->bbc[WHITE] | board->bbc[BLACK]) & super[square];

   while (possible_attackers) {
      int first_attacker = bitscan64(possible_attackers);
      n = board->piece[first_attacker];
      possible_attackers &= ~board->bbp[n];

      bitboard_t bb = board->bbp[n];

      /* Steppers */
      if (bb && piece_capture_flags[n] & MF_STEPPER) {
         int si = (piece_capture_flags[n] & MF_STEPPER) >> 8;
         int d;
         for (d=0; d<8; d++) {
            int c = (stepper_description[si] >> (d*3)) & 7;
            bitboard_t captures = bb;
            /* We have a repetition count, so we do a number of steps one after the other.
             * This can effectively duplicate a slider.
             */
            for ( ; c>0; c--) {
               captures &= step_mask[d];
               if (step_shift[d] > 0) {
                  attacked = captures << step_shift[d];
                  if (attacked & make_bitboard_square(square))
                     attacker |= make_bitboard_square(square - step_shift[d]);
               } else {
                  attacked = captures >> -step_shift[d];
                  if (attacked & make_bitboard_square(square))
                     attacker |= make_bitboard_square(square + step_shift[d]);
               }
            }
         }
      }

      /* Sliders and leapers */
      if (piece_capture_flags[n] & MF_HOPSLIDELEAP)
      while (bb) {
         int from = bitscan64(bb);
         bb ^= make_bitboard_square(from);
         attacked = 0;

         if (piece_capture_flags[n] & MF_SLIDER) {
            /* Get all regular captures */
            if (piece_capture_flags[n] & MF_SLIDER_H) attacked |= get_rank_attacks(hcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_V) attacked |= get_file_attacks(hcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_D) attacked |= get_diagonal_attacks(hcslider, occupied, from);
            if (piece_capture_flags[n] & MF_SLIDER_A) attacked |= get_antidiagonal_attacks(hcslider, occupied, from);
         }

         if (piece_capture_flags[n] & MF_HOPPER) {
            /* Get all regular captures */
            if (piece_capture_flags[n] & MF_HOPPER_H) attacked |= get_rank_attacks(hchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_V) attacked |= get_file_attacks(hchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_D) attacked |= get_diagonal_attacks(hchopper, occupied, from);
            if (piece_capture_flags[n] & MF_HOPPER_A) attacked |= get_antidiagonal_attacks(hchopper, occupied, from);
         }

         if (piece_capture_flags[n] & MF_IS_LEAPER)
            attacked |= get_leaper_moves(piece_move_flags[n], occupied, from);

         if (attacked & make_bitboard_square(square))
            attacker |= make_bitboard_square(from);
      }
   }

   ubitboard_t ret;
   ret.bb = attacker;
   return ret;
}

#endif
