/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef PIECES_H
#define PIECES_H

#include <stdint.h>
#include "bitboard.h"

#define MAX_PIECE_TYPES    16

/* Bitfield colours */
typedef enum sides { NONE=-1, WHITE, BLACK, NUM_SIDES } sides;
static const sides next_side[NUM_SIDES] = { BLACK, WHITE };

typedef uint32_t move_flag_t;

struct piece_description_t {
   /* Define the properties of all piece types.
    * We don't use a struct for a piece type so that we can keep properties of different pieces more closely
    * packed, which is better for cache performance.
    * The index in the arrays here corresponds to the index in the board piece list.
    *
    * Basic entries:
    *    move_flags           (32 bits)
    *    capture_flags        (32 bits)
    *    flags                (32 bits)
    *    name                 (string)
    *    abbreviation         (string)
    *    notation             (string)
    * For evaluation:
    *    piece_value          (16 bit)
    *    centre_weight        (8 bit)
    *    advance_weight       (8 bit)
    *    shield_weight        (8 bit)
    *    mobility_score       (8 bit)
    */
   union {
   bitboard_t        piece_promotion_zone[MAX_PIECE_TYPES];
   large_bitboard_t  piece_large_promotion_zone[MAX_PIECE_TYPES];
   };
   union {
   bitboard_t        piece_optional_promotion_zone[MAX_PIECE_TYPES];
   large_bitboard_t  piece_large_optional_promotion_zone[MAX_PIECE_TYPES];
   };
   union {
   bitboard_t        piece_special_zone[MAX_PIECE_TYPES];
   large_bitboard_t  piece_large_special_zone[MAX_PIECE_TYPES];
   };
   union {
   bitboard_t        piece_drop_zone[MAX_PIECE_TYPES];
   large_bitboard_t  piece_large_drop_zone[MAX_PIECE_TYPES];
   };

   union {
   bitboard_t        pawn_passer_mask[NUM_SIDES][128];
   large_bitboard_t  large_pawn_passer_mask[NUM_SIDES][128];
   };

   union {
   bitboard_t        pawn_front_span[NUM_SIDES][128];
   large_bitboard_t  large_pawn_front_span[NUM_SIDES][128];
   };

   union {
   bitboard_t        pawn_stop_square[NUM_SIDES][128];
   large_bitboard_t  large_pawn_stop_square[NUM_SIDES][128];
   };

   union {
   bitboard_t        prison[MAX_PIECE_TYPES];
   large_bitboard_t  large_prison[MAX_PIECE_TYPES];
   };

   union {
   bitboard_t        attack_from[MAX_PIECE_TYPES][128];
   large_bitboard_t  large_attack_from[MAX_PIECE_TYPES][128];
   };

   int         num_piece_types;
   int         pawn_index[NUM_SIDES]; /* The piece type that will be evaluated as a pawn */
   int         pawn_steps[NUM_SIDES]; /* The number of moves a pawn has, a factor in pawn mobility and passer eval */
   int         castle_piece[NUM_SIDES];/* The piece type that will castle with the king, nominally "rook" */
   move_flag_t piece_move_flags[MAX_PIECE_TYPES];
   move_flag_t piece_capture_flags[MAX_PIECE_TYPES];
   move_flag_t piece_special_move_flags[MAX_PIECE_TYPES];
   uint8_t     piece_flags[MAX_PIECE_TYPES];
   uint16_t    piece_promotion_choice[MAX_PIECE_TYPES];
   uint8_t     piece_maximum[MAX_PIECE_TYPES][NUM_SIDES];
   bool        pieces_can_win[MAX_PIECE_TYPES][MAX_PIECE_TYPES];

   int16_t     piece_value[MAX_PIECE_TYPES];
   int16_t     piece_promotion_value[MAX_PIECE_TYPES];
   int         centre_weight[MAX_PIECE_TYPES];
   int         advance_weight[MAX_PIECE_TYPES];
   int         promo_weight[MAX_PIECE_TYPES];
   int         shield_weight[MAX_PIECE_TYPES];
   int8_t      mobility_score[MAX_PIECE_TYPES][128];
   int         val_perm[MAX_PIECE_TYPES];

   int         king_safety_weight[MAX_PIECE_TYPES];
   int         max_moves[MAX_PIECE_TYPES];
   int         min_moves[MAX_PIECE_TYPES];
   int         avg_moves[MAX_PIECE_TYPES];

   char *      piece_name[MAX_PIECE_TYPES];
   char *      piece_abbreviation[MAX_PIECE_TYPES];
   char *      piece_notation[MAX_PIECE_TYPES];
} __attribute__((aligned(16)));
typedef struct piece_description_t piece_description_t;

/* Piece flags */
#define PF_ROYAL     0x01        /* Piece is royal */
#define PF_CASTLE    0x02        /* Piece may castle */
#define PF_SET_EP    0x04        /* Piece sets "en-passant" square when doing a special move */
#define PF_TAKE_EP   0x08        /* Piece may capture en-passant */

#define PF_NOMATE    0x20        /* This piece, by itself, cannot mate */
#define PF_PAIRBONUS 0x40        /* Piece gets a pair bonus for similar pieces on opposite colour */
#define PF_NORET     0x80        /* Piece cannot return to its original position */

/* For the generalised description of moves, we need to pack the piece type and the colour bit into one bit
 * field. This feels a bit like a hack...
 */
static inline uint8_t piece_for_side(int piece, sides side)
{
   return piece | (side << 4);
}

static inline int get_piece_value(const piece_description_t *pt, int id)
{
   return pt->piece_value[id];
}

#endif
