/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <ctype.h>
#include <allegro5/allegro.h>
#include <allegro5/allegro_image.h>
#include <allegro5/allegro_ttf.h>
#include <allegro5/allegro_native_dialog.h>
#include <allegro5/allegro_primitives.h>

#include "sjaak.h"

#define PROGRAM_NAME "Sjaak"

#define SQUARE_SIZE 80
#define BOARD_WIDTH  (num_files*SQUARE_SIZE)
#define BOARD_HEIGHT (num_ranks*SQUARE_SIZE)
#define BORDER 16
#define ITERATION_FONT_SIZE 12
#define TOP_MARGIN 16
#define BOTTOM_MARGIN 16

static ALLEGRO_DISPLAY *board_display;
static ALLEGRO_DISPLAY *key_display;
static ALLEGRO_BITMAP *figurine[MAX_PIECE_TYPES][NUM_SIDES];
static ALLEGRO_BITMAP *cursor;
static ALLEGRO_FONT *font;
static ALLEGRO_MOUSE_CURSOR *mouse_cursor = NULL;
static ALLEGRO_PATH *resource_path = NULL;

static int window_width = 0;
static int window_height = 0;

static int player_select = -1;
static int player_cursor = E4;

static int num_files = 8;
static int num_ranks = 8;

static int board_x = 0, board_y = TOP_MARGIN;
static int moves_x = 0, moves_y = TOP_MARGIN;
static int mouse_file = -1, mouse_row = -1;
static int mouse_pickup_square = 0;

static int drop_piece = -1;
static large_bitboard_t large_legal_moves;
static large_bitboard_t large_mouse_drop;
static bitboard_t legal_moves;
static bitboard_t mouse_drop;

static movelist_t movelist;

static bool move_in_list(int from, int to)
{
   int n;
   for (n=0; n<movelist.num_moves; n++)
      if (get_move_from(movelist.move[n]) == from && get_move_to(movelist.move[n]) == to)
         return true;

   return false;
}

static move_t move_from_list(int from, int to)
{
   int n;
   for (n=0; n<movelist.num_moves; n++) {
      if (get_move_from(movelist.move[n]) == from && get_move_to(movelist.move[n]) == to)
         return movelist.move[n];
   }

   return 0;
}

static void get_movelist(game_t *game)
{
   movelist_t ml;
   sides side = game->board.side_to_move;

   generate_moves(&ml, &game->board, side);

   /* Filter out moves that leave the king in check */
   int n, k;
   k = 0;
   for (n=0; n<ml.num_moves; n++) {
      move_t move = ml.move[n];
      playmove(game, move);
      if (!player_in_check(game, side)) {
         movelist.move[k] = move;
         k++;
      }
      takeback(game);
   }

   movelist.num_moves = k;
}

void generate_legal_move_bitboard(game_t *game, int file, int rank)
{
   legal_moves = board_empty;
   large_legal_moves = large_board_empty;
   int square = pack_row_file(rank, file);
   int piece = get_piece(&game->board, square);
   sides side = get_piece_colour(&game->board, square);
   if (side == game->board.side_to_move) {
      int n;
      for (n=0; n<movelist.num_moves; n++) {
         if (get_move_from(movelist.move[n]) == square) {
            if (game->large_board)
               large_legal_moves |= large_square_bitboards[get_move_to(movelist.move[n])];
            else
               legal_moves |= make_bitboard_square(get_move_to(movelist.move[n]));
         }
      }
   }
}


void draw_board(game_t *game)
{
   static char title[256];
   large_bitboard_t large_kings = large_board_empty;
   bitboard_t kings = board_empty;
   ALLEGRO_COLOR white = al_map_rgba_f(1, 1, 1, 1);
   ALLEGRO_COLOR black = al_map_rgba_f(0, 0, 0, 1);
   ALLEGRO_COLOR light = al_map_rgba_f(.925, .925, .875, 1);
   ALLEGRO_COLOR dark = al_map_rgba_f(.75, .55, .225, 1);
   ALLEGRO_COLOR bg = al_map_rgba_f(.325, .175, .1125, 1);
   ALLEGRO_COLOR select = al_map_rgba_f(0, .5, 0, .75);
   ALLEGRO_COLOR move_col = al_map_rgba_f(0, .5, 0, .6);
   ALLEGRO_COLOR check = al_map_rgba(220, 20, 60, 192);
   double score;
   int x, y;
   int size;

   al_set_target_backbuffer(board_display);
   al_clear_to_color(light);
   al_draw_filled_rectangle(board_x, board_y,
                            window_width, board_y + BOARD_HEIGHT + 2*BORDER, bg);
   al_draw_filled_rectangle(moves_x, moves_y,
                            moves_x + 199 - BORDER, moves_y + BOARD_HEIGHT, light);

   if (game->board.side_to_move == BLACK) {
      snprintf(title, 255, PROGRAM_NAME" (%s) - Black to play", game->name);
   } else {
      snprintf(title, 255, PROGRAM_NAME" (%s) - White to play", game->name);
   }
   al_set_window_title(board_display, title);

   size = SQUARE_SIZE;

   if (player_in_check(game, game->board.side_to_move)) {
      kings = game->board.royal & game->board.bbc[game->board.side_to_move];
      large_kings = game->board.large_royal & game->board.large_bbc[game->board.side_to_move];
   }

   bitboard_t occ = game->board.bbc[WHITE] | game->board.bbc[BLACK];
   large_bitboard_t large_occ = game->board.large_bbc[WHITE] | game->board.large_bbc[BLACK];
   for (y=0; y<num_ranks; y++) {
      for (x=0; x<num_files; x++) {
         int where = pack_row_file(y, x);
         int draw_x = board_x + x*size + BORDER;
         int draw_y = board_y + (num_ranks-1-y)*size + BORDER;
         bool blink_piece = false;

         al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, (x^y)&1 ? light : dark);

         /* Highlight cursor */
         if (player_cursor == where) {
            al_draw_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, white, 1);
         }

         bool occupied = false;
         if (game->large_board)
            occupied = !is_zero128(large_occ & large_square_bitboards[where]);
         else
            occupied = (occ & make_bitboard_square(where)) != 0;

         /* Piece */
         if (occupied) {
            ALLEGRO_COLOR piece_tint = white;
            int piece = get_piece(&game->board, where);
            int col = WHITE;
            if (game->large_board) {
               if (!is_zero128(game->board.large_bbc[BLACK] & large_square_bitboards[where]))
                  col = BLACK;
            } else {
               if (game->board.bbc[BLACK] & make_bitboard_square(where))
                  col = BLACK;
            }

            if(figurine[piece][col])
            al_draw_tinted_scaled_bitmap(figurine[piece][col],
               piece_tint, 0, 0,
               al_get_bitmap_width(figurine[piece][col]),
               al_get_bitmap_height(figurine[piece][col]),
               draw_x, draw_y, size, size, 0);
         }

         /* Highlight mouse cursor */
         if (x == mouse_file && y == mouse_row) {
            al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, select);
         }

         /* Special squares */
         if (game->large_board) {
            if (!is_zero128(large_mouse_drop & large_square_bitboards[where])) {
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, move_col);
            } else if (!is_zero128(large_legal_moves & large_square_bitboards[where])) {
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, move_col);
            }
            if (!is_zero128(large_kings & large_square_bitboards[where]))
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, check);
         } else {
            if (mouse_drop & make_bitboard_square(where)) {
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, move_col);
            } else if (legal_moves & make_bitboard_square(where)) {
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, move_col);
            }
            if (kings & make_bitboard_square(where))
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, check);
         }
      }
   }

   /* Draw coordinates along the board */
   for (x=0; x<num_files; x++) {
      int draw_x = BORDER + x*size + size/2;
      int draw_y = TOP_MARGIN + BORDER + 8*size;
      al_draw_textf(font, white, draw_x, draw_y, ALLEGRO_ALIGN_CENTRE, "%c", x+'A');
   }

   for (x=0; x<num_ranks; x++) {
      int draw_x = BORDER/2;
      int draw_y = TOP_MARGIN + (num_ranks-1 - x) * size + size/2 + ITERATION_FONT_SIZE;
      al_draw_textf(font, white, draw_x, draw_y, ALLEGRO_ALIGN_CENTRE, "%d", x+1);
   }

   if (game->name_white || game->name_black) {
      int width = BOARD_WIDTH + 2*BORDER;//al_get_display_width(board_display);
      al_draw_textf(font, white, width/2-16, TOP_MARGIN, ALLEGRO_ALIGN_RIGHT, "%s", game->name_white);
      al_draw_textf(font, white, width/2, TOP_MARGIN, ALLEGRO_ALIGN_CENTRE, "-");
      al_draw_textf(font, white, width/2+16, TOP_MARGIN, ALLEGRO_ALIGN_LEFT, "%s", game->name_black);
   }

   /* Draw move list */
   int first_move;
   int n;
   int nl, nm, lines_needed;
   int start, stop;

   /* Number of displayed moves */
   nl = BOARD_HEIGHT / 16 - 1;
   nm = 2 * nl;

   if (nl >= 2) {
      first_move = get_move_player(game->move_list[0])>>7;

      start = 0;
      stop = game->moves_played;
      lines_needed = (first_move + stop - start)/2;

      if ( lines_needed > nl ) {
         start = (first_move + stop - 2*nl) & ~0x1;
         start += 2;
      }

      for(n=start; n<stop; n++) {
         /* Odd plies should normally have been played by black - unless we
          * changed the player, or loaded a position which had "black to play"
          */
         int w = (n&1) ^ first_move;
         int dx = 0;

         x = moves_x + w * 80 + 10;
         y = moves_y + ((n+first_move - start)/2)*16;
         al_draw_textf(font, black, moves_x + 4, y, 0, "% 4d.", (n+first_move)/2+1);
         dx = 32;

         al_draw_textf(font, black, x+dx, y, 0, "%-6s", short_move_string(NULL, game->move_list[n], NULL));
      }
   }

   al_flip_display();
}

bool load_figurine(const game_t *game, const char *symbol, sides side, const char *filename)
{
   int index = get_piece_index(game, symbol);
   if (index >= 0) {
      figurine[index][side] = NULL;
      if (al_filename_exists(filename)) {
         figurine[index][side] = al_load_bitmap(filename);
      } else {
         ALLEGRO_PATH *path = al_create_path(filename);
         al_rebase_path(resource_path, path);
         figurine[index][side] = al_load_bitmap(al_path_cstr(path, ALLEGRO_NATIVE_PATH_SEP));
         al_destroy_path(path);
      }
      if (!figurine[index][side]) {
         return false;
         //al_show_native_message_box(NULL, "File not found", "", "Couldn't load figurine set", "Exit", ALLEGRO_MESSAGEBOX_ERROR);
         //exit(0);
      }
   }

   return true;
}

void reload_figurine_set(const game_t *game)
{
   int n;
   sides side;
   for (n=0; n<MAX_PIECE_TYPES; n++) {
      for (side = WHITE; side<NUM_SIDES; side++)
         if (figurine[n][side])
            al_destroy_bitmap(figurine[n][side]);
   }
   memset(figurine, 0, sizeof(figurine));

   /* Load bitmaps for all pieces */
   for (n=0; n<game->pt.num_piece_types; n++) {
      char filename[2048];
      char *s;
      //printf("%s\n", game->pt.piece_name[n]);

      /* Try generic filenames first */
      snprintf(filename, sizeof filename, "gfx/w%s.png", game->pt.piece_name[n]);
      for (s = filename; s[0]; s++) *s = tolower(*s);
      if (al_filename_exists(filename))
         figurine[n][WHITE] = al_load_bitmap(filename);

      if (!figurine[n][WHITE]) {
         char msg[4096];
         snprintf(msg, sizeof msg, "Couldn't load figurine file '%s'", filename);
         al_show_native_message_box(NULL, "File not found", "", msg, "Exit", ALLEGRO_MESSAGEBOX_ERROR);
         exit(0);
      }

      snprintf(filename, sizeof filename, "gfx/b%s.png", game->pt.piece_name[n]);
      for (s = filename; s[0]; s++) *s = tolower(*s);
      if (al_filename_exists(filename))
         figurine[n][BLACK] = al_load_bitmap(filename);

      if (!figurine[n][BLACK]) {
         char msg[4096];
         snprintf(msg, sizeof msg, "Couldn't load figurine file '%s'", filename);
         al_show_native_message_box(NULL, "File not found", "", msg, "Exit", ALLEGRO_MESSAGEBOX_ERROR);
         exit(0);
      }
   }

#if 0
   load_figurine(game, "K", WHITE, "gfx/wk.png");
   load_figurine(game, "Q", WHITE, "gfx/wq.png");
   load_figurine(game, "R", WHITE, "gfx/wr.png");
   load_figurine(game, "B", WHITE, "gfx/wb.png");
   load_figurine(game, "N", WHITE, "gfx/wn.png");
   load_figurine(game, "P", WHITE, "gfx/wp.png");
   load_figurine(game, "M", WHITE, "gfx/wm.png");
   load_figurine(game, "F", WHITE, "gfx/wf.png");
   load_figurine(game, "E", WHITE, "gfx/we.png");

   load_figurine(game, "A", WHITE, "gfx/warch.png");
   load_figurine(game, "C", WHITE, "gfx/wch.png");

   load_figurine(game, "k", BLACK, "gfx/bk.png");
   load_figurine(game, "q", BLACK, "gfx/bq.png");
   load_figurine(game, "r", BLACK, "gfx/br.png");
   load_figurine(game, "b", BLACK, "gfx/bb.png");
   load_figurine(game, "n", BLACK, "gfx/bn.png");
   load_figurine(game, "p", BLACK, "gfx/bp.png");
   load_figurine(game, "g", BLACK, "gfx/bj.png");
   load_figurine(game, "w", BLACK, "gfx/bg.png");
   load_figurine(game, "c", BLACK, "gfx/bd.png");
   load_figurine(game, "l", BLACK, "gfx/bl.png");
   load_figurine(game, "h", BLACK, "gfx/bh.png");
   load_figurine(game, "m", BLACK, "gfx/bm.png");
   load_figurine(game, "f", BLACK, "gfx/bf.png");
   load_figurine(game, "e", BLACK, "gfx/be.png");

   load_figurine(game, "a", BLACK, "gfx/barch.png");
   load_figurine(game, "c", BLACK, "gfx/bch.png");
#endif
}

void resize_window(void)
{
   int new_width, new_height;
   num_files = large_board_files;
   num_ranks = large_board_ranks;

   moves_x = BOARD_WIDTH + 2*BORDER + 1;
   moves_y = BORDER + TOP_MARGIN;
   new_width  = moves_x + 200;
   new_height = BOARD_HEIGHT + 2*BORDER + TOP_MARGIN + BOTTOM_MARGIN;

   if ((new_width != window_width) || (new_height != window_height))
      al_resize_display(board_display, new_width, new_height);

   window_width = new_width;
   window_height = new_height;
}

int main(void)
{
   game_t *game;
   sides computer_side = BLACK;

   construct_inverse_diagonal_maps();
   initialise_slider_tables();
   initialise_hash_keys();

   playgame_init();

   game = create_standard_game();
   //game = create_spartan_game();
   //game = create_maharaja_game();
   //game = create_knightmate_game();
   //game = create_losers_game();
   //game = create_shatranj_game();
   //game = create_makruk_game();
   //game = create_burmese_game();

   /* Print information about all piece types. */
   print_piece_types(game);

   /* Initialise Allegro */
   ALLEGRO_EVENT_QUEUE *queue;
   ALLEGRO_TIMER *display_timer;
   int n, x, y;
   bool redraw = true;
   bool done = false;

   if (!al_init()) {
      fprintf(stderr, "Couldn't initialise Allegro\n");
      return 1;
   }
   al_install_mouse();
   al_install_keyboard();
   al_init_primitives_addon();
   al_init_image_addon();
   al_init_font_addon();
   al_init_ttf_addon();

   al_set_new_display_option(ALLEGRO_SAMPLE_BUFFERS, 1, ALLEGRO_SUGGEST);
   al_set_new_display_option(ALLEGRO_SAMPLES, 4, ALLEGRO_SUGGEST);

   /* Main (board) display */
   moves_x = BOARD_WIDTH + 2*BORDER + 1;
   moves_y = BORDER + TOP_MARGIN;
   window_width  = moves_x + 200;
   window_height = BOARD_HEIGHT + 2*BORDER + TOP_MARGIN + BOTTOM_MARGIN;

   board_display = al_create_display(window_width, window_height);
   al_set_window_title(board_display, PROGRAM_NAME);
   key_display = board_display;

   resource_path = al_get_standard_path(ALLEGRO_RESOURCES_PATH);

   /* Load bitmaps */
   reload_figurine_set(game);
   const char *fontname = "gfx/DejaVuSansMono.ttf";
   if (al_filename_exists(fontname)) {
      font = al_load_font(fontname, ITERATION_FONT_SIZE, ALLEGRO_TTF_NO_KERNING);
   } else {
      ALLEGRO_PATH *path = al_create_path(fontname);
      al_rebase_path(resource_path, path);
      font = al_load_font(al_path_cstr(path, ALLEGRO_NATIVE_PATH_SEP), ITERATION_FONT_SIZE, ALLEGRO_TTF_NO_KERNING);
      al_destroy_path(path);
   }
   if (!font) {
      al_show_native_message_box(NULL, "File not found", "", "Couldn't load font", "Exit", ALLEGRO_MESSAGEBOX_ERROR);
      exit(0);
   }
   cursor = al_create_bitmap(SQUARE_SIZE, SQUARE_SIZE);

   /* Start a new game */
   start_new_game(game);
   //setup_fen_position(game, "lgkcckwl/hhhhhhhh/8/8/8/8/PPPPPPPP/RNBQKBNR/ w KQ -");
   //setup_fen_position(game, "lgkcckwl/hhhh1hhh/3h4/8/3PP3/8/PPP2PPP/RNBQKBNR/ b KQ -");
   //setup_fen_position(game, "2k2k2/7Q/3K4/8/8/8/R7/8 w - - 0 1");
   //setup_fen_position(game, "8/8/8/8/8/8/R7/R3K2k w Q - 0 1");
   //setup_fen_position(game, "r1b2k1r/p4pbp/nq1p1Q2/2pP4/1p2R2N/6P1/P3PPBP/4K2R/ w - -");
   //setup_fen_position(game, "8/8/Bk6/8/1l1h4/1PR2P1P/hg6/4K3 b - - 0 66");
   //setup_fen_position(game, "8/8/Bk6/8/1l1h4/1PR2P1P/1g6/1g2K3 w - - 0 66");
   //setup_fen_position(game, "1R2l1k1/4g1h1/5R2/8/2P1h3/1P4Q1/P2h2PP/7K b - - 0 46");

   display_timer = al_create_timer(1.0 / 40.0);
   queue = al_create_event_queue();
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)al_get_mouse_event_source());
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)al_get_keyboard_event_source());
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)board_display);
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)display_timer);
   al_start_timer(display_timer);

   /* Initialise random number generator.
    * We add a random number to the move score at the root for the first few plies, so we don't play the same
    * game all the time.
    */
   sgenrand(time(NULL));

   /* Generate all pseudo-legal moves */
   get_movelist(game);

   /* Show position */
   print_bitboards(&game->board);

   bool autoplay = false;
   int time_per_move = 1100;
   while (!done) {
      ALLEGRO_EVENT event;

      if (redraw && al_event_queue_is_empty(queue)) {
         draw_board(game);
         redraw = false;
      }

      if (autoplay && al_event_queue_is_empty(queue)) {
         player_select = -1;
         redraw = true;

         set_time_per_move(game, time_per_move);
         autoplay = computer_play(game, 60);
         game->last_move = game->moves_played;
         get_movelist(game);
      }

      if (game->board.side_to_move == computer_side && al_event_queue_is_empty(queue)) {
         player_select = -1;
         redraw = true;

         set_time_per_move(game, time_per_move);
         computer_play(game, 60);
         game->last_move = game->moves_played;
         get_movelist(game);

         /* Legal moves for the highlighted piece */
         generate_legal_move_bitboard(game, mouse_file, mouse_row);
      }

      al_wait_for_event(queue, &event);
      switch (event.type) {
         case ALLEGRO_EVENT_DISPLAY_EXPOSE:
            redraw = true;
            break;

         case ALLEGRO_EVENT_DISPLAY_CLOSE:
            done = true;
            break;

         case ALLEGRO_EVENT_DISPLAY_SWITCH_IN:
            key_display = event.display.source;
            break;

         case ALLEGRO_EVENT_DISPLAY_SWITCH_OUT:
            key_display = NULL;
            break;

         case ALLEGRO_EVENT_MOUSE_AXES:
            if (event.mouse.x > BORDER &&
                event.mouse.x < BORDER+BOARD_WIDTH &&
                event.mouse.y > BORDER &&
                event.mouse.y < BORDER+BOARD_WIDTH) {
               int file_size = SQUARE_SIZE;
               int row_size = SQUARE_SIZE;
               int new_mouse_file = (event.mouse.x - BORDER) / file_size;
               int new_mouse_row = (BOARD_HEIGHT - (event.mouse.y - BORDER)) / row_size;
               if (new_mouse_file != mouse_file || new_mouse_row != mouse_row) {
                  mouse_file = new_mouse_file;
                  mouse_row = new_mouse_row;
                  redraw = true;

                  /* Legal moves for the highlighted piece */
                  generate_legal_move_bitboard(game, mouse_file, mouse_row);
               }
            } else {
               mouse_file = mouse_row = -1;
               legal_moves = board_empty;
               large_legal_moves = large_board_empty;
               redraw = true;
            }
            break;

         case ALLEGRO_EVENT_MOUSE_BUTTON_UP:
            if (key_display == board_display) {
               al_set_system_mouse_cursor(board_display, ALLEGRO_SYSTEM_MOUSE_CURSOR_DEFAULT);
               if (mouse_file != -1 && mouse_row != -1) {
                  int square = pack_row_file(mouse_row, mouse_file);
                  bool valid_drop = 0;
                  if (game->large_board)
                     valid_drop = !is_zero128(large_mouse_drop & large_square_bitboards[square]);
                  else
                     valid_drop = mouse_drop & make_bitboard_square(square);
                  if (valid_drop) {
                     move_t move = move_from_list(mouse_pickup_square, square);
                     if (move == 0)
                        break;
                     sides side = game->board.side_to_move;
                     playmove(game, move);
                     if (player_in_check(game, side)) {
                        takeback(game);
                        get_movelist(game);
                     }

                     /* Legal moves for the highlighted piece */
                     generate_legal_move_bitboard(game, mouse_file, mouse_row);
                  }

                  redraw = true;
                  mouse_drop = board_empty;
                  large_mouse_drop = large_board_empty;
               }

            }
            break;
         case ALLEGRO_EVENT_MOUSE_BUTTON_DOWN:
            if (mouse_file != -1 && mouse_row != -1) {
               int square = pack_row_file(mouse_row, mouse_file);
               int piece = get_piece(&game->board, square);
               sides side = get_piece_colour(&game->board, square);

               if (side != NONE) {
                  ALLEGRO_STATE state;
                  int size = SQUARE_SIZE;
                  mouse_pickup_square = square;
                  al_store_state(&state, ALLEGRO_STATE_TARGET_BITMAP | ALLEGRO_STATE_BLENDER);

                  al_set_mouse_cursor(board_display, NULL);
                  al_destroy_mouse_cursor(mouse_cursor);
                  al_set_target_bitmap(cursor);
                  al_clear_to_color(al_map_rgba(0,0,0,0));
                  al_draw_scaled_bitmap(figurine[piece][side],
                        0, 0,
                        al_get_bitmap_width(figurine[piece][side]),
                        al_get_bitmap_height(figurine[piece][side]),
                        0, 0, size, size, 0);
                  mouse_cursor = al_create_mouse_cursor(cursor, size/2, size/2);
                  al_set_mouse_cursor(board_display, mouse_cursor);

                  al_restore_state(&state);
                  mouse_drop = legal_moves;
                  large_mouse_drop = large_legal_moves;
               }
               redraw = true;
            }
            break;
         
         case ALLEGRO_EVENT_KEY_DOWN:
         case ALLEGRO_EVENT_KEY_CHAR:
            switch (event.keyboard.keycode) {
               case ALLEGRO_KEY_ESCAPE:
                  done = true;
                  break;
               case ALLEGRO_KEY_F1:
                  end_game(game);
                  game = create_standard_game();
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F2:
                  end_game(game);
                  game = create_spartan_game();
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F3:
                  end_game(game);
                  game = create_maharaja_game();
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F4:
                  end_game(game);
                  game = create_knightmate_game();
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F5:
                  end_game(game);
                  game = create_shatranj_game();
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F6:
                  end_game(game);
                  game = create_makruk_game();
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F7:
                  end_game(game);
                  game = create_capablanca_game();
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F8:
                  end_game(game);
                  game = create_courier_game();
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_DELETE:         /* New game */
                  /* FIXME: this leaks memory, because the internal
                   * pointers in the game struct are not freed, just
                   * overwritten.
                   */
                  start_new_game(game);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;

               case ALLEGRO_KEY_N:
                  break;

               case ALLEGRO_KEY_P:
                  break;

               case ALLEGRO_KEY_FULLSTOP:       /* Switch figurine set */
                  break;
               case ALLEGRO_KEY_COMMA:
                  break;
               case ALLEGRO_KEY_LEFT:
                  if ((player_cursor&7) != 0) player_cursor--;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_RIGHT:
                  if ((player_cursor&7) != 7) player_cursor++;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_UP:
                  if (player_cursor < 56) player_cursor += 8;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_DOWN:
                  if (player_cursor>7) player_cursor -= 8;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_ENTER:
                  autoplay = !autoplay;
                  break;
               case ALLEGRO_KEY_TAB:
                  computer_side = game->board.side_to_move;
                  redo_last_move(game);
                  get_movelist(game);
                  redraw = true;
                  break;
               case ALLEGRO_KEY_HOME:
                  while (game->moves_played)
                     takeback(game);
                  get_movelist(game);
                  redraw = true;
                  break;
               case ALLEGRO_KEY_BACKSPACE:
                  computer_side = game->board.side_to_move;
                  takeback(game);
                  get_movelist(game);
                  redraw = true;
                  break;
               case ALLEGRO_KEY_SPACE:
                  //autoplay = !autoplay;
                  computer_side = game->board.side_to_move;
                  break;
            }
            break;
         case ALLEGRO_EVENT_TIMER:
            if (player_select > -1) redraw = true;
            break;
      }

   }

   end_game(game);

   return 0;
}
