/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdlib.h>
#include <stdint.h>
#include "eval_table.h"

#define HASH_BUCKETS    3

static inline size_t map_key_to_index(uint64_t key, size_t nelem)
{
   return key & (nelem - 1);
}

eval_hash_table_t *create_eval_hash_table(size_t nelem)
{
   eval_hash_table_t *table = calloc(1, sizeof *table);

   table->number_of_elements = nelem;
   table->data = calloc(nelem + HASH_BUCKETS, sizeof *table->data);
   return table; 
}

void destroy_eval_hash_table(eval_hash_table_t *table)
{
   if (table) {
      free(table->data);
      free(table);
   }
}

eval_table_entry_t *query_eval_table_entry(eval_hash_table_t *table, uint64_t key)
{
   size_t index;

   if (!table)
      return NULL;

   /* Map the key onto the array index, check if entry is there */
   index = map_key_to_index(key, table->number_of_elements);

   size_t b = 0;
   for (b = 0; b<HASH_BUCKETS; b++) {
      if (table->data[index+b].key == key) {
         table->data[index+b].generation = table->generation;
         return &table->data[index+b];
      }
   }

   return NULL;
}

void store_eval_table_entry(eval_hash_table_t *table, uint64_t key, int score)
{
   eval_table_entry_t *data;
   eval_table_entry_t *worst_data = NULL;
   size_t index, b;

   if (!table)
      return;

   /* Map the key onto the array index, check if entry is there */
   index = map_key_to_index(key, table->number_of_elements);
   worst_data = data = table->data + index;

   /* Check all buckets */
   for (b=1; b<HASH_BUCKETS; b++) {
      if (data[b].key == key) {
         worst_data = data+b;
         break;
      }

      if (data[b].generation < table->generation || data[b].generation == 0) {
         worst_data = data+b;
      }
   }
   data = worst_data;

   data->key = key;
   data->score = score;
   data->generation = table->generation;
   table->write_count++;
}

void prepare_evaltable_search(eval_hash_table_t *table)
{
   table->generation++;
   table->write_count = 0;
}

void prefetch_evaltable(eval_hash_table_t *table, uint64_t key)
{
   size_t index;

   index = map_key_to_index(key, table->number_of_elements);
   __builtin_prefetch(table->data+index);
}

