/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdint.h>
#include <math.h>
#include "softexp.h"
#include "evaluate.h"
#include "pieces.h"
#include "board.h"
#include "psq.h"
#include "pawn_table.h"

static ubitboard_t palace[2];
typedef union king_safety_pattern_t {
   struct {
      bitboard_t kingb[NUM_SIDES];
      bitboard_t kingab[NUM_SIDES];
      bitboard_t kingabc[NUM_SIDES];
   };

   struct {
      large_bitboard_t large_kingb[NUM_SIDES];
      large_bitboard_t large_kingab[NUM_SIDES];
      large_bitboard_t large_kingabc[NUM_SIDES];
   };
} king_safety_pattern_t;

static inline int min(int x, int y)
{
   return (x<y) ? x : y;
}

static inline int max(int x, int y)
{
   return (x>y) ? x : y;
}

/* Progressive king safety evaluation table. */
static int king_safety_penalty[65] = { 0 };

/* Mapping of the piece-square tables for either side */
static int psq_map[NUM_SIDES][128] = {
 { A1, B1, C1, D1, E1, F1, G1, H1,
   A2, B2, C2, D2, E2, F2, G2, H2,
   A3, B3, C3, D3, E3, F3, G3, H3,
   A4, B4, C4, D4, E4, F4, G4, H4,
   A5, B5, C5, D5, E5, F5, G5, H5,
   A6, B6, C6, D6, E6, F6, G6, H6,
   A7, B7, C7, D7, E7, F7, G7, H7,
   A8, B8, C8, D8, E8, F8, G8, H8 },
 { A8, B8, C8, D8, E8, F8, G8, H8,
   A7, B7, C7, D7, E7, F7, G7, H7,
   A6, B6, C6, D6, E6, F6, G6, H6,
   A5, B5, C5, D5, E5, F5, G5, H5,
   A4, B4, C4, D4, E4, F4, G4, H4,
   A3, B3, C3, D3, E3, F3, G3, H3,
   A2, B2, C2, D2, E2, F2, G2, H2,
   A1, B1, C1, D1, E1, F1, G1, H1 },
};

/* Base centralisation score for each square */
static int centre_table[128];

/* Base advancement score for each square */
static int advance_table[128];

/* Extra advancement score for proximity to promotion zone, for pawns */
static int promo_table[128];

/* End game bonus for driving the defending king to the edge */
static int lone_king_table[128];

/* Base shield score, for pawn shields */
static int shield_table[128];

/* Bonus for passed pawns */
static int passed_table[128];

/* Evaluation table for KNBK ending, assuming a dark-squared bishop.
 * For a light-squared bishop, flip the board.
 * This table gives the score from the perspective of the attacker, so the
 * corner of the bishop colour gets a higher score.
 */
static int8_t knbk_table[128];

int16_t psq[MAX_PIECE_TYPES][NUM_SIDES][128];

/* Manhatten distance between two squares (for evaluating lone king endings) */
static int king_distance(int square1, int square2)
{
   int row1, file1;
   int row2, file2;

   row1 = unpack_rank(square1);
   file1 = unpack_file(square1);
   row2 = unpack_rank(square2);
   file2 = unpack_file(square2);

   return max(abs(row1 - row2), abs(file1-file2));
}

/* Initialise the base tables for the current board size */
static void initialise_base_tables(bool large_board)
{
   int r, f;
   int c = large_board_files;

   if (large_board_ranks > c)
      c = large_board_ranks;

   for (r = 0; r<large_board_ranks; r++) {
      for (f = 0; f<large_board_files; f++) {
         int s = pack_row_file(r, f);

         /* Advance table, a progressive piece advancement */
         advance_table[s] = r;

         centre_table[s] = ((c-1)&~0x1) - (abs(2*r - c + 1)/2 + abs(2*f - c + 1)/2);
         if (large_board_files & 1)
            centre_table[s] = ((c-1)&~0x1) - (abs(2*r - c + 1)/2 + abs(2*f - c + 2)/2);

         promo_table[s] = 0;
         //if (r == large_board_ranks-1) promo_table[s] = 99;
         //if (r == large_board_ranks-2) promo_table[s] = 90;
         //if (r == large_board_ranks-3) promo_table[s] = 50;
         //if (r == large_board_ranks-4) promo_table[s] = 10;

         shield_table[s] = 0;
         //if (f < large_board_files/2 && (r == 1 || r == large_board_ranks-2)) shield_table[s] = 1;
         //if (f > large_board_files/2+1 && (r == 1 || r == large_board_ranks-2)) shield_table[s] = 1;
         if (f < large_board_files/2 && (r == 1)) shield_table[s] = 1;
         if (f > large_board_files/2+1 && (r == 1)) shield_table[s] = 1;

         knbk_table[s] = abs(large_anti_diagonal_nr[s]-large_anti_diagonal_nr[large_board_files-1])*5;

         lone_king_table[s] = 6*(abs(2*r - large_board_ranks + 1) + abs(2*f - large_board_files + 1));

         passed_table[s] = sqrt(PAWN_PASSED_BONUS*PAWN_PASSED_BONUS * r*r*r);

         psq_map[WHITE][s] = s;
         psq_map[BLACK][s] = pack_row_file(large_board_ranks-1 - r, f);
         ///printf("[%2d %2d]  ", lone_king_table[s], 48 -6*centre_table[s]);
      }
      //printf("\n");
   }
}

static void compute_piece_square_tables(const game_t *game)
{
   int palace_tropism[NUM_SIDES][128];
   int board_size = large_board_files * large_board_ranks;
   sides side;
   int n, c;

   /* Initialise the base (centre/advance/promotion) tables */
   initialise_base_tables(game->large_board);

   /* If we're dealing with a constrained king ("palace"), initialise palace tropism terms. */
   memset(&palace, 0, sizeof palace);
   memset(&palace_tropism, 0, sizeof palace_tropism);
   if (game->board.rule_flags & RF_KING_TRAPPED) {
      int royal = 0;
      for (n=0; n<game->pt.num_piece_types; n++) {
         if (game->pt.piece_flags[n] & PF_ROYAL) {
            royal = n;
            break;
         }
      }

      if (game->large_board) {
         palace[WHITE].lbb = large_board_south & game->pt.large_prison[royal];
         palace[BLACK].lbb = large_board_north & game->pt.large_prison[royal];
         for (side = WHITE; side < NUM_SIDES; side++) {
            int max_tropism = 0;
            for (c=0; c<board_size; c++) {
               large_bitboard_t bb = palace[next_side[side]].lbb;
               int dist = 128;
               while (!is_zero128(bb)) {
                  int square = bitscan128(bb);
                  bb ^= large_square_bitboards[square];
                  dist = min(dist, king_distance(square, c));
               }

               palace_tropism[side][c] = dist;
               max_tropism = max(max_tropism, dist);
            }
            for (c=0; c<board_size; c++) {
               palace_tropism[side][c] = max_tropism - palace_tropism[side][c];
               if (palace_tropism[side][c] == max_tropism)
                  palace_tropism[side][c] = max_tropism - 2;
            }
         }
      } else {
         /* TODO */
      }
   }

#if 0
   printf_large_bitboard(palace[WHITE].lbb);
   printf_large_bitboard(palace[BLACK].lbb);
   printf("Palace tropism\n");
   for (side = WHITE; side < NUM_SIDES; side++) {
      printf("%s\n", side ? "BLACK" : "WHITE");
      for (c=0; c<large_board_files * large_board_ranks; c++) {
         if (c && (c % large_board_files) == 0) printf("\n");
         printf(" %+3d", palace_tropism[side][c]);
      }
      printf("\n");
   }
#endif

   for (n=0; n<game->pt.num_piece_types; n++) {
      for (side = WHITE; side < NUM_SIDES; side++) {
         for (c=0; c<board_size; c++) {
            int square = psq_map[side][c];
            psq[n][side][c] = game->pt.advance_weight[n] * advance_table[square] +
                              game->pt.promo_weight[n]   * promo_table[square]   +
                              game->pt.shield_weight[n]  * shield_table[square]  +
                              game->pt.centre_weight[n]  * centre_table[square];
#ifdef PIECE_VALUES_IN_PSQ
            psq[n][side][c] += game->board.piece_types->piece_value[n] * 8;
#endif
            /* Piece value adjustment, for pawns on the edge of the board */
            if (n == game->pt.pawn_index[side]) {
               if (game->large_board && !is_zero128(large_board_edge & large_square_bitboards[square]))
                  psq[n][side][c] -= PAWN_EDGE_PENALTY;
               else if (!game->large_board && !(board_edge & make_bitboard_square(square)))
                  psq[n][side][c] -= PAWN_EDGE_PENALTY;
            }

            if (!(game->pt.piece_flags[n] & PF_ROYAL)) {
               move_flag_t flags = game->pt.piece_move_flags[n];

               /* Enemy palace tropism */
               if (!(flags & MF_SLIDER))
                  psq[n][side][c] += 3*palace_tropism[side][c];

               /* Attacks on the enemy palace */
               if (game->board.rule_flags & RF_KING_TRAPPED) {
                  if (game->large_board) {
                     if (flags & MF_IS_LEAPER) {
                        large_bitboard_t bb = get_large_leaper_moves(flags, large_board_empty, c);
                        if (flags & MF_LEAPER_ASYMM)
                           bb = get_large_aleaper_moves(flags, large_board_empty, c, side);
                        bb &= palace[next_side[side]].lbb;
                        psq[n][side][c] += 2*popcount128(bb);
                     }
                  } else {
                     /* TODO */
                  }
               }

               /* Penalise asymmetric leapers that cannot move backward and are on the last rank */
               if (flags & (MF_LEAPER_ASYMM | MF_STEPPER)) {
                  if (game->large_board) {
                     large_bitboard_t bb;
                     if (flags & MF_LEAPER_ASYMM)
                        bb = get_large_aleaper_moves(flags, large_board_empty, c, side);
                     else
                        bb = get_large_stepper_move_bitboard(flags, large_board_empty, large_square_bitboards[c]);
                     int penalty = 0;
                     int r = unpack_rank(c);
                     int m = popcount128(bb);

                     /* Restricted movement: fewer than the maximum number of steps */
                     if (m == game->pt.min_moves[n]) {
                        int delta = game->pt.max_moves[n] - game->pt.min_moves[n];
                        if (delta)
                           penalty = game->pt.min_moves[n] * game->pt.piece_value[n]/delta;
                     }

                     /* An asymmetric leaper (or really, a stepper) that has no forward or backward moves */
                     if (is_zero128(bb & (large_board_northward[r]|large_board_southward[r])))
                        penalty = game->pt.piece_value[n]/2;

                     psq[n][side][c] -= penalty;
                  } else {
                     /* TODO */
                  }
               }
            }

            /* Flip the sign for black; the evaluation is always from white's point of view */
            if (side == BLACK)
               psq[n][side][c] = -psq[n][side][c];
         }
      }

#if 0
      printf("Piece square table % 2d %s\n", n, game->pt.piece_name[n]);
      printf("a=%d p=%d s=%d c=%d\n", game->pt.advance_weight[n], game->pt.promo_weight[n], game->pt.shield_weight[n], game->pt.centre_weight[n]);
      for (side = WHITE; side < NUM_SIDES; side++) {
         if (n != game->pt.pawn_index[next_side[side]]) {
            printf("%s\n", side ? "BLACK" : "WHITE");
            for (c=0; c<large_board_files * large_board_ranks; c++) {
               if (c && (c % large_board_files) == 0) printf("\n");
               printf(" %+3d", psq[n][side][c]);
            }
            printf("\n");
         }
      }
#endif
   }
}

/* Initialise parameter-driven evaluation tables */
void initialise_evaluation(const game_t *game)
{
   double x;
   int n;
   trace("Initialise evaluation");

   /* King safety */
   x = KING_SAFETY_CEIL / (1.0 + myexp(KING_SAFETY_SLOPE*KING_SAFETY_INFLECTION));
   king_safety_penalty[0] = 0;
   trace("Initialise king safety");
   for (n=1; n<65; n++)
      king_safety_penalty[n] = KING_SAFETY_CEIL / (1.0 + myexp(-KING_SAFETY_SLOPE*(n - KING_SAFETY_INFLECTION))) - x;

   trace("Compute piece square tables");
   compute_piece_square_tables(game);
   trace("Evaluation initialised");
}

/* Compute forward move and attack spans for pawns of both sides.
 * To find the move span, just shift all pawns forward according to their normal move until the last pawn has
 * reached the end of the board (just shift by the number of ranks).
 * To find the attack span, just shift the move span as per the normal attack.
 *
 * Pawns for which the forward move span does not intersect an enemy pawn are open (free to advance).
 * Pawns for which the forward move span does not intersect the enemy move or attack spans are passed.
 * Pawns that are not in their own side's forward attack span, and cannot advance to do so (safely), are backward.
 * Pawns that are in their own colour forward movement span are doubled and get a penalty.
 *
 * We can experiment with "move" and "move_span" in the detection of backward pawns. A pawn that has no
 * defenders in his forward move span is truly backward, a pawn that still have possible defenders but not in
 * its current or advance square needs to move, but is not as bad as the first example.
 *
 * We do need to keep into account tempo. That means we should consider move spans on a per-move basis,
 * where the move span on move N is the union of the move span at move N-1 and all forward moves from that
 * position.
 *
 * For doubled pawns, the penalty should generally be less if pawns have more move options. Standard pawns
 * move in only one direction, but berolina pawns have two possible moves and are not hindered so much by
 * being "doubled". Chinese pawns (after promotion) have three move coices and are not really hindered by
 * eachother at all.
 */
static int evaluate_pawn_structure(game_t *game)
{
   pawn_table_entry_t *hash;
   int ev = 0;

   /* Check if we've evaluated this position before and it's in the pawn hash table */
   hash = query_pawn_table_entry(game->pawn_table, game->board.pawn_hash);
   if (hash)
      return hash->score;

   if (game->large_board) {
      /* Get forward moves, forward attacks, and forward attack and move spans */
      large_bitboard_t pawns[NUM_SIDES] = {large_board_empty, large_board_empty};
      large_bitboard_t doubled[NUM_SIDES] = {large_board_empty, large_board_empty};
      large_bitboard_t backward[NUM_SIDES] = {large_board_empty, large_board_empty};
      large_bitboard_t open[NUM_SIDES] = {large_board_empty, large_board_empty};
      large_bitboard_t passed[NUM_SIDES] = {large_board_empty, large_board_empty};
      large_bitboard_t attack[NUM_SIDES][16];
      large_bitboard_t move[NUM_SIDES][16];
      large_bitboard_t attack_span[NUM_SIDES];
      large_bitboard_t move_span[NUM_SIDES];
      large_bitboard_t occ;
      large_bitboard_t bb;
      uint32_t         move_flags[NUM_SIDES] = {0, 0};
      uint32_t         capture_flags[NUM_SIDES] = {0, 0};

      /* Find pawns for each side */
      if (game->pt.pawn_index[WHITE] >= 0) {
         pawns[WHITE]         = game->board.large_bbp[game->pt.pawn_index[WHITE]] & game->board.large_bbc[WHITE];
         move_flags[WHITE]    = game->pt.piece_move_flags[game->pt.pawn_index[WHITE]];
         capture_flags[WHITE] = game->pt.piece_capture_flags[game->pt.pawn_index[WHITE]];
      }
      if (game->pt.pawn_index[BLACK] >= 0) {
         pawns[BLACK]         = game->board.large_bbp[game->pt.pawn_index[BLACK]] & game->board.large_bbc[BLACK];
         move_flags[BLACK]    = game->pt.piece_move_flags[game->pt.pawn_index[BLACK]];
         capture_flags[BLACK] = game->pt.piece_capture_flags[game->pt.pawn_index[BLACK]];
      }

      /* Occupancy bitboard, taking into account only pawns */
      occ = pawns[WHITE] | pawns[BLACK];

      /* Single move and attack bitboards */
      move[WHITE][0]   = get_large_stepper_move_bitboard(move_flags[WHITE], occ, pawns[WHITE]);
      attack[WHITE][0] = get_large_stepper_attack_bitboard(capture_flags[WHITE], occ, pawns[WHITE]);
      move[BLACK][0]   = get_large_stepper_move_bitboard(move_flags[BLACK], occ, pawns[BLACK]);
      attack[BLACK][0] = get_large_stepper_attack_bitboard(capture_flags[BLACK], occ, pawns[BLACK]);

      /* Move and attack spans */
      int n;
      for (n=1; n<large_board_files-1; n++) {
         move[WHITE][n] = move[WHITE][n-1];
         move[BLACK][n] = move[BLACK][n-1];
         move[WHITE][n] |= get_large_stepper_move_bitboard(move_flags[WHITE], occ, move[WHITE][n-1]);
         move[BLACK][n] |= get_large_stepper_move_bitboard(move_flags[BLACK], occ, move[BLACK][n-1]);
         attack[WHITE][n] = get_large_stepper_attack_bitboard(capture_flags[WHITE], occ, move[WHITE][n]|pawns[WHITE]);
         attack[BLACK][n] = get_large_stepper_attack_bitboard(capture_flags[BLACK], occ, move[BLACK][n]|pawns[BLACK]);
      }
      move_span[WHITE] = move[WHITE][large_board_files-2];
      move_span[BLACK] = move[BLACK][large_board_files-2];
      attack_span[WHITE] = attack[WHITE][large_board_files-2];
      attack_span[BLACK] = attack[BLACK][large_board_files-2];

      /* Backward pawns, or isolated pawns */
      backward[WHITE] = ~(attack_span[WHITE] & (pawns[WHITE] | move_span[WHITE])) & pawns[WHITE];
      backward[BLACK] = ~(attack_span[BLACK] & (pawns[BLACK] | move_span[BLACK])) & pawns[BLACK];

      /* Doubled pawns
       * What is slightly annoying is that this marks the *forward* most pawn as doubled; what we normally
       * want is the *back* pawn of the pair. If we just penalise based on the presence of a doubled pawn this
       * doesn't matter.
       */
      doubled[WHITE] = get_large_stepper_move_span_bitboard(move_flags[WHITE], large_board_empty, pawns[WHITE]) & pawns[WHITE];
      doubled[BLACK] = get_large_stepper_move_span_bitboard(move_flags[BLACK], large_board_empty, pawns[BLACK]) & pawns[BLACK];

      /* Open pawns and passed pawns.
       * We need to loop over all pawns to detect these properly.
       * For open pawns, we intersect with the move span of enemy pawns.
       * For passed pawns we intersect with the attack span of enemy pawns.
       *
       * There are two ways in which we can use the full attack span calculated above to accelerate things.
       * The first is the situation where the intersection is empty. In that case there is no possible way
       * that this pawn can be blocked or stopped. This is always safe.
       * The second is if both pawn types are confined to files (as they are in normal chess). In that case
       * we only need to check against the full pattern, since no pawns can join in.
       *
       * There are, in fact, some other generalised tests we can do as well: if the intersection after N moves
       * from white and black can mask out all new squares reachable between N and N+1, then all pawns are
       * blocked as well and we can avoid this test. We can do this test first to see whether a loop over all
       * pawns is necessary or not.
       */
      /* White */
      bb = pawns[WHITE];
      while(!is_zero128(bb)) {
         int square = bitscan128(bb);
         large_bitboard_t pawn = large_square_bitboards[square];
         large_bitboard_t span;
         bb ^= pawn;
         
         /* get forward move span for this pawn */
         span = get_large_stepper_move_span_bitboard(move_flags[WHITE], occ, pawn);

         /* If it can reach the promotion zone, then it is open */
         if (!is_zero128(span & game->pt.piece_large_promotion_zone[game->pt.pawn_index[WHITE]])) {
            open[WHITE] |= pawn;

            /* Is this pawn passed as well? */
            if (is_zero128(span & attack_span[BLACK]))
               passed[WHITE] |= pawn;

            /* TODO: properly deal with the situation where black and white pawns may not be mirror images of
             * eachother, so we need to take tempo into account. So far this is only relevant for Spartan.
             */
         }
      }

      /* Ditto black */
      bb = pawns[BLACK];
      while(!is_zero128(bb)) {
         int square = bitscan128(bb);
         large_bitboard_t pawn = large_square_bitboards[square];
         large_bitboard_t span;
         bb ^= pawn;
         
         /* get forward move span for this pawn */
         span = get_large_stepper_move_span_bitboard(move_flags[BLACK], occ, pawn);

         /* If it can reach the promotion zone, then it is open */
         if (!is_zero128(span & game->pt.piece_large_promotion_zone[game->pt.pawn_index[BLACK]])) {
            open[BLACK] |= pawn;

            /* Is this pawn passed as well? */
            if (is_zero128(span & attack_span[WHITE]))
               passed[BLACK] |= pawn;

            /* TODO: properly deal with the situation where black and white pawns may not be mirror images of
             * eachother, so we need to take tempo into account. So far this is only relevant for Spartan.
             */
         }

      }

      /* Return score for pawn structure, based on the pawns found. For passed pawns, we really need piece
       * information as well (backup/blocked; TODO).
       */
      /* White */
      ev -= PAWN_WEAK_PENALTY         * popcount128(backward[WHITE]);
      ev -= PAWN_DOUBLED_PENALTY      * popcount128(doubled[WHITE]);
      ev += PAWN_PROTECT_PASSED_BONUS * popcount128(passed[WHITE] & attack[WHITE][0]);
      bb = passed[WHITE];
      while (!is_zero128(bb)) {
         int square = bitscan128(bb);
         bb^=large_square_bitboards[square];
         ev += PAWN_PASSED_BONUS * advance_table[square];
      }

      /* Ditto black */
      ev += PAWN_WEAK_PENALTY         * popcount128(backward[BLACK]);
      ev += PAWN_DOUBLED_PENALTY      * popcount128(doubled[BLACK]);
      ev -= PAWN_PROTECT_PASSED_BONUS * popcount128(passed[BLACK] & attack[BLACK][0]);
      bb = passed[BLACK];
      while (!is_zero128(bb)) {
         int square = bitscan128(bb);
         bb^=large_square_bitboards[square];
         ev -= PAWN_PASSED_BONUS * advance_table[psq_map[BLACK][square]];
      }
      printf_large_bitboard(attack_span[WHITE]);
      printf_large_bitboard(backward[WHITE]);
   } else {
   }

   /* Store hash table entry for later retrieval */
   store_pawn_table_entry(game->pawn_table, game->board.pawn_hash, ev);

   //printf("%d\n", ev);
   //exit(0);
   return ev;
}

static inline int16_t asses_development(const game_t *game)
{
   int16_t dev = 0;

   if (game->large_board) {
      dev += popcount128(game->board.large_init & game->board.large_bbc[BLACK] & ~game->board.large_royal) -
             popcount128(game->board.large_init & game->board.large_bbc[WHITE] & ~game->board.large_royal);
      dev += popcount128(large_short_castle_free[BLACK] & game->board.large_bbc[BLACK]) -
             popcount128(large_short_castle_free[WHITE] & game->board.large_bbc[WHITE]);
      dev += popcount128(large_long_castle_free[BLACK] & game->board.large_bbc[BLACK]) -
             popcount128(large_long_castle_free[WHITE] & game->board.large_bbc[WHITE]);
   } else {
      dev += popcount64(game->board.init & game->board.bbc[BLACK] & ~game->board.royal) -
             popcount64(game->board.init & game->board.bbc[WHITE] & ~game->board.royal);
   }
   dev *= 4;

   /* Evaluate castling status: give a penalty for the current position if
    * castling rights have been lost since the root of the search, but the
    * player has not castled.
    */
   if (!game->board.did_castle[WHITE] && may_castle(&game->root_board, WHITE) && !may_castle(&game->board, WHITE))
      dev -= CASTLE_LOST_PENALTY;

   if (!game->board.did_castle[BLACK] && may_castle(&game->root_board, BLACK) && !may_castle(&game->board, BLACK))
      dev += CASTLE_LOST_PENALTY;

   /* A slightly smaller penalty if the player can castle, but hasn't */
   if (!game->board.did_castle[WHITE] && may_castle(&game->board, WHITE))
      dev -= CASTLE_LOST_PENALTY/2;

   if (!game->board.did_castle[BLACK] && may_castle(&game->board, BLACK))
      dev += CASTLE_LOST_PENALTY/2;

   return dev;
}

/* Dedicated evaluation function for endings with a lone king */
static bool lone_king_evaluation(const game_t *game, sides defender, int16_t *ev)
{
   const board_t *board = &game->board;
   sides attacker = next_side[defender];
   int defender_king_square;
   int safe_zone = 0;
   uint16_t eval = 0;

   /* Check for special score for bare kings */
   if (expect(game->bare_king_score, false))
      return false;

   if (game->large_board) {
      large_bitboard_t lone_king = board->large_bbc[defender] & board->large_royal;

      /* Confirm that this is a lone king ending (as opposed to multiple kings or no king) */
      if (is_zero128(lone_king) || !onebit128(lone_king))
         return false;
      defender_king_square = bitscan128(lone_king);

      /* Where the defender should go depends on the material the attacker has. If the attacker has one colour
       * bound piece, the defender wants to go to the opposite corner. Otherwise, close to the centre is good
       * enough.
       */
      large_bitboard_t safe = large_board_all;
      large_bitboard_t bb = board->large_bbc[attacker];
      while (!is_zero128(bb)) {
         int n = get_piece(board, bitscan128(bb));
         bb &= ~game->board.large_bbp[n];

         if (game->pt.piece_flags[n] & PF_PAIRBONUS) {
            if (!is_zero128(board->large_bbp[n] & large_board_light))
               safe &= large_board_light;
            if (!is_zero128(board->large_bbp[n] & large_board_dark))
               safe &= large_board_dark;
         }
      }
      if (!is_zero128(safe)) safe = safe ^ large_board_all;

      if (is_equal128(safe, large_board_light)) {
         safe_zone = 1;
      } else if (is_equal128(safe, large_board_dark)) {
         safe_zone = 2;
      }

      /* Bring the attacking king closer to the defending king */
      bb = board->large_royal ^ lone_king;
      while (!is_zero128(bb)) {
         int attacker_king_square = bitscan128(bb);
         bb ^= large_square_bitboards[attacker_king_square];
         eval -= 8*king_distance(attacker_king_square, defender_king_square);
      }
   } else {
      bitboard_t lone_king = board->bbc[defender] & board->royal;

      /* Confirm that this is a lone king ending (as opposed to multiple kings or no king) */
      if (!lone_king || !onebit64(lone_king))
         return false;
      defender_king_square = bitscan64(lone_king);

      /* Where the defender should go depends on the material the attacker has. If the attacker has one colour
       * bound piece, the defender wants to go to the opposite corner. Otherwise, close to the centre is good
       * enough.
       */
      bitboard_t safe = board_all;
      bitboard_t bb = board->bbc[attacker];
      while (bb) {
         int n = get_piece(board, bitscan64(bb));
         bb &= ~game->board.bbp[n];

         if (game->pt.piece_flags[n] & PF_PAIRBONUS) {
            if (board->bbp[n] & board_light)
               safe &= board_light;
            if (board->bbp[n] & board_dark)
               safe &= board_dark;
         }
      }
      if (safe) safe = ~safe;

      if (safe == board_light) {
         safe_zone = 1;
      } else if (safe == board_dark) {
         safe_zone = 2;
      }

      /* Bring the attacking king closer to the defending king */
      bb = board->royal ^ lone_king;
      while (bb) {
         int attacker_king_square = bitscan64(bb);
         bb ^= make_bitboard_square(attacker_king_square);
         eval -= 8*king_distance(attacker_king_square, defender_king_square);
      }
   }

   /* Evaluate position of the defending king, from the point of view of the attacker */
   if (safe_zone == 1) {
      eval += 10 * knbk_table[defender_king_square];
   } else if (safe_zone == 2) {
      eval += 10 * knbk_table[flip_horizontal(defender_king_square)];
   }
   eval += lone_king_table[defender_king_square];

   *ev = (attacker == WHITE) ? eval : -eval;
   return true;
}

static void initialise_king_safety_pattern(const board_t *board, king_safety_pattern_t *pattern)
{
   /* Clear */
   memset(pattern, 0, sizeof *pattern);

   if (board->large_board) {
      large_bitboard_t k, sq;
      large_bitboard_t kinga, kingb, kingc;
      k = board->large_bbc[WHITE] & board->large_royal;
      sq = k | shift_large_bitboard_file_left(k) | shift_large_bitboard_file_right(k);
      kinga = (shr128(sq, large_board_files) | sq) ^ k;
      kingb = shl128(sq, large_board_files);
      kingc = shl128(sq, 2*large_board_files);
      pattern->large_kingb[WHITE] = kingb;
      pattern->large_kingab[WHITE] = kinga | kingb;
      pattern->large_kingabc[WHITE] = kinga | kingb | kingc;

      k = board->large_bbc[BLACK] & board->large_royal;
      sq = k | shift_large_bitboard_file_left(k) | shift_large_bitboard_file_right(k);
      kinga = (shl128(sq, large_board_files) | sq) ^ k;
      kingb = shr128(sq, large_board_files);
      kingc = shr128(sq, 2*large_board_files);
      pattern->large_kingb[BLACK] = kingb;
      pattern->large_kingab[BLACK] = kinga | kingb;
      pattern->large_kingabc[BLACK] = kinga | kingb | kingc;
   } else {
      bitboard_t k, sq;
      bitboard_t kinga, kingb, kingc;
      k = board->bbc[WHITE] & board->royal;
      sq = k | shift_bitboard_file_left(k) | shift_bitboard_file_right(k);
      kinga = (sq >> 8 | sq) ^ k;
      kingb = sq << 8;
      kingc = sq << 16;
      pattern->kingb[WHITE] = kingb;
      pattern->kingab[WHITE] = kinga | kingb;
      pattern->kingabc[WHITE] = kinga | kingb | kingc;

      k = board->bbc[BLACK] & board->royal;
      sq = k | shift_bitboard_file_left(k) | shift_bitboard_file_right(k);
      kinga = (sq << 8 | sq) ^ k;
      kingb = sq >> 8;
      kingc = sq >> 16;
      pattern->kingb[BLACK] = kingb;
      pattern->kingab[BLACK] = kinga | kingb;
      pattern->kingabc[BLACK] = kinga | kingb | kingc;
   }
}

int16_t static_evaluation(game_t *game, sides side_to_move, int alpha, int beta)
{
   king_safety_pattern_t king_pattern;
   large_bitboard_t large_attacked[NUM_SIDES] = { large_board_empty, large_board_empty };
   large_bitboard_t large_piece_attacks[NUM_SIDES][MAX_PIECE_TYPES];
   large_bitboard_t large_king_attacks[NUM_SIDES] = { large_board_empty, large_board_empty };
   large_bitboard_t large_hsliders[NUM_SIDES] = { large_board_empty, large_board_empty };
   large_bitboard_t large_vsliders[NUM_SIDES] = { large_board_empty, large_board_empty };
   large_bitboard_t large_dsliders[NUM_SIDES] = { large_board_empty, large_board_empty };
   large_bitboard_t large_pawns = large_board_empty;
   bitboard_t piece_attacks[NUM_SIDES][MAX_PIECE_TYPES];
   bitboard_t attacked[NUM_SIDES] = { board_empty, board_empty };
   bitboard_t king_attacks[NUM_SIDES] = { board_empty, board_empty };
   bitboard_t hsliders[NUM_SIDES] = { board_empty, board_empty };
   bitboard_t vsliders[NUM_SIDES] = { board_empty, board_empty };
   bitboard_t dsliders[NUM_SIDES] = { board_empty, board_empty };
   bitboard_t pawns = board_empty;
   uint16_t closed_files[NUM_SIDES] = { 0, 0 };
   int king_attacked_pattern[NUM_SIDES] = { 0, 0 };
   int king_counter[NUM_SIDES] = { 0, 0 };
   int piece_types[NUM_SIDES] = { 0, 0 };
   int num_pieces[NUM_SIDES] = { 0, 0 };
   int num_royals[NUM_SIDES] = { 0, 0 };
   int minors[NUM_SIDES] = { 0, 0 };
   int num_pawns[NUM_SIDES] = { 0, 0 };
   uint16_t piece_ids[NUM_SIDES] = { 0, 0 };
   int mate_potential[NUM_SIDES] = { 0, 0 };
   int army_attack_strength[NUM_SIDES] = { 0, 0 };
   bool can_win[NUM_SIDES] = { false, false };
   bool only_edge_pawns[NUM_SIDES] = { true, true };
   int16_t pawn = 0;    // Pawn structure
   int16_t dev = 0;     // Piece development
   int16_t mat = 0;     // Material balance
   int16_t mob = 0;     // Piece mobility
   int16_t pt = 0;      // Piece threats
   int16_t ks = 0;      // King safety
   int16_t ps = 0;      // Piece square tables
   int16_t of = 0;      // Open files, semi-open files and coordinated sliders
   int16_t ev = 0;
   sides side;
   int index;
   int n;

   /* See if the evaluation has been recorded in the hash table */
   eval_table_entry_t *eval_tab = query_eval_table_entry(game->eval_table, game->board.hash);
   if (eval_tab) {
      ev = eval_tab->score;

      /* Tapered evaluation when we're about to hit the 50 move counter */
      if (game->fifty_counter[game->moves_played] > 80)
         ev = ev * (101 - game->fifty_counter[game->moves_played]) / 20;

      return ( (side_to_move == WHITE) ? ev : -ev ) + SIDE_TO_MOVE_BONUS;
   }

   /* Count number of pieces on each side */
   if (game->large_board) {
      num_pieces[WHITE] = popcount128(game->board.large_bbc[WHITE]);
      num_pieces[BLACK] = popcount128(game->board.large_bbc[BLACK]);
      num_royals[WHITE] = popcount128(game->board.large_bbc[WHITE] & game->board.large_royal);
      num_royals[BLACK] = popcount128(game->board.large_bbc[BLACK] & game->board.large_royal);
      memset(&large_piece_attacks, 0, sizeof large_piece_attacks);
   } else {
      num_pieces[WHITE] = popcount64(game->board.bbc[WHITE]);
      num_pieces[BLACK] = popcount64(game->board.bbc[BLACK]);
      num_royals[WHITE] = popcount64(game->board.bbc[WHITE] & game->board.royal);
      num_royals[BLACK] = popcount64(game->board.bbc[BLACK] & game->board.royal);
      memset(&piece_attacks, 0, sizeof piece_attacks);
   }

   /* Evaluate material balance */
   mat += game->board.material[WHITE] - game->board.material[BLACK];

   /* Lazy evaluation: if the material balance is already far outside the (alpha,beta) bounds, skip the rest
    * of the evaluation: it can't make up anyway.
    * Exception: when one of the two sides has a lone king, in which case we need the lone king heuristics
    * function to deliver mate.
    */
   if ( (mat + LAZY_MARGIN < alpha || mat - LAZY_MARGIN > beta) && num_pieces[WHITE] > 2 && num_pieces[BLACK] > 2)
      return ((side_to_move == WHITE) ? mat : -mat) + SIDE_TO_MOVE_BONUS;

   /* King safety evaluation.
    * Inspired by the description of Rebel's king safety evaluation by Ed Schroeder.
    * There are three types of square around the king: squares behind
    * and next to the king (A), squares on the row directly in front of
    * the king (B) and squares two rows in front of the king (C).
    * Initialise data structures.
    */
   initialise_king_safety_pattern(&game->board, &king_pattern);



   /* Development */
   dev = asses_development(game);



   /* Loop over all pieces.
    * Gather attack pattern and adjust score for piece square tabes and mobility.
    */
   if (!game->large_board) {     /* Small board */
      bitboard_t occ = game->board.bbc[WHITE] | game->board.bbc[BLACK];
      bitboard_t pieces = occ;
      while (pieces) {
         n = get_piece(&game->board, bitscan64(pieces));
         pieces &= ~game->board.bbp[n];

         for (side = WHITE; side < NUM_SIDES; side++) {
            bitboard_t bb = game->board.bbp[n] & game->board.bbc[side];
            if (bb) {
               if (!(game->pt.piece_flags[n] & PF_ROYAL)) piece_ids[side] |= 1<<n;
               piece_types[side]++;
            }

            /* Non-sliders and pieces that don't promote become weaker in the end game */
            if (bb && (game->pt.piece_move_flags[n] & MF_SLIDER) == 0 && game->pt.piece_promotion_zone[n] == board_empty)
               minors[side]++;

            /* Record sliders, for the purpose of open file, 7th rank, and battery bonuses */
            if (game->pt.piece_move_flags[n] & MF_SLIDER_H)                hsliders[side] |= bb;
            if (game->pt.piece_move_flags[n] & MF_SLIDER_V)                vsliders[side] |= bb;
            if (game->pt.piece_move_flags[n] & (MF_SLIDER_A|MF_SLIDER_D))  dsliders[side] |= bb;

            while (bb) {
               bitboard_t moves;
               int square = bitscan64(bb);
               bb ^= make_bitboard_square(square);

               /* Calculate mobility */
               moves = get_move_bitboard(&game->board, make_bitboard_square(square), side);

               /* Asses mobility */
               int count = popcount64(moves);
               if (side == WHITE)
                  mob += game->pt.mobility_score[n][count];
               else
                  mob -= game->pt.mobility_score[n][count];

               /* Attack bitboard, for king safety evaluation.
                * Record number of attackers on A and B squares.
                */
               bitboard_t attacks = get_attack_bitboard(&game->board, make_bitboard_square(square), side);
               piece_attacks[side][n] |= attacks;
               if (!(game->pt.piece_flags[n] & PF_ROYAL))
                  attacked[side] |= attacks;
               else
                  king_attacks[side] |= attacks;

               /* Attacks on the enemy king position */
               if (attacks & king_pattern.kingabc[next_side[side]])
                  king_attacked_pattern[next_side[side]] += game->pt.king_safety_weight[n];
               army_attack_strength[side] += game->pt.king_safety_weight[n];

               /* Record mate potential */
               if (!(game->pt.piece_flags[n] & PF_NOMATE))
                  mate_potential[side]++;

               /* Pawns */
               if (n == game->pt.pawn_index[side]) {
                  num_pawns[side]++;
                  closed_files[side] |= 1 << unpack_file(square);
                  pawns |= make_bitboard_square(square);

                  if (game->pt.pawn_front_span[side][square] & ~(board_hfile|board_afile))
                     only_edge_pawns[side] = false;
               }


               /* Pawn structure terms, white */
               if (n == game->pt.pawn_index[WHITE]) {
                  /* Doubled pawns */
                  if (bb & game->pt.pawn_front_span[WHITE][square])
                     pawn -= PAWN_DOUBLED_PENALTY;

                  /* Open pawns, possibly passers */
                  int other_pawn = game->pt.pawn_index[BLACK];
                  bool open_pawn = (game->pt.pawn_front_span[WHITE][square] & game->board.bbp[other_pawn]) == 0;
                  if (open_pawn || game->pt.pawn_steps[WHITE]>1) {
                     pawn += PAWN_OPEN_BONUS * advance_table[square] / game->pt.pawn_steps[WHITE];
                     if (!(game->pt.pawn_passer_mask[WHITE][square] & game->board.bbp[game->pt.pawn_index[BLACK]]))
                        pawn += (passed_table[psq_map[WHITE][square]] * game->pt.piece_promotion_value[n])/128;
                  }
               }
               /* Ditto, black */
               if (n == game->pt.pawn_index[BLACK]) {
                  /* Doubled pawns */
                  if (bb & game->pt.pawn_front_span[BLACK][square])
                     pawn += PAWN_DOUBLED_PENALTY;

                  /* Open pawns, possibly passers */
                  int other_pawn = game->pt.pawn_index[WHITE];
                  bool open_pawn = (game->pt.pawn_front_span[BLACK][square] & game->board.bbp[other_pawn]) == 0;
                  if (open_pawn || game->pt.pawn_steps[BLACK]>1) {
                     pawn -= PAWN_OPEN_BONUS * advance_table[square] / game->pt.pawn_steps[BLACK];
                     if (!(game->pt.pawn_passer_mask[BLACK][square] & game->board.bbp[game->pt.pawn_index[WHITE]]))
                        pawn -= (passed_table[psq_map[BLACK][square]] * game->pt.piece_promotion_value[n])/128;
                  }
               }
            }
         }

         /* Pair bonus
          * TODO: replace with a more general "field control" bonus.
          */
         if (game->pt.piece_flags[n] & PF_PAIRBONUS) {
            bitboard_t bb;
            bb = game->board.bbp[n] & game->board.bbc[WHITE];
            mat += PAIR_BONUS * ( (bb & board_light) && (bb & board_dark) );

            bb = game->board.bbp[n] & game->board.bbc[BLACK];
            mat -= PAIR_BONUS * ( (bb & board_light) && (bb & board_dark) );
         }
      }

#ifdef USE_SLIDER_TERMS
      /* Test for sliders on open files */
      bitboard_t open_files = (~(closed_files[WHITE]|closed_files[BLACK]) & 0xff) * board_afile;
      of += OPEN_FILE_BONUS*popcount64(open_files & vsliders[WHITE]);
      of -= OPEN_FILE_BONUS*popcount64(open_files & vsliders[BLACK]);

      /* Semi-open */
      open_files = (~closed_files[WHITE] & 0xff) * board_afile;
      of += SEMI_OPEN_FILE_BONUS*popcount64(open_files & vsliders[WHITE]);
      open_files = (~closed_files[BLACK] & 0xff) * board_afile;
      of -= SEMI_OPEN_FILE_BONUS*popcount64(open_files & vsliders[BLACK]);

      /* (Generalised) 7th rank bonus */
      of += SEVENTH_RANK_BONUS*popcount64(board_rank7 & hsliders[WHITE]);
      of -= SEVENTH_RANK_BONUS*popcount64(board_rank2 & hsliders[BLACK]);

      /* Batteries along files ("doubled rook") */
      for (n=0; n<large_board_files; n++) {
         if (popcount64(board_file[n] & vsliders[WHITE]) > 1) of += FILE_BATTERY;
         if (popcount64(board_file[n] & vsliders[BLACK]) > 1) of -= FILE_BATTERY;
      }

      /* Batteries along a diagonal */
      for (n=0; n<15; n++) {
         if (popcount64(board_a1h8_diagonal[n] & dsliders[WHITE]) > 1) of += DIAGONAL_BATTERY;
         if (popcount64(board_a8h1_diagonal[n] & dsliders[WHITE]) > 1) of += DIAGONAL_BATTERY;
         if (popcount64(board_a1h8_diagonal[n] & dsliders[BLACK]) > 1) of -= DIAGONAL_BATTERY;
         if (popcount64(board_a8h1_diagonal[n] & dsliders[BLACK]) > 1) of -= DIAGONAL_BATTERY;
      }
#endif

   } else {    /* large board */
      large_bitboard_t occ = game->board.large_bbc[WHITE] | game->board.large_bbc[BLACK];
      large_bitboard_t pieces = occ;
      while (!is_zero128(pieces)) {
         n = get_piece(&game->board, bitscan128(pieces));
         pieces &= ~game->board.large_bbp[n];

         for (side = WHITE; side < NUM_SIDES; side++) {
            large_bitboard_t bb = game->board.large_bbp[n] & game->board.large_bbc[side];
            if (!is_zero128(bb)) {
               if (!(game->pt.piece_flags[n] & PF_ROYAL)) piece_ids[side] |= 1<<n;
               piece_types[side]++;
            }

            /* Non-sliders and pieces that don't promote become weaker in the end game */
            if (!is_zero128(bb) && (game->pt.piece_move_flags[n] & MF_SLIDER) == 0 &&
                is_equal128(game->pt.piece_large_promotion_zone[n], large_board_empty))
               minors[side]++;

            /* Record sliders, for the purpose of open file, 7th rank, and battery bonuses */
            if (game->pt.piece_move_flags[n] & MF_SLIDER_H)               large_hsliders[side] |= bb;
            if (game->pt.piece_move_flags[n] & MF_SLIDER_V)               large_vsliders[side] |= bb;
            if (game->pt.piece_move_flags[n] & (MF_SLIDER_A|MF_SLIDER_D)) large_dsliders[side] |= bb;

            while (!is_zero128(bb)) {
               large_bitboard_t moves;
               int count = 0;
               int square = bitscan128(bb);
               bb ^= large_square_bitboards[square];

               /* Mobility */
               moves = get_large_attack_bitboard(&game->board, large_square_bitboards[square], side);
               count = popcount128(moves);
               if (side == WHITE)
                  mob += game->pt.mobility_score[n][count];
               else
                  mob -= game->pt.mobility_score[n][count];

               /* Attack bitboard, for king safety evaluation.
                * Record number of attackers on A and B squares.
                */
               large_bitboard_t attacks = moves;//get_large_attack_bitboard(&game->board, large_square_bitboards[square], side);
               large_piece_attacks[side][n] |= attacks;
               if (!(game->pt.piece_flags[n] & PF_ROYAL))
                  large_attacked[side] |= attacks;
               else
                  large_king_attacks[side] |= attacks;

               /* Attacks on the enemy king position */
               if (!is_zero128(attacks & king_pattern.large_kingabc[next_side[side]])) {
                  king_attacked_pattern[next_side[side]] += game->pt.king_safety_weight[n];
               }

               /* Record mate potential */
               if (!(game->pt.piece_flags[n] & PF_NOMATE))
                  mate_potential[side]++;

               /* Pawns */
               if (n == game->pt.pawn_index[side]) {
                  num_pawns[side]++;
                  closed_files[side] |= 1 << unpack_file(square);
                  large_pawns |= large_square_bitboards[square];

                  if (!is_zero128(game->pt.large_pawn_front_span[side][square] & ~(large_board_east_edge|large_board_west_edge)))
                     only_edge_pawns[side] = false;
               }

               /* Pawn structure terms, white */
               if (n == game->pt.pawn_index[WHITE]) {
                  /* Doubled pawns */
                  if (!is_zero128(bb & game->pt.large_pawn_front_span[WHITE][square]))
                     pawn -= PAWN_DOUBLED_PENALTY;

                  /* Open pawns, possibly passers */
                  int other_pawn = game->pt.pawn_index[BLACK];
                  bool open_pawn = is_zero128(game->pt.large_pawn_front_span[WHITE][square] & game->board.large_bbp[other_pawn]);
                  if (open_pawn || game->pt.pawn_steps[WHITE]>1) {
                     pawn += PAWN_OPEN_BONUS * advance_table[square] / game->pt.pawn_steps[WHITE];
                     if (is_zero128(game->pt.large_pawn_passer_mask[WHITE][square] & game->board.large_bbp[game->pt.pawn_index[BLACK]]))
                        pawn += passed_table[psq_map[WHITE][square]];
                  }
               }
               /* Ditto, black */
               if (n == game->pt.pawn_index[BLACK]) {
                  /* Doubled pawns */
                  if (!is_zero128(bb & game->pt.large_pawn_front_span[BLACK][square]))
                     pawn += PAWN_DOUBLED_PENALTY;

                  /* Open pawns, possibly passers */
                  int other_pawn = game->pt.pawn_index[WHITE];
                  bool open_pawn = is_zero128(game->pt.large_pawn_front_span[BLACK][square] & game->board.large_bbp[other_pawn]);
                  if (open_pawn || game->pt.pawn_steps[BLACK]>1) {
                     pawn -= PAWN_OPEN_BONUS * advance_table[square] / game->pt.pawn_steps[BLACK];
                     if (is_zero128(game->pt.large_pawn_passer_mask[BLACK][square] & game->board.large_bbp[game->pt.pawn_index[WHITE]]))
                        pawn -= passed_table[psq_map[BLACK][square]];
                  }
               }
            }
         }

         /* Pair bonus
          * TODO: replace with a more general "field control" bonus.
          */
         if (game->pt.piece_flags[n] & PF_PAIRBONUS) {
            large_bitboard_t bb;
            bb = game->board.large_bbp[n] & game->board.large_bbc[WHITE];
            mat += PAIR_BONUS * ( !is_zero128(bb & large_board_light) && !is_zero128(bb & large_board_dark) );

            bb = game->board.large_bbp[n] & game->board.large_bbc[BLACK];
            mat -= PAIR_BONUS * ( !is_zero128(bb & large_board_light) && !is_zero128(bb & large_board_dark) );
         }
      }

#ifdef USE_SLIDER_TERMS
      /* Test for sliders on open files */
      uint16_t rank_mask = (1<<large_board_files)-1;
      large_bitboard_t open_files = u128(~(closed_files[WHITE]|closed_files[BLACK]) & rank_mask, 0);
      open_files = large_fill_north(open_files);
      of += OPEN_FILE_BONUS*popcount128(open_files & large_vsliders[WHITE]);
      of -= OPEN_FILE_BONUS*popcount128(open_files & large_vsliders[BLACK]);

      /* Semi-open */
      open_files = u128(~closed_files[WHITE] & rank_mask, 0);
      open_files = large_fill_north(open_files);
      of += SEMI_OPEN_FILE_BONUS*popcount128(open_files & large_vsliders[WHITE]);
      open_files = u128(~closed_files[BLACK] & rank_mask, 0);
      open_files = large_fill_north(open_files);
      of -= SEMI_OPEN_FILE_BONUS*popcount128(open_files & large_vsliders[BLACK]);

      /* (Generalised) 7th rank bonus */
      of += SEVENTH_RANK_BONUS*popcount128(large_board_rank[large_board_ranks-2] & large_hsliders[WHITE]);
      of -= SEVENTH_RANK_BONUS*popcount128(large_board_rank[1] & large_hsliders[BLACK]);

      /* Batteries along files ("doubled rook") */
      for (n=0; n<large_board_files; n++) {
         if (popcount128(large_board_file[n] & large_vsliders[WHITE]) > 1) of += FILE_BATTERY;
         if (popcount128(large_board_file[n] & large_vsliders[BLACK]) > 1) of -= FILE_BATTERY;
      }

      /* Batteries along a diagonal */
      for (n=0; n<large_board_ranks+large_board_files; n++) {
         if (popcount128(large_board_diagonal[n]     & large_dsliders[WHITE]) > 1) of += DIAGONAL_BATTERY;
         if (popcount128(large_board_antidiagonal[n] & large_dsliders[WHITE]) > 1) of += DIAGONAL_BATTERY;
         if (popcount128(large_board_diagonal[n]     & large_dsliders[BLACK]) > 1) of -= DIAGONAL_BATTERY;
         if (popcount128(large_board_antidiagonal[n] & large_dsliders[BLACK]) > 1) of -= DIAGONAL_BATTERY;
      }
#endif
   }



   /* Initialise the king-safety counters with the score from the attack pattern.
    * This is just the normalised sum of the attack potential of the attacking pieces.
    * Cap the value at a critical number.
    */
   king_counter[WHITE] = king_attacked_pattern[WHITE] / KING_SAFETY_PIECE_WEIGHT_NORMALISATION;
   king_counter[BLACK] = king_attacked_pattern[BLACK] / KING_SAFETY_PIECE_WEIGHT_NORMALISATION;
   if (king_counter[WHITE] > KING_SAFETY_PIECE_WEIGHT_MAX) king_counter[WHITE] = KING_SAFETY_PIECE_WEIGHT_MAX;
   if (king_counter[BLACK] > KING_SAFETY_PIECE_WEIGHT_MAX) king_counter[BLACK] = KING_SAFETY_PIECE_WEIGHT_MAX;

   /* Determine whether either side is still able to force a win, based on the material on the board.
    * A side is considered to still be able to win if:
    *  1. It has a piece that can deliver mate
    *  2. It has pawns (TODO: take into account what the pawn may promote into in this case)
    *  3. It has more than 2 pieces remaining
    *  4. It has a pair of pieces that is able to force mate (eg. BB or BN, but not NN)
    * If none of these is true, then the best result is a draw. Of course, even if some of these are true, the
    * position may still be drawish.
    */
   for (side = WHITE; side < NUM_SIDES; side++) {
      if (mate_potential[side] > 1)
         can_win[side] = true;
      else {
         int non_pawn_non_royal = num_pieces[side]-num_pawns[side]-num_royals[side];
         if (num_pawns[side] > 1 || non_pawn_non_royal > 2)
            can_win[side] = true;

         if (num_pawns[side] == 0 && non_pawn_non_royal == 2 && !can_win[side]) {
            int n1, n2;
            n1 = n2 = bitscan16(piece_ids[side]);
            piece_ids[side] ^= 1<<n1;
            if (piece_ids[side])
               n2 = bitscan16(piece_ids[side]);
            if (game->pt.pieces_can_win[n1][n2])
               can_win[side] = true;
         }
      }
   }

   /* If neither side can win, skip the rest of the evaluation and return a draw-ish score. */
   if (expect(!can_win[WHITE] && !can_win[BLACK], false)) {
      ev = game->board.psq / 16;
      return ( (side_to_move == WHITE) ? ev : -ev );
   }

   /* Asses a number of special draw-ish cases. 
    * Edge pawn + wrong colour bound vs. king. Does not apply if kings can drive eachother from the corner.
    */
   if (expect(game->board.rule_flags & RF_KING_CORNERDRIVE, true))
   for (side = WHITE; side < NUM_SIDES; side++) {
      sides other = next_side[side];
      if (only_edge_pawns[side] && num_pieces[other] == 1 && num_pawns[side] == 1 && num_pieces[side] == 3) {
         if (game->large_board) {
            int n = game->pt.pawn_index[side];
            int pawn_square = bitscan128(large_pawns);
            large_bitboard_t promotion = game->pt.piece_large_promotion_zone[n] &
                                         game->pt.large_pawn_front_span[side][pawn_square];
            large_bitboard_t piece = game->board.large_bbc[side] & ~(game->board.large_royal | large_pawns);
            n = get_piece(&game->board, bitscan128(piece));

            if (game->pt.piece_flags[n] & PF_PAIRBONUS) {   /* Colour bound piece */
               large_bitboard_t bb = piece | promotion;
               if (!is_equal128((bb & large_board_light), bb) && !is_equal128((bb & large_board_dark), bb)) {   /* Wrong colour bound */
                  bb = large_king_attacks[other] | (game->board.large_royal & game->board.large_bbc[other]);
                  if (!is_zero128(bb & promotion)) {   /* The defending king is in the corner and can't be driven off, draw */
                     mat = 0;
                     break;
                  }
               }
            }
         } else {
            int n = game->pt.pawn_index[side];
            int pawn_square = bitscan64(pawns);
            bitboard_t promotion = game->pt.piece_promotion_zone[n] & game->pt.pawn_front_span[side][pawn_square];
            bitboard_t piece = game->board.bbc[side] & ~(game->board.royal | pawns);
            n = get_piece(&game->board, bitscan64(piece));

            if (game->pt.piece_flags[n] & PF_PAIRBONUS) {   /* Colour bound piece */
               bitboard_t bb = piece | promotion;
               if (!((bb & board_light) == bb) && !((bb & board_dark) == bb)) {   /* Wrong colour bound */
                  bb = king_attacks[other] | (game->board.royal & game->board.bbc[other]);
                  if (bb & promotion) {   /* The defending king is in the corner and can't be driven off, draw */
                     mat = 0;
                     break;
                  }
               }
            }
      }
      }
   }

   /* Check if one side has a lone king */
   if (piece_types[WHITE] == 1 || piece_types[BLACK] == 1) {
      sides defender = (piece_types[WHITE] == 1) ? WHITE : BLACK;

      /* Call special evaluation to mate a lone king */
      if (lone_king_evaluation(game, defender, &ps)) {
         ev = mat + ps;

         /* Tapered evaluation when we're about to hit the 50 move counter */
         if (game->fifty_counter[game->moves_played] > 80)
            ev = ev * (101 - game->fifty_counter[game->moves_played]) / 20;

         /* Adjust the score: if the side that is nominally ahead can't win, drop the score to 0(ish) */
         if (ev > 0 && !can_win[WHITE]) ev = ps;
         if (ev < 0 && !can_win[BLACK]) ev = ps;
         return (side_to_move == WHITE) ? ev : -ev;
      }
   }


   /* Pre-fetch the pawn table, we're going to need it shortly */
   //prefetch_pawntable(game->pawn_table, game->board.pawn_hash);


   /* Evaluate king positions and king safety */
   if (game->large_board) {   /* Large board */
      /* Fewer than 4 piece types on a side, assume we're in an end game.
       * Royal pieces should come out now.
       * FIXME: if there is more than one king (in particular, two), then one of the two kings is allowed to
       * advance while the other one remains safe. In that case, calculate which king is safest and which king is
       * most active and use those scores.
       * FIXME: We need smooth scaling with "game phase", however we define that.
       */
      if (piece_types[WHITE] < 4 || piece_types[BLACK] < 4) {  /* End game */
         large_bitboard_t bb;

         /* White */
         bb = game->board.large_royal & game->board.large_bbc[WHITE];
         while (!is_zero128(bb)) {
            int square = bitscan128(bb);
            bb ^= large_square_bitboards[square];

            ps += 2*centre_table[square];
         }

         /* Black */
         bb = game->board.large_royal & game->board.large_bbc[BLACK];
         while (!is_zero128(bb)) {
            int square = bitscan128(bb);
            bb ^= large_square_bitboards[square];

            ps -= 2*centre_table[square];
         }
      } else {       /* Not end game */
         large_bitboard_t bb;

         /* Number of attackers on "A" and "B" squares; includes king */
         large_bitboard_t kingb[NUM_SIDES] = { king_pattern.large_kingb[WHITE], king_pattern.large_kingb[BLACK] };
         large_bitboard_t kingab[NUM_SIDES] = { king_pattern.large_kingab[WHITE], king_pattern.large_kingab[BLACK] };
         king_counter[WHITE] += popcount128( (large_attacked[BLACK] | large_king_attacks[BLACK]) & kingab[WHITE]);
         king_counter[BLACK] += popcount128( (large_attacked[WHITE] | large_king_attacks[WHITE]) & kingab[BLACK]);

         /* Penalise more if there are no defenders for "A" and "B" type squares
          * (other than the king itself)
          */
         king_counter[WHITE] += popcount128(large_attacked[BLACK] & ~large_attacked[WHITE] & kingab[WHITE]);
         king_counter[BLACK] += popcount128(large_attacked[WHITE] & ~large_attacked[BLACK] & kingab[BLACK]);

         /* Penalise missing pieces on "B" type squares */
         king_counter[WHITE] += popcount128(kingb[WHITE]) - popcount128(kingb[WHITE] & game->board.large_bbc[WHITE]);
         king_counter[BLACK] += popcount128(kingb[BLACK]) - popcount128(kingb[BLACK] & game->board.large_bbc[BLACK]);

         /* Adjust the counter for the position of the king for white */
         bb = game->board.large_royal & game->board.large_bbc[WHITE];
         if (onebit128(bb)) {     /* One king, keep him safe */
            int square = bitscan128(bb);
            ps -= centre_table[square];

            king_counter[WHITE] += advance_table[psq_map[WHITE][square]];
            king_counter[WHITE] = min(40, king_counter[WHITE]);
            int as2 = army_attack_strength[BLACK]*army_attack_strength[BLACK];
            int kss = KING_SAFTEY_STRENGTH_SCALE*KING_SAFTEY_STRENGTH_SCALE;
            ks -= king_safety_penalty[king_counter[WHITE]] * as2/kss;
         } else {                /* Multiple kings, make them join the fight */
            while (!is_zero128(bb)) {
               int square = bitscan128(bb);
               bb ^= large_square_bitboards[square];

               ps += centre_table[square];
            }
         }

         /* Dito black */
         bb = game->board.large_royal & game->board.large_bbc[BLACK];
         if (onebit128(bb)) {     /* One king, keep him safe */
            int square = bitscan128(bb);
            ps += centre_table[square];

            king_counter[BLACK] += advance_table[psq_map[BLACK][square]];
            king_counter[BLACK] = min(40, king_counter[BLACK]);
            int as2 = army_attack_strength[WHITE]*army_attack_strength[WHITE];
            int kss = KING_SAFTEY_STRENGTH_SCALE*KING_SAFTEY_STRENGTH_SCALE;
            ks += king_safety_penalty[king_counter[BLACK]] * as2/kss;
         } else {                /* Multiple kings, make them join the fight */
            while (!is_zero128(bb)) {
               int square = bitscan128(bb);
               bb ^= large_square_bitboards[square];

               ps -= centre_table[square];
            }
         }
      }
   } else {                   /* Small board */
      /* Fewer than 4 piece types on a side, assume we're in an end game.
       * Royal pieces should come out now.
       * FIXME: if there is more than one king (in particular, two), then one of the two kings is allowed to
       * advance while the other one remains safe. In that case, calculate which king is safest and which king is
       * most active and use those scores.
       * FIXME: We need smooth scaling with "game phase", however we define that.
       */
      if (piece_types[WHITE] < 4 || piece_types[BLACK] < 4) {  /* End game */
         bitboard_t bb;

         /* White */
         bb = game->board.royal & game->board.bbc[WHITE];
         while (bb) {
            int square = bitscan64(bb);
            bb ^= make_bitboard_square(square);

            ps += 2*centre_table[square];
         }

         /* Black */
         bb = game->board.royal & game->board.bbc[BLACK];
         while (bb) {
            int square = bitscan64(bb);
            bb ^= make_bitboard_square(square);

            ps -= 2*centre_table[square];
         }
      } else {       /* Not end game */
         bitboard_t bb;

         /* Number of attackers on "A" and "B" squares; includes king */
         bitboard_t kingb[NUM_SIDES] = { king_pattern.kingb[WHITE], king_pattern.kingb[BLACK] };
         bitboard_t kingab[NUM_SIDES] = { king_pattern.kingab[WHITE], king_pattern.kingab[BLACK] };
         king_counter[WHITE] += popcount64( (attacked[BLACK] | king_attacks[BLACK]) & kingab[WHITE]);
         king_counter[BLACK] += popcount64( (attacked[WHITE] | king_attacks[WHITE]) & kingab[BLACK]);

         /* Penalise more if there are no defenders for "A" and "B" type squares
          * (other than the king itself)
          */
         king_counter[WHITE] += popcount64(attacked[BLACK] & ~attacked[WHITE] & kingab[WHITE]);
         king_counter[BLACK] += popcount64(attacked[WHITE] & ~attacked[BLACK] & kingab[BLACK]);

         /* Penalise missing pieces on "B" type squares */
         king_counter[WHITE] += popcount64(kingb[WHITE]) - popcount64(kingb[WHITE] & game->board.bbc[WHITE]);
         king_counter[BLACK] += popcount64(kingb[BLACK]) - popcount64(kingb[BLACK] & game->board.bbc[BLACK]);

         /* Adjust the counter for the position of the king for white */
         bb = game->board.royal & game->board.bbc[WHITE];
         if (num_royals[WHITE]==1) {     /* One king, keep him safe */
            int square = bitscan64(bb);
            ps -= centre_table[square];

            king_counter[WHITE] += advance_table[psq_map[WHITE][square]];
            king_counter[WHITE] = min(40, king_counter[WHITE]);
            int as2 = army_attack_strength[BLACK]*army_attack_strength[BLACK];
            int kss = KING_SAFTEY_STRENGTH_SCALE*KING_SAFTEY_STRENGTH_SCALE;
            ks -= king_safety_penalty[king_counter[WHITE]] * as2/kss;
         } else {                /* Multiple kings, make them join the fight */
            while (bb) {
               int square = bitscan64(bb);
               bb ^= make_bitboard_square(square);

               ps += centre_table[square];
            }
         }

         /* Dito black */
         bb = game->board.royal & game->board.bbc[BLACK];
         if (num_royals[BLACK]==1) {     /* One king, keep him safe */
            int square = bitscan64(bb);
            ps += centre_table[square];

            king_counter[BLACK] += advance_table[psq_map[BLACK][square]];
            king_counter[BLACK] = min(40, king_counter[BLACK]);
            int as2 = army_attack_strength[WHITE]*army_attack_strength[WHITE];
            int kss = KING_SAFTEY_STRENGTH_SCALE*KING_SAFTEY_STRENGTH_SCALE;
            ks += king_safety_penalty[king_counter[BLACK]] * as2/kss;
         } else {                /* Multiple kings, make them join the fight */
            while (bb) {
               int square = bitscan64(bb);
               bb ^= make_bitboard_square(square);

               ps -= centre_table[square];
            }
         }
      }
   }

#ifdef PENALISE_HANGING_PIECES
   /* Evaluate piece safety (hanging pieces) */
   if (game->large_board) {
      large_bitboard_t en_prise;

      /* Pieces that are truly en-prise should be rare, since the quiescence search should look at these.
       * We only get here if the piece is pinned, or there is some other threat. Still, this represents a
       * weakness we should avoid.
       */

      /* White */
      en_prise = game->board.large_bbc[WHITE] & large_attacked[BLACK] & ~large_attacked[WHITE];
      while (!is_zero128(en_prise)) {
         int square = bitscan128(en_prise);
         en_prise ^= large_square_bitboards[square];
         pt -= game->pt.piece_value[get_piece(&game->board, square)]/8;
      }

      /* Black */
      en_prise = game->board.large_bbc[BLACK] & large_attacked[WHITE] & ~large_attacked[BLACK];
      while (!is_zero128(en_prise)) {
         int square = bitscan128(en_prise);
         en_prise ^= large_square_bitboards[square];
         pt += game->pt.piece_value[get_piece(&game->board, square)]/8;
      }

      /* Pieces that are attacked but defended should still be penalised if they are
       * attacked by a piece of lower value.
       * The penalty is proportional to the difference in value between the two pieces.
       */
      large_bitboard_t threatened;

      threatened = game->board.large_bbc[WHITE] & large_attacked[BLACK] & large_attacked[WHITE];
      while (!is_zero128(threatened)) {
         int square = bitscan128(threatened);
         threatened ^= large_square_bitboards[square];

         int piece = get_piece(&game->board, square);
         int n;
         for (n=game->pt.num_piece_types-1; n>=0; n--) {
            int *perm = game->pt.val_perm;
            int loss = game->pt.piece_value[piece] - game->pt.piece_value[perm[n]];
            if (loss > 0 && !is_zero128(large_piece_attacks[BLACK][perm[n]] & large_square_bitboards[square])) {
               pt -= loss / 8;
            }
         }
      }

      threatened = game->board.large_bbc[BLACK] & large_attacked[BLACK] & large_attacked[WHITE];
      while (!is_zero128(threatened)) {
         int square = bitscan128(threatened);
         threatened ^= large_square_bitboards[square];

         int piece = get_piece(&game->board, square);
         int n;
         for (n=game->pt.num_piece_types-1; n>=0; n--) {
            int *perm = game->pt.val_perm;
            int loss = game->pt.piece_value[piece] - game->pt.piece_value[perm[n]];
            if (loss > 0 && !is_zero128(large_piece_attacks[WHITE][perm[n]] & large_square_bitboards[square])) {
               pt += loss / 8;
            }
         }
      }
   } else {
      bitboard_t en_prise;

      /* Pieces that are truly en-prise should be rare, since the quiescence search should look at these.
       * We only get here if the piece is pinned, or there is some other threat. Still, this represents a
       * weakness we should avoid.
       */

      /* White */
      en_prise = game->board.bbc[WHITE] & attacked[BLACK] & ~attacked[WHITE];
      while (en_prise) {
         int square = bitscan64(en_prise);
         en_prise ^= make_bitboard_square(square);
         pt -= game->pt.piece_value[get_piece(&game->board, square)]/8;
      }

      /* Black */
      en_prise = game->board.bbc[BLACK] & attacked[WHITE] & ~attacked[BLACK];
      while (en_prise) {
         int square = bitscan64(en_prise);
         en_prise ^= make_bitboard_square(square);
         pt += game->pt.piece_value[get_piece(&game->board, square)]/8;
      }

      /* Pieces that are attacked but defended should still be penalised if they are
       * attacked by a piece of lower value.
       * The penalty is proportional to the difference in value between the two pieces.
       */
      bitboard_t threatened;

      threatened = game->board.bbc[WHITE] & attacked[BLACK] & attacked[WHITE];
      while (threatened) {
         int square = bitscan64(threatened);
         threatened ^= make_bitboard_square(square);

         int piece = get_piece(&game->board, square);
         int n;
         for (n=game->pt.num_piece_types-1; n>=0; n--) {
            int *perm = game->pt.val_perm;
            int loss = game->pt.piece_value[piece] - game->pt.piece_value[perm[n]];
            if (loss > 0 && piece_attacks[BLACK][perm[n]] & make_bitboard_square(square)) {
               pt -= loss / 8;
            }
         }
      }

      threatened = game->board.bbc[BLACK] & attacked[BLACK] & attacked[WHITE];
      while (threatened) {
         int square = bitscan64(threatened);
         threatened ^= make_bitboard_square(square);

         int piece = get_piece(&game->board, square);
         int n;
         for (n=game->pt.num_piece_types-1; n>=0; n--) {
            int *perm = game->pt.val_perm;
            int loss = game->pt.piece_value[piece] - game->pt.piece_value[perm[n]];
            if (loss > 0 && piece_attacks[WHITE][perm[n]] & make_bitboard_square(square)) {
               pt += loss / 8;
            }
         }
      }
   }
#endif

   /* Evaluate pawn structure */
   //pawn = evaluate_pawn_structure(game);

   /* Piece square tables */
   ps += game->board.psq;

 //printf("%d %d %d %d %d [%d %d] %d %d\n", side_to_move, dev, mat, mob, ks, king_counter[WHITE], king_counter[BLACK], ps, pt);
   ev = dev + mat + ks + (mob + ps)/4 + pt + of + pawn;

   /* Trade down bonus: if ahead in material, trade pieces. Otherwise, trade pawns. */
   int pieces = num_pieces[WHITE] + num_pieces[BLACK] - num_pawns[WHITE] - num_pawns[BLACK];
   if (pieces < 6) {
      if (mat > 0) {
         ev -= pieces * 2;
         ev += (num_pawns[WHITE]  + num_pawns[BLACK])  * 4;
      } else {
         ev += pieces * 2;
         ev -= (num_pawns[WHITE]  + num_pawns[BLACK])  * 4;
      }
   }

   /* Without pawns it's harder to win, so penalise for having fewer pawns */
   if (num_pawns[WHITE] < 5) ev -= (5 - num_pawns[WHITE] + (num_pawns[WHITE] == 0)) * PAWN_STEP_ENDGAME;
   if (num_pawns[BLACK] < 5) ev += (5 - num_pawns[BLACK] + (num_pawns[BLACK] == 0)) * PAWN_STEP_ENDGAME;

   /* If we're only up against pieces and have only one piece with mate potential, things become more
    * difficult because we can't just exchange one piece.
    */
   if (ev > 0 && num_pawns[WHITE] == 0 && num_pawns[BLACK] == 0 && piece_types[BLACK] > 1 && mate_potential[WHITE] < 2) ev /= 2;
   if (ev < 0 && num_pawns[WHITE] == 0 && num_pawns[BLACK] == 0 && piece_types[WHITE] > 1 && mate_potential[BLACK] < 2) ev /= 2;

   /* Adjust the score: if the side that is nominally ahead can't win, drop the score to 0(ish) */
   if (ev > 0 && !can_win[WHITE]) ev = ps/16;
   if (ev < 0 && !can_win[BLACK]) ev = ps/16;

   /* Store in the eval hash table for later retrieval */
   store_eval_table_entry(game->eval_table, game->board.hash, ev);

   /* Tapered evaluation when we're about to hit the 50 move counter */
   if (game->fifty_counter[game->moves_played] > 80)
      ev = ev * (101 - game->fifty_counter[game->moves_played]) / 20;

   return ( (side_to_move == WHITE) ? ev : -ev ) + SIDE_TO_MOVE_BONUS;
}

