/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include "pieces.h"
#include "movegen.h"
#include "board.h"
#include "see.h"
#include "bits.h"

#undef PRINT_SEE_DEBUG_MESSAGES

static inline int max(int x, int y)
{
   return (x>y)?x:y;
}

static inline int find_least_valued_piece(const board_t *board, ubitboard_t own)
{
   int *perm = board->piece_types->val_perm;
   int n;

   if (!board->large_board) {
      for (n=0; n<board->piece_types->num_piece_types; n++) {
         if (board->bbp[perm[n]] & own.bb)
            return bitscan64(board->bbp[perm[n]] & own.bb);
      }
   } else {
      for (n=0; n<board->piece_types->num_piece_types; n++) {
         if (!is_zero128(board->large_bbp[perm[n]] & own.lbb))
            return bitscan128(board->large_bbp[perm[n]] & own.lbb);
      }
   }

   /* should never get here if everything works correctly for own!=0 */
   assert(!is_zero128(own.lbb));

   return -1;
}

static int move_value(int16_t *piece_value, move_t move, sides side_to_move)
{
   int value = 0;
   int n, c;

   /* First: resolve all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);

      value -= (side == WHITE) ? piece_value[piece] : -piece_value[piece];
   }

   /* Second: resolve all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);

      value += (side == WHITE) ? piece_value[piece] : -piece_value[piece];
   }

   return (side_to_move == WHITE) ? value : -value;
}

/* Perform static exchange evaluation for a given capture move.
 * Uses the "swap" algorithm
 * (http://chessprogramming.wikispaces.com/SEE+-+The+Swap+Algorithm).
 * returns the evaluation in centipawns.
 */
int static_exchange_evaluation(const board_t *const_board, const move_t move)
{
   int score[32];
   int depth;
   int piece = get_move_piece(move);
   int square = get_move_to(move);
   ubitboard_t attackers;
   ubitboard_t own;
   board_t board;

   assert(is_capture_move(move));
   assert(get_move_player(move) == const_board->side_to_move);

   if (const_board->large_board) {
      assert(!is_zero128(const_board->large_bbc[get_move_player(move)] & large_square_bitboards[get_move_from(move)]));
   } else {
      assert((const_board->bbc[get_move_player(move)] & make_bitboard_square(get_move_from(move))));
   }

   /* Clear attackers and own bitboards */
   memset(&attackers, 0, sizeof attackers);
   memset(&own, 0, sizeof own);

   /* Make a copy of the board */
   memcpy(&board, const_board, sizeof *const_board);

   /* Initial gain: the piece on the target square */
   depth = 0;
   score[depth] = move_value(board.piece_types->piece_value, move, board.side_to_move);

   /* Easy: if the piece we use to capture the enemy piece is worth less
    * than the piece we captured, it's always a winning capture, so return a
    * rough estimate for the SEE score.
    */
   if (score[depth] > board.piece_types->piece_value[piece])
      return score[depth];

   /* Perform the capture on the board */
   makemove_see(&board, move);

   /* Get a new list of all attackers/defenders */
   attackers = get_all_attackers(&board, square);

   if (!is_zero128(attackers.lbb)) {
      do {
         /* Consider next move - one ply deeper, flip side */
         depth++;
         assert(depth < 128);
         /* Update score (speculative) */
         score[depth] = get_piece_value(board.piece_types, get_piece(&board, square)) - score[depth-1];
         /* Perform LVA capture */
         if (!board.large_board)
            own.bb = attackers.bb&board.bbc[board.side_to_move];
         else
            own.lbb = attackers.lbb&board.large_bbc[board.side_to_move];
         if (!is_zero128(own.lbb)) {
            /* Find the least valued attacker */
            int origin = find_least_valued_piece(&board, own);

            if (const_board->large_board) {
               assert(!is_zero128(board.large_bbc[board.side_to_move] & large_square_bitboards[origin]));
            } else {
               assert(board.bbc[board.side_to_move] & make_bitboard_square(origin));
            }

            int piece = piece_for_side(board.piece[origin], board.side_to_move);
            int ptaken = piece_for_side(board.piece[square], next_side[board.side_to_move]);
            move_t move = encode_normal_capture(piece, origin, square, ptaken);

            /* Perform the capture on the board */
            makemove_see(&board, move);

            /* Update the list of attackers/defenders */
            //attackers = get_all_attackers(&board, square);
            if (!board.large_board)
               attackers.bb &= ~make_bitboard_square(origin);
            else
               attackers.lbb &= ~large_square_bitboards[origin];
         }
      } while(!is_zero128(own.lbb));

      /* Compute final score */
      while (--depth) {
         score[depth-1] = -max(-score[depth-1], score[depth]);
      }
   }
   return score[depth];
}

