/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include "hashtable.h"

#define HASH_BUCKETS    3

static inline size_t map_key_to_index(uint64_t key, size_t nelem)
{
   return key & ( nelem - 1);
}

hash_table_t *create_hash_table(size_t nelem)
{
   hash_table_t *table = calloc(1, sizeof *table);
   table->data = calloc(nelem + HASH_BUCKETS, sizeof *table->data);
   table->number_of_elements = nelem;
   return table;
}

void destroy_hash_table(hash_table_t *table)
{
   if (table) {
      free(table->data);
      free(table);
   }
}

void store_table_entry(hash_table_t *table, uint64_t key, int depth, int score, unsigned int flags, move_t best_move)
{
   hash_table_entry_t *data = NULL;
   size_t index, b;

   if (!table)
      return;

   /* Map the key onto the array indexi */
   index = map_key_to_index(key, table->number_of_elements);

   /* First step: test all buckets. If we find a match, then store the result. */
   for (b=0; b<HASH_BUCKETS; b++) {
      if (table->data[index + b].key == key) {
         data = table->data+index+b;
         break;
      }
   }

   /* Second step: find an older entry with a lower draft */
   if (!data) {
      int lowest_draft = 1000;
      for (b=0; b<HASH_BUCKETS; b++) {
         if (table->data[index + b].generation < table->generation &&
             table->data[index + b].depth < lowest_draft) {
            lowest_draft = table->data[index + b].depth;
            data = table->data+index+b;
            break;
         }
      }

      /* Third step: replace the lowest-draft entry */
      if (!data) {
         for (b=0; b<HASH_BUCKETS; b++) {
            if (table->data[index + b].depth < lowest_draft) {
               lowest_draft = table->data[index + b].depth;
               data = table->data+index+b;
               break;
            }
         }
      }
   }
   
   assert(data);

   data->key = key;
   data->depth = depth;
   data->score = score;
   data->flags = flags;
   data->best_move = best_move;
   data->generation = table->generation;
   table->write_count++;
}

hash_table_entry_t *query_table_entry(hash_table_t *table, uint64_t key)
{
   size_t index;

   if (!table)
      return NULL;

   /* Map the key onto the array index, check if entry is there */
   index = map_key_to_index(key, table->number_of_elements);

   /* Check all buckets */
   size_t b = 0;
   for (b = 0; b<HASH_BUCKETS; b++) {
      if (table->data[index+b].key == key) {
         table->data[index+b].generation = table->generation;
         return &table->data[index+b];
      }
   }

   return NULL;
}

void prepare_hashtable_search(hash_table_t *table)
{
   if (!table) return;
   table->generation++;
   table->write_count = 0;
}

void prefetch_hashtable(hash_table_t *table, uint64_t key)
{
   size_t index;

   index = map_key_to_index(key, table->number_of_elements);
   __builtin_prefetch(table->data+index);
}

int count_unused_table_entries(hash_table_t *table)
{
   int n, count;
   if (!table)
      return 0;

   count = 0;

   for (n=0; n<table->number_of_elements; n++) {
      if (table->data[n].generation == 0)
         count++;
   }

   return count;
}
