/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <ctype.h>
#include <stdint.h>
#include <allegro5/allegro.h>
#include <allegro5/allegro_image.h>
#include <allegro5/allegro_ttf.h>
#include <allegro5/allegro_native_dialog.h>
#include <allegro5/allegro_primitives.h>

#include "sjaak.h"

#define PROGRAM_NAME "Sjaak"

#define BOARD_WIDTH  (num_files*square_size)
#define BOARD_HEIGHT ((num_ranks + (river != 0))*square_size)

static int square_size = 80;
static int border_size = 16;
static int top_margin = 20;
static int bottom_margin = 12;
static int iteration_font_size = 16;
static int board_font_size = 12;

static ALLEGRO_DISPLAY *board_display;
static ALLEGRO_DISPLAY *key_display;
static ALLEGRO_BITMAP *figurine[MAX_PIECE_TYPES][NUM_SIDES];
static ALLEGRO_BITMAP *cursor;
static ALLEGRO_FONT *board_font;
static ALLEGRO_FONT *font;
static ALLEGRO_MOUSE_CURSOR *mouse_cursor = NULL;
static ALLEGRO_PATH *resource_path = NULL;

static int window_width = 0;
static int window_height = 0;

static int num_files = 8;
static int num_ranks = 8;
static int river = 0;

static int board_x = 0, board_y = 0;
static int moves_x = 0, moves_y = 0;
static int moves_w = 0, moves_h = 0;
static int mouse_file = -1, mouse_row = -1;
static int mouse_pickup_square = 0;
static int wclock_x = 0, wclock_y = 0;
static int bclock_x = 0, bclock_y = 0;
static int clock_w = 0;

static int drop_x = 0;
static int drop_y = 0;
static int drop_w = 0;
static int drop_h = 0;
static int drop_str_w = 32;
static bool mouse_menu = false;
static bool menu_open = false;

static int drop_piece = -1;
static large_bitboard_t large_legal_moves;
static large_bitboard_t large_mouse_drop;
static bitboard_t legal_moves;
static bitboard_t mouse_drop;

static movelist_t movelist;

static int royal_index = 0;

#define total_time (.1*5 * 60 * 1000)
static int64_t clock_time[2] = { total_time, total_time };
bool clock_running = false;

/* Standard "Merida" chess font mapping. Just the standard set. */
static uint32_t std_chess_font_map[] = {
   0x0E254, /* king, filling */
   0x02654, /* white king */
   0x0E255, /* queen, filling */
   0x02655, /* white queen */
   0x0E256, /* rook, filling */
   0x02656, /* white rook */
   0x0E257, /* bishop, filling */
   0x02657, /* white bishop */
   0x0E258, /* knight, filling */
   0x02658, /* white knight */
   0x0E259, /* pawn, filling */
   0x02659, /* white pawn */

   0x0E254, /* king, filling */
   0x0265A, /* black king */
   0x0E255, /* queen, filling */
   0x0265B, /* black queen */
   0x0E256, /* rook, filling */
   0x0265C, /* black rook */
   0x0E257, /* bishop, filling */
   0x0265D, /* black bishop */
   0x0E258, /* knight, filling */
   0x0265E, /* black knight */
   0x0E259, /* pawn, filling */
   0x0265F, /* black pawn */
   0
};

/* Extended "Quivera" chess font mapping. */
static uint32_t ext_chess_font_map[] = {
   /* White pieces */
   0x0265a, /* king filling */
   0x02654, /* king */
   0x0265b, /* queen filling */
   0x02655, /* queen */
   0x0265c, /* rook filling */
   0x02656, /* rook */
   0x0265d, /* bishop filling */
   0x02657, /* bishop */
   0x0265e, /* knight filling */
   0x02658, /* knight */
   0x0265f, /* pawn filling */
   0x02659, /* pawn */

   0x0E011, /* Elephant */
   0x0E010,
   0x0E013, /* Mandarin */
   0x0E012,
   0x0E015, /* Cannon */
   0x0E014,
   0x0E017, /* Chinese pawn */
   0x0E016,
   0x0E019, /* Ferz */
   0x0E018,
   0x0E01B, /* Wazir */
   0x0E01A,
   0x0E01D, /* Giraffe */
   0x0E01C,
   0x0E01F, /* Picket */
   0x0E01E,
   0x0E021, /* Camel */
   0x0E020,
   0x0E023, /* War Engine; Dabbaba */
   0x0E022,
   0x0E025, /* Pawn of pawns */
   0x0E024,
   0x0E027,
   0x0E026,
   0x0E029,
   0x0E028,
   0x0E02B,
   0x0E02A,
   0x0E02D,
   0x0E02C,
   0x0E02F,
   0x0E02E,
   0x0E031,
   0x0E030,
   0x0E033,
   0x0E032,
   0x0E035,
   0x0E034,
   0x0E037,
   0x0E036,
   0x0E039,
   0x0E038,
   0x0E03B, /* Prince */
   0x0E03A,
   0x0E03D, /* Adventitious King */
   0x0E03C,
   0x0E03F, /* Gold General */
   0x0E03E,
   0x0E041, /* Silver General */
   0x0E040,
   0x0E043, /* Lance */
   0x0E042,
   0x0E045, /* Dragon king */
   0x0E044,
   0x0E047, /* Dragon Horse */
   0x0E046,
   0x0E049, /* Promoted silver general */
   0x0E048,
   0x0E04B, /* Promoted lance */
   0x0E04A,
   0x0E04D  /* Promoted Horse */,
   0x0E04C,
   0x0E04F, /* Promoted pawn */
   0x0E04E,

   0x0E141, /* Berolina pawn */
   0x0E140,
   0x0E143,
   0x0E142,
   0x0E145, /* Man; Commoner */
   0x0E144,
   0x0E149, /* Vao */
   0x0E148,
   0x0E14B, /* Windmill */
   0x0E14A,
   0x0E14D,
   0x0E14C,
   0x0E14F,
   0x0E14E,
   0x0E15F,
   0x0E15E,
   0x0E16B,
   0x0E16A,
   0x0E16D,
   0x0E16C,
   0x0E181,
   0x0E180,
   0x0E183,
   0x0E182,
   0x0E185,
   0x0E184,
   0x0E187,
   0x0E186,
   0x0E189,
   0x0E188,
   0x0E191,
   0x0E190,
   0x0E193,
   0x0E192,
   0x0E195,
   0x0E194,
   0x0E197,
   0x0E196,
   0x0E199,
   0x0E198,
   0x0E19B,
   0x0E19A,
   0x0E19D,
   0x0E19C,
   0x0E19F,
   0x0E19E,
   0x0E1A1,
   0x0E1A0,
   0x0E1A3,
   0x0E1A2,
   0x0E1A5,
   0x0E1A4,
   0x0E1A7,
   0x0E1A6,

   /* Misc. other (playing cards) */
   0x02660,
   0x02664,
   0x02665,
   0x02661,
   0x02666,
   0x02662,
   0x02663,
   0x02667,

   0x00020,
   0x0E000,
   0x00020,
   0x0E001,
   0x00020,
   0x0E002,
   0x00020,
   0x0E003,
   0x00020,
   0x0E004,
   0x00020,
   0x0E005,
   0x00020,
   0x0E006,
   0x00020,
   0x0E007,
   0x00020,
   0x0E008,
   0x00020,
   0x0E009,
   0x00020,
   0x0E00A,
   0x00020,
   0x0E00B,
   0x00020,
   0x0E00C,
   0x00020,
   0x0E00D,
   0x00020,
   0x0E00E,

   /* Black pieces */
   0x02654, /* king filling */
   0x0265A, /* king */
   0x00020, /* queen filling */
   0x0265B, /* queen */
   0x02656, /* rook filling */
   0x0265C, /* rook */
   0x02657, /* bishop filling */
   0x0265D, /* bishop */
   0x02658, /* knight filling */
   0x0265E, /* knight */
   0x02659, /* pawn filling */
   0x0265F, /* pawn */

   0x0E010,
   0x0E011,
   0x0E012,
   0x0E013,
   0x0E014,
   0x0E015,
   0x0E016,
   0x0E017,
   0x0E018,
   0x0E019,
   0x0E01A,
   0x0E01B,
   0x0E01C,
   0x0E01D,
   0x0E01E,
   0x0E01F,
   0x0E020,
   0x0E021,
   0x0E022,
   0x0E023,
   0x0E024,
   0x0E025,
   0x0E026,
   0x0E027,
   0x0E028,
   0x0E029,
   0x0E02A,
   0x0E02B,
   0x0E02C,
   0x0E02D,
   0x0E02E,
   0x0E02F,
   0x0E030,
   0x0E031,
   0x0E032,
   0x0E033,
   0x0E034,
   0x0E035,
   0x0E036,
   0x0E037,
   0x0E038,
   0x0E039,
   0x0E03A,
   0x0E03B,
   0x0E03C,
   0x0E03D,
   0x0E03E,
   0x0E03F,
   0x0E040,
   0x0E041,
   0x0E042,
   0x0E043,
   0x0E044,
   0x0E045,
   0x0E046,
   0x0E047,
   0x0E048,
   0x0E049,
   0x0E04A,
   0x0E04B,
   0x0E04C,
   0x0E04D,
   0x0E04E,
   0x0E04F,

   0x0E140,
   0x0E141,
   0x0E142,
   0x0E143,
   0x0E144,
   0x0E145,
   0x0E148,
   0x0E149,
   0x0E14A,
   0x0E14B,
   0x0E14C,
   0x0E14D,
   0x0E14E,
   0x0E14F,
   0x0E15E,
   0x0E15F,
   0x0E16A,
   0x0E16B,
   0x0E16C,
   0x0E16D,
   0x0E180,
   0x0E181,
   0x0E182,
   0x0E183,
   0x0E184,
   0x0E185,
   0x0E186,
   0x0E187,
   0x0E188,
   0x0E189,
   0x0E190,
   0x0E191,
   0x0E192,
   0x0E193,
   0x0E194,
   0x0E195,
   0x0E196,
   0x0E197,
   0x0E198,
   0x0E199,
   0x0E19A,
   0x0E19B,
   0x0E19C,
   0x0E19D,
   0x0E19E,
   0x0E19F,
   0x0E1A0,
   0x0E1A1,
   0x0E1A2,
   0x0E1A3,
   0x0E1A4,
   0x0E1A5,
   0x0E1A6,
   0x0E1A7,

   /* Misc. other (playing cards) */
   0x02664,
   0x02660,
   0x02661,
   0x02665,
   0x02662,
   0x02666,
   0x02667,
   0x02663,

   0x00020,
   0x0E000,
   0x00020,
   0x0E001,
   0x00020,
   0x0E002,
   0x00020,
   0x0E003,
   0x00020,
   0x0E004,
   0x00020,
   0x0E005,
   0x00020,
   0x0E006,
   0x00020,
   0x0E007,
   0x00020,
   0x0E008,
   0x00020,
   0x0E009,
   0x00020,
   0x0E00A,
   0x00020,
   0x0E00B,
   0x00020,
   0x0E00C,
   0x00020,
   0x0E00D,
   0x00020,
   0x0E00E,

   0
};

/* Chess "Leipzig" font */
static uint32_t leip_chess_font_map[] = {
   'l', 'k', 'w', 'q', 't', 'r', 'v', 'b', 'm', 'n', 'o', 'p',
   'k', 'l', 'q', 'w', 'r', 't', 'b', 'v', 'n', 'm', 'p', 'o',
   //' ', 'k', ' ', 'q', ' ', 'r', ' ', 'b', ' ', 'n', ' ', 'p',
   //' ', 'l', ' ', 'w', ' ', 't', ' ', 'v', ' ', 'm', ' ', 'o',
   0
};

/* Chess "Draughts" font */
static uint32_t draughts_font_map[] = {
   'g', 'h', 'b', 'd',
   'h', 'g', 'd', 'b',
   0
};

typedef struct chess_font_map_t {
   char *fontfile;
   uint32_t *map;
   int num_char_side;
} chess_font_map_t;

#define CHESSFONT(name, map) { name, map, (sizeof(map)/sizeof(*map)-1)/4 }

static chess_font_map_t chess_fonts[] = {
   CHESSFONT( "gfx/chess_merida_unicode.ttf", std_chess_font_map ),
   CHESSFONT( "gfx/draughts.ttf", draughts_font_map ),
   CHESSFONT( "gfx/magnetic.ttf", leip_chess_font_map ),
   CHESSFONT( "gfx/leipzig.ttf", leip_chess_font_map ),
   CHESSFONT( "gfx/condal.ttf", leip_chess_font_map ),
   CHESSFONT( "gfx/medieval.ttf", leip_chess_font_map ),
   CHESSFONT( "gfx/kingfont.ttf", leip_chess_font_map ),
   CHESSFONT( "gfx/adventure.ttf", leip_chess_font_map ),
   CHESSFONT( "gfx/Quivira.ttf", ext_chess_font_map ),
};

#define num_font_files (sizeof chess_fonts / sizeof *chess_fonts)
static ALLEGRO_FONT *figurine_font[num_font_files];
static ALLEGRO_FONT *figurine_text_font[num_font_files];

static char chess_char_map[8192];

typedef struct {
   ALLEGRO_FONT *font;
   ALLEGRO_FONT *text_font;
   char *name;
   char *w1, *w2;
   char *b1, *b2;
   uint32_t wi, bi;
} chess_figurine_t;
static chess_figurine_t *font_figurine = NULL;
static int num_font_figurines = 0;
static chess_figurine_t *figurine_glyph[32];

typedef game_t *(*new_game_func_t)(void);
static new_game_func_t create_variant_game[] = {
   create_standard_game,
   create_berolina_game,
   create_pocketknight_game,
   create_chinese_game,
   create_amazon_game,
   create_spartan_game,
   create_maharaja_game,
   create_knightmate_game,
   create_shatranj_game,
   create_makruk_game,

   create_capablanca_game,
   create_gothic_game,
   create_embassy_game,
   create_janus_game,

   create_grand_game,
   create_indiangrand_game,

   create_courier_game,

   NULL
};
static int selected_variant = 0;
static int number_of_variants = sizeof create_variant_game / sizeof *create_variant_game - 1;

static char *variant_names[sizeof create_variant_game / sizeof *create_variant_game];
static char *fairy_file = NULL;
static int fairy_file_games = 0;
static char **fairy_file_names = NULL;

static int max(int i1, int i2) { return (i1>i2) ? i1 : i2; }
static int min(int i1, int i2) { return (i1>i2) ? i2 : i1; }

/* utf8_setc:
 *  Sets a character in a UTF-8 string.
 *  Taken from Allegro 4.
 */
static int utf8_setc(char *s, int c)
{
   int size, bits, b, i;

   if (c < 128) {
      *s = c;
      return 1;
   }

   bits = 7;
   while (c >= (1<<bits))
      bits++;

   size = 2;
   b = 11;

   while (b < bits) {
      size++;
      b += 5;
   }

   b -= (7-size);
   s[0] = c>>b;

   for (i=0; i<size; i++)
      s[0] |= (0x80>>i);

   for (i=1; i<size; i++) {
      b -= 6;
      s[i] = 0x80 | ((c>>b)&0x3F);
   }

   return size;
}

static void index_fairy_file(const char *filename)
{
   FILE *f;

   f = fopen(filename, "r");
   if (!f) return;
   char line[4096];

   while (!feof(f)) {
      fgets(line, sizeof line, f);
      /* Strip away comments */
      char *s = strstr(line, "#");
      if (s) s[0] = '\0';

      /* Snip end-of-line */
      s = strstr(line, "\n");
      if (s) s[0] = '\0';

      /* Strip trailing space */
      s = line+strlen(line)-1;
      while (s > line && isspace(s[0])) { s[0] = '\0'; s--; }

      /* New variant */
      if (strstr(line, "Variant:") == line) {
         s = line + 8;
         while (isspace(*s)) s++;
         fairy_file_names = realloc(fairy_file_names, (fairy_file_games+1)*sizeof *fairy_file_names);
         fairy_file_names[fairy_file_games] = strdup(s);
         fairy_file_games++;
         continue;
      }

   }
}

static void load_figurine_fonts()
{
   int nf = 0;
   int n, k, x;
   num_font_figurines = 0;

   for (nf=0; nf<num_font_files; nf++) {
      char *fontname = chess_fonts[nf].fontfile;

      if (al_filename_exists(fontname)) {
         figurine_font[nf] = al_load_font(fontname, square_size, ALLEGRO_TTF_NO_KERNING);
         figurine_text_font[nf] = al_load_font(fontname, iteration_font_size, ALLEGRO_TTF_NO_KERNING);
      } else {
         ALLEGRO_PATH *path = al_create_path(fontname);
         al_rebase_path(resource_path, path);
         figurine_font[nf] = al_load_font(al_path_cstr(path, ALLEGRO_NATIVE_PATH_SEP), square_size, ALLEGRO_TTF_NO_KERNING);
         figurine_text_font[nf] = al_load_font(al_path_cstr(path, ALLEGRO_NATIVE_PATH_SEP), iteration_font_size, ALLEGRO_TTF_NO_KERNING);
         al_destroy_path(path);
      }
      if (!figurine_font[nf]) {
         //al_show_native_message_box(NULL, "File not found", "", "Couldn't load font", "Exit", ALLEGRO_MESSAGEBOX_ERROR);
         printf("Can't load font file '%s'\n", fontname);
         exit(0);
      }

      /* Count the number of figurines in the font */
      num_font_figurines += chess_fonts[nf].num_char_side;
   }

   /* Now map all fonts */
   font_figurine = realloc(font_figurine, num_font_figurines * sizeof *font_figurine);
   memset(font_figurine, 0, num_font_figurines * sizeof *font_figurine);
   memset(chess_char_map, 0, sizeof chess_char_map);

   n = 0;
   x = 0;
   for (nf=0; nf<num_font_files; nf++) {
      for(k=0; k<chess_fonts[nf].num_char_side; k++) {
         font_figurine[n].text_font = figurine_text_font[nf];
         font_figurine[n].font = figurine_font[nf];
         //printf("%d %s %d\n", nf, chess_fonts[nf].fontfile, chess_fonts[nf].num_char_side);

         char *w1 = chess_char_map + x;
         x += utf8_setc(w1, chess_fonts[nf].map[2*k])+1;
         chess_char_map[x] = '\0';
         x++;

         char *w2 = chess_char_map + x;
         x += utf8_setc(w2, chess_fonts[nf].map[2*k + 1])+1;
         chess_char_map[x] = '\0';
         x++;

         char *b1 = chess_char_map + x;
         x += utf8_setc(b1, chess_fonts[nf].map[2*k + 2*chess_fonts[nf].num_char_side])+1;
         chess_char_map[x] = '\0';
         x++;

         char *b2 = chess_char_map + x;
         x += utf8_setc(b2, chess_fonts[nf].map[2*k + 2*chess_fonts[nf].num_char_side + 1])+1;
         chess_char_map[x] = '\0';
         x++;

         font_figurine[n].wi = chess_fonts[nf].map[2*k + 1];
         font_figurine[n].bi = chess_fonts[nf].map[2*k + 2*chess_fonts[nf].num_char_side + 1];
         font_figurine[n].w1 = w1;
         font_figurine[n].w2 = w2;
         font_figurine[n].b1 = b1;
         font_figurine[n].b2 = b2;
         n++;
      }
   }
}


static bool move_in_list(int from, int to)
{
   int n;
   for (n=0; n<movelist.num_moves; n++)
      if (get_move_from(movelist.move[n]) == from && get_move_to(movelist.move[n]) == to)
         return true;

   return false;
}

static move_t move_from_list(int from, int to)
{
   int n;
   for (n=0; n<movelist.num_moves; n++) {
      if (get_move_from(movelist.move[n]) == from && get_move_to(movelist.move[n]) == to)
         return movelist.move[n];
   }

   return 0;
}

static void get_movelist(game_t *game)
{
   movelist_t ml;
   sides side = game->board.side_to_move;

   generate_moves(&ml, &game->board, side);

   /* Filter out moves that leave the king in check */
   int n, k;
   k = 0;
   for (n=0; n<ml.num_moves; n++) {
      move_t move = ml.move[n];
      playmove(game, move);
      if (!player_in_check(game, side)) {
         movelist.move[k] = move;
         k++;
      }
      takeback(game);
   }

   movelist.num_moves = k;
}

void generate_legal_move_bitboard(game_t *game, int file, int rank)
{
   legal_moves = board_empty;
   large_legal_moves = large_board_empty;
   int square = pack_row_file(rank, file);
   int piece = get_piece(&game->board, square);
   sides side = get_piece_colour(&game->board, square);
   if (side == game->board.side_to_move) {
      int n;
      for (n=0; n<movelist.num_moves; n++) {
         if (get_move_from(movelist.move[n]) == square) {
            if (game->large_board)
               large_legal_moves |= large_square_bitboards[get_move_to(movelist.move[n])];
            else
               legal_moves |= make_bitboard_square(get_move_to(movelist.move[n]));
         }
      }
   }
}


void draw_board(game_t *game, board_t *board)
{
   static char title[256];
   large_bitboard_t large_kings = large_board_empty;
   large_bitboard_t palace = large_board_empty;
   bitboard_t kings = board_empty;
   ALLEGRO_COLOR white = al_map_rgba_f(1, 1, 1, 1);
   ALLEGRO_COLOR black = al_map_rgba_f(0, 0, 0, 1);
   ALLEGRO_COLOR grey = al_map_rgba_f(.75, .75, .75, 1);
   ALLEGRO_COLOR light = al_map_rgba_f(.925, .925, .875, 1);
   ALLEGRO_COLOR dark = al_map_rgba_f(.75, .55, .225, 1);
   ALLEGRO_COLOR bg = al_map_rgba_f(.325, .175, .1125, 1);
   ALLEGRO_COLOR selected = al_map_rgba_f(0.5625, .3625, 0.075, .75);
   ALLEGRO_COLOR select = al_map_rgba_f(0, .5, 0, .75);
   ALLEGRO_COLOR move_col = al_map_rgba_f(0, .5, 0, .6);
   ALLEGRO_COLOR check = al_map_rgba(220, 20, 60, 192);
   double score;
   int x, y;
   int size;

   bool checkered = true;
   int royal = 0;

   /* Chinese-style board */
   if (game->board.rule_flags & RF_KING_TRAPPED) {
      int n;
      for (n=0; n<game->pt.num_piece_types; n++) {
         if (game->pt.piece_flags[n] & PF_ROYAL) {
            royal = n;
            break;
         }
      }

      palace = game->pt.large_prison[royal];
      checkered = false;
   }

   al_set_target_backbuffer(board_display);
   al_clear_to_color(light);
   al_draw_filled_rectangle(0, 0, window_width, top_margin, black);
   al_draw_filled_rectangle(board_x, board_y, window_width, board_y + BOARD_HEIGHT + 2*border_size, bg);
   al_draw_filled_rectangle(moves_x, moves_y, moves_x + moves_w, moves_y + moves_h, bg);
   al_draw_filled_rectangle(moves_x+border_size, moves_y+border_size, moves_x + moves_w - border_size, moves_y + moves_h-border_size, light);
   al_draw_filled_rectangle(wclock_x, wclock_y, wclock_x+clock_w, wclock_y+iteration_font_size, bg);
   al_draw_filled_rectangle(bclock_x, bclock_y, bclock_x+clock_w, wclock_y+iteration_font_size, bg);
   //al_draw_filled_rectangle(0, window_height - iteration_font_size, window_width, window_height, light);

   if (board->side_to_move == BLACK) {
      snprintf(title, 255, PROGRAM_NAME" (%s) - Black to play", game->name);
   } else {
      snprintf(title, 255, PROGRAM_NAME" (%s) - White to play", game->name);
   }
   al_set_window_title(board_display, title);

   size = square_size;

   if (player_in_check_for_board(game, board, board->side_to_move)) {
      kings = board->royal & board->bbc[board->side_to_move];
      large_kings = board->large_royal & board->large_bbc[board->side_to_move];
   }

   bitboard_t occ = board->bbc[WHITE] | board->bbc[BLACK];
   large_bitboard_t large_occ = board->large_bbc[WHITE] | board->large_bbc[BLACK];
   for (y=0; y<num_ranks; y++) {
      for (x=0; x<num_files; x++) {
         ALLEGRO_COLOR colour = light;
         int where = pack_row_file(y, x);
         int draw_x = board_x + x*size + border_size;
         int draw_y = board_y + (num_ranks-1-y)*size + border_size;
         bool blink_piece = false;

         if (y < river) draw_y += size;

         if (checkered && ((x^y)&1) == 0) colour = dark;

         if (!is_zero128(palace & large_square_bitboards[where])) colour = dark;

         al_draw_filled_rectangle(draw_x, draw_y, draw_x+size-1, draw_y+size-1, colour);

         al_draw_rectangle(draw_x, draw_y, draw_x+size-1, draw_y+size-1, black, 1);

         bool occupied = false;
         if (game->large_board)
            occupied = !is_zero128(large_occ & large_square_bitboards[where]);
         else
            occupied = (occ & make_bitboard_square(where)) != 0;

         /* Piece */
         if (occupied) {
            ALLEGRO_COLOR piece_tint = white;
            int piece = get_piece(board, where);
            int col = WHITE;
            if (game->large_board) {
               if (!is_zero128(board->large_bbc[BLACK] & large_square_bitboards[where]))
                  col = BLACK;
            } else {
               if (board->bbc[BLACK] & make_bitboard_square(where))
                  col = BLACK;
            }

            if(figurine_glyph[piece]) {
               ALLEGRO_FONT *font = figurine_glyph[piece]->font;
               char *s1 = figurine_glyph[piece]->w1;
               char *s2 = figurine_glyph[piece]->w2;
               if (col == BLACK) {
                  s1 = figurine_glyph[piece]->b1;
                  s2 = figurine_glyph[piece]->b2;
               }
               al_draw_textf(font, white, draw_x+size/2, draw_y, ALLEGRO_ALIGN_CENTRE, "%s", s1);
               al_draw_textf(font, black, draw_x+size/2, draw_y, ALLEGRO_ALIGN_CENTRE, "%s", s2);
            } else {
               al_draw_textf(figurine_font[8], (col==WHITE)?white:black, draw_x+size/2, draw_y, ALLEGRO_ALIGN_CENTRE, "?");
            }
         }

         /* Highlight mouse cursor */
         if (x == mouse_file && y == mouse_row) {
            al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, select);
         }

         /* Special squares */
         if (game->large_board) {
            if (!is_zero128(large_mouse_drop & large_square_bitboards[where])) {
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, move_col);
            } else if (!is_zero128(large_legal_moves & large_square_bitboards[where])) {
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, move_col);
            }
            if (!is_zero128(large_kings & large_square_bitboards[where]))
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, check);
         } else {
            if (mouse_drop & make_bitboard_square(where)) {
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, move_col);
            } else if (legal_moves & make_bitboard_square(where)) {
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, move_col);
            }
            if (kings & make_bitboard_square(where))
               al_draw_filled_rectangle(draw_x, draw_y, draw_x+size, draw_y+size, check);
         }
      }
   }

   int n;
#if 0
   for (n=0; n<num_font_figurines; n++) {
      int m = 2*n;
      x = ((m*size) % window_width) / size;
      y = ((m*size) / window_width);
      int draw_x = x*size + size/2;
      int draw_y = y*size;

      al_draw_filled_rectangle(draw_x-size/2, draw_y, draw_x+size/2, draw_y+size, dark);
      al_draw_textf(font_figurine[n].font, white, draw_x, draw_y, ALLEGRO_ALIGN_CENTRE, "%s", font_figurine[n].w1);
      al_draw_textf(font_figurine[n].font, black, draw_x, draw_y, ALLEGRO_ALIGN_CENTRE, "%s", font_figurine[n].w2);

      m = 2*n+1;
      x = ((m*size) % window_width) / size;
      y = ((m*size) / window_width);
      draw_x = x*size + size/2;
      draw_y = y*size;
      al_draw_filled_rectangle(draw_x-size/2, draw_y, draw_x+size/2, draw_y+size, dark);
      al_draw_textf(font_figurine[n].font, white, draw_x, draw_y, ALLEGRO_ALIGN_CENTRE, "%s", font_figurine[n].b1);
      al_draw_textf(font_figurine[n].font, black, draw_x, draw_y, ALLEGRO_ALIGN_CENTRE, "%s", font_figurine[n].b2);
   }
#endif

   /* Draw coordinates along the board */
   for (x=0; x<num_files; x++) {
      int draw_x = border_size + x*size + size/2;
      int draw_y = top_margin + border_size + num_ranks*size;
      if (river) draw_y += size;
      al_draw_textf(board_font, white, draw_x, draw_y, ALLEGRO_ALIGN_CENTRE, "%c", x+'A');
   }

   for (x=0; x<num_ranks; x++) {
      int draw_x = border_size/2;
      int draw_y = top_margin + border_size + (num_ranks - x) * size - (size + board_font_size) / 2;
      if (river) draw_y += size;
      al_draw_textf(board_font, white, draw_x, draw_y, ALLEGRO_ALIGN_CENTRE, "%d", x+1);
   }

   /* Player names */
   if (game->name_white || game->name_black) {
      int width = BOARD_WIDTH + 2*border_size;//al_get_display_width(board_display);
      al_draw_textf(board_font, white, width/2-16, top_margin, ALLEGRO_ALIGN_RIGHT, "%s", game->name_white);
      al_draw_textf(board_font, white, width/2, top_margin, ALLEGRO_ALIGN_CENTRE, "-");
      al_draw_textf(board_font, white, width/2+16, top_margin, ALLEGRO_ALIGN_LEFT, "%s", game->name_black);
   }

   /* Clock */
   char bulletstr[5]; x = utf8_setc(bulletstr, 0x25CF); bulletstr[x] = '\0';
   char circlestr[5]; x = utf8_setc(circlestr, 0x25CB); circlestr[x] = '\0';
   char timestr[10];
   int hours, minutes, hseconds;
   hours = clock_time[WHITE] / (1000 * 60 * 60);
   minutes = max((clock_time[WHITE] - hours * (1000 * 60 * 60)) / (1000 * 60), 0);
   hseconds = max((clock_time[WHITE] - (hours * 60 + minutes) * (1000 * 60)), 0);
   if (hours) {
      snprintf(timestr, 10, "%d:%02d:%02d", hours, minutes, hseconds / 1000);
   } else if (minutes || hseconds > 10000) {
      snprintf(timestr, 10, "%02d:%02d", minutes, hseconds / 1000);
   } else {
      snprintf(timestr, 10, "%.2f", (float)hseconds / 1000.);
   }
   al_draw_textf(font, white, wclock_x + border_size, wclock_y, ALLEGRO_ALIGN_LEFT, "%s %s", bulletstr, timestr);
   al_draw_textf(font, black, wclock_x + border_size, wclock_y, ALLEGRO_ALIGN_LEFT, "%s", circlestr);
   hours = clock_time[BLACK] / (1000 * 60 * 60);
   minutes = max((clock_time[BLACK] - hours * (1000 * 60 * 60)) / (1000 * 60), 0);
   hseconds = max((clock_time[BLACK] - (hours * 60 + minutes) * (1000 * 60)), 0);
   if (hours) {
      snprintf(timestr, 10, "%d:%02d:%02d", hours, minutes, hseconds / 1000);
   } else if (minutes || hseconds > 10000) {
      snprintf(timestr, 10, "%02d:%02d", minutes, hseconds / 1000);
   } else {
      snprintf(timestr, 10, "%.2f", (float)hseconds / 1000.);
   }
   al_draw_textf(font, black, bclock_x + border_size, bclock_y, ALLEGRO_ALIGN_LEFT, "%s", bulletstr);
   al_draw_textf(font, white, bclock_x + border_size, bclock_y, ALLEGRO_ALIGN_LEFT, "%s %s", circlestr, timestr);

   /* Display the game drop-down list */
   al_draw_filled_rectangle(drop_x, drop_y, drop_x+drop_w, drop_y+drop_h, (menu_open | mouse_menu) ? selected : bg);
   al_draw_rectangle(drop_x, drop_y, drop_x+drop_w, drop_y+drop_h, light, 1);
   if (game->name) {
      char *s = strdup(game->name);
      if (strlen(s) > drop_str_w) s[drop_str_w] = '\0';
      al_draw_textf(font, white, drop_x+border_size, (drop_h-iteration_font_size)/2-1, ALLEGRO_ALIGN_LEFT, "%s", s);
      free(s);
   }
   char arrowstr[5];
   al_draw_filled_rectangle(drop_x+drop_w - border_size, drop_y, drop_x+drop_w, drop_y+drop_h, bg);
   al_draw_rectangle(drop_x+drop_w - border_size, drop_y, drop_x+drop_w, drop_y+drop_h, light, 1);
   x = utf8_setc(arrowstr, 0x25BC); arrowstr[x] = '\0';
   al_draw_textf(font, black, drop_x+drop_w-border_size/2, (drop_h-iteration_font_size)/2-1, ALLEGRO_ALIGN_CENTRE, "%s", arrowstr);
   x = utf8_setc(arrowstr, 0x25BD); arrowstr[x] = '\0';
   al_draw_textf(font, light, drop_x+drop_w-border_size/2, (drop_h-iteration_font_size)/2-1, ALLEGRO_ALIGN_CENTRE, "%s", arrowstr);

   /* Draw move list */
   int first_move;
   int nl, nm, lines_needed;
   int start, stop;

   /* Number of displayed moves */
   nl = moves_h / iteration_font_size-2;
   nm = 2 * nl;

   if (nl >= 2) {
      first_move = get_move_player(game->move_list[0])>>7;

      start = 0;
      stop = game->root_moves_played;
      lines_needed = (first_move + stop - start)/2;

      if ( lines_needed >= nl ) {
         start = (first_move + stop - 2*nl) & ~0x1;
         start += 2;
      }

      for(n=start; n<stop; n++) {
         /* Odd plies should normally have been played by black - unless we
          * changed the player, or loaded a position which had "black to play"
          */
         char s[128];
         int w = (n&1) ^ first_move;
         int dx = 0;

         snprintf(s, 128, "% 4d.", (n+first_move)/2+1);
         x = moves_x + border_size + w * 80 + 10;
         y = moves_y + border_size + ((n+first_move - start)/2)*iteration_font_size;
         al_draw_textf(font, black, moves_x + 4, y, 0, "%s", s);
         dx = al_get_text_width(font, s);

         short_move_string(NULL, game->move_list[n], s);

         char *p = s;

         dx += 6 * al_get_text_width(font, "M");
         int x_fig = x + dx;
         int x_move = x + dx;
         if (!is_castle_move(game->move_list[n])) {
            int piece = 0;
            while (piece_symbol_string[piece] && (piece_symbol_string[piece] != s[0])) piece++;
            //printf("%d\piece", piece);
            //printf("'%s'\piece", game->pt.piece_name[piece]);

            if (piece_symbol_string[piece] && !isspace(game->pt.piece_notation[piece][0])) {
               char *str = figurine_glyph[piece]->w2;
               //printf("'%s'\piece", game->pt.piece_notation[piece]);
               if (get_move_player(game->move_list[n]) == BLACK)
                  str = figurine_glyph[piece]->b2;
               x_move -= al_get_text_width(font, p+1);
               x_fig = x_move - al_get_text_width(figurine_glyph[piece]->text_font, figurine_glyph[piece]->w2); 
               al_draw_textf(figurine_glyph[piece]->text_font, black, x_fig, y, ALLEGRO_ALIGN_LEFT, "%s", str); 
               p++;
            } else {
               x_move -= al_get_text_width(font, p);
            }
         } else {
            x_move -= al_get_text_width(font, p);
         }

         al_draw_textf(font, black, x_move, y, ALLEGRO_ALIGN_LEFT, "%s", p);
      }
   }

   /* Menu */
   if (menu_open) {
      int y;
      al_draw_filled_rectangle(drop_x, drop_y+drop_h, drop_x+drop_w, window_height-top_margin, bg);
      for (n=0; variant_names[n]; n++) {
         char *s = strdup(variant_names[n]);
         y = drop_y+drop_h + n*iteration_font_size + (drop_h-iteration_font_size)/2-1;
         if (n == selected_variant)
            al_draw_filled_rectangle(drop_x, y, drop_x+drop_w, y+iteration_font_size, dark);
         if (strlen(s) > drop_str_w) s[drop_str_w] = '\0';
         al_draw_textf(font, white, drop_x+border_size, y, ALLEGRO_ALIGN_LEFT, "%s", s);
         free(s);
      }
      int k = n;
      for (int n=0; n<fairy_file_games; n++) {
         y += iteration_font_size;
         if (k+n == selected_variant)
            al_draw_filled_rectangle(drop_x, y, drop_x+drop_w, y+iteration_font_size, dark);
         al_draw_textf(font, white, drop_x+border_size, y, ALLEGRO_ALIGN_LEFT, "%s", fairy_file_names[n]);
      }
   }

   al_flip_display();
}

static void load_fonts(void)
{
   const char *fontname = "gfx/DejaVuSansMono.ttf";

   board_font_size = min(12, square_size * 0.4);
   board_font_size = min(board_font_size, border_size);

   al_destroy_font(font);
   al_destroy_font(board_font);
   font = board_font = NULL;
   if (al_filename_exists(fontname)) {
      font = al_load_font(fontname, iteration_font_size, ALLEGRO_TTF_NO_KERNING);
      board_font = al_load_font(fontname, board_font_size, ALLEGRO_TTF_NO_KERNING);
   } else {
      ALLEGRO_PATH *path = al_create_path(fontname);
      al_rebase_path(resource_path, path);
      font = al_load_font(al_path_cstr(path, ALLEGRO_NATIVE_PATH_SEP), iteration_font_size, ALLEGRO_TTF_NO_KERNING);
      board_font = al_load_font(al_path_cstr(path, ALLEGRO_NATIVE_PATH_SEP), board_font_size, ALLEGRO_TTF_NO_KERNING);
      al_destroy_path(path);
   }
   if (!font) {
      al_show_native_message_box(NULL, "File not found", "", "Couldn't load font", "Exit", ALLEGRO_MESSAGEBOX_ERROR);
      exit(0);
   }
}

void detect_board_features(const game_t *game)
{
   int n;
   river = 0;

   if(game->large_board) {
      for (n=0; n<game->pt.num_piece_types; n++) {
         if (game->pt.piece_flags[n] & PF_ROYAL) continue;  /* Skip royal pieces */
         if (!(is_zero128(game->pt.large_prison[n]))) {
            if (is_zero128(game->pt.large_prison[n] & large_board_south)) {
               river = large_board_ranks / 2;
               break;
            }
         }
      }
   }
}

static void reset_clock(game_t *game)
{
   game->movestotc = 40;
   game->movestogo = 40;
   game->time_inc[0] = game->time_inc[1] = 0;
   clock_time[0] = total_time;
   clock_time[1] = total_time;
   game->time_left[0] = clock_time[0];
   game->time_left[1] = clock_time[1];
   set_time_for_game(game);
   clock_running = false;
}

void reload_figurine_set(const game_t *game)
{
   int n;
   sides side;
   for (n=0; n<MAX_PIECE_TYPES; n++) {
      for (side = WHITE; side<NUM_SIDES; side++)
         if (figurine[n][side])
            al_destroy_bitmap(figurine[n][side]);
   }
   memset(figurine, 0, sizeof(figurine));

   /* Load bitmaps for all pieces */
   if (0)
   for (n=0; n<game->pt.num_piece_types; n++) {
      char filename[2048];
      char *s;
      //printf("%s\n", game->pt.piece_name[n]);

      /* Try generic filenames first */
      snprintf(filename, sizeof filename, "gfx/w%s.png", game->pt.piece_name[n]);
      for (s = filename; s[0]; s++) *s = tolower(*s);
      if (al_filename_exists(filename))
         figurine[n][WHITE] = al_load_bitmap(filename);

      if (!figurine[n][WHITE]) {
         char msg[4096];
         snprintf(msg, sizeof msg, "Couldn't load figurine file '%s'", filename);
         al_show_native_message_box(NULL, "File not found", "", msg, "Exit", ALLEGRO_MESSAGEBOX_ERROR);
         exit(0);
      }

      snprintf(filename, sizeof filename, "gfx/b%s.png", game->pt.piece_name[n]);
      for (s = filename; s[0]; s++) *s = tolower(*s);
      if (al_filename_exists(filename))
         figurine[n][BLACK] = al_load_bitmap(filename);

      if (!figurine[n][BLACK]) {
         char msg[4096];
         snprintf(msg, sizeof msg, "Couldn't load figurine file '%s'", filename);
         al_show_native_message_box(NULL, "File not found", "", msg, "Exit", ALLEGRO_MESSAGEBOX_ERROR);
         exit(0);
      }
   }

   for (n=0; n<num_font_files; n++) {
      al_destroy_font(figurine_font[n]);
      al_destroy_font(figurine_text_font[n]);
   }

   load_figurine_fonts();
   
   memset(figurine_glyph, 0, sizeof figurine_glyph);
   for (n=0; n<game->pt.num_piece_types; n++) {
      /* Try generic filenames first */
      if (strcmp(game->pt.piece_name[n], "King") == 0) {
         figurine_glyph[n] = &font_figurine[0];
      } else if (strcmp(game->pt.piece_name[n], "Queen") == 0) {
         figurine_glyph[n] = &font_figurine[1];
      } else if (strcmp(game->pt.piece_name[n], "Rook") == 0) {
         figurine_glyph[n] = &font_figurine[2];
      } else if (strcmp(game->pt.piece_name[n], "Bishop") == 0) {
         figurine_glyph[n] = &font_figurine[3];
      } else if (strcmp(game->pt.piece_name[n], "Knight") == 0) {
         figurine_glyph[n] = &font_figurine[4];
      } else if (strcmp(game->pt.piece_name[n], "Pawn") == 0) {
         figurine_glyph[n] = &font_figurine[5];
      } else if (strcmp(game->pt.piece_name[n], "White Pawn") == 0) {
         figurine_glyph[n] = &font_figurine[5];
      } else if (strcmp(game->pt.piece_name[n], "Black Pawn") == 0) {
         figurine_glyph[n] = &font_figurine[5];

      } else if (strcmp(game->pt.piece_name[n], "Wolf") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 'h') {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Sheep") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 'h') {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Goose") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 'h') {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Stone") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 'h') {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Archbishop") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E194) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Cardinal") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E194) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Chancellor") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E192) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Marshal") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E192) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Wazir") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E01A) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Man") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E144) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Ferz") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E018) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Guard") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E012) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Elephant") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E010) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Dabbabah") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E022) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Horse") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x02658) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Cannon") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E014) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Amazon") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E190) {//0x0E03A) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Maharaja") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E190) {//0x0E03A) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }

      } else if (strcmp(game->pt.piece_name[n], "Lieutenant") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E01E) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }
      } else if (strcmp(game->pt.piece_name[n], "Captain") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E022) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }
      } else if (strcmp(game->pt.piece_name[n], "General") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E044) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }
      } else if (strcmp(game->pt.piece_name[n], "Warlord") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E046) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }
      } else if (strcmp(game->pt.piece_name[n], "Hoplite") == 0) {
         int k;
         for (k=0; k<num_font_figurines; k++) {
            if (font_figurine[k].wi == 0x0E140) {
               figurine_glyph[n] = &font_figurine[k];
               break;
            }
         }
      }
   }
}

void resize_window(void)
{
   int new_width, new_height;
   num_files = large_board_files;
   num_ranks = large_board_ranks;

   window_width = al_get_display_width(board_display);
   window_height = al_get_display_height(board_display);
   al_set_window_title(board_display, PROGRAM_NAME);
   key_display = board_display;

   /* Layout of the display */
   drop_str_w = 32;
   drop_w = drop_str_w*al_get_text_width(font, "M")+2*border_size;
   drop_w = BOARD_WIDTH + 2*border_size;
   drop_h = max(iteration_font_size+2, top_margin);

   border_size = 16;

   if (window_width > window_height) {
      square_size = (window_height - top_margin - bottom_margin - 2*border_size) / num_ranks;
      if (border_size >= square_size) {
         border_size = square_size / 2;
         square_size = (window_height - top_margin - bottom_margin - 2*border_size) / num_ranks;
      }
      moves_x = BOARD_WIDTH + border_size + 1;
      moves_y = border_size + top_margin;
      moves_w = window_width - moves_x;
      moves_h = BOARD_HEIGHT;

      clock_w = max(moves_w / 2, 7*al_get_text_width(font, "M")+2*border_size);
      if (drop_w + 2*clock_w + border_size > window_width) drop_w = window_width - 2*clock_w - border_size;
      wclock_x = min(moves_x, drop_w);
      wclock_y = 0;
      clock_w = (window_width - wclock_x) / 2;

      bclock_x = wclock_x + clock_w;
      bclock_y = wclock_y;
   } else {
      drop_w = window_width;
      square_size = (window_width - 2*border_size) / num_files;
      if (border_size >= square_size * 0.8) {
         border_size = square_size / 2;
         square_size = (window_width - 2*border_size) / num_files;
      }
      moves_x = 0;
      moves_y = top_margin + 2*border_size + BOARD_HEIGHT + iteration_font_size;
      moves_w = window_width;
      moves_h = window_height - moves_y - bottom_margin;

      clock_w = max(moves_w / 2, 7*al_get_text_width(font, "M")+2*border_size);
      wclock_x = moves_x;
      wclock_y = moves_y - iteration_font_size;

      bclock_x = wclock_x + clock_w;
      bclock_y = wclock_y;
   }
   board_x = 0;
   board_y = top_margin;

   drop_x = BOARD_WIDTH + 2*border_size - drop_w;
   drop_x = 0;
   if (drop_x < 0) drop_x = 0;
   if (drop_x + drop_w > window_width) drop_w = window_width - drop_x;
   drop_str_w = (drop_w - 2*border_size) / al_get_text_width(font, "M");
   drop_y = 0;
}

static bool thinking = false;
static void *think(ALLEGRO_THREAD *thr, void *arg)
{
   game_t *game = arg;

   if (!thinking || !game) return NULL;

   computer_play(game, 60);

   thinking = false;
   return NULL;
}

#define abort_thread()\
   if (thinking_thread) {\
      abort_search = true;\
      al_join_thread(thinking_thread, NULL);\
      al_destroy_thread(thinking_thread);\
      thinking_thread = NULL;\
   }0\

int main(void)
{
   ALLEGRO_THREAD *thinking_thread = NULL;
   board_t board;
   game_t *game;
   sides computer_side = BLACK;

   construct_inverse_diagonal_maps();
   initialise_slider_tables();
   initialise_hash_keys();

   playgame_init();

   for (int n=0; create_variant_game[n]; n++) {
      char *s = variant_names[n] = malloc(128);
      game = create_variant_game[n]();
      snprintf(s, 128, "%s (%dx%d)", game->name, large_board_files, large_board_ranks);
      printf("%s\n", s);
      end_game(game);
   }
   //exit(0);

   //game = create_standard_game();
   //game = create_chinese_game();
   //game = create_spartan_game();
   //game = create_maharaja_game();
   //game = create_knightmate_game();
   //game = create_losers_game();
   //game = create_shatranj_game();
   //game = create_makruk_game();
   //game = create_burmese_game();

   game = create_variant_game[selected_variant]();
   detect_board_features(game);
   reset_clock(game);

   /* Print information about all piece types. */
   print_piece_types(game);

   /* Initialise Allegro */
   ALLEGRO_EVENT_QUEUE *queue;
   ALLEGRO_TIMER *clock_timer;
   int n, x, y;
   bool redraw = true;
   bool done = false;

   /* Initialise chess font character mapping */
   memset(figurine_font, 0, sizeof figurine_font);
   memset(figurine_text_font, 0, sizeof figurine_font);

   if (!al_init()) {
      fprintf(stderr, "Couldn't initialise Allegro\n");
      return 1;
   }
   al_install_mouse();
   al_install_keyboard();
   al_init_primitives_addon();
   al_init_image_addon();
   al_init_font_addon();
   al_init_ttf_addon();

   /* Load the 'variants.txt' description file */
   fairy_file = "variants.txt";
   if (al_filename_exists(fairy_file)) {
      index_fairy_file(fairy_file);
   } else {
      ALLEGRO_PATH *path = al_create_path(fairy_file);
      al_rebase_path(resource_path, path);
      index_fairy_file(al_path_cstr(path, ALLEGRO_NATIVE_PATH_SEP));
      al_destroy_path(path);
   }
   for (int n=0; n<fairy_file_games; n++) {
      printf("%s\n", fairy_file_names[n]);
   }

   al_set_new_display_option(ALLEGRO_SAMPLE_BUFFERS, 1, ALLEGRO_SUGGEST);
   al_set_new_display_option(ALLEGRO_SAMPLES, 4, ALLEGRO_SUGGEST);

   //al_set_new_display_flags(ALLEGRO_FULLSCREEN_WINDOW);
   al_set_new_display_flags(ALLEGRO_RESIZABLE | ALLEGRO_GENERATE_EXPOSE_EVENTS);
   //board_display = al_create_display(800, 480);
   board_display = al_create_display(420, 700);
   //board_display = al_create_display(240, 400);
   //iteration_font_size /= 2;
   //board_font_size /= 2;

   resource_path = al_get_standard_path(ALLEGRO_RESOURCES_PATH);

   load_fonts();

   /* Load figurines */
   resize_window();
   load_fonts();
   reload_figurine_set(game);

   cursor = al_create_bitmap(square_size, square_size);

   /* Start a new game */
   start_new_game(game);
   //setup_fen_position(game, "lgkcckwl/hhhhhhhh/8/8/8/8/PPPPPPPP/RNBQKBNR/ w KQ -");
   //setup_fen_position(game, "lgkcckwl/hhhh1hhh/3h4/8/3PP3/8/PPP2PPP/RNBQKBNR/ b KQ -");
   //setup_fen_position(game, "2k2k2/7Q/3K4/8/8/8/R7/8 w - - 0 1");
   //setup_fen_position(game, "8/8/8/8/8/8/R7/R3K2k w Q - 0 1");
   //setup_fen_position(game, "r1b2k1r/p4pbp/nq1p1Q2/2pP4/1p2R2N/6P1/P3PPBP/4K2R/ w - -");
   //setup_fen_position(game, "8/8/Bk6/8/1l1h4/1PR2P1P/hg6/4K3 b - - 0 66");
   //setup_fen_position(game, "8/8/Bk6/8/1l1h4/1PR2P1P/1g6/1g2K3 w - - 0 66");
   //setup_fen_position(game, "1R2l1k1/4g1h1/5R2/8/2P1h3/1P4Q1/P2h2PP/7K b - - 0 46");

   clock_timer = al_create_timer(1.0 / 40.0);
   queue = al_create_event_queue();
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)al_get_mouse_event_source());
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)al_get_keyboard_event_source());
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)board_display);
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)clock_timer);
   al_start_timer(clock_timer);

   /* Initialise random number generator.
    * We add a random number to the move score at the root for the first few plies, so we don't play the same
    * game all the time.
    */
   sgenrand(time(NULL));

   /* Generate all pseudo-legal moves */
   get_movelist(game);

   /* Show position */
   print_bitboards(&game->board);

   bool autoplay = false;
   int time_per_move = 1100;
   uint64_t start_time;
   memcpy(&board, &game->board, sizeof board);
   while (!done) {
      ALLEGRO_EVENT event;

      if (game && game->moves_played == 0)
         memcpy(&board, &game->board, sizeof board);

      /* Start the player's clock if the computer is not thinking */
      if (!thinking_thread) {
         start_clock(game);
         game->root_moves_played = game->moves_played;
      }

      if (redraw && al_event_queue_is_empty(queue)) {
         draw_board(game, &board);
         redraw = false;
      }

      if (autoplay && al_event_queue_is_empty(queue)) {
         if (!thinking_thread) {
            computer_side = game->board.side_to_move;
         }
      }

      if ( (game->board.side_to_move == computer_side || thinking_thread) && al_event_queue_is_empty(queue)) {
         if (thinking_thread) {
            if (!thinking) {
               clock_time[computer_side] = start_time - peek_timer(game);
               if (clock_time[computer_side] < 0) clock_time[computer_side]  = 0;
               al_join_thread(thinking_thread, NULL);
               al_destroy_thread(thinking_thread);
               thinking_thread = NULL;

               int moves_by_side = game->moves_played/2 + (game->moves_played & computer_side);
               if (moves_by_side && moves_by_side % game->movestotc == 0)
                  clock_time[computer_side] += total_time;

               memcpy(&board, &game->board, sizeof board);
               game->last_move = game->moves_played;
               get_movelist(game);

               /* Legal moves for the highlighted piece */
               generate_legal_move_bitboard(game, mouse_file, mouse_row);
               redraw = true;

               /* Start the player's clock */
               start_clock(game);
            }
         } else {
            //set_time_per_move(game, time_per_move);
            start_time = game->time_left[game->board.side_to_move] = clock_time[game->board.side_to_move];
            if (game->time_left[game->board.side_to_move] > 200)
               game->time_left[game->board.side_to_move] -= 200;
            else if (game->time_left[game->board.side_to_move] < 100)
               game->time_left[game->board.side_to_move] -= 100;
            int moves_by_side = game->root_moves_played/2 + (game->root_moves_played & game->board.side_to_move);
            game->movestogo = game->movestotc - (moves_by_side % game->movestotc);
            //printf("\nMoves to go: %d %d\n\n", game->movestogo, clock_time[game->board.side_to_move]);
            set_time_for_game(game);

            thinking = true;
            thinking_thread = al_create_thread(think, game);
            al_start_thread(thinking_thread);
            redraw = true;
            clock_running = true;
         }
      }

      al_wait_for_event(queue, &event);
      if (thinking_thread) {
         clock_time[computer_side] = start_time - peek_timer(game);
         if (clock_time[computer_side] < 0) clock_time[computer_side]  = 0;
      } else {
         if (clock_running) clock_time[game->board.side_to_move] -= peek_timer(game);
         if (clock_time[game->board.side_to_move] < 0) clock_time[game->board.side_to_move]  = 0;

         /* Reset the player's clock; needed because we want to update the displayed time every once in a
          * while.
          */
         start_clock(game);
      }
      switch (event.type) {
         case ALLEGRO_EVENT_DISPLAY_EXPOSE:
            redraw = true;
            break;

         case ALLEGRO_EVENT_DISPLAY_RESIZE:
            al_acknowledge_resize(event.display.source);
            resize_window();
            reload_figurine_set(game);
            redraw = true;
            break;

         case ALLEGRO_EVENT_DISPLAY_CLOSE:
            done = true;
            break;

         case ALLEGRO_EVENT_DISPLAY_SWITCH_IN:
            key_display = event.display.source;
            break;

         case ALLEGRO_EVENT_DISPLAY_SWITCH_OUT:
            key_display = NULL;
            break;

         case ALLEGRO_EVENT_MOUSE_AXES:
            if (menu_open == false && !thinking_thread) {
               if (event.mouse.x > board_x + border_size &&
                     event.mouse.x < board_x + border_size+BOARD_WIDTH &&
                     event.mouse.y > board_y + border_size &&
                     event.mouse.y < board_y + border_size+BOARD_HEIGHT) {
                  int file_size = square_size;
                  int row_size = square_size;
                  int new_mouse_file = (event.mouse.x - board_x - border_size) / file_size;
                  int new_mouse_row = (BOARD_HEIGHT - (event.mouse.y - board_y - border_size)) / row_size;
                  if (river && new_mouse_row > river) new_mouse_row--;
                  if (new_mouse_file != mouse_file || new_mouse_row != mouse_row) {
                     mouse_file = new_mouse_file;
                     mouse_row = new_mouse_row;
                     redraw = true;

                     /* Legal moves for the highlighted piece */
                     generate_legal_move_bitboard(game, mouse_file, mouse_row);
                  }
               } else {
                  mouse_file = mouse_row = -1;
                  legal_moves = board_empty;
                  large_legal_moves = large_board_empty;
                  redraw = true;
               }
            }


            if (event.mouse.x > drop_x &&
                event.mouse.x < drop_x+drop_w &&
                event.mouse.y > drop_y &&
                event.mouse.y < drop_y+drop_h) {
               mouse_menu = true;
            } else {
               mouse_menu = false;
            }

            if (menu_open && event.mouse.x > drop_x && event.mouse.x <drop_x+drop_w && event.mouse.y > drop_y) {
               int n = (event.mouse.y - drop_y - drop_h - (drop_h-iteration_font_size)/2+1) / iteration_font_size;
               selected_variant = n;
               redraw = true;
            }
            break;

         case ALLEGRO_EVENT_MOUSE_BUTTON_UP:
            if (key_display == board_display) {
               al_set_system_mouse_cursor(board_display, ALLEGRO_SYSTEM_MOUSE_CURSOR_DEFAULT);
               if (mouse_file != -1 && mouse_row != -1 && !thinking_thread) {
                  int square = pack_row_file(mouse_row, mouse_file);
                  bool valid_drop = 0;
                  if (game->large_board)
                     valid_drop = !is_zero128(large_mouse_drop & large_square_bitboards[square]);
                  else
                     valid_drop = mouse_drop & make_bitboard_square(square);
                  if (valid_drop) {
                     move_t move = move_from_list(mouse_pickup_square, square);
                     if (move == 0)
                        break;
                     sides side = game->board.side_to_move;
                     playmove(game, move);
                     if (player_in_check(game, side)) {
                        takeback(game);
                        get_movelist(game);
                     } else {
                        clock_running = true;
                     }

                     /* Legal moves for the highlighted piece */
                     generate_legal_move_bitboard(game, mouse_file, mouse_row);
                     memcpy(&board, &game->board, sizeof board);
                  }

                  redraw = true;
                  mouse_drop = board_empty;
                  large_mouse_drop = large_board_empty;
               }

               if (mouse_menu) {
                  menu_open = true;
                  redraw = true;
               }
            }
            break;
         case ALLEGRO_EVENT_MOUSE_BUTTON_DOWN:
            if (mouse_file != -1 && mouse_row != -1 && !thinking_thread) {
               int square = pack_row_file(mouse_row, mouse_file);
               int piece = get_piece(&game->board, square);
               sides side = get_piece_colour(&game->board, square);

               if (side != NONE) {
                  ALLEGRO_STATE state;
                  int size = square_size;
                  mouse_pickup_square = square;
                  al_store_state(&state, ALLEGRO_STATE_TARGET_BITMAP | ALLEGRO_STATE_BLENDER);

                  al_set_mouse_cursor(board_display, NULL);
                  al_destroy_mouse_cursor(mouse_cursor);
                  al_set_target_bitmap(cursor);
                  al_clear_to_color(al_map_rgba(0,0,0,0));
                  /*
                  al_draw_scaled_bitmap(figurine[piece][side],
                        0, 0,
                        al_get_bitmap_width(figurine[piece][side]),
                        al_get_bitmap_height(figurine[piece][side]),
                        0, 0, size, size, 0);
                  */
                  ALLEGRO_COLOR white = al_map_rgba_f(1, 1, 1, 1);
                  ALLEGRO_COLOR black = al_map_rgba_f(0, 0, 0, 1);
                  ALLEGRO_FONT *font = figurine_glyph[piece]->font;
                  char *s1 = figurine_glyph[piece]->w1;
                  char *s2 = figurine_glyph[piece]->w2;
                  if (side == BLACK) {
                     s1 = figurine_glyph[piece]->b1;
                     s2 = figurine_glyph[piece]->b2;
                  }
                  al_draw_textf(font, white, size/2, 0, ALLEGRO_ALIGN_CENTRE, "%s", s1);
                  al_draw_textf(font, black, size/2, 0, ALLEGRO_ALIGN_CENTRE, "%s", s2);
                  mouse_cursor = al_create_mouse_cursor(cursor, size/2, size/2);
                  al_set_mouse_cursor(board_display, mouse_cursor);

                  al_restore_state(&state);
                  mouse_drop = legal_moves;
                  large_mouse_drop = large_legal_moves;
               }
               redraw = true;
            }

            if (menu_open) {
               menu_open = false;
               redraw = true;

               if (selected_variant < number_of_variants) {
                  //printf("%s\n", variant_names[selected_variant]);
                  abort_thread();
                  end_game(game);
                  game = create_variant_game[selected_variant]();
                  reset_clock(game);
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
               } else if ((selected_variant - number_of_variants) < fairy_file_games) {
                  int n = selected_variant - number_of_variants;

                  abort_thread();
                  end_game(game);
                  game = create_game_from_file(fairy_file, fairy_file_names[n]);
                  reset_clock(game);
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
               }
            }
            break;
         
         case ALLEGRO_EVENT_KEY_UP:
            switch (event.keyboard.keycode) {
               case ALLEGRO_KEY_ESCAPE:
                  done = true;
                  break;
               case ALLEGRO_KEY_F1:
                  abort_thread();
                  end_game(game);
                  game = create_standard_game();
                  reset_clock(game);
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F2:
                  abort_thread();
                  end_game(game);
                  game = create_spartan_game();
                  reset_clock(game);
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F3:
                  abort_thread();
                  end_game(game);
                  game = create_maharaja_game();
                  reset_clock(game);
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F4:
                  abort_thread();
                  end_game(game);
                  game = create_knightmate_game();
                  reset_clock(game);
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F5:
                  abort_thread();
                  end_game(game);
                  game = create_shatranj_game();
                  reset_clock(game);
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F6:
                  abort_thread();
                  end_game(game);
                  game = create_grand_game();
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  reset_clock(game);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F7:
                  abort_thread();
                  end_game(game);
                  game = create_capablanca_game();
                  reset_clock(game);
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F8:
                  abort_thread();
                  end_game(game);
                  game = create_courier_game();
                  reset_clock(game);
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_F9:
                  abort_thread();
                  end_game(game);
                  game = create_chinese_game();
                  reset_clock(game);
                  detect_board_features(game);
                  resize_window();
                  reload_figurine_set(game);
                  start_new_game(game);
                  autoplay = false;
                  memcpy(&board, &game->board, sizeof board);
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  redraw = true;
                  break;
               case ALLEGRO_KEY_DELETE:         /* New game */
                  /* FIXME: this leaks memory, because the internal
                   * pointers in the game struct are not freed, just
                   * overwritten.
                   */
                  abort_thread();
                  start_new_game(game);
                  autoplay = false;
                  get_movelist(game);
                  computer_side = next_side[game->board.side_to_move];
                  reset_clock(game);
                  redraw = true;
                  break;

               case ALLEGRO_KEY_N:
                  break;

               case ALLEGRO_KEY_P:
                  break;

               case ALLEGRO_KEY_COMMA:
                  break;
               case ALLEGRO_KEY_ENTER:
                  autoplay = !autoplay;
                  break;
               case ALLEGRO_KEY_TAB:
                  if (!thinking_thread) {
                     computer_side = game->board.side_to_move;
                     redo_last_move(game);
                     get_movelist(game);
                     redraw = true;
                  }
                  break;
               case ALLEGRO_KEY_HOME:
                  if (!thinking_thread) {
                     while (game->moves_played)
                        takeback(game);
                     get_movelist(game);
                     redraw = true;
                  }
                  break;
               case ALLEGRO_KEY_BACKSPACE:
                  if (!thinking_thread) {
                     computer_side = game->board.side_to_move;
                     takeback(game);
                     get_movelist(game);
                     redraw = true;
                  }
                  break;
               case ALLEGRO_KEY_SPACE:
                  //autoplay = !autoplay;
                  if (!thinking_thread) {
                     computer_side = game->board.side_to_move;
                  } else {
                     abort_search = true;
                  }
                  break;
            }
            break;
         case ALLEGRO_EVENT_TIMER:
            redraw = true;
            break;
      }

   }

   abort_thread();

   end_game(game);

   return 0;
}
