/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>

#include "sjaak.h"

uint64_t opening_positions = 0;

static FILE *file = NULL;

#define repmask  0x7fffff
static int quick_repetition_count[repmask + 1] = { 0 };

static inline void mark_position(const game_t *game)
{
   quick_repetition_count[game->board.hash&repmask]++;
}

static inline bool position_repeated(const game_t *game)
{
   //printf("%016llx %d\n", game->board.hash, quick_repetition_count[game->board.hash&0xFFFF]);
   if (quick_repetition_count[game->board.hash&repmask]) {
      return true;
   }
   return false;
}

uint64_t perft(game_t *game, int target_depth, int depth, int root)
{
   movelist_t movelist;
   sides me = game->board.side_to_move;
   uint64_t nodes = 0;
   int n, ev;

   if (depth == target_depth) {
      /* Test for suitable opening positions */

      /* Eliminate unbalanced positions */
      //ev = abs(static_evaluation(game, game->board.side_to_move, -CHECKMATE, CHECKMATE));
      ev = abs(game->board.material[WHITE]-game->board.material[BLACK]);
      if (ev > 0)
         return 1;

      /* Eliminate positions where the side-to-move is in check */
      if (player_in_check(game, game->board.side_to_move))
         return 1;

      /* Castle rights lost */
      //if (!game->board.did_castle[WHITE] && !may_castle(&game->board, WHITE)) return 1;
      //if (!game->board.did_castle[BLACK] && !may_castle(&game->board, BLACK)) return 1;

      /* q-search test */
      if (abs(search(game, ev-50, ev+1000, 0, 0)) > 100)
         return 1;

      game->board.side_to_move = next_side[game->board.side_to_move];
      int res = search(game, ev-50, ev+1000, 0, 0);
      game->board.side_to_move = next_side[game->board.side_to_move];
      if (abs(res) > 100)
         return 1;

      /* Normal search test */
#if 0
      if (abs(search(game, ev, ev+1, 2, 0)) > 100)
         return 1;

      game->board.side_to_move = next_side[game->board.side_to_move];
      res = search(game, ev, ev+1, 2, 0);
      game->board.side_to_move = next_side[game->board.side_to_move];
      if (abs(res) > 100)
         return 1;
#endif

      fprintf(file, "%s\n", make_fen_string(game, NULL));
      opening_positions++;

      return 1;
   }

   /* Check if previous move left the player in check */
   generate_moves(&movelist, &game->board, me);

   for (n=0; n<movelist.num_moves; n++) {
      uint64_t count = 0;
      move_t move = movelist.move[n];

      /* Prune "uninteresting" moves */

      /* Moving the king, losing castling rights */
      if (!is_castle_move(move) && game->pt.piece_flags[get_move_piece(move)] & PF_ROYAL)
         continue;

      /* Moving a rook, losing castling rights */
      if (get_move_from(move) == 0 || get_move_from(move) == 9) continue;
      if (get_move_from(move) == 70 || get_move_from(move) == 79) continue;

      /* Moving an already developed piece, except when it's a capture */
      if (!is_capture_move(move) && !(game->board.large_init & large_square_bitboards[get_move_from(move)]))
         continue;

      /* Not restoring material balance after a capture */
      if (!is_capture_move(move) && (abs(game->board.material[WHITE]-game->board.material[BLACK]) > 100))
         continue;

      /* Captures with negative SEE */
      if (is_capture_move(move) && static_exchange_evaluation(&game->board, move)<0)
         continue;

      /* Knight on the rim (or any other piece really) */
      if (large_board_edge & large_square_bitboards[get_move_to(move)])
         continue;

      /* Pushing pawns on the flank (not necessarily bad, but prune them anyway for now */
      switch (get_move_from(move)) {
         case 10:
            continue;
         case 11:
            continue;
         case 12:
            continue;
         //case 13:
         //   continue;

         case 16:
            continue;
         case 17:
            continue;
         case 18:
            continue;
         case 19:
            continue;

         case 60:
            continue;
         case 61:
            continue;
         case 62:
            continue;
         //case 63:
         //   continue;

         case 66:
            continue;
         case 67:
            continue;
         case 68:
            continue;
         case 69:
            continue;
      }

      /* Don't move major pieces in the first 4 plies */
      if (depth < 4 && game->pt.piece_value[get_move_piece(move)] > 400) continue;

      /* On the first two plies only consider pawn pushes */
      if (depth < 2 && game->pt.piece_value[get_move_piece(move)] > 100) continue;

      /* First ply: only consider double pawn pushes, not single */
      if (depth == 0 && (get_move_to(move)==24 || get_move_to(move)==25)) continue;
      if (depth == 0 && get_move_to(move)==23) continue;

      playmove(game, move);

      /* Disqualify positions with a backward pawn */
      board_t *board = &game->board;
      if (board->large_init & large_square_bitboards[14] && !(board->large_init & (large_square_bitboards[13]|large_square_bitboards[15]))) {
         takeback(game);
         continue;
      }
      if (board->large_init & large_square_bitboards[64] && !(board->large_init & (large_square_bitboards[63]|large_square_bitboards[65]))) {
         takeback(game);
         continue;
      }

      /* Prune positions that have been visited before, we're only interested in generating unique leaf-node
       * positions, not in enumerating the paths to get there.
       * The method is crude (using only the low bits of the hash key to decide that a position has been
       * repeated, but it is quick and we don't care too much if we prune too many).
       */
      if (position_repeated(game)) {
         takeback(game);
         continue;
      }
      mark_position(game);

      /* Prune checking moves, just to keep the number of positions down. It's probably bad this early in the
       * game anyway.
       */
      if (player_in_check(game, next_side[me])) {
         takeback(game);
         continue;
      }

      if (!player_in_check(game, me))  /* Don't count illegal moves */
         count = perft(game, target_depth, depth+1, root - 1);
      nodes += count;
      if (root > 0)
         printf("%8s %10lld %10lld\n", move_string(movelist.move[n], NULL), nodes, opening_positions);
      takeback(game);
   }
   return nodes;
}


int main(void)
{
   game_t *game;

   construct_inverse_diagonal_maps();
   initialise_slider_tables();
   initialise_hash_keys();

   playgame_init();

   //game = create_standard_game();
   //game = create_spartan_game();
   game = create_capablanca_game();
   //game = create_gothic_game();

   /* Print information about all piece types. */
   print_piece_types(game);

   /* Start a new game */
   start_new_game(game);
   //setup_fen_position(game, "rnabqkbcnr/p1pppppppp/10/1p8/P9/10/1PPPPPPPPP/RNABQKBCNR w KQkq -");
   //setup_fen_position(game, "rnabqkbcnr/pppppppppp/10/10/10/P9/1PPPPPPPPP/RNABQKBCNR b KQkq -");
   //setup_fen_position(game, "rnabqkbcnr/p1pppppppp/1p8/10/10/P9/1PPPPPPPPP/RNABQKBCNR w KQkq -");
   //setup_fen_position(game, "rnabqkbcnr/p1pppppppp/1p8/10/10/PP8/2PPPPPPPP/RNABQKBCNR b KQkq -");
   //setup_fen_position(game, "rn1bqkbcnr/p1pppppppp/ap8/10/10/PP8/2PPPPPPPP/RNABQKBCNR w KQkq -");

   /* Show position */
   print_bitboards(&game->board);

   file = fopen("capablanca_open_pos_candidate.epd", "w");

   printf("\nVariant: %s\n", game->name);
   /* Perft timing */
   uint64_t t;
   int n = 8;
   opening_positions = 0;
   t = get_timer();
   uint64_t nodes = perft(game, n, 0, 1);
   uint64_t tt = get_timer();
   if (tt == t) tt++;
   printf("%2d %10lld %10lld %5.2f %12.2fnps\n", n, nodes, opening_positions,
         (tt - t)/1000000.0,nodes*1.0e6/(tt-t));
   t = tt;

   fclose (file);

   end_game(game);

   return 0;
}
