#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include "assert.h"
#include "pipe2.h"

/* Bidirectional pipe.
 * See
 *  http://www.unixwiz.net/techtips/remap-pipe-fds.html
 * and
 *  http://www.linuxquestions.org/questions/programming-9/where-do-i-get-a-bidirectional-popen-pipe-320699/
 */

pipe2_t *p2open(const char *cmd, char *const argv[])
{
   pipe2_t *p2 = NULL;
   int writepipe[2] = {-1,-1},	/* parent -> child */
       readpipe [2] = {-1,-1};	/* child -> parent */
   pid_t	child_pid;

   /* Open read and write pipes */
   if (pipe(readpipe) < 0)
      goto done;
   if (pipe(writepipe) < 0) {
      close(readpipe[0]);
      close(readpipe[1]);
      goto done;
   }

   /* Convenient defines to make code easier to read */
#define PARENT_READ	readpipe[0]
#define CHILD_WRITE	readpipe[1]
#define CHILD_READ	writepipe[0]
#define PARENT_WRITE	writepipe[1]

   /* Spawn child process */
   child_pid = fork();

   /* Failed to launch child process */
   if (child_pid < 0) {
      close(readpipe[0]);
      close(readpipe[1]);
      close(writepipe[0]);
      close(writepipe[1]);
      goto done;
   }

   /* Child process */
   if (child_pid == 0) {
      /* Close parent file descriptors */
      close(PARENT_WRITE);
      close(PARENT_READ);

      /* Attach input and output pipes to stdin, stdout */
      dup2(CHILD_READ,  STDIN_FILENO);  close(CHILD_READ);
      dup2(CHILD_WRITE, STDOUT_FILENO);  close(CHILD_WRITE);
      execvp(cmd, argv);
      exit(EXIT_FAILURE);
   }

   /* Parent */
	close(CHILD_READ);
	close(CHILD_WRITE);

   FILE *in, *out;
   if (!(in=fdopen(PARENT_READ,"r"))) {
      close(PARENT_READ);
      close(PARENT_WRITE);
      goto done;
   }
   if (!(out=fdopen(PARENT_WRITE,"w"))) {
      fclose(out);
      close(PARENT_WRITE);
      goto done;
   }

   /* Turn off buffering for pipes */
   setvbuf(in, NULL, _IONBF, 0);
   setvbuf(out, NULL, _IONBF, 0);

   p2 = malloc(sizeof *p2);

   p2->in = in;
   p2->in_fd = PARENT_READ;
   p2->out = out;
   p2->out_fd = PARENT_WRITE;
   p2->pid = child_pid;

done:
   return p2;
}

int p2close(pipe2_t *pipe)
{
   if (pipe) {
      int res1 = fclose(pipe->in);
      int res2 = fclose(pipe->out);
      if (res1==EOF || res2==EOF) return EOF;

      int status;
      while (waitpid(pipe->pid,&status,0)<0)
         if (errno != EINTR) return EOF;
   }

   return 0;
}

char *p2gets(char *s, int n, pipe2_t *pipe)
{
   assert(pipe);
   assert(s);
#if 0
   return fgets(s, n, pipe->in);

#else

   size_t off = 0;

   memset(s, 0, n);
   while(p2_input_waiting(pipe) || (s[0] == '\0' || s[strlen(s)-1] != '\n')) {
      ssize_t res = read(pipe->in_fd, s+off, n - 1-off);

      if (res < 0)
         return NULL;

      off += res;
      s[off+1] = '\0';

      int status;
      if (waitpid(pipe->pid, &status, WNOHANG) != 0)
         break;
   }
   return s;
#endif
}

bool input_waiting(FILE *file)
{
   //printf("Enter\n");
   if (!file) return false;

   struct timeval timeout;
   fd_set readfds;

   FD_ZERO(&readfds);
   FD_SET(fileno(file), &readfds);
   /* Set to timeout immediately */
   timeout.tv_sec = 0;
   timeout.tv_usec = 0;
   select(fileno(file)+1, &readfds, 0, 0, &timeout);

   //printf("Done\n");
   return (FD_ISSET(fileno(file), &readfds));
}

bool p2_input_waiting(pipe2_t *pipe)
{
   if (!pipe) return false;

   return input_waiting(pipe->in);
}

#if 0
void wait_input(int fd1, int fd2)
{
   fd_set readfds, errfds;

   FD_ZERO(&readfds);
   FD_SET(fd1, &readfds);
   FD_SET(fd2, &readfds);

   FD_ZERO(&errfds);
   FD_SET(fd1, &errfds);
   FD_SET(fd2, &errfds);

   //printf("Waiting for input...\n");
   select(32, &readfds, NULL, &errfds, NULL);
   if (FD_ISSET(fd1, &errfds)) {
      printf("Error on 1\n");
   }
   if (FD_ISSET(fd2, &errfds)) {
      printf("Error on 2\n");
   }

   //printf("Received input\n");
}

static void child_signal_handler(int sig)
{
   printf("Child terminated\n");
   exit(0);
   if (sig == SIGCHLD) {
      signal(sig, child_signal_handler);
   }
}

int main(void)
{
   pipe2_t *f;
   char *cmd = "./xsjaak";
   char *argv[2];

   argv[0] = cmd;
   argv[1] = NULL;

   setvbuf(stdin, NULL, _IONBF, 0);
   setvbuf(stdout, NULL, _IONBF, 0);

   signal(SIGCHLD, child_signal_handler);

   f = p2open(cmd, argv);
   if (!f) {
      printf("Can't open pipe\n");
      return 0;
   }

   while (1) {
      char s[4096];
      wait_input(f->in_fd, fileno(stdin));
      //printf("%d %d\n", input_waiting(stdin), p2_input_waiting(f));
      while(p2_input_waiting(f)) {
         //printf("Read: ");
         fflush(stdout);
         memset(s, 0, sizeof s);
         if (!p2gets(s, sizeof s, f))
            perror(NULL);
         printf("%s", s);
         fflush(stdout);
      }
      while (input_waiting(stdin)) {
         fgets(s, sizeof s, stdin);
         write(f->out_fd, s, strlen(s));
         //fprintf(f->out, "%s", s);
         //fflush(f->out);
      }
   }

   p2close(f);
   return 0;
}
#endif
