/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdint.h>
#include "assert.h"
#include "move.h"
#include "board.h"
#include "movegen.h"
#include "attack_table.h"

/* Initialise attack tables */
void initialise_attack_tables(board_t *board)
{
   assert(board);
   int n;
   sides side;

   for (side = WHITE; side < NUM_SIDES; side++) {
      for (n=0; n<board->piece_types->num_piece_types; n++) {
         if (board->large_board) {
            large_bitboard_t bb = board->large_bbc[side] & board->large_bbp[n];
            board->large_attack_table[n][side] = get_large_attack_bitboard(board, bb, side);
         } else {
            bitboard_t bb = board->bbc[side] & board->bbp[n];
            board->attack_table[n][side] = get_attack_bitboard(board, bb, side);
         }
      }
   }
}

/* Regenerate the attack table for pieces whose attack table intersects with the target square and
 * for pieces identified by the bitpattern.
 */
static void update_large_attack_table_for_move_bitboard(board_t *board, large_bitboard_t delta_occ, uint32_t update_pieces)
{
   assert(board);
   int n;
   sides side;
   
   board->large_attacks[WHITE] = large_board_empty;
   board->large_attacks[BLACK] = large_board_empty;

   /* Check which attack tables need to be updated */
   for (n=0; n<board->piece_types->num_piece_types; n++) {
      for (side = WHITE; side < NUM_SIDES; side++) {
         uint32_t piece_flag = 1 << (n + 16*side);
         if (!is_zero128(board->large_attack_table[n][side] & delta_occ) || (update_pieces & piece_flag)) {
            large_bitboard_t bb = board->large_bbc[side] & board->large_bbp[n];
            board->large_attack_table[n][side] = get_large_attack_bitboard(board, bb, side);
         }
         board->large_attacks[side] |= board->large_attack_table[n][side];
      }
   }
}

/* Regenerate the attack table for pieces whose attack table intersects with the target square and
 * for pieces identified by the bitpattern.
 */
static void update_attack_table_for_move_bitboard(board_t *board, bitboard_t delta_occ, uint32_t update_pieces)
{
   assert(board);
   int n;
   sides side;
   
   board->attacks[WHITE] = board_empty;
   board->attacks[BLACK] = board_empty;

   /* Check which attack tables need to be updated */
   for (n=0; n<board->piece_types->num_piece_types; n++) {
      for (side = WHITE; side < NUM_SIDES; side++) {
         uint32_t piece_flag = 1 << (n + 16*side);
         if ((board->attack_table[n][side] & delta_occ) || (update_pieces & piece_flag)) {
            bitboard_t bb = board->bbc[side] & board->bbp[n];
            board->attack_table[n][side] = get_attack_bitboard(board, bb, side);
         }
         board->attacks[side] |= board->attack_table[n][side];
      }
   }
}

/* Update attack tables for the move that has just been made/unmade */
void update_attack_table(board_t *board, move_t move)
{
#if !defined USE_ATTACK_TABLE
   return;
#endif

   assert(board);

   /* null move */
   if (move == 0) return;

   /* Construct bitboards for squares for which the occupancy state has changed */
   large_bitboard_t large_delta_occ = large_board_empty;
   bitboard_t delta_occ = board_empty;
   uint32_t update_pieces = 0;
   int n, c;

    /* First: mark all pickups */
    n = get_move_pickups(move);
    for (c=0; c<n; c++) {
       uint16_t p = get_move_pickup(move, c);
       int square = decode_pickup_square(p);
       int piece  = decode_pickup_piece(p);
       sides side = decode_pickup_side(p);
       int shift = piece + side * 16;

       update_pieces |= 1<<shift;

       delta_occ ^= make_bitboard_square(square);
       large_delta_occ ^= large_square_bitboards[square];
    }

    /* Second: mark all drops */
    n = get_move_drops(move);
    for (c=0; c<n; c++) {
       uint16_t p = get_move_drop(move, c);
       int square = decode_drop_square(p);
       int piece  = decode_drop_piece(p);
       sides side = decode_drop_side(p);
       int shift = piece + side * 16;

       update_pieces |= 1<<shift;

       delta_occ ^= make_bitboard_square(square);
       large_delta_occ ^= large_square_bitboards[square];
    }

    /* TODO: for a capture, the occupancy state of the capture square does not change,
     * but we should still update the attack tables for the captured piece type.
     * Similarly, the occupancy of the square where the capturing piece originated is empty and will not
     * be picked up when attack tables are updated.
     *
     * The attack tables that should be updated, then, are those for which the move bit pattern intersects
     * the attack map and those for the pieces directly involved in the move (there may be overlap here).
     * With 16 piece types and two sides, the piece pattern should fit nicely in a 32-bit integer.
     */

   if (board->large_board) {
      update_large_attack_table_for_move_bitboard(board, large_delta_occ, update_pieces);
   } else {
      update_attack_table_for_move_bitboard(board, delta_occ, update_pieces);
   }
}

