/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include <ctype.h>
#include "board.h"
#include "pieces.h"

void clear_board(board_t *board)
{
   piece_description_t *ps = board->piece_types;
   bitboard_t drop_zone[NUM_SIDES];
   uint32_t rule_flags = board->rule_flags;
   bool use_holdings = board->use_holdings;
   bool large_board = board->large_board;

   memcpy(drop_zone, board->drop_zone, sizeof drop_zone);

   memset(board, 0, sizeof *board);

   board->piece_types = ps;
   board->rule_flags = rule_flags;
   memcpy(board->drop_zone, drop_zone, sizeof drop_zone);
   board->use_holdings = use_holdings;
   board->large_board = large_board;
}



/**********************************************************************
 *                       output functions                             *
 **********************************************************************/
static const char out_char[] = { '.', '1' };
void print_uint8(uint8_t b)
{
   int c;

   for (c=0; c<8; c++) {
      printf ("%c", out_char[(b>>c)&1]);
   }
}

static void print_uint16(uint16_t b)
{
   int c;
   
   for (c=0; c<large_board_files; c++) {
      printf ("%c", out_char[(b>>c)&1]);
   }
}

static void print_large_bitboards(const board_t *board)
{
   int c, n;

   if (!board->large_board)
      return;

   printf("White pieces\tBlack pieces\tUnmoved pieces\tRoyal\t\tep\n");
   large_bitboard_t ep = large_board_empty;
   if (board->ep) ep = large_square_bitboards[board->ep];
   for (c=large_board_ranks-1; c>=0; c--) {
      print_uint16(get_large_bitboard_row(board->large_bbc[0], c));
      printf("\t");
      print_uint16(get_large_bitboard_row(board->large_bbc[1], c));
      printf("\t");
      print_uint16(get_large_bitboard_row(board->large_init, c));
      printf("\t");
      print_uint16(get_large_bitboard_row(board->large_royal, c));
      printf("\t");
      print_uint16(get_large_bitboard_row(ep, c));
      printf("\n");
   }

   if (!board->piece_types)
      return;

   for (n=0; n<board->piece_types->num_piece_types; n+=7) {
      printf("\n");
      int k;
      for (k=0; k<7; k++) {
         if (n+k >= board->piece_types->num_piece_types) break;
         char *s = board->piece_types->piece_name[n+k];
         printf("%*s", -large_board_files-2, s);
      }
      printf("\n");
      for (c=large_board_ranks-1; c>=0; c--) {
         for (k=0; k<7; k++) {
            if (n+k >= board->piece_types->num_piece_types) break;
            print_uint16(get_large_bitboard_row(board->large_bbp[n+k], c));
            printf("  ");
         }
         printf("\n");
      }
   }

   if (board->use_holdings) {
      for (c=0; c<NUM_SIDES; c++) {
         printf("%s holdings [ ", c ? "Black" : "White");
         for (n=0; n<board->piece_types->num_piece_types; n++)
            if (board->holdings[n][c])
               printf("%s: %02d ", board->piece_types->piece_notation[n], board->holdings[n][c]);
         printf("]\n");
      }
   }
}


void print_bitboards(const board_t *board)
{
   int c, n;

   if (board->large_board) {
      print_large_bitboards(board);
      return;
   }

   if (board->side_to_move == WHITE)
      printf("White to move\n");
   else
      printf("Black to move\n");

   printf("White pieces\tBlack pieces\tUnmoved pieces\tRoyal\t\tep\n");
   bitboard_t ep = board_empty;
   if (board->ep) ep = square_bitboards[board->ep];
   for (c=7; c>=0; c--) {
      print_uint8(get_bitboard_row(board->bbc[0], c));
      printf("\t");
      print_uint8(get_bitboard_row(board->bbc[1], c));
      printf("\t");
      print_uint8(get_bitboard_row(board->init, c));
      printf("\t");
      print_uint8(get_bitboard_row(board->royal, c));
      printf("\t");
      print_uint8(get_bitboard_row(ep, c));
      printf("\n");
   }

   if (!board->piece_types)
      return;

   for (n=0; n<board->piece_types->num_piece_types; n+=7) {
      printf("\n");
      int k;
      for (k=0; k<7; k++) {
         if (n+k >= board->piece_types->num_piece_types) break;
         char *s = board->piece_types->piece_name[n+k];
         printf("%-10s", s);
      }
      printf("\n");
      for (c=7; c>=0; c--) {
         for (k=0; k<7; k++) {
            if (n+k >= board->piece_types->num_piece_types) break;
            print_uint8(get_bitboard_row(board->bbp[n+k], c));
            printf("  ");
         }
         printf("\n");
      }
   }

   if (board->use_holdings) {
      for (c=0; c<NUM_SIDES; c++) {
         printf("%s holdings [ ", c ? "Black" : "White");
         for (n=0; n<board->piece_types->num_piece_types; n++)
            if (board->holdings[n][c])
               printf("%s: %02d ", board->piece_types->piece_notation[n], board->holdings[n][c]);
         printf("]\n");
      }
   }
}

void print_board(const board_t *board)
{
   bitboard_t unocc = ~(board->bbc[WHITE] | board->bbc[BLACK]);
   int c, n;

   for (c=large_board_ranks-1; c>=0; c--) {
      for (n=0; n<large_board_files; n++) {
         int square = pack_row_file(c, n);
         int piece = get_piece(board, square);
         if (make_bitboard_square(square) & unocc) {
            if ( (c^n)&1 )
               printf(". ");
            else
               printf("+ ");
         } else {
            char cc = board->piece_types->piece_abbreviation[piece][0];
            bool white = true;
            if (cc == ',') cc = board->piece_types->piece_abbreviation[piece][1];
            if (board->large_board) {
               white = !is_zero128(board->large_bbc[WHITE] & large_square_bitboards[pack_row_file(c, n)]);
            } else {
               white = board->bbc[WHITE] & make_bitboard_square(pack_row_file(c, n));
            }
            if (white) {
               cc = toupper(cc);
            } else {
               cc = tolower(cc);
            }
            printf("%c ", cc);
         }
      }
      printf("\n");
   }
}

