/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdarg.h>
#include <ctype.h>
#include <math.h>
#include "softexp.h"
#include "square.h"
#include "board.h"
#include "move.h"
#include "game.h"
#include "evaluate.h"
#include "moveflags.h"
#include "attack_table.h"
#include "pawn_table.h"
#include "eval_table.h"

static inline int min(int x, int y)
{
   return (x<y) ? x : y;
}

static inline int max(int x, int y)
{
   return (x>y) ? x : y;
}

static char buf[65536];
static void printfstderr(const char *msg, ...)
{
   va_list ap;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   fprintf(stderr, "%s", buf);
}

static void (*default_iteration_output)(const char *, ...) = NULL;
static void (*default_uci_output)(const char *, ...) = NULL;
static void (*default_xboard_output)(const char *, ...) = NULL;
static void (*default_error_output)(const char *, ...) = printfstderr;

#define pstderr if (game && game->error_output) game->error_output

void set_board_size(game_t *game, int files, int ranks)
{
   if (files > 8 || ranks > 8) {
      /* Initialise bitmasks for the large bitboard */
      initialise_large_bitboards(files, ranks);

      /* Initialised data structures for a large board */
      initialise_large_slider_tables();

      game->large_board = game->board.large_board = true;

      /* Castle flags, these are stubs and need to be corrected for the variant in question */
      short_castle_king_dest[WHITE] = files/2;
      short_castle_king_dest[BLACK] = files/2 + (ranks-1)*files;
      long_castle_king_dest[WHITE]  = files/4;
      long_castle_king_dest[BLACK]  = files/4 + (ranks-1)*files;
   } else {
      initialise_bitboards(8, 8);

      /* Initialise slider data structures */
      initialise_slider_tables();

      game->large_board = game->board.large_board = false;
      large_board_ranks = ranks;
      large_board_files = files;

      /* Castle flags */
      short_castle_king_dest[WHITE] = G1;
      short_castle_king_dest[BLACK] = G8;
      long_castle_king_dest[WHITE]  = C1;
      long_castle_king_dest[BLACK]  = C8;

      short_castle_mask[WHITE] = E1_MASK|H1_MASK;
      short_castle_free[WHITE] = G1_MASK|F1_MASK;
      short_castle_safe[WHITE] = G1_MASK|F1_MASK|E1_MASK;
      long_castle_mask[WHITE]  = E1_MASK|A1_MASK;
      long_castle_free[WHITE]  = B1_MASK|C1_MASK|D1_MASK;
      long_castle_safe[WHITE]  = C1_MASK|D1_MASK|E1_MASK;

      short_castle_mask[BLACK] = E8_MASK|H8_MASK;
      short_castle_free[BLACK] = G8_MASK|F8_MASK;
      short_castle_safe[BLACK] = G8_MASK|F8_MASK|E8_MASK;
      long_castle_mask[BLACK]  = E8_MASK|A8_MASK;
      long_castle_free[BLACK]  = B8_MASK|C8_MASK|D8_MASK;
      long_castle_safe[BLACK]  = C8_MASK|D8_MASK|E8_MASK;
   }
   load_square_names(files, ranks);
}

void remove_square(game_t *game, int square)
{
   assert(game);

   if (game->large_board) {
      large_bitboard_t bb = ~large_square_bitboards[square];
      int n;

      large_board_all &= bb;
      large_board_edge &= bb;
      large_board_east_edge &= bb;
      large_board_west_edge &= bb;
      large_board_north_edge &= bb;
      large_board_south_edge &= bb;
      large_board_south &= bb;
      large_board_north &= bb;
      for (n=0; n<16; n++) {
         large_board_rank[n] &= bb;
         large_board_file[n] &= bb;
         large_board_northward[n] &= bb;
         large_board_southward[n] &= bb;
      }
   } else {
      bitboard_t bb = ~make_bitboard_square(square);

      board_all     &= bb;
      board_edge    &= bb;
      board_dark    &= bb;
      board_light   &= bb;
      board_rank1   &= bb;
      board_rank2   &= bb;
      board_rank3   &= bb;
      board_rank4   &= bb;
      board_rank5   &= bb;
      board_rank6   &= bb;
      board_rank7   &= bb;
      board_rank8   &= bb;
   }
}

void place_flag(game_t *game, sides side, int square)
{
   assert(game);

   if (game->large_board)
      game->flag[side].lbb |= large_square_bitboards[square];
   else
      game->flag[side].bb |= make_bitboard_square(square);
}

void playgame_init(void)
{
   default_iteration_output = (void *)printf;
   default_uci_output = NULL;
}

void playgame_shutdown(void)
{
}

void set_default_output_function(void (*func)(const char *, ...))
{
   default_iteration_output = func;
}

void set_uci_output_function(void (*func)(const char *, ...))
{
   default_uci_output = func;
}

void set_xboard_output_function(void (*func)(const char *, ...))
{
   default_xboard_output = func;
}

void set_error_output_function(void (*func)(const char *, ...))
{
   default_error_output = func;
}

void set_transposition_table_size(game_t *game, uint64_t nelem)
{
   if (!game)
      return;

   /* Round to the next-lowest power of 2 */
   nelem |= (nelem >> 1);
   nelem |= (nelem >> 2);
   nelem |= (nelem >> 4);
   nelem |= (nelem >> 8);
   nelem |= (nelem >> 16);
   nelem |= (nelem >> 32);
   nelem >>= 1;
   nelem++;

   game->default_hash_size = nelem;

   /* Reallocate table if it was already allocated */
   if (game->default_hash_size) {
      destroy_hash_table(game->transposition_table);
      game->transposition_table = create_hash_table(game->default_hash_size);
   }
}


game_t *create_game(void)
{
   /* Make sure memory is aligned on a 16-byte (128 bit) boundary */
   void *p = calloc(1, sizeof(game_t)+15);
   game_t *game = p;
   if ((uint64_t)(game) & 15) game = (game_t *)(((uint64_t)game + 15) & ~15ull);
   game->board.piece_types = &game->pt;
   game->board.rule_flags = 0;
   game->board.use_holdings = false;
   game->board.drop_zone[WHITE] = game->board.drop_zone[BLACK] = board_all;
   game->mate_score = -CHECKMATE;
   game->stale_score = -STALEMATE;
   game->rep_score = -STALEMATE;
   game->no_piece_score = -CHECKMATE;
   game->flag_score = -CHECKMATE;
   game->bare_king_score = 0;
   game->default_hash_size = HASH_TABLE_SIZE;
   game->real_base = p;
   return game;
}

/* Initialize the game datastructure to correspond to a new game */
void start_new_game(game_t *game)
{
   game->max_moves = 1000;
   free(game->score);
   free(game->in_check);
   free(game->did_castle);
   free(game->fifty_counter);
   free(game->ep_capture);
   free(game->ep);
   free(game->hash_key);
   free(game->init);
   free(game->move_list);
   destroy_hash_table(game->transposition_table);
   destroy_pawn_hash_table(game->pawn_table);
   destroy_eval_hash_table(game->eval_table);
   game->score          = calloc(game->max_moves, sizeof *game->score);
   game->in_check       = calloc(game->max_moves, sizeof *game->in_check);
   game->did_castle     = calloc(game->max_moves, sizeof *game->did_castle);
   game->fifty_counter  = calloc(game->max_moves, sizeof *game->fifty_counter);
   game->ep_capture     = calloc(game->max_moves, sizeof *game->ep_capture);
   game->ep             = calloc(game->max_moves, sizeof *game->ep);
   game->hash_key       = calloc(game->max_moves, sizeof *game->hash_key);
   game->init           = calloc(game->max_moves, sizeof *game->init);
   game->move_list      = calloc(game->max_moves, sizeof *game->move_list);

   game->transposition_table = NULL;

   clear_board(&game->board);
   if (game->start_fen)
      setup_fen_position(game, game->start_fen);

   if (game->default_hash_size == 0) game->default_hash_size = HASH_TABLE_SIZE;
   game->transposition_table = create_hash_table(game->default_hash_size);
   game->pawn_table = create_pawn_hash_table(1024*1024);
   game->eval_table = create_eval_hash_table(1024*64);

   game->analysing = false;
   game->pondering = false;
   game->ponder_move = 0;

   game->name_white = NULL;
   game->name_black = NULL;

   game->output_iteration = default_iteration_output;
   game->uci_output = default_uci_output;
   game->xboard_output = default_xboard_output;
   game->error_output = default_error_output;
}

/* Frees the memory used by a game */
void end_game(game_t *game)
{
   if (game) {
      destroy_hash_table(game->transposition_table);
      destroy_pawn_hash_table(game->pawn_table);
      destroy_eval_hash_table(game->eval_table);
      free(game->start_fen);
      free(game->xb_setup);
      free(game->name_white);
      free(game->name_black);
      free(game->score);
      free(game->in_check);
      free(game->did_castle);
      free(game->fifty_counter);
      free(game->ep_capture);
      free(game->ep);
      free(game->hash_key);
      free(game->init);
      free(game->real_base);
   }
}

bool add_piece_type(game_t *game, move_flag_t move_flags, move_flag_t capture_flags, uint8_t piece_flags,
                     ubitboard_t promotion_zone, const char *promotion_string,
                     const char *name, const char *symbol, const char *notation)
{
   if (!game || game->pt.num_piece_types>=MAX_PIECE_TYPES)
      return false;

   int n = game->pt.num_piece_types;
   uint16_t promotion_choice = 0;

   /* Store allowed pieces for promotion in a bit field */
   if (promotion_string && promotion_string[0]) {
      const char *s = promotion_string;
      while (*s) {
         int c;
         for (c=0; c<game->pt.num_piece_types; c++) {
            char *p = strchr(game->pt.piece_abbreviation[c], *s);
            if (p) {
               promotion_choice |= 1<<c;
            }
         }
         s++;
      }
   }
   free(game->pt.piece_name[n]);
   free(game->pt.piece_abbreviation[n]);
   free(game->pt.piece_notation[n]);

   game->pt.piece_move_flags[n] = move_flags;
   game->pt.piece_capture_flags[n] = capture_flags;
   game->pt.piece_flags[n] = piece_flags;
   game->pt.piece_special_zone[n] = 0;
   if (game->board.large_board) {
      game->pt.piece_large_promotion_zone[n] = promotion_zone.lbb;
      game->pt.large_prison[n] = large_board_all;
   } else {
      game->pt.piece_promotion_zone[n] = promotion_zone.bb;
      game->pt.prison[n] = board_all;
   }
   game->pt.piece_drop_zone[n] = board_all;
   game->pt.piece_promotion_choice[n] = promotion_choice;
   game->pt.piece_name[n] = strdup(name);
   game->pt.piece_abbreviation[n] = strdup(symbol);
   game->pt.piece_notation[n] = strdup(notation);
   if (piece_flags & PF_ROYAL) {
      game->pt.piece_maximum[n][WHITE] = 1;
      game->pt.piece_maximum[n][BLACK] = 1;
   } else {
      game->pt.piece_maximum[n][WHITE] = 128;
      game->pt.piece_maximum[n][BLACK] = 128;
   }
   game->pt.num_piece_types++;

   piece_symbol_string[n] = notation[0];
   piece_symbol_string[n+1] = '\0';
   return true;
}

void set_maximum_number_of_kings(game_t *game, sides side, int count)
{
   int n;
   for (n=0; n<game->pt.num_piece_types; n++)
      if (game->pt.piece_flags[n] & PF_ROYAL)
         game->pt.piece_maximum[n][side] = count;
}

void set_maximum_number_of_pieces(game_t *game, sides side, const char *symbol, int count)
{
   int n;
   for (n=0; n<game->pt.num_piece_types; n++) {
      char *p = strchr(game->pt.piece_abbreviation[n], *symbol);
      if (p)
         game->pt.piece_maximum[n][side] = count;
   }
}

void add_rule_flag(game_t *game, move_flag_t rule_flag)
{
   game->board.rule_flags |= rule_flag;
}

void remove_rule_flag(game_t *game, move_flag_t rule_flag)
{
   game->board.rule_flags &= ~rule_flag;
}

bool add_special_move(game_t *game, const char *symbol, ubitboard_t zone, move_flag_t move_flags)
{
   int n;

   if (!symbol)
      return false;

   for (n=0; n<game->pt.num_piece_types; n++) {
      char *p = strchr(game->pt.piece_abbreviation[n], *symbol);
      if (p) {
         if (game->board.large_board)
            game->pt.piece_large_special_zone[n] = zone.lbb;
         else
            game->pt.piece_special_zone[n] = zone.bb;
         game->pt.piece_special_move_flags[n] = move_flags;
         return true;
      }
   }

   return false;
}

bool set_piece_drop_zone(game_t *game, const char *symbol, ubitboard_t zone)
{
   int n;

   if (!symbol)
      return false;

   for (n=0; n<game->pt.num_piece_types; n++) {
      char *p = strchr(game->pt.piece_abbreviation[n], *symbol);
      if (p) {
         game->pt.piece_drop_zone[n] = zone.bb;
         return true;
      }
   }

   return false;
}

void set_player_drop_zone(game_t *game, sides side, ubitboard_t zone)
{
   assert(game);
   assert(side == WHITE || side == BLACK);

   game->board.drop_zone[side] = zone.bb;
}

bool set_piece_value(game_t *game, const char *symbol, uint16_t value)
{
   int n;

   if (!symbol)
      return false;

   for (n=0; n<game->pt.num_piece_types; n++) {
      char *p = strchr(game->pt.piece_abbreviation[n], *symbol);
      if (p) {
         game->pt.piece_value[n] = value;
         return true;
      }
   }

   return false;
}

/* Only works for leapers at the moment. */
bool limit_piece_movement_region(game_t *game, const char *symbol, ubitboard_t region)
{
   int n;

   if (!symbol)
      return false;

   for (n=0; n<game->pt.num_piece_types; n++) {
      char *p = strchr(game->pt.piece_abbreviation[n], *symbol);
      if (p) {
         if (game->large_board) {
            move_flag_t mf = set_large_leaper_mask(game->pt.piece_move_flags[n], region.lbb);
            move_flag_t cf = mf;
            if (game->pt.piece_move_flags[n] != game->pt.piece_capture_flags[n])
               cf = set_large_leaper_mask(game->pt.piece_capture_flags[n], region.lbb);
            if (mf == -1 || cf == -1) return false;
            game->pt.piece_move_flags[n] &= ~MF_LEAPER_FLAGS;
            game->pt.piece_capture_flags[n] &= ~MF_LEAPER_FLAGS;
            game->pt.piece_move_flags[n] |= mf;
            game->pt.piece_capture_flags[n] |= cf;
            game->pt.large_prison[n] = region.lbb;
         } else {
            move_flag_t mf = set_leaper_mask(game->pt.piece_move_flags[n], region.bb);
            move_flag_t cf = mf;
            if (game->pt.piece_move_flags[n] != game->pt.piece_capture_flags[n])
               cf = set_leaper_mask(game->pt.piece_capture_flags[n], region.bb);
            if (mf == -1 || cf == -1) return false;
            game->pt.piece_move_flags[n] &= ~MF_LEAPER_FLAGS;
            game->pt.piece_capture_flags[n] &= ~MF_LEAPER_FLAGS;
            game->pt.piece_move_flags[n] |= mf;
            game->pt.piece_capture_flags[n] |= cf;
            game->pt.prison[n] = region.bb;
         }
         if (game->pt.piece_flags[n] & PF_ROYAL)
            game->board.rule_flags |= RF_KING_TRAPPED;
         return true;
      }
   }

   return false;
}

int get_piece_index(const game_t *game, const char *symbol)
{
   int n;

   if (!symbol)
      return -1;

   for (n=0; n<game->pt.num_piece_types; n++) {
      char *p = strchr(game->pt.piece_abbreviation[n], *symbol);
      if (p)
         return n;
   }

   return -1;
}

/* Load the standard evaluation weights for each piece type.
 * These are based on some general heuristics:
 *  - sliders should be centralised and above else have good mobility
 *  - leapers should be centralised and advanced and have good mobility
 *  - steppers should be centralised, advanced, and form a shield
 */
void load_piece_evaluation_terms(game_t *game)
{
   // *    centre_weight        (8 bit)
   // *    advance_weight       (8 bit)
   // *    shield_weight        (8 bit)
   // *    mobility_score       (8 bit)
   int n, c;

   for (n=0; n<game->pt.num_piece_types; n++) {
      game->pt.advance_weight[n] = 0;
      game->pt.promo_weight[n] = 0;
      game->pt.shield_weight[n] = 0;
      game->pt.centre_weight[n] = 0;
      for (c=0; c<128; c++)
         game->pt.mobility_score[n][c] = 0;

      if (game->pt.piece_flags[n] & PF_ROYAL) {
         //game->pt.centre_weight[n] = -1;
         continue;
      }

      if (game->pt.piece_promotion_zone[n] != board_empty) {
         game->pt.promo_weight[n] = 4;
      }

      /* Determine the piece with the highest value that each piece can promote to.
       * We use this to scale the value of a passed pawn, which is calibrated using the value of a queen
       * in normal chess. Rescale so the weight is on a 0..128 scale.
       */
      game->pt.piece_promotion_value[n] = 0;
      c = game->pt.piece_promotion_choice[n];
      while (c) {
         int tpiece = bitscan16(c);
         c ^= (1<<tpiece);
         if (game->pt.piece_value[tpiece] > game->pt.piece_promotion_value[n])
            game->pt.piece_promotion_value[n] = game->pt.piece_value[tpiece];
      }
      game->pt.piece_promotion_value[n] *= 128. / 950.;

      if (game->pt.piece_move_flags[n] & MF_STEPPER) {
         /* Bonus for advancing this piece type */
         game->pt.advance_weight[n] = 1;

         /* The bonus for the "pawn shield" should be larger, taking into accoutn the PSQ tables */
         game->pt.shield_weight[n] = 8;

         /* Centralisation of this piece type is somewhat important as well */
         game->pt.centre_weight[n] = 2;
      }

      if (game->pt.piece_move_flags[n] & MF_IS_LEAPER) {
         /* Bonus for advancing this piece type */
         game->pt.advance_weight[n] = 1;

         /* No shield bonus */
         game->pt.shield_weight[n] = 0;

         /* Centralisation of this piece type is rather important as well */
         game->pt.centre_weight[n] = 3;

         /* If the king is trapped, then it's less important to centralise and more important to attack the
          * prison ("palace").
          */
         if (game->board.rule_flags & RF_KING_TRAPPED) {
            game->pt.advance_weight[n]++;
            game->pt.centre_weight[n] -= 2;
         }
      }

      if (game->pt.piece_move_flags[n] & MF_SLIDER) {
         /* No bonus for advancing this piece type */
         game->pt.advance_weight[n] += 1;

         /* No shield bonus */
         game->pt.shield_weight[n] += 0;

         /* Centralisation of this piece type is rather important as well */
         if (game->pt.piece_move_flags[n] & (MF_SLIDER_D|MF_SLIDER_A))
            game->pt.centre_weight[n] += 2;
      }

      /* Progressive mobility evaluation */
      if ((game->pt.piece_move_flags[n] & (MF_IS_LEAPER|MF_SLIDER)) && !(game->pt.piece_flags[n] & PF_ROYAL)) {
         for (c=0; c<128; c++) {
            int board_size = large_board_files * large_board_ranks;
            int numer = game->pt.max_moves[n] - game->pt.min_moves[n];
            if (numer == 0) numer = game->pt.max_moves[n];
            assert(numer);
            double x = (double)(c - game->pt.min_moves[n]) / numer;
            double inflection = MOBILITY_INFLECTION;
            double slope = MOBILITY_SLOPE;
            double ceil = MOBILITY_CEIL;

            /* Specific terms for different piece types */
            if ((game->pt.piece_move_flags[n] & MF_SLIDER) == game->pt.piece_move_flags[n]) {
               inflection = MOBILITY_SLIDER_INFLECTION;
               slope = MOBILITY_SLIDER_SLOPE;
               ceil = MOBILITY_SLIDER_CEIL;
            }
            if ((game->pt.piece_move_flags[n] & MF_LEAPER_FLAGS) == game->pt.piece_move_flags[n]) {
               inflection = MOBILITY_LEAPER_INFLECTION;
               slope = MOBILITY_LEAPER_SLOPE;
               ceil = MOBILITY_LEAPER_CEIL;
            }

            double y = (x - inflection) * slope;
            double f = 2.0*(1./(1. + myexp(-y)) - 0.5);
            int mob = ceil * board_size / game->pt.avg_moves[n] * f;

            if (c > game->pt.max_moves[n]) break;

            if (mob > 127) mob = 127;
            if (mob <-127) mob = -127;
            game->pt.mobility_score[n][c] = mob;

            //printf("%-10s %d %d\n", game->pt.piece_name[n], c, game->pt.mobility_score[n][c]);
         }
      }
   }

   /* Asses whether pieces can mate or not */
   asses_mating_potential(game);
   asses_mating_pairs(game);
}

bool set_evaluation_terms(game_t *game, const char *symbol, int advance, int shield, int centre)
{
   int n;

   if (!symbol)
      return false;

   for (n=0; n<game->pt.num_piece_types; n++) {
      char *p = strchr(game->pt.piece_abbreviation[n], *symbol);
      if (p) {
         game->pt.advance_weight[n] = advance;
         game->pt.shield_weight[n] = shield;
         game->pt.centre_weight[n] = centre;
         return true;
      }
   }

   return false;
}

static int sort_value[MAX_PIECE_TYPES];
static int piece_value_sorter(const void *p1, const void *p2)
{
   int i1 = *(int *)(p1);
   int i2 = *(int *)(p2);

   return sort_value[i1] - sort_value[i2];
}

/* Set up the permutation list of piece values (in increasing order) */
void sort_piece_values(game_t *game)
{
   int n;

   for (n=0; n<game->pt.num_piece_types; n++) {
      sort_value[n] = game->pt.piece_value[n];
      game->pt.val_perm[n] = n;
      if (game->pt.piece_flags[n] & PF_ROYAL)
         sort_value[n] += CHECKMATE;
   }
   qsort(game->pt.val_perm, game->pt.num_piece_types, sizeof *game->pt.val_perm, piece_value_sorter);
}

/* Asses the mating potential of single pieces, to decide how to evaluate positions of lone king vs. a piece
 * of this type.
 */
void asses_mating_potential(game_t *game)
{
   int corner = (large_board_ranks-1)*large_board_files;
   board_t board;
   bitboard_t king1 = A8_MASK;
   bitboard_t king2 = board_all & ~king1;
   large_bitboard_t large_king1 = large_square_bitboards[corner];
   large_bitboard_t large_king2 = large_board_all & ~large_king1;
   large_bitboard_t large_bb;
   bitboard_t bb;
   int royal_index = 0;

   /* Find the index of the royal piece
    * FIXME: black and white may not have the same type of royal piece!
    */
   for (royal_index = 0; royal_index<game->pt.num_piece_types; royal_index++)
      if (game->pt.piece_flags[royal_index] & PF_ROYAL) break;

   /* Break out if there is no royal piece */
   if (royal_index == game->pt.num_piece_types)
      return;

   /* Set up the board */
   memcpy(&board, &game->board, sizeof board);
   clear_board(&board);

   if (board.large_board) {
      board.large_bbc[WHITE] |= large_king1;
      board.large_royal |= large_king1;
      board.large_bbp[royal_index] |= large_king1;
      board.piece[corner] = royal_index;

      /* Get all allowed squares for the enemy king */
      large_bb = get_large_attack_bitboard(&board, large_king1, WHITE);
      large_king2 &= ~large_bb;
      large_king1 |= large_bb;
   } else {
      board.bbc[WHITE] |= king1;
      board.royal |= king1;
      board.bbp[royal_index] |= king1;
      board.piece[corner] = royal_index;

      /* Get all allowed squares for the enemy king */
      bb = get_attack_bitboard(&board, king1, WHITE);
      king2 &= ~bb;
      king1 |= bb;
   }

   /* Loop over all piece types */
   int n;
   for (n=0; n<game->pt.num_piece_types; n++) {
      int ps, ks;
      bool mate_potential = false;

      /* Skip royal pieces */
      if (n == royal_index)
         continue;

      /* Skip pawns */
      if (game->pt.piece_flags[n] & PF_NORET)
         continue;

      /* Skip pieces that can promote */
      if (!board.large_board && game->pt.piece_promotion_zone[n]) continue;
      if (board.large_board && !is_zero128(game->pt.piece_large_promotion_zone[n])) continue;

      /* Loop over all squares */
      int board_size = large_board_files * large_board_ranks;
      for (ps = 0; ps < board_size; ps++) {
         large_bitboard_t large_pbb = large_square_bitboards[ps];
         bitboard_t pbb = make_bitboard_square(ps);

         if (ps == corner) continue;

         if (mate_potential) break;

         /* Place the piece on the board */
         if (board.large_board) {
            board.large_bbc[BLACK] |= large_pbb;
            board.large_bbp[n] |= large_pbb;
         } else {
            board.bbc[BLACK] |= pbb;
            board.bbp[n] |= pbb;
         }
         board.piece[ps] = n;

         if (board.large_board) {
            large_bitboard_t kk = large_king2 & ~large_pbb;
            large_bb = get_large_attack_bitboard(&board, large_pbb, BLACK);

            if (!is_zero128(large_bb & large_king1)) {
               /* Now place the second piece on any of the remaining squares and test whether the position is
                * mate.
                */
               while (!is_zero128(kk) && !mate_potential) {
                  ks = bitscan128(kk);
                  large_bitboard_t kbb = large_square_bitboards[ks];
                  kk ^= kbb;

                  board.large_bbc[BLACK] |= kbb;
                  board.large_royal |= kbb;
                  board.large_bbp[royal_index] |= kbb;
                  board.piece[ks] = royal_index;

                  /* Check if piece is attacked by the defending king, if so, it must be protected */
                  if (!is_zero128(large_king1 & large_pbb)) {
                     large_bitboard_t ka = get_large_attack_bitboard(&board, kbb, BLACK);
                     if (is_zero128(large_pbb & ka)) {
                        board.large_bbc[BLACK] ^= kbb;
                        board.large_royal ^= kbb;
                        board.large_bbp[royal_index] ^= kbb;
                        continue;
                     }
                  }

                  /* Check if the defending king is under attack and unable to escape */
                  large_bitboard_t a = get_large_attack_bitboard(&board, board.large_bbc[BLACK], BLACK) | large_pbb;

                  if ( is_equal128(a & large_king1, large_king1) )
                     mate_potential = true;

                  board.large_bbc[BLACK] ^= kbb;
                  board.large_royal ^= kbb;
                  board.large_bbp[royal_index] ^= kbb;
               }
            }

            /* Remove the piece from the board */
            board.large_bbc[BLACK] ^= large_pbb;
            board.large_bbp[n] ^= large_pbb;
         } else {
            bitboard_t kk = king2 & ~pbb;
            bb = get_attack_bitboard(&board, pbb, BLACK);

            if (bb & king1) {
               /* Now place the second piece on any of the remaining squares and test whether the position is
                * mate.
                */
               while (kk && !mate_potential) {
                  ks = bitscan64(kk);
                  bitboard_t kbb = make_bitboard_square(ks);
                  kk ^= kbb;

                  board.bbc[BLACK] |= kbb;
                  board.royal |= kbb;
                  board.bbp[royal_index] |= kbb;
                  board.piece[ks] = royal_index;

                  /* Check if piece is attacked by the defending king, if so, it must be defended */
                  if (king1 & pbb) {
                     bitboard_t ka = get_attack_bitboard(&board, kbb, BLACK);
                     if ((pbb & ka) == 0) {
                        board.bbc[BLACK] ^= kbb;
                        board.royal ^= kbb;
                        board.bbp[royal_index] ^= kbb;
                        continue;
                     }
                  }

                  /* Check if one king stalemates the other */
                  bitboard_t kabb1 = get_attack_bitboard(&board, king1, WHITE);
                  bitboard_t kabb2 = get_attack_bitboard(&board, kbb, BLACK);
                  if ((kabb1 & kabb2) != kabb1) {

                     /* Check if the defending king is under attack and unable to escape */
                     bitboard_t a = get_attack_bitboard(&board, board.bbc[BLACK], BLACK) | pbb;
                     if ( (a & king1) == king1 ) mate_potential = true;

                  }
                  board.bbc[BLACK] ^= kbb;
                  board.royal ^= kbb;
                  board.bbp[royal_index] ^= kbb;
               }
            }

            /* Remove the piece from the board */
            board.bbc[BLACK] ^= pbb;
            board.bbp[n] ^= pbb;
         }
      }

      if (!mate_potential)
         game->pt.piece_flags[n] |= PF_NOMATE;
   }
}


/* Test whether a lone king is in check in the given position */
static bool lone_king_is_checked(board_t *board, sides side)
{
   if (board->large_board) {
      large_bitboard_t king = board->large_royal & board->large_bbc[side];
      large_bitboard_t attacks = get_large_attack_bitboard(board, large_board_all, next_side[side]);
      if (!is_zero128(king & attacks)) return true;
   } else {
      bitboard_t king = board->royal & board->bbc[side];
      bitboard_t attacks = get_attack_bitboard(board, board_all, next_side[side]);
      if (king & attacks) return true;
   }
   return false;
}

/* Test whether a lone king is mated in the given position */
static bool lone_king_is_mated(board_t *board, sides side)
{
   if (board->large_board) {
      large_bitboard_t king = board->large_royal & board->large_bbc[side];
      large_bitboard_t attacks = get_large_attack_bitboard(board, large_board_all, next_side[side]);
      large_bitboard_t moves = get_large_attack_bitboard(board, large_board_all, side);
      large_bitboard_t bb = moves | king;
      if (is_equal128(bb & attacks, bb)) return true;
   } else {
      bitboard_t king = board->royal & board->bbc[side];
      bitboard_t attacks = get_attack_bitboard(board, board_all, next_side[side]);
      bitboard_t moves = get_attack_bitboard(board, board_all, side);
      bitboard_t bb = moves | king;
      if ((bb & attacks) == bb) return true;
   }
   return false;
}

/* Assess mating potential of pairs of pieces */
void asses_mating_pairs(game_t *game)
{
   board_t bboard;
   int n1, n2;
   int ps1, ps2, ks1, ks2, ks2i;
   int dks[] = { 0,1,large_board_files+1,   2,large_board_files+2,2*large_board_files+3,  large_board_files };
   int board_size = large_board_files * large_board_ranks;
   int royal_index = 0;

   assert(game);

   /* Clear list */
   memset(game->pt.pieces_can_win, 0, sizeof(game->pt.pieces_can_win));

   /* Find the index of the royal piece
    * FIXME: black and white may not have the same type of royal piece!
    */
   for (royal_index = 0; royal_index<game->pt.num_piece_types; royal_index++)
      if (game->pt.piece_flags[royal_index] & PF_ROYAL) break;

   /* Break out if there is no royal piece */
   if (royal_index == game->pt.num_piece_types)
      return;

   /* Back up the board */
   memcpy(&bboard, &game->board, sizeof bboard);
   clear_board(&game->board);

   /* Try all combinations */
   for (n1 = 0; n1<game->pt.num_piece_types; n1++) {
      /* Skip pawns and royals */
      if (game->pt.pawn_index[WHITE] == n1 || game->pt.pawn_index[BLACK] == n1) continue;
      if (n1 == royal_index) continue;
      for (n2 = n1; n2<game->pt.num_piece_types; n2++) {

         /* If either piece can deliver mate on its own, continue */
         if (!(game->pt.piece_flags[n1] & PF_NOMATE) || !(game->pt.piece_flags[n2] & PF_NOMATE)) {
            game->pt.pieces_can_win[n1][n2] = game->pt.pieces_can_win[n2][n1] = true;
            continue;
         }

         /* Skip pawns */
         if (game->pt.pawn_index[WHITE] == n2 || game->pt.pawn_index[BLACK] == n2) continue;
         if (n2 == royal_index) continue;

         /* Now we need to find out whether a pair of pieces neither of which can deliver mate on their own
          * can do so by retrograde analysis. We restrict the defending king to a corner initially.
          */
         for (ks2i = 0; ks2i<3; ks2i++) {
            if (game->pt.pieces_can_win[n1][n2]) break;
            ks2 = dks[ks2i];
            place_piece(&game->board, BLACK, royal_index, ks2);
            for (ks1 = 0; ks1 < board_size; ks1++) {
               if (game->pt.pieces_can_win[n1][n2]) break;
               if (ks1 == ks2) continue;
               place_piece(&game->board, WHITE, royal_index, ks1);
               if (player_in_check(game, WHITE)) { remove_piece(&game->board, ks1); continue; }

               for (ps1 = 0; ps1 < board_size; ps1++) {
                  /* Break out early if we've determined that this combination of pieces can win */
                  if (game->pt.pieces_can_win[n1][n2]) break;
                  /* Skip if two pieces are on top of eachother */
                  if (ps1 == ks1 || ps1 == ks2) continue;
                  /* Skip if in this location the corner is not under attack */
                  bool does_attack_corner = false;
                  int n;
                  for (n=0; n<6; n++) {
                     if (game->large_board) {
                        if (!is_zero128(game->pt.large_attack_from[n1][dks[n]]&large_square_bitboards[ps1]))
                           does_attack_corner = true;
                     } else {
                        if (game->pt.attack_from[n1][dks[n]] & square_bitboards[ps1])
                           does_attack_corner = true;
                     }
                  }
                  if (!does_attack_corner) continue;
                  place_piece(&game->board, WHITE, n1, ps1);
                  int first = 0;
                  if (n1 == n2) first = ps1+1;
                  for (ps2 = first; ps2 < board_size; ps2++) {
                     if (ps2 == ps1 || ps2 == ks1 || ps2 == ks2) continue;
                     /* Skip if in this location the corner is not under attack */
                     bool does_attack_corner = false;
                     int n;
                     for (n=0; n<6; n++) {
                        if (game->large_board) {
                           if (!is_zero128(game->pt.large_attack_from[n2][dks[n]]&large_square_bitboards[ps2]))
                              does_attack_corner = true;
                        } else {
                           if (game->pt.attack_from[n2][dks[n]] & square_bitboards[ps2])
                              does_attack_corner = true;
                        }
                     }
                     if (!does_attack_corner) continue;
                     place_piece(&game->board, WHITE, n2, ps2);

                     /* Is this mate? */
                     if (lone_king_is_mated(&game->board, BLACK)) {
                        /* Retrograde analysis: was the mate forced? Proceed as follows:
                         * 1. Remove the checking piece, but mark squares from which it could also have
                         *    delivered mate (this can be a little tricky). Skip positions where the king is
                         *    still in check after removing the checking piece (too complicated)
                         * 2. Find all possible progenitor squares for the defending king. This may include
                         *    squares where he is checked.
                         * 3. For each progenitor square, find the alternative squares that the king could
                         *    have moved to other than the square where he was mated.
                         * 4. Find all squares where the removed piece could have attacked those squares to
                         *    prevent the king from escaping.
                         * 5. Intersect sets from (1) and (4). If non-empty, the mate was forced.
                         *
                         * This will fail to detect situations where the king only has a choice of where he is
                         * mated. Not sure if that ever applies in practice...
                         */
                        int checking_piece = ps1;
                        int other_piece = ps2;
                        remove_piece(&game->board, ps1);
                        if (lone_king_is_checked(&game->board, BLACK)) checking_piece = ps2;
                        place_piece(&game->board, WHITE, n1, ps1);
                        if (checking_piece == ps2) {
                           /* Verify that we're not dealing with double check */
                           remove_piece(&game->board, ps2);
                           if (lone_king_is_checked(&game->board, BLACK)) continue;

                           place_piece(&game->board, WHITE, n2, ps2);
                           int other_piece = ps1;
                        }

                        /* Find progenitor squares for the defending king. Exclude squares where the two kings
                         * would attack eachother.
                         */
                        if (game->large_board) {
                           large_bitboard_t taboo=get_large_attack_bitboard(&game->board, large_square_bitboards[ks1], WHITE);
                           large_bitboard_t prevk=get_large_attack_bitboard(&game->board, large_square_bitboards[ks2], BLACK);
                           large_bitboard_t escape=large_board_empty;
                           prevk &= ~(taboo | game->board.large_bbc[WHITE]);

                           /* Find alternative escape squares */
                           remove_piece(&game->board, ks2);
                           while(!is_zero128(prevk)) {
                              int square = bitscan128(prevk);
                              prevk ^= large_square_bitboards[square];
                              place_piece(&game->board, BLACK, royal_index, square);
                              escape |= get_large_attack_bitboard(&game->board, large_square_bitboards[square], BLACK);
                              remove_piece(&game->board, square);
                           }
                           place_piece(&game->board, BLACK, royal_index, ks2);

                           /* Exclude square on which the king was mated */
                           escape &= ~(game->board.large_bbc[BLACK] | taboo);

                           /* If there are no escape squares, break out early - the position could not have
                            * been reached.
                            */
                           if (is_zero128(escape)) continue;

                           /* Find out if there is any square where the checking piece could have removed all
                            * alternative escape squares.
                            */
                           large_bitboard_t sentry = large_board_all;
                           while(!is_zero128(escape)) {
                              int square = bitscan128(escape);
                              escape ^= large_square_bitboards[square];
                              sentry &= game->pt.large_attack_from[get_piece(&game->board, checking_piece)][square];
                           }

                           /* Find squares from where the checking piece could have moved to the square where
                            * it delivered mate, but where it would not have attacked the mated king.
                            */
                           int n = get_piece(&game->board, checking_piece);
                           sentry &= game->pt.large_attack_from[n][checking_piece];
                           sentry &= ~game->pt.large_attack_from[n][ks2];
                           sentry &= ~(game->board.large_bbc[WHITE] | large_square_bitboards[ks2]);
                           
                           /* If this set is not empty, then we could have delivered mate */
                           if (!is_zero128(sentry))
                              game->pt.pieces_can_win[n1][n2] = game->pt.pieces_can_win[n2][n1] = true;

                           if (n1 == n2 && n1 == 7 && game->pt.pieces_can_win[n1][n2]) {
                              exit(0);
                           }

                        } else {
                           bitboard_t taboo = get_attack_bitboard(&game->board, square_bitboards[ks1], WHITE);
                           bitboard_t prevk = get_attack_bitboard(&game->board, square_bitboards[ks2], BLACK);
                           bitboard_t escape = board_empty;
                           prevk &= ~(taboo | game->board.bbc[WHITE]);

                           /* Find previous progenitors */
                           remove_piece(&game->board, ks2);
                           while(prevk) {
                              int square = bitscan64(prevk);
                              prevk ^= square_bitboards[square];
                              place_piece(&game->board, BLACK, royal_index, square);
                              escape |= get_attack_bitboard(&game->board, square_bitboards[square], BLACK);
                              remove_piece(&game->board, square);
                           }
                           place_piece(&game->board, BLACK, royal_index, ks2);

                           /* Exclude square on which the king was mated */
                           escape &= ~(game->board.bbc[BLACK] | taboo);

                           /* If there are no escape squares, break out early - the position could not have
                            * been reached.
                            */
                           if (!escape) continue;

                           /* Find out if there is any square where the checking piece could have removed all
                            * alternative escape squares.
                            */
                           bitboard_t sentry = board_all;
                           while(escape) {
                              int square = bitscan64(escape);
                              escape ^= square_bitboards[square];
                              sentry &= game->pt.attack_from[get_piece(&game->board, checking_piece)][square];
                           }

                           /* Find squares from where the checking piece could have moved to the square where
                            * it delivered mate, but where it would not have attacked the mated king.
                            */
                           int n = get_piece(&game->board, checking_piece);
                           sentry &= game->pt.attack_from[n][checking_piece];
                           sentry &= ~game->pt.attack_from[n][ks2];
                           sentry &= ~(game->board.bbc[WHITE] | square_bitboards[ks2]);
                           
                           /* If this set is not empty, then we could have delivered mate */
                           if (sentry) game->pt.pieces_can_win[n1][n2] = game->pt.pieces_can_win[n2][n1] = true;

                        }

                        //game->pt.pieces_can_win[n1][n2] = game->pt.pieces_can_win[n2][n1] = true;
                        //break;
                     }
                     remove_piece(&game->board, ps2);
                  }
                  remove_piece(&game->board, ps1);
               }

               remove_piece(&game->board, ks1);
            }
            remove_piece(&game->board, ks2);
            break;
         }
      }
   }

   /* Restore the board */
   memcpy(&game->board, &bboard, sizeof bboard);
}

void identify_castle_partner(game_t *game)
{
   /* Identify partner in castling moves (for decoding FENs that don't
    * include the file of the partner in random setups).
    */
   game->pt.castle_piece[WHITE] = -1;
   game->pt.castle_piece[BLACK] = -1;

   if (!game->start_fen)
      return;

   /* Setup the initial position */
   setup_fen_position(game, game->start_fen);

   if (game->large_board) {
      large_bitboard_t bb = large_short_castle_mask[WHITE] | large_long_castle_mask[WHITE];
      bb &= ~game->board.large_royal;
      if (!is_zero128(bb)) {
         int square = bitscan128(bb);
         game->pt.castle_piece[WHITE] = get_piece(&game->board, square);
      }

      bb = large_short_castle_mask[BLACK] | large_long_castle_mask[BLACK];
      bb &= ~game->board.large_royal;
      if (!is_zero128(bb)) {
         int square = bitscan128(bb);
         game->pt.castle_piece[BLACK] = get_piece(&game->board, square);
      }
   } else {
      bitboard_t bb = short_castle_mask[WHITE] | long_castle_mask[WHITE];
      bb &= ~game->board.royal;
      if (bb) {
         int square = bitscan64(bb);
         game->pt.castle_piece[WHITE] = get_piece(&game->board, square);
      }

      bb = short_castle_mask[BLACK] | long_castle_mask[BLACK];
      bb &= ~game->board.royal;
      if (bb) {
         int square = bitscan64(bb);
         game->pt.castle_piece[BLACK] = get_piece(&game->board, square);
      }
   }

   clear_board(&game->board);
}

void get_mobility_statistics(game_t *game)
{
   board_t board;
   int board_size = large_board_files * large_board_ranks;

   /* Set up the board */
   memcpy(&board, &game->board, sizeof board);

   /* Initialise pawn-indices */
   game->pt.pawn_index[WHITE] = -1;
   game->pt.pawn_index[BLACK] = -1;

   /* Clear attack-from database */
   memset(&game->pt.large_attack_from, 0, sizeof game->pt.large_attack_from);

   /* Loop over all piece types */
   int n, ps;
   for (n=0; n<game->pt.num_piece_types; n++) {
      bool colour_bound = true;
      sides side = WHITE;

      /* Calculate "attack from" bitmasks */
      clear_board(&board);
      for (ps=0; ps<board_size; ps++) {
         /* Put the piece on the board */
         place_piece(&board, side, n, ps);

         if (game->large_board) {
            /* TODO: place obstructions around hoppers so that they will actually generate an attack pattern. */
            large_bitboard_t attacks = get_large_attack_bitboard(&board, large_board_all, side);
            while (!is_zero128(attacks)) {
               int square = bitscan128(attacks);
               assert(square >= 0);
               assert(square < 128);
               attacks ^= large_square_bitboards[square];
               game->pt.large_attack_from[n][square] |= large_square_bitboards[ps];
            }
         } else {
            /* TODO: place obstructions around hoppers so that they will actually generate an attack pattern. */
            bitboard_t attacks = get_attack_bitboard(&board, board_all, side);
            while (attacks) {
               int square = bitscan64(attacks);
               attacks ^= square_bitboards[square];
               game->pt.attack_from[n][square] |= square_bitboards[ps];
            }
         }
         remove_piece(&board, ps);
      }

      /* Weight of this piece for king attacks */
      game->pt.king_safety_weight[n] = 1;

      /* Use reasoning to decide whether a piece can return to its original position. This is important for
       * things like the 50-move draw rule.
       */
      game->pt.piece_flags[n] &= ~PF_NORET;
      if ( (game->pt.piece_move_flags[n] & ~MF_STEPPER) == 0) {
         /* Stepper directions: test whether this piece can return to its original square. There are weird
          * combinations that we'll mis-evaluate here, but we'll accept that (at least for now).
          * We're also missing out on the possibility that the direction of captures may change things.
          */
         int si = (game->pt.piece_move_flags[n] & MF_STEPPER) >> 8;
         uint32_t dirs = stepper_description[si];

         if ( (dirs & MF_STEPPER_N) && !((dirs & MF_STEPPER_S) || ((dirs&MF_STEPPER_SE) && (dirs&MF_STEPPER_SW))) )
            game->pt.piece_flags[n] |= PF_NORET;
         if ( (dirs & MF_STEPPER_S) && !((dirs & MF_STEPPER_N) || ((dirs&MF_STEPPER_NE) && (dirs&MF_STEPPER_NW))) )
            game->pt.piece_flags[n] |= PF_NORET;
         if ( (dirs & MF_STEPPER_E) && !((dirs & MF_STEPPER_W) || ((dirs&MF_STEPPER_NW) && (dirs&MF_STEPPER_SW))) )
            game->pt.piece_flags[n] |= PF_NORET;
         if ( (dirs & MF_STEPPER_W) && !((dirs & MF_STEPPER_E) || ((dirs&MF_STEPPER_NE) && (dirs&MF_STEPPER_SE))) )
            game->pt.piece_flags[n] |= PF_NORET;

         if ( (dirs & MF_STEPPER_NE) && !((dirs & MF_STEPPER_SW) || ((dirs&MF_STEPPER_S) && (dirs&MF_STEPPER_W))) )
            game->pt.piece_flags[n] |= PF_NORET;
         if ( (dirs & MF_STEPPER_SE) && !((dirs & MF_STEPPER_NW) || ((dirs&MF_STEPPER_N) && (dirs&MF_STEPPER_W))) )
            game->pt.piece_flags[n] |= PF_NORET;
         if ( (dirs & MF_STEPPER_NW) && !((dirs & MF_STEPPER_SE) || ((dirs&MF_STEPPER_S) && (dirs&MF_STEPPER_E))) )
            game->pt.piece_flags[n] |= PF_NORET;
         if ( (dirs & MF_STEPPER_SW) && !((dirs & MF_STEPPER_NE) || ((dirs&MF_STEPPER_N) && (dirs&MF_STEPPER_E))) )
            game->pt.piece_flags[n] |= PF_NORET;
      }

      /* A pawn is a piece that cannot return to its starting position, has only stepper moves, and
       * can promote (but we leave out that test as a simplification).
       */
      if ( (game->pt.piece_move_flags[n] & ~MF_STEPPER) == 0 && game->pt.piece_flags[n] & PF_NORET) {
         int si = (game->pt.piece_move_flags[n] & MF_STEPPER) >> 8;
         uint32_t dirs = stepper_description[si];
         sides side = WHITE;

         /* South-moving pawns are black */
         if ( dirs & (MF_STEPPER_S | MF_STEPPER_SE | MF_STEPPER_SW) ) side = BLACK;

         game->pt.pawn_index[side] = n;

         /* Forward span for pawns */
         for (ps = 0; ps < board_size; ps++) {
            if (board.large_board) {
               large_bitboard_t moves = get_large_stepper_move_bitboard(game->pt.piece_move_flags[n], large_board_empty, large_square_bitboards[ps]);
               game->pt.large_pawn_front_span[side][ps] = moves;
               game->pt.large_pawn_stop_square[side][ps] = moves;
               int c;
               for (c=0; c<large_board_ranks-1; c++) {
                  moves = get_large_stepper_move_bitboard(game->pt.piece_move_flags[n], large_board_empty, moves);
                  game->pt.large_pawn_front_span[side][ps] |= moves;
               }
            } else {
               bitboard_t moves = get_stepper_move_bitboard(game->pt.piece_move_flags[n], board_empty, square_bitboards[ps]);
               game->pt.pawn_front_span[side][ps] = moves;
               game->pt.pawn_stop_square[side][ps] = moves;
               int c;
               for (c=0; c<large_board_ranks-1; c++) {
                  moves = get_stepper_move_bitboard(game->pt.piece_move_flags[n], board_empty, moves);
                  game->pt.pawn_front_span[side][ps] |= moves;
               }
            }
         }
      }


      /* Skip pawns */
      if (game->pt.piece_flags[n] & PF_NORET)
         continue;

      /* Loop over all squares */
      int sum_m = 0;
      int max_m = 0;
      int min_m = board_size;
      for (ps = 0; ps < board_size; ps++) {
         sides side = WHITE;
         int moves = 0;

         /* If a piece is defined as "BLACK", then best test it as black, it may have asymmetric moves. */
         if (game->pt.piece_abbreviation[n][0] == ',')
            side = BLACK;

         clear_board(&board);
         if (board.large_board) {
            /* Place the piece on the board */
            large_bitboard_t pbb = large_square_bitboards[ps];
            board.large_bbc[side] |= pbb;
            board.large_bbp[n] |= pbb;
            board.piece[ps] = n;

            /* Fudge: set capture flags to move flags to also generate mobility information for hoppers */
            move_flag_t cf = game->pt.piece_capture_flags[n];
            game->pt.piece_capture_flags[n] = game->pt.piece_move_flags[n];
            pbb = get_large_attack_bitboard(&board, large_board_all, side);
            game->pt.piece_capture_flags[n] = cf;

            moves = popcount128(pbb);

            pbb |= large_square_bitboards[ps];
            if ( !is_zero128(pbb & large_board_light) && !is_zero128(pbb & large_board_dark) )
               colour_bound = false;
         } else {
            /* Place the piece on the board */
            bitboard_t pbb = make_bitboard_square(ps);
            board.bbc[side] |= pbb;
            board.bbp[n] |= pbb;
            board.piece[ps] = n;

            pbb = get_attack_bitboard(&board, board_all, side);

            moves = popcount64(pbb);

            pbb |= make_bitboard_square(ps);
            if ( (pbb & board_light) && (pbb & board_dark) )
               colour_bound = false;
         }

         sum_m += moves;
         if (moves > max_m) max_m = moves;
         if (moves < min_m) min_m = moves;
      }

      game->pt.max_moves[n] = max_m;
      game->pt.min_moves[n] = min_m;
      game->pt.avg_moves[n] = sum_m / board_size;

      /* Check whether a piece is colour-bound and should get a pair bonus */
      game->pt.piece_flags[n] &= ~PF_PAIRBONUS;
      if (colour_bound)
         game->pt.piece_flags[n] |= PF_PAIRBONUS;

      /* Determine evaluation weight for king safety evaluation. This is a measure of a piece's
       * short-range attack potential, as measured on a 5x5 board and a 3x3 board.
       * This gives a total initial attack strength for the normal chess array of 96 (assigning 1 for the
       * pawns), 104 for the Spartans in Spartan chess and 138 for the normal array in Capablanca chess.
       * If we scale the king safety evaluation with the attack strength, then this results in a slightly
       * higher weight for king safety for white in Spartan chess and a much higher weight for king safety in
       * Capablanca chess.
       * This seems reasonable.
       * A reasonable first guess for when to reduce the weight of the king safety to 0 is at 2/3 of the
       * initial strength (64 for normal chess).
       */
      ps = pack_row_file(large_board_ranks/2, large_board_files/2);

      /* If this piece is a leaper, it may have a constraint and therefore we should search the place on the
       * board where it attacks the most squares.
       */
      if ( game->pt.piece_capture_flags[n] & MF_LEAPER_ASYMM) {
         uint32_t flags = game->pt.piece_capture_flags[n];
         sides side = WHITE;
         int maxmoves = 0;
         int s;

         /* If a piece is defined as "BLACK", then best test it as black, it may have asymmetric moves. */
         if (game->pt.piece_abbreviation[n][0] == ',')
            side = BLACK;

         for (s=0; s<board_size; s++) {
            int m;
            if (board.large_board) {
               large_bitboard_t bb = get_large_aleaper_moves(flags, large_board_empty, s, side);
               m = popcount128(bb);

               /* Test if this piece, once advanced, will be unable to return to its previous position.
                * If so, clear the 50-move flag after this piece is advanced.
                */
               if (is_zero128(bb & large_board_southward[unpack_rank(s)]) ||
                   is_zero128(bb & large_board_northward[unpack_rank(s)]))
                  game->pt.piece_flags[n] |= PF_NORET;

            } else {
               bitboard_t bb = get_aleaper_moves(flags, board_empty, s, side);
               m = popcount64(bb);
            }
            if (m > maxmoves) {
               maxmoves = m;
               ps = s;
            }
         }
      }

      clear_board(&board);
      if (board.large_board) {
         /* Create a 5x5 mask */
         large_bitboard_t mask5 = large_square_bitboards[ps-1] | large_square_bitboards[ps-2] |
            large_square_bitboards[ps] |
            large_square_bitboards[ps+1] | large_square_bitboards[ps+2];
         mask5 |= shr128(mask5, large_board_files);
         mask5 |= shr128(mask5, large_board_files);
         mask5 |= shl128(mask5, large_board_files);
         mask5 |= shl128(mask5, large_board_files);

         /* Create a 3x3 mask */
         large_bitboard_t mask3 = large_square_bitboards[ps-1] | large_square_bitboards[ps] | large_square_bitboards[ps+1];
         mask3 |= shr128(mask3, large_board_files);
         mask3 |= shl128(mask3, large_board_files);

         /* Place the piece on the board */
         large_bitboard_t pbb = large_square_bitboards[ps];
         board.large_bbc[WHITE] |= pbb;
         board.large_bbp[n] |= pbb;
         board.piece[ps] = n;

         pbb = get_large_attack_bitboard(&board, large_board_all, WHITE);

         /* Get evaluation weight.
          * Penalise short-range pieces for which the 3x3 and 5x5 attack sets coincide.
          * Penalise colour-bound pieces.
          */
         int count5 = popcount128(pbb & mask5);
         int count3 = popcount128(pbb & mask3);
         if (count3 == count5) count3 = 0;
         if (colour_bound) count3 = 0;

         /* Special term for hoppers, which have no attacks on an empty board */
         if (game->pt.piece_capture_flags[n] & MF_HOPPER) {
            board.large_bbc[BLACK] |= mask3;
            pbb = get_large_attack_bitboard(&board, large_board_all, WHITE);
            int count = popcount128(pbb & mask5);
            count5 += count;
            count3 += count;
         }

         game->pt.king_safety_weight[n] = count5 + count3;
      } else {
         /* Create a 5x5 mask */
         bitboard_t mask5 = square_bitboards[ps-1] | square_bitboards[ps-2] |
                            square_bitboards[ps] |
                            square_bitboards[ps+1] | square_bitboards[ps+2];
         mask5 |= mask5 >> large_board_files;
         mask5 |= mask5 >> large_board_files;
         mask5 |= mask5 << large_board_files;
         mask5 |= mask5 << large_board_files;

         /* Create a 3x3 mask5 */
         bitboard_t mask3 = square_bitboards[ps-1] | square_bitboards[ps] | square_bitboards[ps+1];
         mask3 |= mask3 >> large_board_files;
         mask3 |= mask3 << large_board_files;

         /* Place the piece on the board */
         bitboard_t pbb = make_bitboard_square(ps);
         board.bbc[WHITE] |= pbb;
         board.bbp[n] |= pbb;
         board.piece[ps] = n;

         pbb = get_attack_bitboard(&board, board_all, WHITE);

         /* Get evaluation weight.
          * Penalise short-range pieces for which the 3x3 and 5x5 attack sets coincide.
          * Penalise colour-bound pieces.
          */
         int count5 = popcount64(pbb & mask5);
         int count3 = popcount64(pbb & mask3);
         if (count3 == count5) count3 = 0;
         if (colour_bound) count3 = 0;
         game->pt.king_safety_weight[n] = count5 + count3;
      }

#if 0
      double m_avg = (double)sum_m / board_size;
      double x = (double)(min_m-m_avg)/(max_m-m_avg);
      if (min_m == m_avg && max_m == m_avg) x = 0;
      printf("Statistics for piece type %d (%s)\n", n, game->pt.piece_name[n]);
      printf("   Maximum number of moves %2d\n", max_m);
      printf("   Minimum number of moves %2d\n", min_m);
      printf("   Average number of moves %2d\n", sum_m / board_size);
      printf("   Weight ratio            %.2f\n", (double)(board_size * board_size) / sum_m);
      printf("   x range                 %.2f < x < 1\n", x);
#endif

      /* If this piece is royal, then check whether one king can driver the other out of a corner. It it can,
       * then special evaluation of edge pawns should be disabled.
       */
      if (game->pt.piece_flags[n] & PF_ROYAL) {
      game->board.rule_flags &= ~RF_KING_CORNERDRIVE;
      for (ps = 0; ps < board_size; ps++) {
         int corner = 0;
         int ops;

         if (game->board.rule_flags & RF_KING_CORNERDRIVE) break;
         clear_board(&board);
         if (board.large_board) {
            /* Place the first king on the board */
            large_bitboard_t atk1, atk2;
            large_bitboard_t pbb = large_square_bitboards[ps];
            board.large_bbc[WHITE] |= pbb;
            board.large_bbp[n] |= pbb;
            board.piece[ps] = n;

            /* Does it attack the corner? */
            atk1 = get_large_attack_bitboard(&board, pbb, WHITE);
            if (is_zero128(atk1 & large_square_bitboards[corner])) continue;

            /* Place the second king */
            for (ops = 0; ops < board_size; ops++) {
               if (game->board.rule_flags & RF_KING_CORNERDRIVE) break;
               if (ops == ps) continue;

               /* If the kings attack eachother, break out */
               if (!is_zero128(atk1 & large_square_bitboards[ops])) continue;

               board.large_bbc[BLACK] |= large_square_bitboards[ops];
               board.large_bbp[n] |= large_square_bitboards[ops];
               board.piece[ops] = n;
               atk2 = get_large_attack_bitboard(&board, large_square_bitboards[ops], BLACK);

               if (!is_zero128(atk2 & large_square_bitboards[corner]))
                  game->board.rule_flags |= RF_KING_CORNERDRIVE;
            }
         } else {
            /* Place the first king on the board */
            bitboard_t atk1, atk2;
            bitboard_t pbb = square_bitboards[ps];
            board.bbc[WHITE] |= pbb;
            board.bbp[n] |= pbb;
            board.piece[ps] = n;

            /* Does it attack the corner? */
            atk1 = get_attack_bitboard(&board, pbb, WHITE);
            if (!(atk1 & square_bitboards[corner])) continue;

            /* Place the second king */
            for (ops = 0; ops < board_size; ops++) {
               if (game->board.rule_flags & RF_KING_CORNERDRIVE) break;
               if (ops == ps) continue;

               /* If the kings attack eachother, break out */
               if (atk1 & square_bitboards[ops]) continue;

               board.bbc[BLACK] |= square_bitboards[ops];
               board.bbp[n] |= square_bitboards[ops];
               board.piece[ops] = n;
               atk2 = get_attack_bitboard(&board, square_bitboards[ops], BLACK);

               if (atk2 & square_bitboards[corner])
                  game->board.rule_flags |= RF_KING_CORNERDRIVE;
            }
         }
      }
      }

   }

   /* For pawn types, determine the number of possible steps. This determines how we evaluate passers. */
   int pawn_steps[NUM_SIDES] = { 0, 0 };
   sides side;
   for (side = WHITE; side < NUM_SIDES; side++) {
      if (game->pt.pawn_index[side] > -1) {
         int n = game->pt.pawn_index[side];
         int si = (game->pt.piece_move_flags[n] & MF_STEPPER) >> 8;
         uint32_t dirs = stepper_description[si];

         if ( dirs & MF_STEPPER_N  ) pawn_steps[side]++;
         if ( dirs & MF_STEPPER_NE ) pawn_steps[side]++;
         if ( dirs & MF_STEPPER_E  ) pawn_steps[side]++;
         if ( dirs & MF_STEPPER_SE ) pawn_steps[side]++;
         if ( dirs & MF_STEPPER_S  ) pawn_steps[side]++;
         if ( dirs & MF_STEPPER_SW ) pawn_steps[side]++;
         if ( dirs & MF_STEPPER_W  ) pawn_steps[side]++;
         if ( dirs & MF_STEPPER_NW ) pawn_steps[side]++;
      }
   }
   game->pt.pawn_steps[WHITE] = max(0, pawn_steps[WHITE]);
   game->pt.pawn_steps[BLACK] = max(0, pawn_steps[BLACK]);

   /* Determine passer test boards.
    * We mark the location of any enemy pawn that will be able to stop the current pawn either by attacking
    * it promotion path, or by intervention (which of course can't happen for normal chess pawns).
    */
   memset(&game->pt.large_pawn_passer_mask, 0, NUM_SIDES*128*sizeof(large_bitboard_t));
   if (game->pt.pawn_index[WHITE]>-1 && game->pt.pawn_index[BLACK]>-1) {
   for (side = WHITE; side < NUM_SIDES; side++) {
      sides me = side;
      sides other = next_side[me];
      uint32_t attack_flags = game->pt.piece_capture_flags[game->pt.pawn_index[other]];
      for (ps=0; ps<board_size; ps++) {
         int ops;

         /* For very mobile pawns (eg, berolina) disable passer scoring unless they reach the 5th rank. Before
          * then, there are simply too many possible paths to the back rank to meaningfully test whether the
          * pawn might be passed. We simply set the board to the full board in this case, which means that any
          * pawn is considered able to stop the pawn, effectively disabling the passed pawn evaluation unless
          * the other side has no pawns left.
          */
         if (pawn_steps[side] > 1) {
            static const int passed_test_rank = 2;
            if (side == WHITE && unpack_rank(ps) < (large_board_ranks - 1 - passed_test_rank)) {
               if (game->large_board) game->pt.large_pawn_passer_mask[me][ps] = large_board_all;
               else                   game->pt.pawn_passer_mask[me][ps]       = board_all;
               continue;
            }
            if (side == BLACK && unpack_rank(ps) > passed_test_rank) {
               if (game->large_board) game->pt.large_pawn_passer_mask[me][ps] = large_board_all;
               else                   game->pt.pawn_passer_mask[me][ps]       = board_all;
               continue;
            }
         }

         for (ops = 0; ops<board_size; ops++) {
            if (game->large_board) {
               /* Find the attack span of the enemy pawn */
               large_bitboard_t move_span;
               large_bitboard_t attack_span;
               move_span = game->pt.large_pawn_front_span[other][ops] | large_square_bitboards[ops];
               attack_span = get_large_stepper_attack_bitboard(attack_flags, large_board_empty, move_span);

               /* If the attack or move spans of the enemy pawn intersect with the forward move span of this
                * pawn, then it may be possible for our pawn to be stopped.
                */
               if (!is_zero128( (attack_span | move_span) & game->pt.large_pawn_front_span[me][ps])) {
                  /* Test whether the enemy pawn blocks or attacks the path of our pawn on its current square.
                   * If not, calculate successor positions for both pawns. Repeat for all ranks. (TODO)
                   * We're pessimistic and assume that the enemy has the move.
                   */
                  game->pt.large_pawn_passer_mask[me][ps] |= large_square_bitboards[ops];
               }
            } else {
               /* Find the attack span of the enemy pawn */
               bitboard_t move_span;
               bitboard_t attack_span;
               move_span = game->pt.pawn_front_span[other][ops] | square_bitboards[ops];
               attack_span = get_stepper_attack_bitboard(attack_flags, board_empty, move_span);

               /* If the attack or move spans of the enemy pawn intersect with the forward move span of this
                * pawn, then it may be possible for our pawn to be stopped.
                */
               if ((attack_span | move_span) & game->pt.pawn_front_span[me][ps]) {
                  /* Test whether the enemy pawn blocks or attacks the path of our pawn on its current square.
                   * If not, calculate successor positions for both pawns. Repeat for all ranks. (TODO)
                   * We're pessimistic and assume that the enemy has the move.
                   */
                  game->pt.pawn_passer_mask[me][ps] |= square_bitboards[ops];
               }
            }
         }
      }
   }

   /* Are promotions optional or compulsory? This can be overwritten by a flag, but the rule of thumb is this:
    * if a pawn, inside the promotion zone, can make a *step* and still be in the promotion zone, the promotion
    * is optional.
    */
   for (side = WHITE; side < NUM_SIDES; side++) {
      int n = game->pt.pawn_index[side];
      int move_flags = game->pt.piece_move_flags[n];
      game->pt.piece_large_optional_promotion_zone[n] = large_board_empty;
      for (ps=0; ps<board_size; ps++) {
         if (game->large_board) {
            large_bitboard_t bb = large_square_bitboards[ps];
            if (!is_zero128(bb & game->pt.piece_large_promotion_zone[n]) && 
                !is_zero128(game->pt.large_pawn_stop_square[side][ps] & game->pt.piece_large_promotion_zone[n]))
               game->pt.piece_large_optional_promotion_zone[n] |= bb;
         } else {
            bitboard_t bb = square_bitboards[ps];
            if ((bb & game->pt.piece_promotion_zone[n]) && 
                (game->pt.pawn_stop_square[side][ps] & game->pt.piece_promotion_zone[n]))
               game->pt.piece_optional_promotion_zone[n] |= bb;
         }
      }
   }
   }
}

static void record_castle_state(game_t *game, char state, bitboard_t *castle_init, large_bitboard_t *large_castle_init)
{
   /* Deal with different type of castle flags */

   switch(state) {
      case '-':
         return;

      case 'K':
         /* Starting at the right edge of the board, find the first castle
          * piece.
          */
         state = '-';
         for (int n = large_board_files - 1; n>=0; n--) {
            if (get_piece(&game->board, n) == game->pt.castle_piece[WHITE]) {
               state = 'A' + unpack_file(n);
               break;
            }

         }
         record_castle_state(game, state, castle_init, large_castle_init);
         return;

      case 'Q':
         /* Starting at the left edge of the board, find the first castle
          * piece.
          */
         state = '-';
         for (int n = 0; n<large_board_files; n++) {
            if (get_piece(&game->board, n) == game->pt.castle_piece[WHITE]) {
               state = 'A' + unpack_file(n);
               break;
            }

         }
         record_castle_state(game, state, castle_init, large_castle_init);
         return;

      case 'k':
         /* Starting at the right edge of the board, find the first castle
          * piece.
          */
         state = '-';
         for (int n = large_board_files - 1; n>=0; n--) {
            int s = large_board_files * (large_board_ranks-1);
            if (get_piece(&game->board, s + n) == game->pt.castle_piece[BLACK]) {
               state = 'a' + unpack_file(n);
               break;
            }

         }
         record_castle_state(game, state, castle_init, large_castle_init);
         return;

      case 'q':
         /* Starting at the left edge of the board, find the first castle
          * piece.
          */
         state = '-';
         for (int n = 0; n<large_board_files; n++) {
            int s = large_board_files * (large_board_ranks-1);
            if (get_piece(&game->board, s + n) == game->pt.castle_piece[BLACK]) {
               state = 'a' + unpack_file(n);
               break;
            }

         }
         record_castle_state(game, state, castle_init, large_castle_init);
         return;
   }

   /* Fischer-random style castling. This identifies the file of the piece with which
    * we can castle. We can find the file of the king by looking at the royal bitboard, and we
    * know the king destination after castling from the rule description. From this, we can work
    * out the required bitmasks.
    */
   sides side = BLACK;
   if (isupper(state)) side = WHITE;
   int king_from, king_to, rook_from;
   int rook_file;
   if (game->large_board) {
      king_from = bitscan128(game->board.large_royal & game->board.large_bbc[side]);
   } else {
      king_from = bitscan64(game->board.royal & game->board.bbc[side]);
   }
   rook_file = tolower(state) - 'a';
   rook_from = pack_row_file( (side == WHITE) ? 0 : (large_board_ranks-1), rook_file);
   king_to = long_castle_king_dest[side];
   if (rook_from > king_from) king_to = short_castle_king_dest[side];

   //printf("%d (%s %s) (%s)\n", side, square_names[king_from], square_names[king_to], square_names[rook_from]);
   deduce_castle_flags(game->large_board, side, king_from, king_to, rook_from);

   if (game->large_board) {
      *large_castle_init |= large_square_bitboards[king_from] | large_square_bitboards[rook_from];
   } else {
      *castle_init |= square_bitboards[king_from] | square_bitboards[rook_from];
   }

   /* Clear castle flags by default */
   if (game->large_board) {
      large_bitboard_t bb = game->board.large_bbp[get_piece(&game->board, rook_from)] |
         game->board.large_bbc[side];
      game->board.large_init &= ~bb;
   } else {
      bitboard_t bb = game->board.bbp[get_piece(&game->board, rook_from)] | game->board.bbc[side];
      game->board.init &= ~bb;
   }

   return;
}

void setup_fen_position(game_t *game, const char *str)
{
   const char *s = str;
   int n;
   int square = A8;
   int prev_rank = 16;

   if (game->board.large_board) {
      prev_rank = 2*large_board_files;
      square = large_board_files * (large_board_ranks - 1);
   }

   assert(game);
   if (!s) return;
   game->moves_played = 0;
   clear_board(&game->board);
   memset(game->repetition_hash_table, 0, sizeof(game->repetition_hash_table));

   /* Parse first record: piece positions */
   while(*s && (*s != ' ') && (*s != '[') && square>=0) {
      switch (*s) {
         case '/':
            square -= prev_rank;
            break;
         case '1':
            if (isdigit(s[1])) {
               square += 10;
               break;
            }
         case '2':
         case '3':
         case '4':
         case '5':
         case '6':
         case '7':
         case '8':
         case '9':
         case '0':
            square += (*s - '1')+1;
            break;

         default:
            for (n=0; n<game->pt.num_piece_types; n++) {
               char *p = strchr(game->pt.piece_abbreviation[n], *s);
               int colour = WHITE;
               if (p) {
                  /* Determine piece colour */
                  if (p != game->pt.piece_abbreviation[n])
                     colour = BLACK;

                  if (!game->board.large_board) {
                     bitboard_t bb = make_bitboard_square(square);
                     game->board.bbc[colour] |= bb;
                     game->board.bbp[n] |= bb;
                     game->board.init |= bb;
                     game->board.piece[square] = n;
                     if (game->pt.piece_flags[n] & PF_ROYAL)
                        game->board.royal |= bb;
                  } else {
                     large_bitboard_t bb = large_square_bitboards[square];
                     game->board.large_bbc[colour] |= bb;
                     game->board.large_bbp[n] |= bb;
                     game->board.large_init |= bb;
                     game->board.piece[square] = n;
                     if (game->pt.piece_flags[n] & PF_ROYAL)
                        game->board.large_royal |= bb;
                  }
                  game->board.hash ^= piece_key[n][colour][square];
                  if (n == game->pt.pawn_index[colour]) game->board.pawn_hash ^= piece_key[n][colour][square];
#ifndef PIECE_VALUES_IN_PSQ
                  game->board.material[colour] += game->board.piece_types->piece_value[n];
#endif
                  game->board.psq += psq[n][colour][square];
                  square++;
                  break;
               }
            }
            if (n == game->pt.num_piece_types) {
               pstderr("Error: unknown piece type '%c' (bad FEN %s)\n", s[0], str);
               return;
            }
            break;
      }
      s++;
   }

   /* Optional: check for holdings */
   while(*s && (*s == ' ')) s++;
   if (*s == '[') {
      s++;
      while (*s != ']') {
         for (n=0; n<game->pt.num_piece_types; n++) {
            char *p = strchr(game->pt.piece_abbreviation[n], *s);
            int colour = WHITE;
            if (p) {
               /* Determine piece colour */
               if (p != game->pt.piece_abbreviation[n])
                  colour = BLACK;

               game->board.holdings[n][colour]++;
               break;
            }
         }
         s++;
      }
      s++;
   }

   /* Second record: side to move */
   while(*s && (*s == ' ')) s++;
   game->board.side_to_move = WHITE;
   if (*s == 'b') {
      game->board.side_to_move = BLACK;
      game->board.hash ^= side_to_move_key;
   }
   if (*s) s++;

   /* Initialise attack map */
   initialise_attack_tables(&game->board);

   /* Third record: castling rights.
    * Skip if the next entry is a number, in which case the game doesn't have castling.
    */
   while(*s && (*s == ' ')) s++;
   if (!isdigit(*s)) {
      bitboard_t castle_init = board_empty;
      large_bitboard_t large_castle_init = large_board_empty;

      /* Clear castle flags by default */
      if (game->large_board) {
         game->board.large_init &= ~game->board.large_royal;
      } else {
         game->board.init &= ~game->board.royal;
      }

      while(*s && (*s != ' ')) {
         record_castle_state(game, *s, &castle_init, &large_castle_init);
         s++;
      }
      if (*s) s++;

      /* Set castle flags.
       * Do a paranoid sanity check to make sure that only occupied squares have their init-bits set.
       */
      if (game->large_board) {
         game->board.large_init |= large_castle_init;
      } else {
         game->board.init |= castle_init;
      }
   }

   /* Make sure the initial flags are at least somewhat sane by making sure only occupied squares have their
    * init bits set.
    */
   if (!game->board.large_board) {
      game->board.init &= game->board.bbc[WHITE] | game->board.bbc[BLACK];
   } else {
      game->board.large_init &= game->board.large_bbc[WHITE] | game->board.large_bbc[BLACK];
   }

   /* Fourth record: En-passant square
    * If this record is a number, then the game doesn't have en-passant capture and we skip it.
    */
   while(*s && (*s == ' ')) s++;
   if (!isdigit(*s)) {
      if (*s && (*s != '-')) {
         int col = *s - 'a';
         int row = s[1] - '1';
         s+=2;
         /* En-passant move-to square */
         game->board.ep = pack_row_file(row, col);
         /* En-passant capture square, this may be encoded in the FEN */
         if (!isspace(*s)) {
            int col = *s - 'a';
            int row = s[1] - '1';
            s+=2;
            game->board.ep_capture = pack_row_file(row, col);
         } else {
            /* Assume we have normal pawns, in which case we can simply derive it from the move-to square */
            if (game->board.side_to_move == WHITE)
               game->board.ep_capture = game->board.ep - large_board_files;
            else
               game->board.ep_capture = game->board.ep + large_board_files;
         }
      }
   }

   /* Fifth record: half-move counter (50 move counter) */ 
   while(*s && (*s == ' ')) s++;
   sscanf(s, "%d\n", &n);
   if(game->fifty_counter) game->fifty_counter[0] = n;
   while(*s && (*s != ' ')) s++;

   if (game->repetition_hash_table) {
      memset(game->repetition_hash_table, 0, sizeof game->repetition_hash_table);
      game->repetition_hash_table[game->board.hash&0xFFFF] = 1;
   }

   if (!game->board.large_board)
      game->board.init &= board_rank1 | board_rank8;
}

char *make_fen_string(const game_t *game, char *buffer)
{
   static char static_buffer[4096];
   bitboard_t occ;
   large_bitboard_t large_occ;
   char *fen = buffer;
   int n = 0;
   int r, f, s;

   if (!fen) fen = static_buffer;
   fen[0] = '\0';

   occ = game->board.bbc[WHITE] | game->board.bbc[BLACK];
   large_occ = game->board.large_bbc[WHITE] | game->board.large_bbc[BLACK];

   /* First record: board position */
   /* Scan all ranks */
   for (r = large_board_ranks-1; r>=0; r--) {
      int count = 0;
      for (f = 0; f < large_board_files; f++) {
         s = pack_row_file(r, f);

         /* Empty? */
         if (game->large_board) {
            if (is_zero128(large_occ & large_square_bitboards[s])) {
               count++;
               continue;
            }
         } else {
            if ((occ & make_bitboard_square(s)) == 0) {
               count++;
               continue;
            }
         }

         /* Not empty, do we have a count? */
         if (count) n += snprintf(fen+n, 4096 - n, "%d", count);
         count = 0;

         /* Print piece */
         int side = WHITE;
         if (game->large_board) {
            if (!is_zero128(game->board.large_bbc[BLACK] & large_square_bitboards[s])) {
               side = BLACK;
            }
         } else {
            if ((game->board.bbc[BLACK] & make_bitboard_square(s))) {
               side = BLACK;
            }
         }

         int piece = get_piece(&game->board, s);
         if (side == WHITE) {
            n += snprintf(fen+n, 4096-n, "%c", game->pt.piece_abbreviation[piece][0]);
         } else {
            int l = 2;
            if (game->pt.piece_abbreviation[piece][0] == ',') l = 1;
            n += snprintf(fen+n, 4096-n, "%c", game->pt.piece_abbreviation[piece][l]);
         }
      }
      if (count) n += snprintf(fen+n, 4096 - n, "%d", count);
      if (r) n += snprintf(fen+n, 4096 - n, "/");
   }

   n += snprintf(fen+n, 4096 - n, " ");

   /* Second record: side to move */
   if (game->board.side_to_move == WHITE)
      n += snprintf(fen+n, 4096 - n, "w ");
   else
      n += snprintf(fen+n, 4096 - n, "b ");

   /* Third record: castling rights
    * FIXME: check for kingside/queenside castling!
    */
   if (may_castle(&game->board, WHITE)) n += snprintf(fen+n, 4096 - n, "KQ");
   if (may_castle(&game->board, BLACK)) n += snprintf(fen+n, 4096 - n, "kq");
   n += snprintf(fen+n, 4096 - n, " ");

   /* Fourth record: en-passant square */
   if (game->board.ep)
      n += snprintf(fen+n, 4096 - n, "%s", square_names[game->board.ep]);
   else
      n += snprintf(fen+n, 4096 - n, "-");
   n += snprintf(fen+n, 4096 - n, " ");

   /* Fifth and sixth record: half-move counter and full-move counter */
   int par = (game->moves_played&1) && next_side[game->board.side_to_move];
   n += snprintf(fen+n, 4096 - n, "%d ", game->fifty_counter[game->moves_played]);
   n += snprintf(fen+n, 4096 - n, "%d", (int)(game->moves_played + par)/2 + 1);

   return fen;
}

void print_piece_types(const game_t *game)
{
   int n;

   for (n=0; n<game->pt.num_piece_types; n++) {
      int mate_helpers = 0;
      if (game->pt.piece_flags[n] & PF_NOMATE) {
         int n2;
         for (n2=0; n2<game->pt.num_piece_types; n2++) {
            if (!(game->pt.piece_flags[n2] & PF_NOMATE)) continue;
            if (game->pt.pieces_can_win[n][n2]) mate_helpers |= (1<<n2);
         }
      }
      printf("% 2d %s %s (%s)\n", n, game->pt.piece_name[n], game->pt.piece_notation[n], game->pt.piece_abbreviation[n]);
      if (game->pt.piece_move_flags[n] & MF_SLIDER)
         printf("   Slider  %08x (move)\n", game->pt.piece_move_flags[n] & MF_SLIDER);
      if (game->pt.piece_move_flags[n] & MF_HOPPER)
         printf("   Hopper  %08x (move)\n", game->pt.piece_move_flags[n] & MF_HOPPER);
      if (game->pt.piece_move_flags[n] & MF_IS_LEAPER)
         printf("   Leaper  %08x (move)\n", game->pt.piece_move_flags[n] & 0xffff0000);
      if (game->pt.piece_move_flags[n] & MF_STEPPER)
         printf("   Stepper %08x (move)\n", game->pt.piece_move_flags[n] & MF_STEPPER);

      if (game->pt.piece_capture_flags[n] & MF_SLIDER)
         printf("   Slider  %08x (capture)\n", game->pt.piece_capture_flags[n] & MF_SLIDER);
      if (game->pt.piece_capture_flags[n] & MF_HOPPER)
         printf("   Hopper  %08x (capture)\n", game->pt.piece_capture_flags[n] & MF_HOPPER);
      if (game->pt.piece_capture_flags[n] & MF_IS_LEAPER)
         printf("   Leaper  %08x (capture)\n", game->pt.piece_capture_flags[n] & 0xffff0000);
      if (game->pt.piece_capture_flags[n] & MF_STEPPER)
         printf("   Stepper %08x (capture)\n", game->pt.piece_capture_flags[n] & MF_STEPPER);

      if (game->pt.piece_flags[n] & PF_ROYAL)
         printf("   Royal\n");
      if (game->pt.piece_flags[n] & PF_PAIRBONUS)
         printf("   Pair bonus\n");
      if (game->pt.piece_flags[n] & PF_NORET)
         printf("   No return\n");
      if (game->pt.piece_flags[n] & PF_NOMATE)
         printf("   Cannot deliver mate alone\n");
      if (mate_helpers) {
         printf("   Can deliver mate with help of ");
         while (mate_helpers) {
            int n = bitscan16(mate_helpers);
            mate_helpers ^= (1<<n);
            printf("%s ", game->pt.piece_notation[n]);
         };
         printf("\n");
      }

      if (game->pt.pawn_index[WHITE] == n)
         printf("   White pawn\n");
      if (game->pt.pawn_index[BLACK] == n)
         printf("   Black pawn\n");

      printf("   King attack weight %d\n", game->pt.king_safety_weight[n]);
   }
   //if (game->board.rule_flags & RF_KING_TRAPPED)
   //   printf("Trapped kings\n");
   //if (game->board.rule_flags & RF_KING_CORNERDRIVE)
   //   printf("Kings can drive\n");
}

static const char *file_names[] = { "a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p" };
static const char *row_names[] = { "1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11","12","13","14","15","16" };
static char static_output_buffer[256];

char *short_move_string(const game_t *game, const move_t move, char *buffer)
{
   movelist_t movelist;
   char *s = buffer;
   const char *piece = "";
   const char *suffix = "";
   const char *token = "";
   const char *origin = "";
   int p = get_move_piece(move);

   if (!s) s = static_output_buffer;

   if (game)
      generate_moves(&movelist, &game->board, game->board.side_to_move);

   if (move == 0) {
      snprintf(s, 15, "(null)");
      return s;
   }

   if (is_castle_move(move)) {
      int to = unpack_file(get_move_to(move));
      if (to >= large_board_files/2)
         snprintf(s, 256, "%s", kingside_castle);
      else
         snprintf(s, 256, "%s", queenside_castle);
      return s;
   }

   if (is_drop_move(move)) {
      uint16_t p = get_move_drop(move, 0);
      char fp = piece_symbol_string[decode_drop_piece(p)];
      int to = decode_drop_square(get_move_drop(move, 0));
      snprintf(s, 256, "%c@%s", fp, square_names[to]);
      return s;
   }

   int from, to;
   int c, k;
   char fp = '\0', tp = '\0';

   if (is_promotion_move(move))
      tp = piece_symbol_string[decode_drop_piece(get_move_drop(move, 0))];

   from = to = decode_drop_square(get_move_drop(move, 0));
   k = get_move_pickups(move);
   for (c=0; c<k; c++) {
      uint16_t p = get_move_pickup(move, c);
      if (decode_pickup_side(p) == decode_drop_side(get_move_drop(move, 0))) {
         from = decode_pickup_square(p);
         fp = piece_symbol_string[decode_pickup_piece(p)];
         break;
      }
   }

   if (is_capture_move(move)) token = "x";
   if (fp == ' ' && is_capture_move(move)) 
      fp = file_names[unpack_file(get_move_from(move))][0];

   /* Test whether the combination piece/to is unambiguous, otherwise dismbiguate */
   if (game == NULL) goto disambiguous;
   int count = 0;
   int n;
   for (n=0; n<movelist.num_moves; n++) {
      if (get_move_piece(move) == get_move_piece(movelist.move[n]) &&
            to == get_move_to(movelist.move[n]) && !is_drop_move(move)) count++;
   }
   if (count <= 1) goto disambiguous;

   /* Try to disambiguate by file */
   count = 0;
   for (n=0; n<movelist.num_moves; n++) {
      if (get_move_piece(move) == get_move_piece(movelist.move[n]) &&
            to == get_move_to(movelist.move[n]) &&
            unpack_file(from) == unpack_file(get_move_from(movelist.move[n])))
         count++;
   }
   if (count == 1) {
      origin = file_names[unpack_file(from)];
   } else {
      /* Disambiguate by row */
      origin = row_names[unpack_rank(from)];
   }
disambiguous:

   snprintf(s, 256, "%c%s%s%s%c", fp, origin, token, square_names[to], tp);
   return s;
}
