/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include <stdint.h>
#include "square.h"
#include "bitboard.h"

large_bitboard_t large_square_bitboards[128];
large_bitboard_t large_board_empty;
large_bitboard_t large_board_all;
large_bitboard_t large_board_edge;
large_bitboard_t large_board_east_edge;
large_bitboard_t large_board_west_edge;
large_bitboard_t large_board_north_edge;
large_bitboard_t large_board_south_edge;
large_bitboard_t large_board_light;
large_bitboard_t large_board_dark;
large_bitboard_t large_board_rank[16];
large_bitboard_t large_board_file[16];
large_bitboard_t large_board_south;
large_bitboard_t large_board_north;
large_bitboard_t large_board_northward[16];
large_bitboard_t large_board_southward[16];

large_bitboard_t large_board_diagonal[32];
large_bitboard_t large_board_antidiagonal[32];

large_bitboard_t large_short_castle_mask[2];
large_bitboard_t large_short_castle_free[2];
large_bitboard_t large_short_castle_safe[2];
large_bitboard_t large_long_castle_mask[2];
large_bitboard_t large_long_castle_free[2];
large_bitboard_t large_long_castle_safe[2];

/* Bitmask for a row, only extract bits that actually belong to that row, mask the rest */
uint16_t large_row_mask;

void initialise_large_bitboards(int files, int ranks)
{
   int size = files * ranks;
   int n;

   /* Sanity checks */
   assert(files < 16);
   assert(ranks < 16);

   large_board_files = files;
   large_board_ranks = ranks;

   large_row_mask = (1<<files)-1;

   large_square_bitboards[0] = u128(1,0);
   large_board_light = large_board_dark = large_board_empty = large_board_east_edge = large_board_west_edge = large_board_north_edge = large_board_south_edge = large_board_all = u128(0,0);
   memset(large_board_diagonal, 0, sizeof(large_board_diagonal));
   memset(large_board_antidiagonal, 0, sizeof(large_board_antidiagonal));
   memset(large_board_rank, 0, sizeof(large_board_rank));
   memset(large_board_file, 0, sizeof(large_board_file));

   /* Castle flags */
   for (n=0; n<2; n++) {
      large_short_castle_mask[n] = u128(0, 0);
      large_short_castle_free[n] = u128(0, 0);
      large_short_castle_safe[n] = u128(0, 0);
      large_long_castle_mask[n] = u128(0, 0);
      large_long_castle_free[n] = u128(0, 0);
      large_long_castle_safe[n] = u128(0, 0);
   }

   //for (n=1; n<128; n++)
   //   large_square_bitboards[n] = shl128(large_square_bitboards[n-1], 1);
   for (n=0; n<64; n++)
      large_square_bitboards[n] = u128(1ull<<n, 0);
   for (n=0; n<64; n++)
      large_square_bitboards[64+n] = u128(0, 1ull<<n);

   for (n=0; n<128; n++) {
      int f = n % files;
      int r = n / files;
      large_board_rank[r] |= large_square_bitboards[n];
      large_board_file[f] |= large_square_bitboards[n];
      if ((f^r)&1)
         large_board_light |= large_square_bitboards[n];
      else
         large_board_dark |= large_square_bitboards[n];

      if (f == 0)
         large_board_west_edge |= large_square_bitboards[n];

      if (f == (files-1))
         large_board_east_edge |= large_square_bitboards[n];

      if (r == 0)
         large_board_south_edge |= large_square_bitboards[n];

      if ((r == (ranks-1)) || n > size)
         large_board_north_edge |= large_square_bitboards[n];
   }
   large_board_edge = large_board_south_edge | large_board_north_edge | large_board_east_edge | large_board_west_edge;

   for (n=0; n<size; n++) {
      large_board_all |= large_square_bitboards[n];
   }

   /* Set up diagonals and anti-diagonals
    * For the mapping of diagonal numbers, we pretend the board is square; it may not be, but this is easiest.
    */
   int s = files;
   if (ranks > files) s = ranks;
   for (n = 0; n<size; n++) {
      int f = n % files;
      int r = n / files;
      large_diagonal_nr[n] = r + (s-1) - f;
      large_anti_diagonal_nr[n] = r + f;

      large_board_diagonal[large_diagonal_nr[n]] |= large_square_bitboards[n];
      large_board_antidiagonal[large_anti_diagonal_nr[n]] |= large_square_bitboards[n];
   }

   /* North/south bitboards */
   large_board_south = large_board_north = large_board_empty;
   for (n=0; n<ranks/2; n++)
      large_board_south |= large_board_rank[n];
   for (; n<ranks; n++)
      large_board_north |= large_board_rank[n];

   /* Northward/southward bitboards */
   for (n=0; n<ranks; n++) {
      int c;
      large_board_northward[n] = large_board_empty;
      large_board_southward[n] = large_board_empty;

      for (c=0; c<n; c++)       large_board_southward[n] |= large_board_rank[c];
      for (c=n+1; c<ranks; c++) large_board_northward[n] |= large_board_rank[c];
   }

#if 0
   int r, f;
   for (r=ranks-1; r>=0; r--) {
      for (f = 0; f< files; f++) {
         n = f + r * files;

         printf("%3d ", large_anti_diagonal_nr[n]);
      }
      printf("\n");
   }
#endif
}

static void print_uint16_t(uint16_t b)
{
   int c;
   
   for (c=0; c<large_board_files; c++) {
      printf ("%d", (b>>c)&1);
   }
}

/* This function prints a bitboard in such a way that 'A1' ends up on the
 * lower left corner and 'H8' ends up at the upper right corner.
 */
void printf_large_bitboard(large_bitboard_t b)
{
   int c;

   printf("+");
   for (c=0; c<large_board_files; c++) printf("-");
   printf("+\n");
   for (c=large_board_ranks-1; c>=0; c--) {
      uint16_t bb = get_large_bitboard_row(b, c);
      printf("|");
      print_uint16_t(bb);
      printf("|\n");
   }
   printf("+");
   for (c=0; c<large_board_files; c++) printf("-");
   printf("+\n");
}


void print_large_bitboard_squares(large_bitboard_t b)
{
   int square;

   //printf("Bitboard squares: ");
   while(!is_zero128(b)) {
      square = bitscan128(b);
      b ^= large_square_bitboards[square];
      printf("%s ", square_names[square]);
   }
   //printf("\n");
}



