/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "alphabeta.h"
#include "game.h"
#include "evaluate.h"
#include "see.h"

#undef DEBUG
#ifdef DEBUG
#define trace_tree printf
static void indent(int depth)
{
   int c;
   for(c=0; c<depth; c++) printf(" ");
}
#else
#define trace_tree(s,...)
#define indent(...)
#endif

volatile bool abort_search = 0;
size_t positions_evaluated;
size_t moves_searched;
int positions_in_hashtable;
int branches_pruned;

static inline int min(int x, int y)
{
   return (x<y) ? x : y;
}

static inline int max(int x, int y)
{
   return (x>y) ? x : y;
}

/* When taking back a move here, we don't have to be afraid of going back
 * beyond the first move in the game - so it's safe to skip the test for
 * that in the takeback() function.
 */
//#define takeback takeback_no_check

/* Return the number of repetitions of the current position in the game
 * tree.
 */
int count_repetition(const game_t *game)
{
   int n;
   int count;
   count = 0;

   /* FIXME: should detect whether a stepper has an irreversible move.
    * This is the case if one of its directions is singular.
    */
   for (n=game->moves_played-1; n>=0; n--) {
      if (is_irreversible_move(game->move_list[n]))
         return count;

      count += (game->hash_key[n] == game->board.hash);
   }

   return count;
}


static inline bool position_repeated(const game_t *game)
{
   return game->repetition_hash_table[game->board.hash&0xFFFF]>=2 && count_repetition(game)>=1;
}

static inline bool material_draw(const board_t *board)
{
   return false;
}

static inline bool legal_draw(const game_t *game)
{
   return game->fifty_counter[game->moves_played] >= 101 ||
          material_draw(&game->board);
}

bool game_is_draw(game_t *game)
{
   assert(game_ended(game));
   if (legal_draw(game) || material_draw(&game->board))
      return true;

   if (position_repeated(game))
      return true;

   /* Check for capture-the-flag type victory conditions */
   if ((game->board.rule_flags & RF_CAPTURE_THE_FLAG) &&
       player_captured_flags(game, next_side[game->board.side_to_move]))
      return game->flag_score == 0;

   int bare_king_score = score_bare_kings(game, game->board.side_to_move);
   if (bare_king_score != -1)
      return bare_king_score == 0;

   return !player_in_check(game, game->board.side_to_move);
}

sides get_winner(game_t *game)
{
   assert(game_ended(game));
   sides win = WHITE;

   int bare_king_score = score_bare_kings(game, game->board.side_to_move);
   if (bare_king_score != -1 && bare_king_score != 0) {
      if ( (game->board.bbc[WHITE] & game->board.royal) == (game->board.bbc[WHITE]) )
         return BLACK;
      else
         return WHITE;
   }

   if ((game->board.rule_flags & RF_CAPTURE_THE_FLAG) && player_captured_flags(game, WHITE))
      return WHITE;

   if ((game->board.rule_flags & RF_CAPTURE_THE_FLAG) && player_captured_flags(game, BLACK))
      return BLACK;

   if (player_in_check(game, WHITE))
      return BLACK;

   return win;
}

bool is_mate(game_t *game)
{
   if (player_in_check(game, game->board.side_to_move)) {
      movelist_t movelist;
      sides me = game->board.side_to_move;
      generate_moves(&movelist, &game->board, me);
      int legal_moves = movelist.num_moves;
      int n;
      for (n=0; n<movelist.num_moves; n++) {
         move_t move = movelist.move[n];
         playmove(game, move);
         if (player_in_check(game, me))
            legal_moves--;
         takeback(game);
      }

      if (legal_moves == 0)
         return true;
   }

   return false;
}

bool game_ended(game_t *game)
{
   if (legal_draw(game) || material_draw(&game->board))
      return true;

   if (position_repeated(game))
      return true;

   /* Check for capture-the-flag type victory conditions */
   if ((game->board.rule_flags & RF_CAPTURE_THE_FLAG) &&
       player_captured_flags(game, next_side[game->board.side_to_move]))
      return true;

   int bare_king_score = score_bare_kings(game, game->board.side_to_move);
   if (bare_king_score != -1)
      return true;

   /* Check whether the current position is checkmate */
   return is_mate(game);
}

/* Function call to check how much time we have left
 * We do the check every time clock_positions nodes have been searched.
 * Don't do this during QS - there aren't that many QS nodes anyway.
 */
static const int clock_positions = 0x00007FFF;
#define check_clock() \
   if (game->max_nodes && (moves_searched == game->max_nodes)) abort_search |= 1;\
   if (game->check_clock && ((positions_evaluated & clock_positions)==0))\
      abort_search |= game->check_clock(game)

#define in_check(n) (game->in_check[game->moves_played + n])

/* quicksort (qsort) callback function */
static const int *qsort_score_ptr;
static int movelist_sorter_qsort(const void *mi1, const void *mi2)
{
   int *score = (int*)qsort_score_ptr;
   int i1 = *(int*)mi1;
   int i2 = *(int*)mi2;

   return score[i2] - score[i1];
}

static void sort_moves(int *restrict move_perm, int num_moves, const int *restrict score)
{
#undef QSORT
#ifdef QSORT
   qsort_score_ptr = score;
   qsort(move_perm, num_moves, sizeof *move_perm, movelist_sorter_qsort);
#else
   int n, m;
   for (n=1; n<num_moves; n++) {
      for (m=0; m<=n; m++) {
         if (score[move_perm[m]] <= score[move_perm[n]]) {
            int k = move_perm[n];
            int c;
            for (c=n; c>m; c--)
               move_perm[c] = move_perm[c-1];
            move_perm[m] = k;
            break;
         }
      }
   }
#endif
}

/* Store a "combo move"; this is a depth-independent extension of the
 * killer principle.
 */
static void inline store_combo_move(game_t *game, int depth, move_t prev_move, move_t move)
{
#ifdef USE_COMBINATION_MOVES
   if (in_check(0) || is_promotion_move(move) || is_capture_move(move) || !prev_move) {
      return;
   }
   int c = get_move_player(move) >> 7;
   game->combo[get_move_from(prev_move)][get_move_to(prev_move)][c] = move;
#endif
}

static bool inline is_combo(const game_t *game, int depth, move_t prev_move, move_t move)
{
#ifdef USE_COMBINATION_MOVES
   int c = get_move_player(move) >> 7;
   return game->combo[get_move_from(prev_move)][get_move_to(prev_move)][c] == move;
#else
   return false;
#endif
}

/* Store a "counter move"; this is a depth-independent extension of the
 * killer principle.
 */
static void inline store_counter_move(game_t *game, int depth, move_t prev_move, move_t move)
{
   if (in_check(0) || is_promotion_move(move) || is_capture_move(move) || !prev_move)
      return;

   int c = get_move_player(move);
   game->counter[get_move_from(prev_move)][get_move_to(prev_move)][c] = move;
}

static bool inline is_counter(const game_t *game, int depth, move_t prev_move, move_t move)
{
   int c = get_move_player(move) >> 7;
   return game->counter[get_move_from(prev_move)][get_move_to(prev_move)][c] == move;
}

static bool is_killer(const game_t *game, int depth, move_t prev_move, move_t move) __attribute__((unused));
static bool is_killer(const game_t *game, int depth, move_t prev_move, move_t move)
{
   if (moves_are_equal(game->killer[0][depth], move) ||
       moves_are_equal(game->killer[1][depth], move))
      return true;
   return false;
}

static bool is_good_capture(const game_t *game, int depth, move_t move)
{
   if (moves_are_equal(game->good_capture[0][depth], move) ||
       moves_are_equal(game->good_capture[1][depth], move))
      return true;
   return false;
}

/* Store good captures. To qualify as a good capture, the move
 *  - must be a capture (obviously)
 *  - have a nominally bad SEE score
 *  - improve on the actual score
 */
static inline void store_good_capture(game_t *game, move_t move, int score, int depth)
{
   if (!is_capture_move(move))
      return;

   if (score > 1100 && !is_good_capture(game, depth, move)) {
      return;
   }
   //if (is_good_capture(game, depth, move))
   //   printf("Used good capture %s at depth %d\n", move_string(move, NULL), depth);

   if (moves_are_equal(move, game->good_capture[0][depth])) {
      /* The move was the first capture - do nothing */
   } else {
      /* This was either the last capture (out of 2 or 3), or it's a new
       * move. Either way, Degrade first capture to second capture (etc) and
       * store the new first capture.
       */
      game->good_capture[1][depth]=game->good_capture[0][depth];
      game->good_capture[0][depth]=move;
      //printf("Stored good capture %s at depth %d\n", move_string(move, NULL), depth);
   }
}

/* Store moves that kill a branch in the killer slots, but only if:
 *  - we were not in check at the beginning of this move
 *  - the move is not a promotion (already high in the tree)
 *  - the move was not a capture (already high in the tree)
 */
static inline void store_killer(game_t *game, move_t move, int depth)
{
   if (in_check(0) || is_promotion_move(move) || is_capture_move(move)) {
      return;
   }

   if (moves_are_equal(move, game->killer[0][depth])) {
      /* The move was the first killer - do nothing */
   } else {
      /* This was either the last killer (out of 2 or 3), or it's a new
       * move. Either way, Degrade first killer to second killer (etc) and
       * store the new first killer.
       */
      game->killer[1][depth]=game->killer[0][depth];
      game->killer[0][depth]=move;
   }
}


static inline void store_mate_killer(game_t *game, move_t move, int depth, int best_score)
{
   if (best_score >= (CHECKMATE-1000))
      game->mate_killer[depth] = move;
}

static void store_null_killer(game_t *game, int depth, move_t move)
{
   if (!is_capture_move(move))
      game->null_killer[depth] = move;
}


static inline void update_history(game_t *game, move_t move, int depth, int draft)
{
#ifdef USE_HISTORY_HEURISTIC
   int history_score;
   int player = get_move_player(move)>>7;

   if (depth > 10 || in_check(0) || is_promotion_move(move) || is_capture_move(move)) {
      return;
   }

   game->history_score[player][get_move_piece(move)][get_move_to(move)] += draft*draft;
   //game->history_score[player][get_move_piece(move)][get_move_to(move)] += depth*depth;
   history_score = game->history_score[player][get_move_piece(move)][get_move_to(move)];
   if (history_score > game->max_history)
      game->max_history = history_score;
#endif
}


static inline bool score_moves(const game_t *game, const movelist_t *movelist, int depth, const move_t hash_move, const move_t prev_move, const move_t my_prev_move, const move_t threat_move, int *restrict score, int *restrict move_perm)
{
   int n;
   bool have_hash_move = false;

   for (n=0; n<movelist->num_moves; n++) {
      move_t move = movelist->move[n];
      int from_row, to_row;
      sides colour = game->board.side_to_move;
      int piece = get_move_piece(move);
      int from = get_move_from(move);
      int to = get_move_to(from);

      score[n] = 0;
      move_perm[n] = n;

      if (game->board.large_board) {
      assert(is_drop_move(move)||!is_zero128(game->board.large_bbc[get_move_player(move)] & large_square_bitboards[get_move_from(move)]));
      } else {
      assert(is_drop_move(move)||(game->board.bbc[get_move_player(move)] & make_bitboard_square(get_move_from(move))));
      }

      /* Give the highest priority to the move from the hash table, if it
       * exists.
       */
      if (moves_are_equal(hash_move, move)) {
         score[n] += CHECKMATE;
         have_hash_move = true;
         continue;
      }

      /* TODO: advances of free pawn, advances of candidate pawns */

      /* Mate killer: a killer move that produced a mate score */
      if (moves_are_equal(game->mate_killer[depth], move)) {
         score[n] += CHECKMATE-100;
         continue;
      }

      /* Promotion moves */
      if (is_promotion_move(move)) {
         score[n] += 1150+game->pt.piece_value[get_move_promotion_piece(move)];
         continue;
      }

      /* Regular captures
       * The score is 1100+the SSE score, which sorts winning captures below
       * promotion moves (with a score >= 1150) and losing captures with a
       * score <=1000 below everything else.
       */
      if (is_capture_move(move)) {
         if (is_good_capture(game, depth, move)) {
            score[n] += 1050;
         } else {
            int see_score = static_exchange_evaluation(&game->board, move);
            if (see_score > 0)
               score[n] += 1100+see_score;
            else
               score[n] += see_score;
         }
         continue;
      }

      /* Award killers a higher score, so they are searched first
       * Given the weights here, winning captures are normally searched
       * before killer moves, losing captures are searched after killer
       * moves.
       */
      if (moves_are_equal(game->killer[0][depth], move)) {
         score[n] += 1100;
         continue;
      }
      if (moves_are_equal(game->killer[1][depth], move)) {
         score[n] += 1090;
         continue;
      }

#if defined HISTORY_REDUCTION
      /* Moves that have historically scored badly get sorted lower down on
       * the list. This is important when using the counter-moev heuristic,
       * because at larger search depth it may turn up moves that don't
       * work so well in other parts of the tree.
       */
      if (game->history_reduce[colour][piece_idx][get_destination(move)] < 75) {
         int s = game->history_reduce[get_move_player(move)>>7][piece_idx][get_destination(move)];
         score[n] -= (100 - s) * 4;
      }
#endif

      /* Counter moves, a generalisation of the Killer concept.
       * The question is whether these should come before, or after
       * killers...
       * Doesn't seem to make much of a difference, but it sometimes hurts
       * badly if they are sorted before the killer moves, so they should
       * probably go between the killers and the losing captures.
       */
      if (is_counter(game, depth, prev_move, move)) {
         score[n] += 1000;
         continue;
      }

      /* Combo moves, a similar idea to a counter move.
       * The question is whether these should come before, or after
       * killers...
       * Doesn't seem to make much of a difference, but it sometimes hurts
       * badly if they are sorted before the killer moves, so they should
       * probably go between the killers and the losing captures.
       */
      if (is_combo(game, depth, my_prev_move, move)) {
         score[n] += 1000;
         continue;
      }

      /* Null killer: a move that failed high after a null move */
      if (moves_are_equal(game->null_killer[depth], move)) {
         score[n] += 1000;
         //continue;
      }

      /* Also look at killers from earlier positions. This doesn't make a
       * big difference, but it speeds things up very slightly still and
       * isn't likely to hurt.
       */
      if (depth>2 && moves_are_equal(game->killer[0][depth-2], move)) {
         score[n] += 1080;
         continue;
      }
      if (depth>2 && moves_are_equal(game->killer[1][depth-2], move)) {
         score[n] += 1070;
         continue;
      }

      /* Castling moves: these should come before other king moves */
      if (is_castle_move(move)) {
         score[n] += 500;
         continue;
      }

      /* Adjust move sorting based on PSQ change.
       * NB: PSQ values are scored from WHITE's point of view, so
       * we must flip the sign if we want BLACK's point of view.
       */
      if (colour == WHITE)
         score[n] += psq[piece][colour][to] - psq[piece][colour][from];
      else
         score[n] -= psq[piece][colour][to] - psq[piece][colour][from];

      /* Idea: pawn advances are typically bad, so penalise pawn pushes
       * (except maybe to the 7th rank) and sort them lower down.
       */

#ifdef USE_HISTORY_HEURISTIC
      /* Finally, add history scores to the move */
      if (game->max_history && !in_check(0) && !is_promotion_move(move) && !is_capture_move(move)) {
         int history_score = game->history_score[colour][get_move_piece(move)][get_move_to(move)];
         score[n] += 50*history_score/game->max_history;
      }
#endif
   }

   return have_hash_move;
}

/* Asses whether zugzwang is likely to be an issue in the current position;
 * if it is, we will avoid a null-move search.
 * This simply tests whether the side to move has fewer than 4 different piece
 * types. This is *very* crude, but probably about the best we can do without
 * knowing more about the particular game being played.
 */
bool zugzwang_threat(const board_t *board)
{
   return count_piece_types(board, board->side_to_move) < 4;
}

/* Reduce the search depth for "uninteresting" moves.
 * Any move with a score < 1000 is a candidate, that is, any move sorted
 * below the winning captures and the killer moves.
 *
 * The code below reduces moves that seem poor by 2 ply, but it will only
 * reduce losing captures by 1 ply. This seems to be a good compromise. In
 * addition, we don't reduce moves where the opponent was in check on the
 * move before quite as much. This means that if this move is a capture
 * following an out-of-check move, we do not reduce at all.
 *
 * If late move pruning is used, then the reductions are applied two plies
 * earlier, because the last two plies of the search are a generalised
 * quiescence search.
 */
static int reduce_late_moves(game_t *game, move_t move, int score, int draft, move_t prev_move, move_t my_prev_move, bool mate_threat, bool pv_node)
{
   const int lmr_depth = 2;
   int reduction = 0;
   int critical_score = 1000;

   if (score < critical_score &&
       !mate_threat &&
       draft > lmr_depth &&
       prev_move != 0) {
      reduction = -lmr_depth;
      if (is_promotion_move(move) || is_capture_move(move) || in_check(0) || in_check(-1))
         reduction = -1;
      return reduction;
   }
   return 0;
}

static inline void update_history_reduction_table(game_t *game, move_t move, move_t prev_move, move_t my_prev_move, int score, int alpha)
{
   int piece_idx = get_move_piece(move);
   int side = game->board.side_to_move;
   if (score > alpha)
      game->history_reduce[piece_idx][side][get_move_to(move)] += 5;
   else
      game->history_reduce[piece_idx][side][get_move_to(move)]--;
}

/* Heuristics to decide whether this move is a candidate for reduction,
 * based on how similar moves have behaved in the past.
 */
static inline bool reduce_from_history(game_t *game, move_t move, move_t prev_move, move_t my_prev_move, int static_score, int alpha)
{
   int piece_idx;
   int side;
   if (is_capture_move(move) || in_check(0) || in_check(-1) || static_score >= alpha+25)
      return false;

   piece_idx = get_move_piece(move);
   side = get_move_player(move);
   if (game->history_reduce[piece_idx][side][get_move_to(move)] < 50) {
      game->history_reduce[piece_idx][side][get_move_to(move)] = 50;
      return true;
   }

   return false;
}


int qsearch(game_t *game, int alpha, int beta, int draft, int depth)
{
   movelist_t movelist;
   int score[MAX_MOVES];
   int perm[MAX_MOVES];
   int static_score = alpha;
   int legal_moves;
   int best_score = -CHECKMATE;
   int n;

   assert(draft >= -32);
   assert(draft <= 0);

   indent(depth);
   trace_tree("Entering quiescence search for depth %d (distance from root %d)\n", -draft, depth);

   game->quiescence_depth_of_variation[depth+1] = -draft;
   truncate_principle_variation(game, depth);

   /* Check whether our search time for this move has expired */
   check_clock();
   if (abort_search) return 0;

   sides me = game->board.side_to_move;
   bool check = player_in_check(game, me);
   game->in_check[game->moves_played] = check;

   /* Do static evaluation, and decide whether we accept the score here as-is, assuming any move we do
    * will do no worse.
    * If the result is actually very bad (much below alpha), then return immediately as well. This is a
    * (dangerous!) form of futility pruning, but it greatly reduces the size of the tree. The idea is
    * discussed in Ed Schroeder's writeup for Rebel.
    */
   if (!check) {
      static_score = static_evaluation(game, me, alpha, beta);
      indent(depth); trace_tree("Static evaluation %d [%d %d]\n", static_score, alpha, beta);
      positions_evaluated++;
      if (static_score > alpha) {
         alpha = static_score;
         if (alpha >= beta || draft < HARD_HORIZON) {
            indent(depth); trace_tree("Taking stand-pat cutoff\n");
            return alpha;
         }
      } else if (static_score < alpha-500 && !check) {
         indent(depth); trace_tree("Taking futility cutoff (1)\n");
         return static_score;
      }
   }

   /* Generate moves, then sort them based on their estimated value: the
    * move generator gives a crude guess for this, and we can use the
    * killer tables to refine this.
    * The 0-th element of the killer tables is at the root of the tree,
    * rather than at the leaves. The reason for this is simple: we want to
    * reuse the killers as much as possible between iterative deepening.
    */
   if (check) {
      game->board.in_check = check;
      generate_moves(&movelist, &game->board, me);
   } else {
      generate_quiescence_moves(&movelist, &game->board, me);
   }
   legal_moves = movelist.num_moves;
   if (legal_moves == 0) {
      indent(depth); trace_tree("No captures, return %d\n", static_score);
      return static_score;
   }


   /* Sort moves */
   int max_gain = 0;
   int best_move_score = -CHECKMATE;
   for (n=0; n<movelist.num_moves; n++) {
      move_t move = movelist.move[n];
      perm[n] = n;
      score[n] = 0;

      if (game->board.large_board) {
         assert(!is_zero128(game->board.large_bbc[get_move_player(move)] & large_square_bitboards[get_move_from(move)]));
      } else {
         assert(game->board.bbc[get_move_player(move)] & make_bitboard_square(get_move_from(move)));
      }

      /* Capture; use SEE to estimate relative worth of captures (but these moves sould all be captures) */
      if (is_capture_move(move)) {
         int gain = static_exchange_evaluation(&game->board, move);
         score[n] += gain;
         if (gain > max_gain)
            max_gain = gain;
         if (score[n] > best_move_score)
            best_move_score = score[n];
         continue;
      }

      /* Capture; use SEE to estimate relative worth of captures (but these moves sould all be captures) */
      if (is_promotion_move(move)) {
         int gain = game->pt.piece_value[get_move_promotion_piece(move)]-game->pt.piece_value[get_move_piece(move)];
         score[n] += gain;
         if (gain > max_gain)
            max_gain = gain;
         if (score[n] > best_move_score)
            best_move_score = score[n];
         continue;
      }

      /* The only way we should be able to get here is if we are in check; otherwise we don't generate normal
       * moves at all.
       * There is one exception: in games with optional promotions, we might get here anyway...
       */
      assert(check);
      if (!check) {
         //printf("Non-capture move in q-search!!! %s\n", move_string(movelist.move[n], NULL));
         //exit(0);
      }

      score[n] = -100;

      if (score[n] > best_move_score)
         best_move_score = score[n];
   }

   /* Check if the best move (estimated through SEE) looks like it might
    * win material. If not, just return the static score, since we don't expect
    * to gain much here. This is not "safe", but this is the quiescence
    * search, and this suggests that we can't improve our score by
    * captures.
    */
   if (best_move_score <= 0 && !check) {
      indent(depth); trace_tree("Taking no-winning captures cutoff\n");
      return static_score;
   }

   /* Futility pruning during quiescence: if the expected gain from any
    * move in the current position cannot improve alpha, then this node is
    * poor and we might as well skip the rest of QS.
    */
   if (static_score+max_gain+500 < alpha && !check) {
      indent(depth); trace_tree("Taking futility cutoff\n");
      return static_score+max_gain+500;
   }

   sort_moves(perm, movelist.num_moves, score);

   alpha = best_score = static_score;
   for (n=0; n<movelist.num_moves; n++) {
      move_t move = movelist.move[perm[n]];
      int local_beta = beta;

      /* Principle variation search: use full window only for first move. */
      if (n>0) local_beta = alpha+1;

      /* Don't look at losing captures */
      if (score[perm[n]] < 0 && !check)
         continue;

      /* Play move and evaluate the resulting position recursively -
       * assuming the move was legal.
       */
      playmove(game, move);
      prefetch_evaltable(game->eval_table, game->board.hash);

      if (player_in_check(game, me)) { /* Illegal move! */
         legal_moves--;
         takeback(game);
         continue;
      }
      moves_searched++;

      indent(depth);
      trace_tree("Searching move %s...\n", move_string(move, NULL));

      int score = -qsearch(game, -local_beta, -alpha, draft-1, depth+1);
      takeback(game);

      indent(depth);
      trace_tree("Search returned score %d\n", score);

      /* Abort if we've exceeded our allowed time to think */
      if (abort_search) return 0;

      if (score > best_score) {
         best_score = score;

         if (best_score > alpha) {           /* New best line found */
            alpha = best_score;

            if (best_score >= local_beta) {       /* Beta cutoff */
               branches_pruned++;
               break;
            }

            /* Update and store principle variation */
            game->quiescence_depth_of_variation[depth+1] = -draft+1;
            backup_principle_variation(game, depth + 1, move);
         }
      }
   }

   /* Detect forced check mates: if the position is a check mate and all
    * moves leading up to this position have been check evasions, the line
    * was actually forced (since all moves are generated when considering
    * check evasions). There might be a simpler mate in the position, but
    * this is better than not detecting the forced line in the first place.
    * If the line is not based on check evasions at every move, we assume
    * that the line was not forced and simply return the static score.
    * Alternatively, we should maybe flag this branch as "dangerous" and
    * let upstream decide what to do about that (eg, research).
    */
   if (legal_moves == 0 && in_check(0) && !(abs(draft)&1)) {
      /* If the side to play encountered checks at all nodes up to this
       * point during QS, then the sequence of moves leading to this
       * position was forced and we can return the mate score.
       */
      for (n=draft; n<=0; n+=2)
         if (!in_check(draft - n))
            return best_score;

      best_score = game->mate_score;
#ifdef ABSOLUTE_MATE_SCORES
      if (mate_score(best_score)) best_score += depth;
#endif
   }

   indent(depth); trace_tree("qsearch found best score %d\n", best_score);
   return best_score;
}

int search(game_t *game, int alpha, int beta, int draft, int depth)
{
   movelist_t movelist;
   int score[MAX_MOVES];
   int *move_score = score;
   int perm[MAX_MOVES];
   int static_score;
   int legal_moves;
   int best_score = -CHECKMATE;
   int hash_flag = HASH_TYPE_ALPHA;
   const int alpha_in = alpha, beta_in = beta;
   hash_table_entry_t *hash;
   move_t prev_move = 0;
   move_t my_prev_move = 0;
   move_t threat_move = 0;
   move_t hash_move = 0;
   bool avoid_null = false;
   bool pv_node = (beta > alpha+1);
   bool mate_threat = false;
   int n;

   indent(depth);
   trace_tree("Entering recursive search with remaining depth %d (depth from root %d)\n", draft, depth);

   /* Truncate the principle variation at this depth here, otherwise we may
    * end up with rubbish at the end of the PV.
    */
   truncate_principle_variation(game, depth);

   /* The very first thing we do is to check if this position has occurred
    * three times in the game tree, or whether no irreversible move has
    * been played in 50 moves (100 ply). Either of these conditions means
    * we should flag a draw and return immediately.
    * During the search, we simply score positions as a draw if they have
    * occurred at least once before. From the top level, we don't claim a
    * draw until three repetitions have been made.
    *
    * For games that disallow perpetual check or chases (like Xiangqi) we need to do
    * some more work when a repetition is detected (TODO). For now, we simply return
    * a large negative score for those games.
    */
   if (legal_draw(game))
      return 0;

   bool possible_chase = position_repeated(game);
   if (possible_chase && !game->rep_score) {
      best_score = game->rep_score;
#ifdef ABSOLUTE_MATE_SCORES
      if (mate_score(best_score)) best_score += depth;
#endif
      return best_score;
   }

   /* Check for bare kings and possible special scores that have to be returned in this case. */
   int bare_king_score = score_bare_kings(game, game->board.side_to_move);
   if (bare_king_score != -1) {
      best_score = bare_king_score;
#ifdef ABSOLUTE_MATE_SCORES
      if (mate_score(best_score)) best_score += depth;
#endif
      return best_score;
   }

   /* Check whether our search time for this move has expired */
   check_clock();
   if (abort_search) return 0;

#ifndef ABSOLUTE_MATE_SCORES
   /* Adjust the window in case we're dealing with a mate score */
   if (alpha < -CHECKMATE+1000) alpha--;
   if (beta  < -CHECKMATE+1000) beta--;
   if (alpha >  CHECKMATE-1000) alpha++;
   if (beta  >  CHECKMATE-1000) beta++;
#endif

   /* Check whether an earlier branch in the tree has given us a mate
    * score; if so, we need to adjust the window
    */
   if (!pv_node) {
      int mate_value = CHECKMATE - depth - 1;
      if (mate_value < beta)
         beta = mate_value;
      if (alpha >= beta)
         return mate_value;

      mate_value = -CHECKMATE + depth;
      if (mate_value > alpha)
         alpha = mate_value;
      if (alpha >= beta)
         return mate_value;
   }

   /* Check for capture-the-flag type victory conditions */
   if (expect(game->board.rule_flags & RF_CAPTURE_THE_FLAG, false)) {
      if (expect(player_captured_flags(game, next_side[game->board.side_to_move]), false)) {
         best_score = game->flag_score;
#ifdef ABSOLUTE_MATE_SCORES
         if (mate_score(best_score)) best_score += depth;
#endif
         return best_score;
      }
   }

   /* Look up this position in the transposition table */
   hash = query_table_entry(game->transposition_table, game->board.hash);
   if (hash) {
      int new_score;
      hash_move = hash->best_move;
      new_score = score_from_hashtable(hash->score, depth);
      positions_in_hashtable++;
      /* If we searched this position before from the same depth or better,
       * just return the previous score. Otherwise we can still use the
       * recorded move to improve our move ordering.
       * Don't return scores from the hash table if we're close to a 50-move draw.
       */
      if (hash->depth > draft && game->fifty_counter[game->moves_played] < 80) {
         if (hash->flags & HASH_TYPE_EXACT) {
            game->hash_hit[depth+1] = true;
            backup_principle_variation(game, depth + 1, hash_move);

            return new_score;
         } else if ((hash->flags & HASH_TYPE_ALPHA) && (new_score<=alpha)) {
            game->hash_hit[depth+1] = true;

            return new_score;
         } else if ((hash->flags & HASH_TYPE_BETA) && (new_score>=beta)) {
            game->hash_hit[depth+1] = true;

            return new_score;
         }
      } else {
         /* We may still be able to learn something from the transposition
          * table here... for instance, whether a NULL-move search would
          * make sense!
          */
         if (hash->depth >= 4 && draft < 4 &&
            hash->flags&(HASH_TYPE_EXACT | HASH_TYPE_BETA) &&
            new_score < beta) {
            avoid_null = true;
         }

         if (hash->flags & HASH_TYPE_ALPHA)
            hash_move = 0;
      }
   }

   sides me = game->board.side_to_move;

   /* Record whether the side to move is in check or not */
   bool check = player_in_check(game, me);
   game->in_check[game->moves_played] = check;

   /* Check extension: if we're in check, extend the search by one ply */
   if (check)
      draft++;

   if (possible_chase && !check)
      return game->rep_score;

   if (draft <= 0)
      return qsearch(game, alpha, beta, 0, depth);

   static_score = game->board.material[me] - game->board.material[next_side[me]];
   static_score += (me == WHITE)? game->board.psq : -game->board.psq;
   //static_score = static_evaluation(game, me, alpha, beta);
   positions_evaluated++;

   /* We don't have a move from the transposition table, which is bad... try to get a replacement
    * internal iterative deepening.
    * FIXME: only in PV nodes?
    */
   if (hash_move == 0 && draft > 4) {
      int iid_draft = draft / 2 + 1;

      game->best_move[depth] = 0;
      search(game, alpha_in, beta_in, iid_draft, depth);
      hash_move = game->best_move[depth];
   }

   /* Store the previous move: this is used to change move ordering on this
    * move. If there is no previous move, then this move will be null.
    */
   prev_move = 0;
   if (game->moves_played>0)
      prev_move = game->move_list[game->moves_played-1];

   my_prev_move = 0;
   if (game->moves_played>1)
      my_prev_move = game->move_list[game->moves_played-2];

    /* Before examining things in detail, see whether a NULL move will cause a cut-off.
     * Disallow NULL moves if
     *  - we're in check (obviously)
     *  - the last move was a move out-of-check (too dangerous, we may have
     *    something much better that we could otherwise miss)
     *  - the previous move was a null move (the search would collapse into
     *    nothing and we'd just end up researching the position at a
     *    shallower depth - which we've already done thanks to ID)
     *  - we're too close to the horizon (don't skip to QS from a NULL
     *    move)
     *  - there's a mate treat (the null-move is pointless)
     *  - there's a high risk of zugzwang
     */
    avoid_null = avoid_null ||
                 pv_node ||
                 in_check(0) ||
                 in_check(-1) ||
                 prev_move == 0 ||
                 mate_score(beta) ||
                 zugzwang_threat(&game->board);
    if (!avoid_null) {
       move_t move;
       int r = 4;
       if (draft < 6) r = 3;
       move = 0;

       /* play null-move */
       play_null_move(game);
       int new_score = -search(game, -beta, 1-beta, draft-r, depth+1);
       takeback(game);
       if (new_score >= beta && !mate_score(new_score)) { /* null-move fails high, reduce */
          hash_move = 0;
          store_table_entry(game->transposition_table, game->board.hash, 
                            draft, new_score, HASH_TYPE_ALPHA, hash_move);
          return new_score;
       } else {
          /* Retrieve move, store as null-killer on the next ply */
          threat_move = game->best_move[depth + 1];
          store_null_killer(game, depth + 1, threat_move);
       }

       /* Mate threat detection and extension */
       if (mate_score(new_score) && draft <= 5) {
          threat_move = game->best_move[depth + 1];
          store_mate_killer(game, threat_move, depth+1, new_score);
          mate_threat = true;
       }
    }

   /* Generate moves, then sort them based on their estimated value: the
    * move generator gives a crude guess for this, and we can use the
    * killer tables to refine this.
    * The 0-th element of the killer tables is at the root of the tree,
    * rather than at the leaves. The reason for this is simple: we want to
    * reuse the killers as much as possible between iterative deepening.
    */
   game->board.in_check = check;
   generate_moves(&movelist, &game->board, me);
   legal_moves = movelist.num_moves;
   if (legal_moves == 0) {
      best_score = 0;
      if (check) {
         best_score = game->mate_score;
      } else {
         /* Special case: no pieces left at all! */
         if (expect(count_piece_types(&game->board, me) == 0, false))
            best_score = game->no_piece_score;
         else
            best_score = game->stale_score;
      }
#ifdef ABSOLUTE_MATE_SCORES
      if (mate_score(best_score)) best_score += depth;
#endif
      return best_score;
   }

   /* Sort moves */
   bool have_hash_move = score_moves(game, &movelist, depth, hash_move, prev_move,
                                 my_prev_move, threat_move, score, perm);

   sort_moves(perm, movelist.num_moves, score);

   bool pruning_candidate = !check && !mate_threat && draft < 4 && !mate_score(beta);
   bool ok_to_prune = false;
   for (n=0; n<movelist.num_moves; n++) {
      move_t move = movelist.move[perm[n]];
      int local_beta = beta;
      int r = mate_threat;

      assert(check == in_check(0));

      /* Principle variation search: use full window only for first move. */
      if (n>0) local_beta = alpha+1;

      /* Is it ok to do forward pruning? */
      static const int pm[] = { 0, 200, 400, 800, 1200 };
      if (pruning_candidate && ok_to_prune && score[perm[n]] < 1000 && static_score + pm[draft] <= alpha)
         r = -draft;

      /* Late move reductions: reduce the depth for poor moves */
      if ((!pv_node || n>0) && ok_to_prune)
         r = reduce_late_moves(game, move, score[perm[n]], draft, prev_move, my_prev_move, mate_threat, pv_node);

      /* Check for history reductions. This reduces moves that have
       * historically failed low in the current search. Discussed by Ed
       * Schroeder at http://www.top-5000.nl/authors/rebel/hr.htm
       * We avoid the reduction in the following cases:
       *
       * - This move has already been extended (or reduced).
       * - This move is a capture
       * - This move is a check or an escape-from-check
       * - The move is not "bad enough" that it has reached the threshold yet.
       * - The static score for the current position is not below alpha.
       */
      if (r == 0 && ok_to_prune && score[perm[n]] < 1000 && draft > 1 && reduce_from_history(game, move, prev_move, my_prev_move, static_score, alpha))
         r = -1;

      /* Play move and evaluate the resulting position recursively -
       * assuming the move was legal.
       */
      playmove(game, move);
      if (player_in_check(game, me)) { /* Illegal move! */
         legal_moves--;
         takeback(game);
         continue;
      }
      moves_searched++;
      ok_to_prune = true;

      /* If we are in check and have a winning capture, then the checking move must have had SEE<0. We do not
       * want to extend these checks, in fact, we want to reduce them.
       */
      //if (check && is_capture_move(move) && score[perm[n]] > 1000)
      //   r = -1;

      indent(depth);
      trace_tree("Searching move %s...\n", move_string(move, NULL));

      int score = -search(game, -local_beta, -alpha, draft-1+r, depth+1);

      if (score > alpha && r < 0)
         score = -search(game, -local_beta, -alpha, draft-1, depth+1);

      if (score > alpha && score < beta && n>0)
         score = -search(game, -beta, -alpha, draft-1, depth+1);

      /* For forced mates further down the tree, we should adjust the score
       * by one ply: mate one node down is a mate-in-one-ply from this ply
       * and a mate-in-two-ply one node further up.
       * We need two checks: one to see if we're mating and one to see if
       * we're being mated.
       */
#ifndef ABSOLUTE_MATE_SCORES
      score -= (score >=  CHECKMATE-1000);
      score += (score <= -CHECKMATE+1000);
#endif
      takeback(game);

      indent(depth);
      trace_tree("Search returned score %d\n", score);

      /* Abort if we've exceeded our allowed time to think */
      if (abort_search) return 0;

      /* Record the result in the history reduction table */
      update_history_reduction_table(game, move, prev_move, my_prev_move, score, alpha);

      if (score > best_score) {
         best_score = score;
         hash_move = move;

         if (best_score > alpha) {           /* New best line found */
            hash_flag = HASH_TYPE_EXACT;
            alpha = best_score;

            /* Store moves that improve alpha in the counter and
             * cmobination move slots. This seems to perform slightly
             * better than following the same logic as killer moves,
             * probably because this means we still get a slightly better
             * move ordering even if the move does not cause a cut-off.
             */
            store_counter_move(game, depth, prev_move, move);
            store_combo_move(game, depth, my_prev_move, move);

            /* Update and store principle variation */
            backup_principle_variation(game, depth + 1, move);

            if (best_score >= beta) {       /* Beta cutoff */
               /* Store good moves in the killer slots, but only if:
                *  - we were not in check at the beginning of this move
                *  - the move is not a promotion (already high in the list)
                *  - the move was not a capture (already high in the list)
                */
               store_killer(game, move, depth);
               store_mate_killer(game, move, depth, best_score);

               /* Store good captures */
               store_good_capture(game, move, move_score[perm[n]], depth);

               update_history(game, move, depth, draft);
               hash_flag = HASH_TYPE_BETA;
               branches_pruned++;
               break;
            }
         }
      }
   }

   if (legal_moves == 0) {
      hash_flag = HASH_TYPE_EXACT;
      hash_move = 0;
      if (check)
         best_score = game->mate_score;
      else
         best_score = game->stale_score;
#ifdef ABSOLUTE_MATE_SCORES
      if (mate_score(best_score)) best_score += depth;
#endif
   }

   if (hash_flag == HASH_TYPE_ALPHA)
      hash_move = 0;

   /* Store best move for retrieval from one level up in the search (for IID) */
   game->best_move[depth] = hash_move;

   /* store evaluation in the transposition table */
   if (legal_moves && !abort_search)
      store_table_entry(game->transposition_table, game->board.hash, draft,
                        score_to_hashtable(best_score, depth), hash_flag, hash_move);


   return best_score;
}
