/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <ctype.h>
#include <stdarg.h>
#include <inttypes.h>
#include <limits.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include "pipe2.h"
#include "sjaak.h"

#define streq(s1, s2) (strcmp((s1), (s2)) == 0)

typedef struct {
   pipe2_t *f;
   char *name;
   uint32_t state;
   int64_t clock;       /* Clock time, in msec */
   int moves;           /* Number of moves played */
   int id;
   int win;
   int draw;
} program_t;

/* Program state flags */
#define PF_INIT      0x0001
#define PF_DEAD      0x0002
#define PF_PING      0x0004
#define PF_TIME      0x0008
#define PF_VARS      0x0010
#define PF_FORCE     0x0020
#define PF_THINK     0x0040
#define PF_FORFEIT   0x0080
#define PF_CLAIMD    0x0100
#define PF_CLAIMW    0x0200
#define PF_CLAIMB    0x0400
#define PF_CLAIM     (PF_CLAIMD | PF_CLAIMW | PF_CLAIMB)
#define PF_RESIGN    0x0800
#define PF_FLAG      0x1000

static game_t *(*new_variant_game)(void) = create_standard_game;

static program_t *prog[2];

static uint64_t start_time = 10000;  /* Msec */
static int moves_per_tc = 40;
static int time_inc = 0;
static int min_time_per_move = 0;

static bool check_children;
static FILE *logf = NULL;
static char *buf;
#define BUF_SIZE  65536

static void child_signal_handler(int sig)
{
   //printf("Child terminated\n");
   //exit(0);
   check_children = true;
   if (sig == SIGCHLD)
      signal(sig, child_signal_handler);
}

static void send_to_program(program_t *prog, const char *msg, ...)
{
   va_list ap;
   static bool newline = true;
   va_start(ap, msg);
   vsnprintf(buf, BUF_SIZE-1, msg, ap);
   va_end(ap);

   if (logf) {
      if (newline)
         fprintf(logf, "%d> ", prog->id);
      newline = false;
      fprintf(logf, "%s", buf);
      if (strstr(buf, "\n"))
         newline = true;
      fflush(logf);
   }

   ssize_t res = 0;

   if (strlen(buf)) {
      do {
         res = write(prog->f->out_fd, buf, strlen(buf));
      } while (!res);
      if (res < 0) {
         int e = errno;
         perror(NULL);
         if (logf)
            fprintf(logf, "Write error to program %d (%s): %s\n",
                  prog->id, prog->name, strerror(e));
         exit(0);
      }
   }
}

void msleep(int msec)
{
   struct timespec timeout;

   timeout.tv_sec = msec / 1000;
   timeout.tv_nsec = (msec % 1000)*1000000;
   nanosleep(&timeout, NULL);
}

void wait_input(int fd1, int fd2, int musec)
{
   struct timeval timeout;
   fd_set readfds, errfds;

   FD_ZERO(&readfds);
   FD_SET(fd1, &readfds);
   FD_SET(fd2, &readfds);

   FD_ZERO(&errfds);
   FD_SET(fd1, &errfds);
   FD_SET(fd2, &errfds);

   //printf("Waiting for input...\n");
   timeout.tv_sec = musec / 1000000;
   timeout.tv_usec = musec % 1000000;
   select(32, &readfds, NULL, &errfds, &timeout);

   //printf("Received input\n");
}

bool child_is_alive(program_t * const prog)
{
   int status;
   pid_t result = waitpid(prog->f->pid, &status, WNOHANG);
   if (result == 0) {
      return true;
   } else if (result == -1) {
      perror(NULL);
   }

   // Child exited
   return false;
}

/* Play a move on the board
 * TODO: recognise SAN moves as well as the weird "n. ... move" format.
 */
bool input_move(game_t *game, char *move)
{
   movelist_t movelist;
   int from, to, n;
   char *s;

   /* Xboard counts from 0 for large boards with > 9 ranks */
   int dr = (large_board_ranks > 9);

   s = move;
   if(s && *s) {
      generate_moves(&movelist, &game->board, game->board.side_to_move);
      while (*s == ' ') s++;
      if (*s) {
         n = -1;
         if (s[1] == '@') {   /* Drop */
            int col = s[2] - 'a';
            int row = s[3] - '1' + dr;
            to = pack_row_file(row, col);

            const char *p = strchr(piece_symbol_string, s[0]);
            if (p == NULL)
               p = strchr(piece_symbol_string, toupper(s[0]));

            int drop_piece = p - piece_symbol_string;

            int k;
            for (k=0; k<movelist.num_moves; k++) {
               int p = get_move_drop(movelist.move[k], 0);
               int square = decode_drop_square(p);
               int piece = decode_drop_piece(p);
               
               if (piece == drop_piece && square == to) {
                  n = k;
                  break;
               }
            }
         } else if (s[0] == 'O' && strstr(s, "O-O") == s) {
            if (streq(s, queenside_castle)) {
               int k;
               for (k=0; k<movelist.num_moves; k++) {
                  if (is_castle_move(movelist.move[k])) {
                     from = get_move_from(movelist.move[k]);
                     to = get_move_to(movelist.move[k]);
                     if (unpack_file(to) < large_board_files/2) {
                        n = k;
                        break;
                     }
                  }
               }
            } else {
               int k;
               for (k=0; k<movelist.num_moves; k++) {
                  if (is_castle_move(movelist.move[k])) {
                     from = get_move_from(movelist.move[k]);
                     to = get_move_to(movelist.move[k]);
                     if (unpack_file(to) >= large_board_files/2) {
                        n = k;
                        break;
                     }
                  }
               }
            }
         } else {
            int col = *s - 'a'; s++;
            int row = *s - '1'; s++;
            from = pack_row_file(row + dr, col);
            col = *s - 'a'; s++;
            row = *s - '1'; s++;
            to = pack_row_file(row + dr, col);

            //printf("Read move from %d (%s) to %d (%s), validating\n", from, square_names[from], to, square_names[to]);
            n = validate_move(&movelist, from, to);
            //printf("Move validated as number %d ('%s')\n", n, move_string(movelist.move[n], NULL));

            if (is_promotion_move(movelist.move[n])) {
               /* Find the move that promotes to the correct piece type */
               const char *p = strchr(piece_symbol_string, *s);
               if (p == NULL)
                  p = strchr(piece_symbol_string, toupper(*s));

               int promotion_piece = p - piece_symbol_string;
               for ( ; n<movelist.num_moves; n++)
                  if (get_move_promotion_piece(movelist.move[n]) == promotion_piece)
                     break;

               if (n == movelist.num_moves)  /* Promotion to unknown piece type */
                  return false;

               s++;
            }
         }
         if (n<0) {
            if (logf) {
               int k;
               fprintf(logf, "# Legal moves: ");
               for (k=0; k<movelist.num_moves; k++)
                  fprintf(logf, "%s ", move_string(movelist.move[k], NULL));
               fprintf(logf, "\n");
               fflush(logf);
            }
            return false;
         }
         playmove(game, movelist.move[n]);
         //print_bitboards(&game->board);
      }
   } else {
      return false;
   }

   return true;
}

static void parse_engine_input(program_t *prog, game_t *game, char *input)
{
   char *line = input;
   char *token;
   char *next = NULL;

   if (!input || input[0] == '\0') return;

   line = input;

   /* Remove leading white space */
   while (isspace(*line)) line++;

   /* Break up lines at linebreaks */
   next = strstr(line, "\n");
   if (next) {
      next[0] = '\0';
      next++;
   }

   if (logf) {
      fprintf(logf, "%d< %s\n", prog->id, line);
      fflush(logf);
   }

   /* Remove comments */
   token = strstr(line, "#");
   if (token) *token = '\0';

   /* Remove trailing white space */
   token = line+strlen(line);
   while (token>line && isspace(token[-1])) {
      token--; *token='\0';
   }

   /* Skip empty line */
   if (!line[0]) goto done;

   //printf(">%s<\n", line);
   /* Parse engine input */
   if (strstr(line, "move") == line) {
      /* Stop the clock */
      uint64_t time = peek_timer(game);

      /* Play the move on the board. If it is illegal, the engine forfeits the game. */
      bool legal = input_move(game, line+5);
      prog->clock -= time;
      prog->moves++;
      prog->state &= ~PF_THINK;

      /* Update the engine's clock */
      prog->clock += time_inc;
      if (moves_per_tc && prog->moves > moves_per_tc-1 && (prog->moves % moves_per_tc) == 0) {
         if (logf) fprintf(logf, "Time control: %"PRIi64" -> %"PRIu64"\n", prog->clock, prog->clock+start_time);
         prog->clock += start_time;
      }
      if (prog->clock < 0) {
         prog->state |= PF_FLAG;
         prog->clock = 0;
      }
      if (legal) {
         if (logf) {
            fprintf(logf, "Move '%s' legal, time = %"PRIu64" (%"PRIu64" remaining), move %d/%d\n",
                  line+5, time, prog->clock, prog->moves, 40);
            fflush(logf);
         }
      } else {
         prog->state |= PF_FORFEIT;
         if (logf) {
            fprintf(logf, "Illegal move: '%s'\n", line+5);
            fflush(logf);
         }
      }
   } else if (strstr(line, "#") == line) {
      /* Comment, ignored */
   } else if (strstr(line, "tellusererror") == line) {
      fprintf(stderr, "%s\n", line+9);
   } else if (strstr(line, "telluser") == line) {
      printf("%s\n", line+9);
   } else if (strstr(line, "resign") == line) {
      prog->state &= ~PF_THINK;
      prog->state |= PF_RESIGN;
   } else if (strstr(line, "1-0") == line) {
      prog->state &= ~PF_THINK;
      prog->state |= PF_CLAIMW;
   } else if (strstr(line, "0-1") == line) {
      prog->state &= ~PF_THINK;
      prog->state |= PF_CLAIMB;
   } else if (strstr(line, "1/2-1/2") == line) {
      prog->state &= ~PF_THINK;
      prog->state |= PF_CLAIMD;
   } else if (strstr(line, "Error") == line) {
   } else if (strstr(line, "Illegal move") == line) {
      /* Verify claim (TODO)
       * For now, we just forfeit the engine, since we checked moves for legality before sending them on
       */
      prog->state &= ~PF_THINK;
      prog->state |= PF_FORFEIT;
      if (logf) {
         fprintf(logf, "False illegal move claim '%s' by %s in position %s\n",
            line+13, prog->name, make_fen_string(game, NULL));
         fflush(logf);
      }
   } else if (strstr(line, "feature") == line) {
      /* Parse the different feature options */
      char *ss = strstr(line, " ");
      while (ss && ss[0]) {
         char *feature;
         char *value;

         feature = ss;
         while (*feature && isspace(*feature)) feature++;

         value = strstr(feature, "=");
         if (!value) break;
         value[0] = '\0';
         value++;

         /* String or numeric? */
         if (value[0] == '"') {
            value++;

            char *s = value;
            while (*s && *s != '"') s++;
            s[0] = '\0';
            ss = s+1;
         } else {
            char *s = value;
            while (*s && !isspace(*s)) s++;
            ss = s+1;
         }

         bool accept = false;
         if (streq(feature, "myname")) {
            int len = strlen(value);
            prog->name = realloc(prog->name, len+1);
            int n;
            for (n=0; n<len; n++)
               prog->name[n] = value[n];
            prog->name[len] = '\0';
            accept = true;
            if (logf) {
               fprintf(logf, "Set name '%s' for program %d\n", prog->name, prog->id);
               fflush(logf);
            }
         } else if (streq(feature, "sigint")) {
            accept = true;
         } else if (streq(feature, "setboard")) {
            accept = true;
         } else if (streq(feature, "time")) {
            accept = true;
            prog->state |= PF_TIME;
         } else if (streq(feature, "variants")) {
            accept = true;
            prog->state |= PF_VARS;
         } else if (streq(feature, "ping")) {
            accept = true;
            prog->state |= PF_PING;
         } else if (streq(feature, "done")) {
            accept = true;
            int res;
            sscanf(value, "%d", &res);
            if (res == 0)
               prog->state |= PF_INIT;
            else
               prog->state &= ~PF_INIT;
         }
         send_to_program(prog, "%s %s\n", accept?"accepted":"rejected", feature);
      }
   } else {    /* Thinking or ponder output? */
   }

done:
   if (next)
      parse_engine_input(prog, game, next);
}

int main(int argc, char **argv)
{
   FILE *pgnf = NULL;
   char *fcp = NULL;
   char *scp = NULL;
   char *variant_name = "normal";
   char *logfile = NULL;
   char *epdfile = NULL;
   char finit[65536];
   char sinit[65536];
   char *finit_str = finit;
   char *sinit_str = sinit;
   char host[255];
   int mg = 1;
   int epd_count = 0;
   movelist_t movelist;
   game_t *game = NULL;
   
   gethostname(host, sizeof host);
   char *s = strstr(host, ".");
   if (s) *s = '\0';

   finit_str[0] = '\0';
   sinit_str[0] = '\0';

   buf = malloc(BUF_SIZE);

   construct_inverse_diagonal_maps();
   initialise_slider_tables();
   initialise_hash_keys();
   playgame_init();

   new_variant_game = create_standard_game;

   /* Parse command-line options. */
   if (argc>1) {
      int n;
      for (n=1; n<argc; n++) {
         if (strstr(argv[n], "-variant")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: -variant passed, but no variant specified");
               exit(0);
            }
            n++;

            char *input = argv[n];
            variant_name = input;
            new_variant_game = create_variant_from_name(input);
            if (new_variant_game == NULL) {
               fprintf(stderr, "error: variant '%s' unknown\n", input);
               exit(0);
            }

         } else if (strstr(argv[n], "-mg")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: number of games/match not specified\n");
               exit(0);
            }
            n++;
            sscanf(argv[n], "%d", &mg);
         } else if (strstr(argv[n], "-fcp")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: first program not specified\n");
               exit(0);
            }
            n++;
            fcp = argv[n];
         } else if (strstr(argv[n], "-scp")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: second program not specified\n");
               exit(0);
            }
            n++;
            scp = argv[n];
         } else if (strstr(argv[n], "-finit")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: init line for first program not specified\n");
               exit(0);
            }
            n++;
            snprintf(finit_str, sizeof finit - (finit_str - finit), "%s\n", argv[n]);
            finit_str = finit + strlen(finit);
         } else if (strstr(argv[n], "-sinit")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: init line for second program not specified\n");
               exit(0);
            }
            n++;
            snprintf(sinit_str, sizeof sinit - (sinit_str - sinit), "%s\n", argv[n]);
            sinit_str = sinit + strlen(sinit);
         } else if (strstr(argv[n], "-tc")) {
            /* Either just a total time in [minutes:]seconds, or moves/time+increment */
            if (n+1 >= argc) {
               fprintf(stderr, "error: second program not specified\n");
               exit(0);
            }
            n++;
            char *s = argv[n];
            char *p = s;
            if (strstr(s, "/")) {
               p = strstr(s, "/");
               sscanf(s, "%d", &moves_per_tc);
               p[0] = '\0';
               p++;
            }

            int minutes = 0, seconds=0;
            if (strstr(p,":")) {
               sscanf(p, "%d:%d", &minutes, &seconds);
               seconds += 60*minutes;
            } else {
               sscanf(p, "%d", &seconds);
            }
            start_time = seconds * 1000;

            if (strstr(p, "+")) {
               p = strstr(p, "+");
               p++;
               sscanf(p, "%d", &time_inc);
               time_inc *= 1000;
            }
         } else if (strstr(argv[n], "-inc")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: no increment specified\n");
               exit(0);
            }
            n++;
            sscanf(argv[n], "%d", &time_inc);
            time_inc *= 1000;
         } else if (strstr(argv[n], "-mps")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: moves per session not specified\n");
               exit(0);
            }
            n++;
            sscanf(argv[n], "%d", &moves_per_tc);
         } else if (strstr(argv[n], "-mtpm") ){
            if (n+1 >= argc) {
               fprintf(stderr, "error: minimum time per move not specified\n");
               exit(0);
            }
            n++;
            sscanf(argv[n], "%d", &min_time_per_move);
         } else if (strstr(argv[n], "-sgf")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: no safe file specified\n");
               exit(0);
            }
            n++;
            pgnf = fopen(argv[n], "a");
         } else if (strstr(argv[n], "-log")) {
            logfile = "sjef.log";
            if (n+1 < argc && argv[n+1][0] != '-') {
               logfile = argv[n+1];
               n++;
            }
         } else if (strstr(argv[n], "-epd")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: no EPD file specified\n");
               exit(0);
            }
            n++;
            epdfile = argv[n];
         } else {
            fprintf(stderr, "Unknown option: %s\n", argv[n]);
            exit(0);
         }
      }
   }

   if (moves_per_tc < 0) moves_per_tc = 0;
   if (time_inc < 0) time_inc = 0;

   if (!fcp || !scp) {
      fprintf(stderr, "error: need two programs to play\n");
      exit(0);
   }

   if (!mg) {
      printf("No games to play\n");
      exit;
   }

   /* Start a game from a (random) position in the epd file.
    * Count the number of positions in the file.
    */
   if (epdfile) {
      FILE *f = fopen(epdfile, "r");
      if (f) {
         while(fgets(buf, BUF_SIZE, f))
            epd_count++;
         sgenrand(time(NULL));
         fclose(f);
      } else {
         fprintf(stderr, "error: cannot read EPD file %s\n", epdfile);
         exit(0);
      }
   }

   check_children = true;
   signal(SIGCHLD, child_signal_handler);

   if (logfile)
      logf = fopen(logfile, "w");

   pipe2_t *first, *second;
   char *args[2];
   args[1] = NULL;

   args[0] = fcp;
   printf("Starting %s\n", fcp);
   first = p2open(fcp, args);

   args[0] = scp;
   printf("Starting %s\n", scp);
   second = p2open(scp, args);

   prog[0] = malloc(sizeof *prog[0]);
   prog[1] = malloc(sizeof *prog[1]);

   prog[0]->f = first;
   prog[0]->name = strdup(fcp);
   prog[0]->state = PF_INIT;
   prog[0]->id = 1;
   prog[0]->win = 0;
   prog[0]->draw = 0;

   prog[1]->f = second;
   prog[1]->name = strdup(scp);
   prog[1]->state = PF_INIT;
   prog[1]->id = 2;
   prog[1]->win = 0;
   prog[1]->draw = 0;

   /* Send setup */
   send_to_program(prog[0], "xboard\nprotover 2\n");
   send_to_program(prog[1], "xboard\nprotover 2\n");

   /* Send init strings (if needed) */
   if (logf)
      fprintf(logf, "init1 = '%s'\ninit2 = '%s'\n", finit, sinit);
   if (finit[0])
      send_to_program(prog[0], "%s", finit);
   if (sinit[0])
      send_to_program(prog[1], "%s", sinit);

   /* Parse feature options */
   for (int i=0; i<2; i++) {
      printf("Intialising program %d...", i+1);
      fflush(stdout);
      while(prog[i]->state & PF_INIT) {
         while(!p2_input_waiting(prog[i]->f));
         while(p2_input_waiting(prog[i]->f)) {
            if (!p2gets(buf, BUF_SIZE, prog[i]->f))
               perror(NULL);

            parse_engine_input(prog[i], NULL, buf);
         }
      }
      printf("done\n");
   }

   int n;
   char tc_string[256];
   snprintf(tc_string, sizeof tc_string, "%d/%"PRIu64":%02"PRIu64"+%d", moves_per_tc, (start_time/60000), (start_time%60000) / 1000, time_inc/1000);

   printf("Time control: %s\n", tc_string);
   if (logf)
      fprintf(logf, "Time control: %s\n", tc_string);
   if (moves_per_tc * time_inc) printf("Warning: both moves per session and increment specified\n");
   for (n=0; n<mg; n++) {
      /* Decide who plays white */
      int white = (n&1);
      int black = 1 - white;

      /* Start a new game of the appropriate variant */
      printf("Starting variant '%s' (%s - %s) game %d of %d\n", variant_name, prog[white]->name, prog[black]->name, n+1, mg);
      game = new_variant_game();
      start_new_game(game);
      set_infinite_time(game);

      prog[0]->state = 0;
      prog[0]->clock = start_time;
      prog[0]->moves = 0;

      prog[1]->state = 0;
      prog[1]->clock = start_time;
      prog[1]->moves = 0;

      /* Inform the engines about the new variant */
      send_to_program(prog[0], "new\n");
      send_to_program(prog[1], "new\n");

      if (!streq(variant_name, "normal")) {
         send_to_program(prog[0], "variant %s\n", variant_name);
         send_to_program(prog[1], "variant %s\n", variant_name);
      }

      /* Randomise initial moves */
      send_to_program(prog[0], "random\n");
      send_to_program(prog[1], "random\n");

      /* Switch engines to "force" mode, switch off pondering. */
      send_to_program(prog[0], "force\neasy\npost\n");
      send_to_program(prog[1], "force\neasy\npost\n");
      prog[0]->state |= PF_FORCE;
      prog[1]->state |= PF_FORCE;

      /* Setup time control */
      send_to_program(prog[0], "level %d %"PRIu64":%02"PRIu64" %d\n",
         moves_per_tc, (start_time)/(60000), (start_time)%(60000)/1000, time_inc);
      send_to_program(prog[1], "level %d %"PRIu64":%02"PRIu64" %d\n",
         moves_per_tc, (start_time)/(60000), (start_time)%(60000)/1000, time_inc);

      /* Send initial position */
      if (epdfile && epd_count) {
         int pos = genrandf()*epd_count;
         FILE *f = fopen(epdfile, "r");
         int n;
         for (n=0; n<=pos; n++)
            fgets(buf, BUF_SIZE, f);
         fclose(f);

         char *fen = strdup(buf);
         char *s = strstr(fen, "\n");
         if (s) *s = '\0';

         send_to_program(prog[0], "setboard %s\n", fen);
         send_to_program(prog[1], "setboard %s\n", fen);
         setup_fen_position(game, fen);

         if (logf)
            fprintf(logf, "Loaded position #%d from %s (FEN: %s)\n", pos, epdfile, fen);

         free(fen);
      }

      /* Play out the game */
      bool game_is_decided = false;
      while (1) {
         char s[4096];

         if (logf) fflush(logf);
         //fprintf(logf, "Wait input\n");
         //fflush(logf);
         wait_input(prog[0]->f->in_fd, prog[1]->f->in_fd, 100);

         /* Check whether the engines are both still alive */
         if (check_children) {
            bool done = false;
            for (int i=0; i<2; i++) {
               if(!child_is_alive(prog[i])) {
                  prog[i]->state |= PF_DEAD;
                  done = true;
                  printf("Child %d died\n", prog[i]->id);
               }
            }
            if (done) break;//exit(0);
            check_children = false;
         }

         /* Parse input from the two engines */
         for (int i=0; i<2; i++) {
            while(p2_input_waiting(prog[i]->f)) {
               if (!p2gets(buf, BUF_SIZE, prog[i]->f))
                  perror(NULL);
               parse_engine_input(prog[i], game, buf);
            }
         }

         /* If either program has performed an illegal move or claim, it has forfeited the game */
         if ((prog[0]->state & PF_FORFEIT) || (prog[1]->state & PF_FORFEIT)) {
            game_is_decided = true;
            break;
         }

         /* If either program resigned, we're likewise done */
         if ((prog[0]->state & PF_RESIGN) || (prog[1]->state & PF_RESIGN)) {
            game_is_decided = true;
            break;
         }

         /* A program claimed the game was over, see whether we agree but abort the game anyway */
         if ((prog[0]->state & PF_CLAIM) || (prog[1]->state & PF_CLAIM)) {
            game_is_decided = true;
            if (logf) {
               bool end = game_ended(game);
               const char *side_string[2] = { "1-0", "0-1" };
               fprintf(logf, "Claim that game ended, verify: %s (%s)\n",
                  end? "true": "false",
                  end?( game_is_draw(game)?"draw": side_string[get_winner(game)]) : "-");
               fprintf(logf, "Claim made in position %s\n", make_fen_string(game, NULL));
            }
            break;
         }

         /* If neither program is thinking... */
         if (!(prog[0]->state & PF_THINK) && !(prog[1]->state & PF_THINK)) {
            /* Check for end-of-game conditions */
            if (game_ended(game)) {
               game_is_decided = true;
               if (logf) {
                  char *reason = "checkmate";
                  const char *side_string[2] = { "1-0", "0-1" };
                  sides winner = WHITE;
                  if (game_is_draw(game)) {
                     if (game->fifty_counter[game->moves_played] >= 100) {
                        reason = "50-move rule";
                     } else if (count_repetition(game)>=1) {
                        reason = "3-fold repetition";
                     } else {
                        reason = "draw";
                     }
                  } else {
                     winner = get_winner(game);
                     if (is_mate(game)) {
                        if (winner == WHITE)
                           reason = "white mates";
                        else
                           reason = "black mates";
                     } else {
                        if (winner == WHITE)
                           reason = "black forfeits";
                        else
                           reason = "white forfeits";
                     }
                  }
                  fprintf(logf, "Game ended (%s, %s)\n",
                        game_is_draw(game)?"draw": side_string[winner], reason);
                  fprintf(logf, "Final position %s\n", make_fen_string(game, NULL));
               }
               break;
            }

            sides stm = game->board.side_to_move;
            int ptm = (stm + white) & 1;

            /* Update program's clock */
            int time = prog[ptm]->clock;
            if (time < min_time_per_move) time = min_time_per_move+9;
            send_to_program(prog[ptm^1], "otim %d\n", prog[ptm^1]->clock/10);
            send_to_program(prog[ptm], "time %d\n", time/10);

            /* Send the last move played in the game (as needed) */
            prog[ptm]->state |= PF_THINK;
            if (game->moves_played) {
               move_t last_move = game->move_list[game->moves_played-1];
               if (is_drop_move(last_move) || is_castle_move(last_move)) {
                  send_to_program(prog[ptm], "%s\n", move_string(last_move, NULL));
               } else {
                  send_to_program(prog[ptm], "%s%s",
                        square_names[get_move_from(last_move)], square_names[get_move_to(last_move)]);
                  if (is_promotion_move(last_move))
                     send_to_program(prog[ptm], "%c",
                        tolower(piece_symbol_string[get_move_promotion_piece(last_move)]));
                  send_to_program(prog[ptm], "\n");
               }
            }

            /* Switch off force mode if needed */
            if (prog[ptm]->state & PF_FORCE) {
               prog[ptm]->state &= ~PF_FORCE;
               send_to_program(prog[ptm], "go\n");
            }

            /* Start the referee's clock */
            start_clock(game);
         }
      }

      char *result_str = "*";
      if (game_is_decided) {
         if ((prog[0]->state & PF_RESIGN)) {
            result_str = "0-1 {White resigns}";
            prog[1]->win++;
         } else if ((prog[1]->state & PF_RESIGN)) {
            result_str = "1-0 {Black resigns}";
            prog[0]->win++;
         } else if ((prog[0]->state & PF_FORFEIT)) {
            result_str = "0-1 {White forfeits due to illegal move or illegal move claim}";
            prog[1]->win++;
         } else if ((prog[1]->state & PF_FORFEIT)) {
            result_str = "1-0 {Black forfeits due to illegal move or illegal move claim}";
            prog[0]->win++;
         } else if (game_is_draw(game)) {
            result_str = "1/2-1/2";
            prog[0]->draw++;
            prog[1]->draw++;
         } else {
            sides win = get_winner(game);
            int ptw = (win + white) & 1;
            if (win == WHITE)
               result_str = "1-0 {White mates}";
            else
               result_str = "0-1 {Black mates}";

            prog[ptw]->win++;
         }
      }

      printf("Game result: %s\n", result_str);

      /* Write the game to a .pgn file */
      if (pgnf) {
         char *short_result_str = strdup(result_str);
         char *s = strstr(short_result_str, " ");
         if (s) *s = '\0';
         fprintf(pgnf,
                "[Event \"Computer Match\"]\n"
                "[Site \"%s\"]\n"
                "[Date \"\"]\n"
                "[Round \"%d\"]\n"
                "[White \"%s\"]\n"
                "[Black \"%s\"]\n"
                "[Result \"%s\"]\n"
                "[TimeControl \"%s\"]\n"
                "[Variant \"%s\"]\n"
                "[FEN \"%s\"]\n",
                host, n+1,
                prog[white]->name, prog[black]->name, short_result_str, tc_string, variant_name, game->start_fen);

         move_t moves[game->moves_played];
         int n, nm;
         nm = game->moves_played;
         for (n=0; n<nm; n++)
            moves[n] = game->move_list[n];
         while (game->moves_played)
            takeback(game);

         fprintf(pgnf, "\n");
         int l = 0;
         for (n=0; n<nm; n++) {
            char s[128];
            int k = 0;

            if ((n&1) == 0) {
               snprintf(s, sizeof s, "%d. ", n / 2 + 1);
               k = strlen(s);
            }

            snprintf(s+k, sizeof s - k, "%s ", short_move_string(game, moves[n], NULL));
            if ( (l + strlen(s)) > 80) {
               fprintf(pgnf, "\n");
               l = 0;
            }
            l += strlen(s);
            fprintf(pgnf, "%s", s);
            playmove(game, moves[n]);
         }
         fprintf(pgnf, "\n%s\n\n", short_result_str);
         fflush(pgnf);
         free(short_result_str);
      }

      end_game(game);
   }

   printf("Match result: + %d - %d = %d (%.1f-%.1f)\n", prog[0]->win, prog[1]->win, prog[0]->draw,
      1.0*prog[0]->win+0.5*prog[0]->draw, 1.0*prog[1]->win+0.5*prog[1]->draw);

   if (pgnf)
      fclose(pgnf);

   /* Shutdown. Avoid waiting indefinitely by setting an alarm. A timeout of 10s should be plenty. */
   alarm(10);

   /* Flush buffers */
   for (int i=0; i<2; i++) {
      while(p2_input_waiting(prog[i]->f)) {
         p2gets(buf, BUF_SIZE, prog[i]->f);
      }
   }

   /* Shut down children */
   send_to_program(prog[0], "quit\n");
   msleep(10);
   send_to_program(prog[1], "quit\n");
   msleep(10);
   wait_input(prog[0]->f->in_fd, prog[1]->f->in_fd, 50000);
   if (check_children) {
      check_children = false;
      for (int i=0; i<2; i++) {
         if(!(prog[i]->state & PF_DEAD) && child_is_alive(prog[i])) {
            kill(prog[i]->f->pid, SIGTERM);
            if (logf) fprintf(logf, "Send terminate signal to %s (%d)\n", prog[i]->name, prog[i]->id);
         } else {
            prog[i]->state |= PF_DEAD;
            if (logf) fprintf(logf, "%s (%d) exited\n", prog[i]->name, prog[i]->id);
         }
      }
   }
   for (int i=0; i<2; i++) {
      if(!(prog[i]->state & PF_DEAD)) {
         msleep(10);
         if (child_is_alive(prog[i])) {
            if (logf) fprintf(logf, "Send kill signal to %s (%d)\n", prog[i]->name, prog[i]->id);
            kill(prog[i]->f->pid, SIGKILL);
            while (child_is_alive(prog[i]));
         }
      }
   }

   if (logf)
      fclose(logf);
   return 0;
}
